(**
   Implements horizontal and vertical sliders.

  TODO
  * No highlighting of arrows, should change
**)

MODULE VOSlider;

(*
    Implements a scroll gadget.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT D  := VODisplay,
       E  := VOEvent,
       F  := VOFrame,
       G  := VOGUIObject,
       O  := VOObject,
       S  := VOScale,
       U  := VOUtil,
       V  := VOVecImage,
       VM := VOValue;

CONST
  movedMsg     * = 0;
  moveStartMsg * = 1; (* ActionMsg *)
  moveEndMsg   * = 2; (* ActionMsg *)

TYPE
  Prefs*     = POINTER TO PrefsDesc;

  (**
    In this class all preferences stuff of the button is stored.
  **)

  PrefsDesc* = RECORD (G.PrefsDesc)
                 hKnob*,
                 vKnob*      : LONGINT;
                 frame*      : LONGINT;
               END;


  Slider*     = POINTER TO SliderDesc;
  SliderDesc* = RECORD (G.GadgetDesc)
                  prefs       : Prefs;
                  knob        : V.VecImage;
                  frame       : F.Frame;
                  scale       : S.Scale;
                  from,to     : LONGINT;
                  offset,corr : LONGINT;
                  pos         : VM.ValueModel;
                  vert,
                  selected    : BOOLEAN;
                  useScale    : BOOLEAN;
                END;

  (* messages *)

  MovedMsg*     = POINTER TO MovedMsgDesc;

  (**
    The PressedMsg generated everytime the button get clicked.
  **)

  MovedMsgDesc* = RECORD (O.MessageDesc)
                    model* : VM.ValueModel;
                  END;

VAR
  prefs* : Prefs;
  action : O.Action;

  PROCEDURE (p : Prefs) Init*;

  BEGIN
    p.Init^;

    p.vKnob:=V.vSlider;
    p.hKnob:=V.hSlider;
    p.frame:=F.double3DIn;
  END Init;

  PROCEDURE (p : Prefs) SetPrefs(s : Slider);

  BEGIN
    s.prefs:=p;   (* We set the prefs *)

    IF p.background#NIL THEN
      s.SetBackgroundObject(p.background.Copy());
      s.backgroundObject.source:=s;
    END;
  END SetPrefs;

  PROCEDURE (s : Slider) Init*;

  BEGIN
    s.Init^;

    prefs.SetPrefs(s);

    INCL(s.flags,G.canFocus);

    s.vert:=TRUE;
    s.useScale:=TRUE;

    s.pos:=NIL;
    s.scale:=NIL;

    s.selected:=FALSE;

    s.from:=0;
    s.to:=100;
  END Init;

  (**
    Set the direction (horizontal or vertical) of the slider.
  **)

  PROCEDURE (s : Slider) Set*(vert : BOOLEAN);

  BEGIN
    s.vert:=vert;
  END Set;

  (**
    Define, if we should use a scale, or not..
  **)

  PROCEDURE (s : Slider) UseScale*(use : BOOLEAN);

  BEGIN
    s.useScale:=use;
  END UseScale;

  (**
    Set the minimal and maximal value of the slider.
  **)

  PROCEDURE (s : Slider) SetRange*(from,to : LONGINT);

  BEGIN
    IF (s.from#from) OR (s.to#to) THEN
      s.from:=from;
      s.to:=to;
      IF s.visible THEN
        s.Redraw;
      END;
    END;
  END SetRange;

  (**
    Set the integer model to the slider. The slider will
    always represent the value of the model.
  **)

  PROCEDURE (s : Slider) SetModel*(pos : VM.ValueModel);

  BEGIN
    IF s.pos#NIL THEN
      s.UnattachModel(s.pos);
    END;
    s.pos:=pos;
    IF s.pos#NIL THEN
      s.AttachModel(pos);
    END;
  END SetModel;

  PROCEDURE (s : Slider) CalcSize*(display : D.Display);

  BEGIN
    NEW(s.knob);
    s.knob.Init;
    s.knob.SetFlags({G.horizontalFlex,G.verticalFlex});

    IF s.vert THEN
      s.knob.Set(s.prefs.vKnob);
    ELSE
      s.knob.Set(s.prefs.hKnob);
    END;

    IF ~s.knob.StdFocus() & s.MayFocus() THEN
      EXCL(s.flags,G.stdFocus);
      INCL(s.knob.flags,G.mayFocus);
    END;

    s.knob.CalcSize(display);

    IF s.vert THEN
      s.width:=s.knob.oWidth;
      s.height:=3*s.width;
      s.offset:=s.knob.oHeight DIV 2;
    ELSE
      s.height:=s.knob.oHeight;
      s.width:=3*s.height;
      s.offset:=s.knob.oWidth DIV 2;
    END;

    IF s.useScale THEN
      NEW(s.scale);
      s.scale.Init;
      s.CopyBackground(s.scale);
      s.scale.SetDirection(s.vert);
      s.scale.SetInterval(s.from,s.to);
      s.scale.SetFlags({G.horizontalFlex,G.verticalFlex});
      s.scale.CalcSize(display);

      IF s.vert THEN
        INC(s.width,s.scale.width);
        s.height:=G.MaxLong(s.height,s.scale.height);
        INC(s.width,display.spaceWidth);
      ELSE
        INC(s.height,s.scale.height);
        s.width:=G.MaxLong(s.width,s.scale.width);
        INC(s.height,display.spaceHeight);
      END;
    END;

    NEW(s.frame);
    s.frame.Init;
    s.frame.SetFlags({G.horizontalFlex,G.verticalFlex});
    s.frame.SetInternalFrame(s.prefs.frame);
    s.frame.CalcSize(display);

    INC(s.width,s.frame.leftBorder+s.frame.rightBorder);
    INC(s.height,s.frame.topBorder+s.frame.bottomBorder);

    IF s.vert THEN
      INC(s.offset,s.frame.topBorder);
    ELSE
      INC(s.offset,s.frame.leftBorder);
    END;

    s.minWidth:=s.width;
    s.minHeight:=s.height;

    s.CalcSize^(display);
  END CalcSize;

  PROCEDURE (s : Slider) DrawKnob;

  VAR
    kStart,
    bSize,
    offset : LONGINT;

  BEGIN
    IF (s.pos=NIL) OR s.pos.IsNull() THEN
      s.knob.Hide;
      RETURN;
    END;

    IF s.knob.visible THEN
      s.knob.Hide;
    END;

    IF s.selected THEN
      s.draw.mode:={D.selected};
    END;

    IF s.vert THEN

      IF s.useScale THEN
        offset:=s.scale.width+s.display.spaceWidth;
      ELSE
        offset:=0;
      END;

      INC(offset,s.frame.leftBorder);

      bSize:=s.height-2*s.offset-1;
      IF s.to=s.from THEN
        kStart:=s.frame.topBorder;
      ELSE
        kStart:=bSize-U.RoundDiv((s.pos.GetLongint()-s.from)*bSize,s.to-s.from)+s.frame.topBorder;
      END;

      s.knob.Draw(s.x+offset,s.y+kStart,s.draw);
    ELSE

      IF s.useScale THEN
        offset:=s.scale.height+s.display.spaceHeight;
      ELSE
        offset:=0;
      END;

      INC(offset,s.frame.topBorder);

      bSize:=s.width-2*s.offset-1;
      IF s.to=s.from THEN
        kStart:=s.frame.leftBorder;
      ELSE
        kStart:=U.RoundDiv((s.pos.GetLongint()-s.from)*bSize,s.to-s.from)+s.frame.leftBorder;
      END;

      s.knob.Draw(s.x+kStart,s.y+offset,s.draw);
    END;
    s.draw.mode:={};
  END DrawKnob;

  PROCEDURE (s : Slider) GetFocus*(event : E.Event):G.Object;

  BEGIN
    IF ~s.visible OR s.disabled THEN
      RETURN NIL;
    END;

    IF (s.pos=NIL) OR s.pos.IsNull() THEN
      RETURN NIL;
    END;

    WITH event : E.MouseEvent DO
      IF (event.type=E.mouseDown) & s.PointIsIn(event.x,event.y)
      & (event.qualifier={}) & (event.button=E.button1) THEN
        IF s.knob.PointIsIn(event.x,event.y) THEN

          IF s.vert THEN
            s.corr:=event.y-s.knob.y;
          ELSE
            s.corr:=event.x-s.knob.x;
          END;
          s.selected:=TRUE;
          s.DrawKnob;

          action.action:=moveStartMsg;
          s.Send(action,O.actionMsg);

          RETURN s;
        ELSE
          IF s.vert THEN
            IF event.y<s.knob.y THEN
              IF s.pos.GetLongint()>s.from THEN
                s.pos.Inc;
              END;
              RETURN s;
            ELSIF event.y>s.knob.y+s.knob.oHeight THEN
              IF s.pos.GetLongint()<s.to THEN
                s.pos.Dec;
              END;
              RETURN s;
            END;
          ELSE
            IF event.x<s.knob.x THEN
              IF s.pos.GetLongint()>s.from THEN
                s.pos.Dec;
              END;
              RETURN s;
            ELSIF event.x>s.knob.x+s.knob.oWidth THEN
              IF s.pos.GetLongint()<s.to THEN
                s.pos.Inc;
              END;
              RETURN s;
            END;
          END;
        END;
      END;
    ELSE
    END;
    RETURN NIL;
  END GetFocus;

  PROCEDURE (s : Slider) HandleEvent*(event : E.Event):BOOLEAN;

  VAR
    new   : LONGINT;
    moved : MovedMsg;

  BEGIN
    WITH event : E.MouseEvent DO
      IF (event.type=E.mouseUp) & (event.button=E.button1) THEN
        s.selected:=FALSE;
        s.DrawKnob;
        NEW(moved);
        moved.model:=s.pos;
        s.Send(moved,movedMsg);

        action.action:=moveEndMsg;
        s.Send(action,O.actionMsg);

        RETURN TRUE;
      END;
    | event : E.MotionEvent DO
      IF s.selected THEN
        IF s.vert THEN
          new:=s.to-(event.y-s.y-s.corr)*(s.to-s.from+1) DIV (s.height-2*s.offset)+s.from;
        ELSE
          new:=(event.x-s.x-s.corr)*(s.to-s.from+1) DIV (s.width-2*s.offset)+s.from;
        END;
        IF (new>=s.from) & (new<=s.to) THEN
          s.pos.SetLongint(new);
        ELSIF new<s.from THEN
          s.pos.SetLongint(s.from);
        ELSIF new>s.to THEN
          s.pos.SetLongint(s.to);
        END;
      ELSE
      END;
    ELSE
    END;

    RETURN FALSE;
  END HandleEvent;

  PROCEDURE (s : Slider) HandleFocusEvent*(event : E.KeyEvent):BOOLEAN;

  VAR
    keysym,
    old,
    tmp    : LONGINT;
    moved  : MovedMsg;

  BEGIN
    IF event.type=E.keyDown THEN
      keysym:=event.GetKey();
      old:=s.pos.GetLongint();
      IF s.vert & (keysym=E.up) THEN
        IF old>s.from THEN
          s.pos.Dec;
        END;
      ELSIF s.vert & (keysym=E.down) THEN
        IF old<s.to THEN
          s.pos.Inc;
        END;
      ELSIF ~s.vert & (keysym=E.left) THEN
        IF old>s.from THEN
          s.pos.Dec;
        END;
      ELSIF ~s.vert & (keysym=E.right) THEN
        IF old<s.to THEN
          s.pos.Inc;
        END;
      ELSIF keysym=E.home THEN
        s.pos.SetLongint(s.from);
      ELSIF keysym=E.end THEN
        s.pos.SetLongint(s.to);
      ELSIF keysym=E.pageUp THEN
        tmp:=old;
        DEC(tmp,(s.to-s.from+1) DIV 10);
        IF tmp<s.from THEN
          tmp:=s.from;
        END;
        s.pos.SetLongint(tmp);
      ELSIF keysym=E.pageDown THEN
        tmp:=old;
        INC(tmp,(s.to-s.from+1) DIV 10);
        IF tmp>s.to THEN
          tmp:=s.to;
        END;
        s.pos.SetLongint(tmp);
      ELSE
        RETURN FALSE;
      END;

      (* We have changed the value of the model *)
      IF old#s.pos.GetLongint() THEN
        NEW(moved);
        moved.model:=s.pos;
        s.Send(moved,movedMsg);
      END;
      RETURN TRUE;
    END;
    RETURN FALSE;
  END HandleFocusEvent;

  PROCEDURE (s : Slider) Draw*(x,y : LONGINT; draw : D.DrawInfo);

  BEGIN
    s.Draw^(x,y,draw);

    s.DrawBackground(s.x,s.y,s.width,s.height);

    IF s.useScale THEN
      IF s.vert THEN
        s.scale.Resize(-1,s.height-2*s.offset);
        s.scale.Draw(s.x,s.y+s.offset,draw);

        s.frame.Resize(s.knob.oWidth+s.frame.leftBorder+s.frame.rightBorder,s.height);
        s.frame.Draw(s.x+s.scale.oWidth+s.display.spaceWidth,s.y,draw);
      ELSE
        s.scale.Resize(s.width-2*s.offset,-1);
        s.scale.Draw(s.x+s.offset,s.y,draw);

        s.frame.Resize(s.width,s.knob.oHeight+s.frame.topBorder+s.frame.bottomBorder);
        s.frame.Draw(s.x,s.y+s.scale.oHeight+s.display.spaceHeight,draw);
      END;
    ELSE
      IF s.vert THEN
        s.frame.Resize(s.knob.oWidth+s.frame.leftBorder+s.frame.rightBorder,s.height);
        s.frame.Draw(s.x,s.y,draw);
      ELSE
        s.frame.Resize(s.width,s.knob.oHeight+s.frame.topBorder+s.frame.bottomBorder);
        s.frame.Draw(s.x,s.y,draw);
      END;
    END;

    s.draw.FillBackground(s.frame.x+s.frame.leftBorder,s.frame.y+s.frame.topBorder,
                          s.frame.oWidth-s.frame.leftBorder-s.frame.rightBorder,
                          s.frame.oHeight-s.frame.topBorder-s.frame.bottomBorder);

    s.DrawKnob;

    IF s.disabled THEN
      s.DrawDisabled;
    END;
  END Draw;

  PROCEDURE (s : Slider) Hide*;

  BEGIN
    IF s.visible THEN
      s.knob.Hide;
      s.DrawHide;
      s.Hide^;
    END;
  END Hide;

  PROCEDURE (s : Slider) DrawFocus*;

  BEGIN
    IF ~s.knob.StdFocus() THEN
      s.knob.DrawFocus;
      IF s.selected THEN (* little hack, to get the knob selected when it display the focus *)
        s.DrawKnob;
      END;
    ELSE
      (* Delegate drawing to the baseclass *)
      s.DrawFocus^;
    END;
  END DrawFocus;

  PROCEDURE (s : Slider) HideFocus*;

  BEGIN
    IF ~s.knob.StdFocus() THEN
      s.knob.HideFocus;
    ELSE
      (* Delegate drawing to the baseclass *)
      s.HideFocus^;
    END;
  END HideFocus;

  PROCEDURE (s : Slider) Resync*(model : O.Model; msg : O.ResyncMsg);

  BEGIN
    IF s.visible & ~s.disabled THEN
      s.DrawKnob;
    END;
  END Resync;

BEGIN
  NEW(prefs);
  prefs.Init;

  NEW(action);
END VOSlider.
