(**
   A nice digital clock, to be placed everywhere you like.
**)

MODULE VOStringClock;

(*
    A digital clock.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT D  := VODisplay,
       E  := VOEvent,
       G  := VOGUIObject,
       O  := VOObject,

       C  := Calendar,
       SC := SysClock;

CONST
  callPeriod = 1;

  font       = D.fixedFont;

TYPE
  StringClock*     = POINTER TO StringClockDesc;
  StringClockDesc* = RECORD (G.GadgetDesc)
                       timeOut    : D.TimeOut;
                       foreground : D.Color;
                       font       : D.Font;
                       textWidth  : LONGINT;
                       selected   : BOOLEAN;
                     END;

  PROCEDURE (d : StringClock) Init*;

  BEGIN
    d.Init^;

    d.foreground:=D.textColor;
    d.selected:=FALSE;
    d.timeOut:=NIL;
    d.font:=NIL;
  END Init;

  PROCEDURE (d : StringClock) SetForeground*(color : D.Color);

  BEGIN
    d.foreground:=color;
  END SetForeground;

  PROCEDURE (d : StringClock) CalcSize*(display : D.Display);

  VAR
    string : ARRAY 2 OF CHAR;

  BEGIN
    d.font:=display.GetFont(font);
    string:="X";
    d.textWidth:=8*d.font.TextWidth(string,1,{});

    d.width:=d.textWidth;
    d.height:=d.font.height;

    d.minWidth:=d.width;
    d.minHeight:=d.height;

    d.CalcSize^(display);
  END CalcSize;

  (**
    This method gets called when the window gets an event and looks for
    someone that processes it.

    If GetFocus return an object, that objets HandleEvent-method
    get called untill it gives away the focus.
  **)

  PROCEDURE (d : StringClock) GetFocus*(event : E.Event):G.Object;

  BEGIN
    (* It makes no sense to get the focus if we are currently not visible *)
    IF ~d.visible OR d.disabled THEN
      RETURN NIL;
    END;

    (*
      When the left mousebutton gets pressed without any qualifier
      in the bounds of our button...
   *)

    WITH event : E.MouseEvent DO
      IF (event.type=E.mouseDown) & d.PointIsIn(event.x,event.y) & (event.button=E.button1) THEN
        (* We change our state to pressed and redisplay ourself *)
        d.selected:=TRUE;
        d.Redraw;

        (*
          Since we want the focus for waiting for buttonup we return
          a pointer to ourself.
        *)
        RETURN d;
      END;
    ELSE
    END;
    RETURN NIL;
  END GetFocus;

  (**
    Handles all events after GetFocus catched the focus for us.
  **)

  PROCEDURE (d : StringClock) HandleEvent*(event : E.Event):BOOLEAN;

  BEGIN
    (*
      If the user releases the left mousebutton...
    *)
    WITH event : E.MouseEvent DO
      IF (event.type=E.mouseUp) & (event.button=E.button1) THEN
        (* We get unselected again and must redisplay ourself *)
        d.selected:=FALSE;
        d.Redraw;
        RETURN TRUE;
      END;
    ELSE
    END;

    RETURN FALSE; (* No message we are interested in, get more msgs *)
  END HandleEvent;

  PROCEDURE (d : StringClock) Draw*(x,y : LONGINT; draw : D.DrawInfo);

  VAR
    sTime : SC.DateTime;
    text  : ARRAY 9 OF CHAR;

  BEGIN
    d.Draw^(x,y,draw);

    d.DrawBackground(d.x,d.y,d.width,d.height);

    SC.GetClock(sTime);

    IF d.selected THEN
      C.TimeToStr(sTime,"%d.%m.%y",text);
    ELSE
      C.TimeToStr(sTime,"%H:%M:%S",text);
    END;
    draw.PushFont(font,{});
    draw.PushForeground(d.foreground);
    draw.DrawString(d.x+(d.width-d.textWidth) DIV 2,
                    d.y+d.font.ascent+(d.height-d.font.height) DIV 2,text,8);
    draw.PopForeground;
    draw.PopFont;

    IF d.timeOut=NIL THEN
      d.timeOut:=d.display.AddTimeOut(callPeriod,0,d);
    END;
  END Draw;

  PROCEDURE (d : StringClock) Hide*;

  BEGIN
    IF d.timeOut#NIL THEN
      d.display.RemoveTimeOut(d.timeOut);
      d.timeOut:=NIL;
    END;
    IF d.visible THEN
      d.DrawHide;
      d.Hide^;
    END;
  END Hide;

  PROCEDURE (d : StringClock) Receive*(msg : O.Message);

  BEGIN
    WITH msg:
      D.TimeOutMsg DO
        IF d.visible THEN
          d.timeOut:=NIL;
          d.Redraw;
        END;
    ELSE
    END;
  END Receive;

END VOStringClock.