(**
   Implements windows.

  TODO
  * When unmaped, HandleEvents calls top.Hide, which then tries to draw in an
    allready unmaped window. Solution: object.Hide has to check window.maped.

  * Cleanup window-generation code, divide it into more methods

  * Redesign eventloop handling, on method for each event, more functionality
    in baseclase. Maybe some abstraction mechanism for certain events.

  * Support future Object.Refresh
**)

MODULE VOWindow;

(*
    Windows.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT D   := VODisplay,
       E   := VOEvent,
       G   := VOGUIObject,
       K   := VOKeyHandler,
       O   := VOObject;

CONST
  openMsg*   = 0;
  closeMsg*  = 1;

TYPE
  Window*       = POINTER TO WindowDesc;
  WindowDesc*   = RECORD (D.WindowDesc)
                    top         : G.Object;
                    current     : G.Object;
                    keyHandler- : K.KeyHandler;

                  END;

  OpenMsg*        = POINTER TO OpenMsgDesc;
  OpenMsgDesc*    = RECORD (O.MessageDesc)
                    END;

  CloseMsg*       = POINTER TO CloseMsgDesc;
  CloseMsgDesc*   = RECORD (O.MessageDesc)
                    END;

  DeleteMsg*      = POINTER TO DeleteMsgDesc;
  DeleteMsgDesc*  = RECORD (O.MessageDesc)
                   END;

  (* Some messagehandler *)

  Msg2Open*       = POINTER TO Msg2OpenDesc;
  Msg2OpenDesc*   = RECORD (O.HandlerDesc)
                    END;

  Msg2Close*      = POINTER TO Msg2CloseDesc;
  Msg2CloseDesc*  = RECORD (O.HandlerDesc)
                    END;

  Msg2Delete*     = POINTER TO Msg2DeleteDesc;
  Msg2DeleteDesc* = RECORD (O.HandlerDesc)
                    END;


  (**
    Initialize the window object.
  **)

  PROCEDURE (w : Window) Init*;

  BEGIN
    w.Init^;

    w.top:=NIL;
    w.current:=NIL;
    w.keyHandler:=NIL;
  END Init;

  (**
    Set the top object for this window.
  **)

  PROCEDURE(w : Window) SetTop*(top : G.Object);

  BEGIN
    w.top:=top;
  END SetTop;

  (**
     Creates a window on the display with top as top object
     and with title as title.
  **)

  PROCEDURE (w : Window) PreInit*;

  BEGIN
    w.top.CalcSize(w.display);

    w.SetSize(w.top.oWidth,w.top.oHeight);
    IF G.horizontalFlex IN w.top.flags THEN
      w.SetMinSize(w.top.oMinWidth,-1);
      w.SetMaxSize(w.top.oMaxWidth,-1);
    ELSE
      w.SetMinSize(w.width,-1);
      w.SetMaxSize(w.width,-1);
    END;
    IF G.verticalFlex IN w.top.flags THEN
      w.SetMinSize(-1,w.top.oMinHeight);
      w.SetMaxSize(-1,w.top.oMaxHeight);
    ELSE
      w.SetMinSize(-1,w.height);
      w.SetMaxSize(-1,w.height);
    END;

    w.PreInit^;
  END PreInit;

  PROCEDURE (w : Window) ReinitWindow*;

  BEGIN
    IF w.maped THEN
      w.top.Hide;
    END;

    w.top.CalcSize(w.display);

    w.Resize(w.top.oWidth,w.top.oHeight);

    (* TODO: Calc new bounds *)

    w.top.Resize(w.width,w.height);

    IF w.maped THEN
      w.top.Redraw;
      IF w.keyHandler#NIL THEN
        w.keyHandler.Refocus;
      END;
    ELSE
      w.top.Resize(w.width,w.height);
    END;
  END ReinitWindow;

  PROCEDURE (w : Window) AddKeyHandler*(handler : K.KeyHandler);

  BEGIN
    w.keyHandler:=handler;
  END AddKeyHandler;

  (* -- handler for some common windowevents -- *)

  PROCEDURE (w : Window) ClosePressed*;

  VAR
    close : CloseMsg;

  BEGIN
    NEW(close);
    w.Send(close,closeMsg);
  END ClosePressed;

  PROCEDURE (w : Window) Unmaped*;

  BEGIN
    w.Unmaped^;
    w.top.Hide;
  END Unmaped;

  PROCEDURE (w : Window) FocusIn*;

  BEGIN
    w.FocusIn^;
    IF w.keyHandler#NIL THEN
      w.keyHandler.Activate;
    END;
  END FocusIn;

  PROCEDURE (w : Window) FocusOut*;

  BEGIN
    w.FocusOut^;
    IF w.keyHandler#NIL THEN
      w.keyHandler.Deactivate;
    END;
  END FocusOut;

  (**
    Draw the window contents.
  **)

  PROCEDURE (w : Window) Draw*;

  BEGIN
    w.top.Draw(0,0,w.draw);
  END Draw;

  (**
    Redraw the given area of the window.
  **)

  PROCEDURE (w : Window) Redraw*(x,y,width,height : LONGINT);

  BEGIN
    w.top.Refresh(x,y,width,height);
  END Redraw;

  (**
    Hide the window.
  **)

  PROCEDURE (w : Window) Hide*;

  BEGIN
    w.top.Hide;
  END Hide;

  (**
    Tell the window that it has been resized. The window resizes in turn
    its top object and redraws itself.
  **)

  PROCEDURE (w : Window) Resized*(width,height : LONGINT);

  BEGIN
    w.top.Resize(width,height);
    w.top.Redraw;
    IF w.keyHandler#NIL THEN
      IF w.current#NIL THEN
        w.keyHandler.SetCurrentObject(w.current(G.Gadget));
      ELSE
        w.keyHandler.Refocus;
      END;
    END;
  END Resized;

  (**
    Returns the object under the cursor to supports the given mode.
  **)

  PROCEDURE (w : Window) GetPosObject(mode : LONGINT):G.Object;

  VAR
    rx,ry,
    wx,wy : LONGINT;

  BEGIN
    w.GetMousePos(rx,ry,wx,wy);
    RETURN w.top.GetPosObject(wx,wy,mode);
  END GetPosObject;

  (**
    Returns the object that coveres the given point and that supports
    dragging of data.

    If drag is TRUE, when want to find a object that we can drag data from,
    else we want an object to drop data on.
  **)

  PROCEDURE (w : Window) GetDnDObject*(x,y : LONGINT; drag : BOOLEAN):G.Object;

  BEGIN
    RETURN w.top.GetDnDObject(x,y,drag);
  END GetDnDObject;

  (**
    Show the context help of an object under the cursor.
  **)

  PROCEDURE (w : Window) ContextHelp*;

  VAR
    object : G.Object;
    help   : D.Window;

  BEGIN
    IF w.current=NIL THEN
      object:=w.GetPosObject(G.helpGadget);
      IF object#NIL THEN
        help:=object.GetHelpObject();
        IF help#NIL  THEN
          help.Open;
        END;
      END;
    END;
  END ContextHelp;

  (**
    Open the context menu for an object under the cursor.
  **)

  PROCEDURE (w : Window) ContextMenu*():BOOLEAN;

  VAR
    object : G.Object;
    menu   : D.Window;

  BEGIN
    object:=w.GetPosObject(G.menuGadget);
    IF object#NIL THEN
      menu:=object.GetMenuObject();
      IF menu#NIL THEN
        menu.Open;
        RETURN TRUE;
      END;
    END;
    RETURN FALSE;
  END ContextMenu;

  (**
    Move the keyboard focus to the next valid object.
  **)

  PROCEDURE (w : Window) FocusNext*;

  BEGIN
    IF w.keyHandler#NIL THEN
      w.keyHandler.NextObject;
    END;
  END FocusNext;

  (**
    Since the object with the keyboard focus hides, we find another one.
  **)

  PROCEDURE (w : Window) FocusObjectHides*;

  BEGIN
    ASSERT(w.keyHandler#NIL);
    w.keyHandler.FindNewObject;
  END FocusObjectHides;

  (**
    The handler, that handles all events not dealed with in the special
    handler-methods
  **)

  PROCEDURE (w : Window) HandleEvent*(event : E.Event):BOOLEAN;

  BEGIN
    IF w.HandleEvent^(event) THEN
      RETURN TRUE;
    END;

    IF (event IS E.KeyEvent) & (w.keyHandler#NIL) THEN
      IF w.keyHandler.HandleEvent(event(E.KeyEvent),TRUE) THEN
        RETURN TRUE;
      END;
    END;

    IF w.current#NIL THEN
      IF w.current.HandleEvent(event) THEN
        w.current:=NIL;
      END;
    ELSE
      w.current:=w.top.GetFocus(event);
      IF w.current#NIL THEN
        IF w.keyHandler#NIL THEN
          w.keyHandler.SetCurrentObject(w.current(G.Gadget));
        END;
      ELSE
        IF (event IS E.KeyEvent) & (w.keyHandler#NIL) THEN
          IF w.keyHandler.HandleEvent(event(E.KeyEvent),FALSE) THEN END;
        END;
      END;
    END;
    RETURN TRUE;
  END HandleEvent;

  (**
    The Handler for messages by user and other objects
  **)

  PROCEDURE (w : Window) Receive*(message : O.Message);

  BEGIN
    WITH
      message: OpenMsg DO
        w.Open;
    | message: CloseMsg DO
        w.Close;
    | message: DeleteMsg DO
        w.Delete;
    ELSE
    END;
  END Receive;

  (*
    some predefined message-converter
  *)

  PROCEDURE (h : Msg2Open) Convert*(message : O.Message):O.Message;

  VAR
    new : OpenMsg;

  BEGIN
    NEW(new);
    RETURN new;
  END Convert;

  PROCEDURE (h : Msg2Close) Convert*(message : O.Message):O.Message;

  VAR
    new : CloseMsg;

  BEGIN
    NEW(new);
    RETURN new;
  END Convert;

  PROCEDURE (h : Msg2Delete) Convert*(message : O.Message):O.Message;

  VAR
    new : DeleteMsg;

  BEGIN
    NEW(new);
    RETURN new;
  END Convert;

END VOWindow.
