(**
  A layout group, that layout menues and the top group object within a window.
**)

MODULE VOWindowGroup;

(*
    A layout group, that layouts a number of buttons horizontaly.
    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT D := VODisplay,
       E := VOEvent,
       G := VOGUIObject;

TYPE
  Prefs*     = POINTER TO PrefsDesc;

  (**
    In this class all preferences stuff of the button is stored.
  **)

  PrefsDesc* = RECORD (G.PrefsDesc)
               END;

  WindowGroup*     = POINTER TO WindowGroupDesc;
  WindowGroupDesc* = RECORD(G.GroupDesc)
                     prefs       : Prefs;
                     men,top,
                     tool,status : G.Object;
                     space       : BOOLEAN;
                   END;


VAR
  prefs* : Prefs;

  PROCEDURE (p : Prefs) SetPrefs(w : WindowGroup);

  BEGIN
    w.prefs:=p;   (* We set the prefs *)

    IF p.background#NIL THEN
      w.SetBackgroundObject(p.background.Copy());
      w.backgroundObject.source:=w;
    END;
  END SetPrefs;

  (**
    WindowGroup inherits horizontalFlex & verticalFlex from top.
  **)

  PROCEDURE (w : WindowGroup) Init*;

  BEGIN
    w.Init^;

    prefs.SetPrefs(w);

    w.men:=NIL;
    w.top:=NIL;
    w.tool:=NIL;
    w.status:=NIL;
    w.space:=TRUE;
  END Init;

  PROCEDURE (w : WindowGroup) Set*(menu, top : G.Object; space : BOOLEAN);

  BEGIN
    w.men:=menu;
    IF w.men#NIL THEN
      w.men.parent:=w;
    END;

    w.top:=top;
    IF w.top#NIL THEN
      w.top.parent:=w;
    END;
  END Set;

  PROCEDURE (w : WindowGroup) SetToolbar*(toolbar : G.Object);

  BEGIN
    w.tool:=toolbar;
    IF w.tool#NIL THEN
      w.tool.parent:=w;
    END;
  END SetToolbar;

  PROCEDURE (w : WindowGroup) SetStatusLine*(status : G.Object);

  BEGIN
    w.status:=status;
    IF w.status#NIL THEN
      w.status.parent:=w;
    END;
  END SetStatusLine;

  PROCEDURE (w : WindowGroup) CalcSize*(display : D.Display);

  BEGIN
    w.width:=0;
    w.height:=0;
    w.minWidth:=0;
    w.minHeight:=0;

    IF w.men#NIL THEN
      w.men.CalcSize(display);
      w.width:=w.men.oWidth;
      w.height:=w.men.oHeight;
      w.minWidth:=w.men.oMinWidth;
      w.minHeight:=w.men.oMinHeight;
    END;

    IF w.tool#NIL THEN
      w.tool.CalcSize(display);
      INC(w.height,w.tool.oHeight);
      INC(w.minHeight,w.tool.oMinHeight);
      w.width:=G.MaxLong(w.width,w.tool.oWidth);
      w.minWidth:=G.MaxLong(w.minWidth,w.tool.oMinWidth);
    END;

    IF w.status#NIL THEN
      w.status.CalcSize(display);
      INC(w.height,w.status.oHeight);
      INC(w.minHeight,w.status.oMinHeight);
      w.width:=G.MaxLong(w.width,w.status.oWidth);
      w.minWidth:=G.MaxLong(w.minWidth,w.status.oMinWidth);
    END;

    IF w.top#NIL THEN
      w.top.CalcSize(display);
      INC(w.height,w.top.oHeight);
      INC(w.minHeight,w.top.oMinHeight);

      IF w.space THEN
        INC(w.height,2*display.spaceHeight);
        INC(w.minHeight,2*display.spaceHeight);
        w.width:=G.MaxLong(w.width,w.top.oWidth+2*display.spaceWidth);
        w.minWidth:=G.MaxLong(w.minWidth,w.top.oMinWidth+2*display.spaceWidth);
      ELSE
        w.width:=G.MaxLong(w.width,w.top.oWidth);
        w.minWidth:=G.MaxLong(w.minWidth,w.top.oMinWidth);
      END;

      IF (G.horizontalFlex IN w.top.flags) THEN
        w.SetFlags({G.horizontalFlex});
      END;
      IF G.verticalFlex IN w.top.flags THEN
        w.SetFlags({G.verticalFlex});
      END;
    END;
    w.CalcSize^(display);
  END CalcSize;


  PROCEDURE (w : WindowGroup) GetFocus*(event : E.Event):G.Object;

  VAR
    help : G.Object;

  BEGIN
    IF w.men#NIL THEN
      help:=w.men.GetFocus(event);
      IF help#NIL THEN
        RETURN help;
      END;
    END;
    IF w.tool#NIL THEN
      help:=w.tool.GetFocus(event);
      IF help#NIL THEN
        RETURN help;
      END;
    END;
    IF w.top#NIL THEN
      help:=w.top.GetFocus(event);
      IF help#NIL THEN
        RETURN help;
      END;
    END;
    IF w.status#NIL THEN
      help:=w.status.GetFocus(event);
      IF help#NIL THEN
        RETURN help;
      END;
    END;
    RETURN NIL;
  END GetFocus;


  PROCEDURE (w : WindowGroup) GetPosObject*(x,y : LONGINT; type : LONGINT):G.Object;

  VAR
    help  : G.Object;

  BEGIN
    IF w.men#NIL THEN
      help:=w.men.GetPosObject(x,y,type);
      IF help#NIL THEN
        RETURN help;
      END;
    END;
    IF w.tool#NIL THEN
      help:=w.tool.GetPosObject(x,y,type);
      IF help#NIL THEN
        RETURN help;
      END;
    END;
    IF w.top#NIL THEN
      help:=w.top.GetPosObject(x,y,type);
      IF help#NIL THEN
        RETURN help;
      END;
    END;
    IF w.status#NIL THEN
      help:=w.status.GetPosObject(x,y,type);
      IF help#NIL THEN
        RETURN help;
      END;
    END;
    RETURN w.GetPosObject^(x,y,type); (** Hoo, not that good *)
  END GetPosObject;

  PROCEDURE (w : WindowGroup) GetDnDObject*(x,y : LONGINT; drag : BOOLEAN):G.Object;

  VAR
    help  : G.Object;

  BEGIN
    IF w.men#NIL THEN
      help:=w.men.GetDnDObject(x,y,drag);
      IF help#NIL THEN
        RETURN help;
      END;
    END;
    IF w.tool#NIL THEN
      help:=w.tool.GetDnDObject(x,y,drag);
      IF help#NIL THEN
        RETURN help;
      END;
    END;
    IF w.top#NIL THEN
      help:=w.top.GetDnDObject(x,y,drag);
      IF help#NIL THEN
        RETURN help;
      END;
    END;
    IF w.status#NIL THEN
      help:=w.status.GetDnDObject(x,y,drag);
      IF help#NIL THEN
        RETURN help;
      END;
    END;
    RETURN NIL;
  END GetDnDObject;


  PROCEDURE (w : WindowGroup) Draw*(x,y : LONGINT; draw : D.DrawInfo);

  VAR
    yPos,height : LONGINT;

  BEGIN
    w.Draw^(x,y,draw);

    w.DrawBackground(w.x,w.y,w.width,w.height);

    yPos:=w.y;
    height:=w.height;

    IF w.men#NIL THEN
      w.men.Resize(w.width,-1);
      w.men.Draw(w.x,yPos,draw);
      INC(yPos,w.men.oHeight);
      DEC(height,w.men.oHeight);
    END;

    IF w.tool#NIL THEN
      w.tool.Resize(w.width,-1);
      w.tool.Draw(w.x,yPos,draw);
      INC(yPos,w.tool.oHeight);
      DEC(height,w.tool.oHeight);
    END;

    IF w.status#NIL THEN
      w.status.Resize(w.width,-1);
      w.status.Draw(w.x,w.height-w.status.oHeight,draw);
      DEC(height,w.status.oHeight);
    END;

    IF w.top#NIL THEN
      IF w.space THEN
        w.top.Resize(w.width-2*w.display.spaceWidth,height-2*w.display.spaceWidth);
        w.top.Draw(w.x+(w.width-w.top.oWidth) DIV 2,yPos+w.display.spaceHeight,draw);
      ELSE
        w.top.Resize(w.width,height);
        w.top.Draw(w.x+(w.width-w.top.oWidth) DIV 2,yPos,draw);
      END;
    END;
  END Draw;

  PROCEDURE (win : WindowGroup) Refresh*(x,y,w,h : LONGINT);

  BEGIN
    IF win.visible & win.Intersect(x,y,w,h) THEN

      win.RestrictToBounds(x,y,w,h);
      win.DrawBackground(x,y,w,h);

      IF win.men#NIL THEN
        win.men.Refresh(x,y,w,h);
      END;
      IF win.tool#NIL THEN
        win.tool.Refresh(x,y,w,h);
      END;
      IF win.top#NIL THEN
        win.top.Refresh(x,y,w,h);
      END;
      IF win.status#NIL THEN
        win.status.Refresh(x,y,w,h);
      END;
    END;
  END Refresh;


  PROCEDURE (w : WindowGroup) Hide*;

  BEGIN
    IF w.visible THEN
      IF w.men#NIL THEN
        w.men.Hide;
      END;
      IF w.tool#NIL THEN
        w.tool.Hide;
      END;
      IF w.top#NIL THEN
        w.top.Hide;
      END;
      IF w.status#NIL THEN
        w.status.Hide;
      END;
      w.DrawHide;
      w.Hide^;
    END;
  END Hide;

BEGIN
  NEW(prefs);
  prefs.Init;
END VOWindowGroup.