#include "branchpropwindow.h"

#include <QColorDialog>

#include "frameobj.h"
#include "settings.h"

extern Settings settings;
extern QString vymName;


BranchPropertyWindow::BranchPropertyWindow (QWidget *parent): QDialog (parent)
{
	ui.setupUi (this);

	setCaption(vymName +" - " +tr ("Property Editor","Window caption"));

	branch=NULL;
	mapEditor=NULL;

	ui.tabWidget->setEnabled(false);

	penColor=QColor (Qt::black);
	brushColor=QColor (Qt::black);
    QPixmap pix( 16,16);
    pix.fill (penColor);
	ui.framePenColorButton->setPixmap (pix);
	ui.frameBrushColorButton->setPixmap (pix);


	// Load Settings
	resize (settings.value ( "/satellite/propertywindow/geometry/size", QSize(450,600)).toSize());
	move   (settings.value ( "/satellite/propertywindow/geometry/pos", QPoint (250,50)).toPoint());
	
	if (settings.value ( "/satellite/propertywindow/showWithMain",true).toBool())
		show();
	else	
		hide();

}

BranchPropertyWindow::~BranchPropertyWindow ()
{
	settings.setValue( "/satellite/propertywindow/geometry/size", size() );
	settings.setValue( "/satellite/propertywindow/geometry/pos", pos() );
	settings.setValue( "/satellite/propertywindow/showWithMain",isVisible() );
}

void BranchPropertyWindow::setBranch (BranchObj *bo)
{
	disconnectSignals();
	branch=bo;
	if (bo) 
	{
		ui.tabWidget->setEnabled (true);

		// Frame
		FrameObj::FrameType t=branch->getFrameType();
		if (t==FrameObj::NoFrame)
		{
			ui.frameTypeCombo->setCurrentIndex (0);
			penColor=Qt::white;
			brushColor=Qt::white;
			ui.colorGroupBox->setEnabled (false);
			ui.framePaddingSpinBox->setEnabled (false);
			ui.frameWidthSpinBox->setEnabled (false);
			ui.framePaddingLabel->setEnabled (false);
			ui.frameBorderLabel->setEnabled (false);
		} else	
		{
			penColor=bo->getFramePenColor();
			brushColor=bo->getFrameBrushColor();
			QPixmap pix( 16,16);
			pix.fill (penColor);
			ui.framePenColorButton->setPixmap (pix);
			pix.fill (brushColor);
			ui.frameBrushColorButton->setPixmap (pix);
			ui.colorGroupBox->setEnabled (true);
			ui.framePaddingSpinBox->setEnabled (true);
			ui.framePaddingSpinBox->setValue (bo->getFramePadding());
			ui.frameWidthSpinBox->setEnabled (true);
			ui.frameWidthSpinBox->setValue (bo->getFrameBorderWidth());
			ui.framePaddingLabel->setEnabled (true);
			ui.frameBorderLabel->setEnabled (true);

			switch (t)
			{
				case FrameObj::Rectangle: 
					ui.frameTypeCombo->setCurrentIndex (1);
					break;
				case FrameObj::Ellipse: 
					ui.frameTypeCombo->setCurrentIndex (2);
					break;
				default: 
					break;
			}
		}	
		
		// Link
		if (branch->getHideLinkUnselected())
			ui.hideLinkIfUnselected->setCheckState (Qt::Checked);
		else	
			ui.hideLinkIfUnselected->setCheckState (Qt::Unchecked);

		// Layout
		if (branch->getIncludeImagesVer())
			ui.incImgVer->setCheckState (Qt::Checked);
		else	
			ui.incImgVer->setCheckState (Qt::Unchecked);
		if (branch->getIncludeImagesHor())
			ui.incImgHor->setCheckState (Qt::Checked);
		else	
			ui.incImgHor->setCheckState (Qt::Unchecked);

		// Finally activate signals
		connectSignals();
	} else
	{
		ui.tabWidget->setEnabled (false);
	}
}

void BranchPropertyWindow::setMapEditor (MapEditor *me)
{
	mapEditor=me;
	if (mapEditor) 
		setBranch (mapEditor->getSelectedBranch() );
	else
		ui.tabWidget->setEnabled (false);
		
}

void BranchPropertyWindow::frameTypeChanged (int i)
{
	if (mapEditor)
	{
		switch (i)
		{
			case 0: mapEditor->setFrameType (FrameObj::NoFrame); break;
			case 1: mapEditor->setFrameType (FrameObj::Rectangle); break;
			case 2: mapEditor->setFrameType (FrameObj::Ellipse); break;
		}
		setBranch (branch);
	}	
}

void BranchPropertyWindow::framePenColorClicked()
{
	if (mapEditor) 
	{	
		QColor col = QColorDialog::getColor( penColor, this );
		if ( col.isValid() ) 
		{
			penColor=col;
			mapEditor->setFramePenColor (penColor);
		}	
	}
}

void BranchPropertyWindow::frameBrushColorClicked()
{
	if (mapEditor) 
	{
		QColor col = QColorDialog::getColor( brushColor, this );
		if ( col.isValid() ) 
		{
			brushColor=col;
			mapEditor->setFrameBrushColor (brushColor);
		}	
	}	
}

void BranchPropertyWindow::framePaddingChanged(int i)
{
	if (mapEditor) mapEditor->setFramePadding (i);
}

void BranchPropertyWindow::frameBorderWidthChanged(int i)
{
	if (mapEditor) mapEditor->setFrameBorderWidth(i);
}

void BranchPropertyWindow::linkHideUnselectedChanged (int i)
{
	if (!branch) return;
	mapEditor->setHideLinkUnselected(i);
}

void BranchPropertyWindow::incImgVerChanged (int  i)
{
	if (mapEditor) mapEditor->setIncludeImagesVer (i);
}

void BranchPropertyWindow::incImgHorChanged (int  i)
{
	if (mapEditor) mapEditor->setIncludeImagesHor (i);
}

void BranchPropertyWindow::closeEvent( QCloseEvent* ce )
{
    ce->accept();	// can be reopened with show()
	hide();
	emit (windowClosed() );
    return;
}


void BranchPropertyWindow::connectSignals()
{
	// Frame
	connect ( 
		ui.framePenColorButton, SIGNAL (clicked()), 
		this, SLOT (framePenColorClicked()));
	connect ( 
		ui.framePaddingSpinBox, SIGNAL (valueChanged( int)), 
		this, SLOT (framePaddingChanged (int)));
	connect ( 
		ui.frameWidthSpinBox, SIGNAL (valueChanged( int)), 
		this, SLOT (frameBorderWidthChanged (int)));
	connect ( 
		ui.frameBrushColorButton, SIGNAL (clicked()), 
		this, SLOT (frameBrushColorClicked()));
	connect ( 
		ui.frameTypeCombo, SIGNAL (currentIndexChanged( int)), 
		this, SLOT (frameTypeChanged (int)));


	// Link	
	connect ( 
		ui.hideLinkIfUnselected, SIGNAL (stateChanged( int)), 
		this, SLOT (linkHideUnselectedChanged (int)));

	// Layout	
	connect ( 
		ui.incImgVer, SIGNAL (stateChanged( int)), 
		this, SLOT (incImgVerChanged (int)));
	connect ( 
		ui.incImgHor, SIGNAL (stateChanged( int)), 
		this, SLOT (incImgHorChanged (int)));
}


void BranchPropertyWindow::disconnectSignals()
{
	// Frame 
	disconnect ( ui.frameTypeCombo, 0,0,0);
	disconnect ( ui.framePenColorButton, 0,0,0);
	disconnect ( ui.framePaddingSpinBox, 0,0,0);
	disconnect ( ui.frameWidthSpinBox, 0,0,0);
	disconnect ( ui.frameBrushColorButton, 0,0,0);

	// Link	
	disconnect ( ui.hideLinkIfUnselected, 0,0,0);

	// Layout	
	disconnect ( ui.incImgVer, 0,0,0);
	disconnect ( ui.incImgHor, 0,0,0);
}


