/*
 * Copyright (c) 1997 Loughborough University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the LUTCHI Research
 *      Centre at Loughborough University.
 * 4. Neither the name of the University nor of the Centre may be used
 *    to endorse or promote products derived from this software without
 *    specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE UNIVERSITY OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */


/*
 * Command history database.
 *
 * J.C.Highfield, 4/97.
 *
 */

#define MULTICAST
#define LEN	1024

#ifndef WIN32
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <sys/stat.h>
#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef HAVE_SYS_TIME_H
    #include <sys/time.h>
#else
    #include <time.h>
#endif
#include <pwd.h>
#include <unistd.h>
#else
#include <assert.h>
#include <io.h>
#include <process.h>
#include <fcntl.h>
#include <windows.h>
#include <malloc.h>
#include <string.h>
#include <stdio.h>
#include <time.h>
#include <winsock.h>
#endif

#include <tcl.h>
#include <tk.h>
#include "wb.h"
#include "db.h"
#include "proto.h"

extern Tcl_Interp *tcl;

extern int maxhops;

struct page *page_root   = NULL;

struct mqueue *transmit   = NULL;
struct mqueue *request    = NULL;
struct mqueue *repair     = NULL;

/* The CurrentPageCount. */
uint32 page_num_max = 0;

/* The Current Page. */
struct page *CurrentPage = NULL;

/* Updated once a second. */
static int activeSenders = 1;

/* Turn off the rate-limiting system. */
/*#define DEBUG_NO_QUEUES*/

/*************************************************************/
/*                                                           */
/* Queue routines.                                           */
/*                                                           */
/*************************************************************/

struct mqueue *queue_head = NULL;
struct sender *sender_head = NULL;

/* Random, exponential delay. */
int DelayCalc (int ttl, int senders)
{
    static int init = 0;
    long delay;
    long random = rand();

    if (!init)
    {
        srand (this_ip() & 0xffff);
        random = rand();
        init = 1;
    }

    /* Just a rule of thumb for now. */
    delay = ttl * senders * random / 32768;

    return delay;
}

void QueueAppend (struct mqueue *element)
{
    struct mqueue *ptr;
    struct mqueue *old;

    if (queue_head == NULL)
    {
        queue_head = element;
        return;
    }

    ptr = queue_head;
    old = NULL;
    while (ptr != NULL)
    {
        old = ptr;
        ptr = ptr->next;
    }

    old->next = element;
}

/* Remove an item from the queue. */
void QueueRemove (struct mqueue *element)
{
    struct mqueue *ptr;
    struct mqueue *old;

    if (queue_head == NULL)
        return;

    if (queue_head == element)
    {
        queue_head = element->next;
        free (element);
        return;
    }

    ptr = queue_head;
    old = NULL;
    while (ptr != NULL)
    {
        if (ptr == element)
        {
            old->next = element->next;
            free (element);
            return;
        }

        old = ptr;
        ptr = ptr->next;
    }

    /* Not reached unless element isn't in the list. */
    return;
}

/* Add a message to the queue. */
int QueueAddTo (void *mesg, uint32 size, int type)
{

#ifndef DEBUG_NO_QUEUES

    struct wb_packet *Packet = (struct wb_packet *) mesg;
    struct mqueue    *q      = NULL;

    if (mesg == NULL)
        return 0;

    SenderAdd (this_ip(), time_stamp(), time_stamp(), 
                ntohl(Packet->pk_type), ascii_source(NULL));

    /* State messages aren't queued at all. */
    if (type == Q_STATE)
    {
        NetSend (mesg, size);
        return 0;
    }

    /* Build the queue element. */
    q = (struct mqueue *) malloc (sizeof (struct mqueue));
    if (q == NULL)
    {
        perror ("malloc failed in QueueAddTo");
        return -1;
    }

    q->type  = type;
    q->size  = size;
    q->data  = mesg;
    q->next  = NULL;

    if (type == Q_DRAW_MSG)
        /* Draw Messages aren't delayed. */
        q->delay = 0;
    else
        /* Only Request & Repair traffic needs delays. */
        q->delay = DelayCalc (maxhops, activeSenders);

    /* Put it on the end of the queue. */
    QueueAppend(q);

#else

    NetSend (mesg, size);

#endif

    return 0;
}


/* Check to see if a repair is already queued. */
struct mqueue *QueueCheckRepair (struct pk_rpy_draw *DrawRpy1)
{
    struct pk_rpy_draw *DrawRpy2;
    struct mqueue *ptr;
    struct mqueue *old;

#ifndef DEBUG_NO_QUEUES

    ptr = queue_head;
    while (ptr != NULL)
    {
        if (ptr->type == Q_REPAIR)
        {
            DrawRpy2 = (struct pk_rpy_draw *) ptr->data;

            if ( (DrawRpy1->page_ip == DrawRpy2->page_ip) &&
                 (DrawRpy1->page_num == DrawRpy2->page_num) )
                /* Same page. */
                if (DrawRpy1->drawer_ip == DrawRpy2->drawer_ip)
                  /* Probably should check for sub-ranges, etc.. */
                  if ( (DrawRpy1->first_cmd == DrawRpy2->first_cmd) ||
                       (DrawRpy1->last_cmd == DrawRpy2->last_cmd) )
                    return ptr;
        }

        ptr = ptr->next;
    }

#endif

    return NULL;
}


/* Check to see if a request is already queued. */
struct mqueue *QueueCheckRequest (struct pk_req_draw *DrawReq1)
{
    struct pk_req_draw *DrawReq2;
    struct mqueue *ptr;
    struct mqueue *old;

#ifndef DEBUG_NO_QUEUES

    ptr = queue_head;
    while (ptr != NULL)
    {
        if (ptr->type == Q_REQUEST)
        {
            DrawReq2 = (struct pk_req_draw *) ptr->data;

            if ( (DrawReq1->page_ip == DrawReq2->page_ip) &&
                 (DrawReq1->page_num == DrawReq2->page_num) )
                /* Same page. */
                if (DrawReq1->drawer_ip == DrawReq2->drawer_ip)
                  /* Probably should check for sub-ranges, etc.. */
                  if ( (DrawReq1->first_cmd == DrawReq2->first_cmd) ||
                       (DrawReq1->last_cmd == DrawReq2->last_cmd) )
                    return ptr;
        }

        ptr = ptr->next;
    }

#endif

    return NULL;
}


/* Check to see if a page report is already queued. */
struct mqueue *QueueCheckReport (struct pk_page_rpt *PageRpt1)
{
    struct pk_page_rpt *PageRpt2;
    struct page_desc   *pd1;
    struct page_desc   *pd2;
    struct mqueue *ptr;
    struct mqueue *old;

#ifndef DEBUG_NO_QUEUES

    pd1 = (struct page_desc *)(sizeof (struct pk_page_rpt) + 
                                   (char *)PageRpt1);

    ptr = queue_head;
    while (ptr != NULL)
    {
        if (ptr->type == Q_REPORT)
        {
            PageRpt2 = (struct pk_page_rpt *) ptr->data;
            pd2 = (struct page_desc *)
                    (sizeof (struct pk_page_rpt) + 
                                   (char *)PageRpt2);

            if ((pd1->page_id == pd2->page_id) &&
                (pd1->page_num == pd2->page_num))
                    return ptr;
        }

        ptr = ptr->next;
    }

#endif

    return NULL;
}

/* Check to see if a page report is already queued. */
struct mqueue *QueueCheckAskPage (struct pk_ask_page *AskPage1)
{
    struct pk_ask_page *AskPage2;
    struct mqueue *ptr;
    struct mqueue *old;

#ifndef DEBUG_NO_QUEUES

    ptr = queue_head;
    while (ptr != NULL)
    {
        if (ptr->type == Q_ASKPAGE)
        {
            AskPage2 = (struct pk_ask_page *) ptr->data;

            if ((AskPage1->page_ip == AskPage2->page_ip) &&
                (AskPage1->page_num == AskPage2->page_num))
                    return ptr;
        }

        ptr = ptr->next;
    }

#endif

    return NULL;
}


void CheckQueues (void)
{
    static unsigned long sLastTime = 0;
    static unsigned long qLastTime = 0;
    static unsigned long pLastTime = 0;
    static unsigned long allowed   = 0;
    unsigned long nowTime;
    extern int datarate;
    struct pk_session *session;
    struct mqueue     *q;
    struct mqueue     *p;
    uint32 size;
    char arg[LEN];
	int count=0;

    /* Disable session announcements during testing. */
    nowTime = time (NULL);

    /* How many active in the last 30s? */
    activeSenders = SenderActive (3000);

    if ((nowTime - sLastTime) >= 10)
    {
        sLastTime = nowTime;

        /* Send a Session announcement. */
        session = build_session(&size);
        QueueAddTo ((char *)session, size, Q_STATE);

        /* Check for needing repairs on the current page. */
        CheckCurrent();

        /* Check for unknown pages. */
        CheckPages();

	/* Update the control panel. */
	sprintf (arg, "UpdatePanel [Sender count]");
        Tcl_Eval(tcl, arg);
    }

#ifndef DEBUG_NO_QUEUES

    /* Reset the per-second bandwidth allowance? */
    if (nowTime > qLastTime)
    {
        qLastTime = nowTime;

        if (queue_head == NULL)
            /* Don't accumulate allowance for an empty queue. */
            allowed = datarate / (8 * activeSenders);
        else
            /* Only for queues stalled by packets bigger than */
            /* the per-second allowance.                      */
            allowed += datarate / (8 * activeSenders);
    }

    /* Send queued messages if bandwidth is available. */
    q = queue_head;
    p = NULL;

	/* only process a couple at a time.... */
    while (q != NULL && count < 5)
    {
		count++;
        /* Messages are first randomly delayed... */
        if (q->delay > 0)
        {
            q->delay--;

            p = q;
            q = q->next;
        }
        else
        {
            /* ...then sent when bandwidth is available. */
            if (allowed > q->size)
            {
                if (p == NULL)
                    queue_head = q->next;
                else
                    p->next = q->next;

                NetSend ((char *)q->data, q->size);
                allowed -= q->size;

                free (q);

                if (p == NULL)
                    q = queue_head;
                else
                    q = p->next;
            }
            else
            {
                p = q;
                q = q->next;
            }
        }
    }

#endif

    if (nowTime > pLastTime)
    {
        /* Also, check to see if the page needs redrawing. */
        PageDraw (CurrentPage, 0);

        /* PageDraw can be slow, so this is done afterwards. */
        pLastTime = time (NULL);
    }
}

/*************************************************************/
/*                                                           */
/* List of conference members.                               */
/*                                                           */
/*************************************************************/


int SenderAdd (uint32 sender_ip, uint32 sent, uint32 seen, 
                                 uint32 pktype, char *name)
{
    struct sender *ptr;
    struct sender *old;

    /* Look for an existing entry. */
    ptr = sender_head;
    old = NULL;
    while (ptr != NULL)
    {
        if (ptr->sender_ip == sender_ip)
        {
            /* Update the existing entry. */
            ptr->last_sent  = sent;
            if (pktype == PK_DRAW_MSG)
                ptr->last_drawn = sent;
            ptr->last_seen  = seen;
            if (name != NULL)
            {
                strncpy (ptr->name, name, SENDER_NAME_LENGTH);
                ptr->name[SENDER_NAME_LENGTH-1] = '\0';
            }

            return 0;
        }

        old = ptr;
        ptr = ptr->next;
    }

    /* Make new entry. */
    ptr = (struct sender *) malloc (sizeof(struct sender));
    if (ptr == NULL)
    {
        perror ("Malloc failed in SenderAdd");
        return -1;
    }
    ptr->sender_ip  = sender_ip;
    ptr->last_sent  = sent;
    ptr->last_drawn = sent;
    ptr->last_seen  = seen;
    if (name != NULL)
    {
        strncpy (ptr->name, name, SENDER_NAME_LENGTH);
        ptr->name[SENDER_NAME_LENGTH-1] = '\0';
    }
    else
        strcpy (ptr->name, "");
    ptr->next      = NULL;

    if (old == NULL)
        sender_head = ptr;
    else
        old->next = ptr;

    return 0;
}

/* Count senders active in the last 'period' 0.01s. */
int SenderActive (uint32 period)
{
    struct sender *ptr;
    uint32 nowTime;
    int count = 0;

    nowTime = time_stamp();

    ptr = sender_head;
    while (ptr != NULL)
    {
        if ((nowTime - ptr->last_sent) > period)
            count++;

        ptr = ptr->next;
    }

    if (count == 0)
        count = 1;

    return count;
}

/*************************************************************/
/*                                                           */
/* Page history routines.                                    */
/*                                                           */
/*************************************************************/

struct pk_draw_msg *MessageCopy (struct pk_draw_msg *message)
{
    struct pk_draw_msg *copy = NULL;
    long   length;

    if (message == NULL)
        return NULL;

    length = sizeof (struct pk_draw_msg) +
                    ntohs(message->draw_cmd_length);

    copy = malloc (length);
    if (copy != NULL)
        memcpy (copy, message, length);

    return copy;
}


struct page *PageMake (uint32 page_num, uint32 page_ip,
                       struct page *prev, struct page *next)
{
    struct page *ptr;

    ptr = (struct page *) malloc (sizeof (struct page));
    if (ptr != NULL)
    {
        ptr->page_number = page_num;
        ptr->creator_ip  = page_ip;
        ptr->seq_num_max = 0;
        ptr->next        = next;
        ptr->prev        = prev;
        ptr->writers     = NULL;
        ptr->actions     = NULL;
    }

    return ptr;
}

struct writer *WriterMake (uint32 writer_ip, struct writer *prev, 
                                            struct writer *next)
{
    struct writer *ptr;

    ptr = (struct writer *) malloc (sizeof (struct writer));
    if (ptr != NULL)
    {
        ptr->writer_ip   = writer_ip;
        ptr->seq_num_max = 0;
        ptr->next        = next;
        ptr->prev        = prev;
    }

    return ptr;
}

/* This routine now owns the message storage. */
struct action *ActionMake (struct pk_draw_msg *message, 
                       struct action *prev, struct action *next)
{
    struct action *ptr;

    ptr = (struct action *) malloc (sizeof (struct action));
    if (ptr != NULL)
    {
        ptr->writer_ip  = ntohl(message->sender);
        ptr->seq_num    = ntohl(message->draw_cmd_num);
        ptr->timestamp  = ntohl(message->timestamp);
        ptr->next       = next;
        ptr->prev       = prev;
        ptr->message    = message;
    }
    else
        free (message);

    return ptr;
}


/* Returns the page pointer if the page already exists. */
struct page *PageFind (uint32 page_num, uint32 page_ip)
{
    struct page *ptr;
    struct page *result = NULL;

    ptr = page_root;
    while (ptr != NULL)
    {
        if (ptr->page_number == page_num)
        {
/* Doesn't matter if the creator matches, just return the pointer - KH */
/*            if (page_ip == (uint32) NULL)
                result = ptr;
            else if (ptr->creator_ip == page_ip)*/
                result = ptr;
        }
 
        if (ptr->page_number > page_num)
            break;

        ptr = ptr->next;
    }

    return result;
}


/* Creates the page if it does not already exist.  */
/* Page list is ordered by page number.            */
struct page *PageAdd (uint32 page_num, uint32 page_ip)
{
    struct page *ptr;
    struct page *pptr;
    struct page *nptr;
	char msg[80];

#ifdef DEBUG && WIN32
	sprintf (msg,"Page_num:%ld  Page_ip:%ld\n",page_num,page_ip);
	OutputDebugString(msg);
#endif

    /* Keep the global counter accurate. */
    if (page_num > PageNumMax())
        PageNumMaxSet (page_num);

    /* Reject pages numbered zero! */
    if (page_num == 0 || page_ip ==0)
        return NULL;

    if (page_root == NULL)
    {
        /* New list. */
        page_root = ptr = PageMake (page_num, page_ip, NULL, NULL);
    }
    else
    {
        /* Find the right place. */
        pptr = NULL;
        nptr = page_root;
        while (nptr != NULL)
        {
            if (nptr->page_number >= page_num)
/* why do the pages need to be from the same creator ? KH*/
/*                if ((nptr->creator_ip == 0) || 
                   (nptr->creator_ip == page_ip))*/
                    break;
            pptr = nptr;
            nptr  = nptr->next;
        }

        if ( (nptr != NULL) && (nptr->page_number == page_num) ) /*&&
            ((nptr->creator_ip == 0) || (nptr->creator_ip == page_ip)) )*/
        {
            /* Page already exists. */
            ptr = nptr;

            /* Check that the page IP address has been set. */
            /* This really shouldn't happen any more.       */
            if (ptr->creator_ip == 0)
                ptr->creator_ip = page_ip;
        }
        else
        {
            /* Insert the new page. */
            ptr = PageMake (page_num, page_ip, pptr, nptr);
            if (pptr == NULL)
                page_root = ptr;
            else
                pptr->next = ptr;
            if (nptr != NULL)
                nptr->prev = ptr;
        }
    }

    /* Update the page select scrollbar. */
    PageScrollSet (PageCurrent());

    return ptr;
}

/* Delete an entire page of the action history. */
struct page *PageDelete (struct page *pptr)
{
    struct page   *pnptr;
    struct writer *wptr;
    struct writer *wnptr;
    struct action *aptr;
    struct action *anptr;

    if (pptr == NULL)
        return NULL;

    /* Delete the writers list. */
    wptr = pptr->writers;
    while (wptr != NULL)
    {
        wnptr = wptr;
        free (wptr);
        wptr = wnptr;
    }

    /* Delete the actions list. */
    aptr = pptr->actions;
    while (aptr != NULL)
    {
        anptr = aptr;
        if (aptr->message != NULL)
            free (aptr->message);
        free (aptr);
        aptr = anptr;
    }

    pnptr = pptr->next;
    free (pptr);

    return pnptr;
}


/* Returns the writer pointer if the writer already exists. */
struct writer *WriterFind (int page_num, uint32 page_ip, uint32 writer_ip)
{
    struct page   *page_ptr;
    struct writer *nptr;

    page_ptr = PageFind (page_num, page_ip);
    if (page_ptr == NULL)
        return NULL;

    if (page_ptr->writers == NULL)
        return NULL;

    nptr = page_ptr->writers;
    while (nptr != NULL)
    {
        if (nptr->writer_ip >= writer_ip)
            break;
        nptr = nptr->next;
    }

    if ((nptr != NULL) && (nptr->writer_ip == writer_ip))
    {
        /* Writer exists. */
        return nptr;
    }

    return NULL;
}

/* Add a writer to the list. */
/* Writer list is ordered by writer's IP address. */
struct writer *WriterAdd (int page_num, uint32 page_ip, uint32 writer_ip)
{
    struct page *page_ptr;
    struct writer *ptr;
    struct writer *pptr;
    struct writer *nptr;

    page_ptr = PageFind (page_num, page_ip);
    if (page_ptr == NULL)
        return NULL;

    if (page_ptr->writers == NULL)
    {
        /* New list. */
        page_ptr->writers = ptr = WriterMake (writer_ip, NULL, NULL);
    }
    else
    {
        /* Find the right place. */
        pptr = NULL;
        nptr = page_ptr->writers;
        while (nptr != NULL)
        {
            if (nptr->writer_ip >= writer_ip)
                break;
            pptr = nptr;
            nptr = nptr->next;
        }

        if ((nptr != NULL) && (nptr->writer_ip == writer_ip))
        {
            /* Writer already exists. */
            ptr = nptr;
        }
        else
        {
            /* Insert the new writer. */
            ptr = WriterMake (writer_ip, pptr, nptr);
            if (pptr == NULL)
                page_ptr->writers = ptr;
            else
                pptr->next = ptr;
            if (nptr != NULL)
                nptr->prev = ptr;
        }
    }

    return ptr;
}

/* Find an action. */
struct action *ActionFind (uint32 page_num, uint32 page_ip,
                           uint32 drawer_ip, uint32 seq_no)
{
    struct page   *page   = NULL;
    struct action *action = NULL;

    page = PageFind (page_num, page_ip);
    if (page == NULL)
        return NULL;

    action = page->actions;
    while (action != NULL)
    {
        if ((action->writer_ip == drawer_ip) &&
            (action->seq_num == seq_no) )
                return action;

        action = action->next;
    }

    return action;
}

/* Add an action to the history. */
/* Action list is ordered by action's timestamp.    */
struct action *ActionAdd (struct pk_draw_msg *message)
{
    struct writer *writer_ptr;
    struct page   *page_ptr;
    struct action *ptr;
    struct action *pptr;
    struct action *nptr;

    uint32  writer_ip, seq_num, timestamp ;
    uint32  page_num, page_ip;

    if (message == NULL)
    {
        perror ("Null draw packet in ActionAdd");
        return NULL;
    }

    /* Extract the key information. */
    writer_ip   = ntohl(message->sender);
    seq_num     = ntohl(message->draw_cmd_num);
    timestamp   = ntohl(message->timestamp);
    page_num    = ntohl(message->page_num);
    page_ip     = ntohl(message->page_ip);

    /* Ensure the page exists in the history. */
    page_ptr = PageAdd (page_num, page_ip);
    if (page_ptr == NULL)
    {
        perror ("failed to add a page to the history");
        return NULL;
    }

    /* Ensure the writer exists in the history. */
    writer_ptr = WriterAdd (page_num, page_ip, writer_ip);
    if (writer_ptr == NULL)
    {
        perror ("failed to add a writer to the history");
        return NULL;
    }

    /* Copy the message. */
    message = MessageCopy (message);
    if (message == NULL)
    {
        perror ("failed to copy a message");
        return NULL;
    }

    /* Add the action to the history. */
    if (page_ptr->actions == NULL)
    {
        /* New list. */
        page_ptr->actions = ptr = ActionMake (message, NULL, NULL);
        if (ptr == NULL)
        {
            perror ("failed to add an action to the history");
            return NULL;
        }
    }
    else
    {
        /* Find the right place. */
        pptr = NULL;
        nptr = page_ptr->actions;

        /* ..first sender */
        while (nptr != NULL)
        {
            if (nptr->writer_ip >= ntohl(message->sender))
                break;

            pptr = nptr;
            nptr = nptr->next;
        }

        /* ..and then draw sequence number */
        while ( (nptr != NULL) && 
				(nptr->writer_ip <= ntohl(message->sender)) )
        {
            if (nptr->seq_num >= ntohl(message->draw_cmd_num))
                break;

            pptr = nptr;
            nptr = nptr->next;
        }

        /* Either the right message or the wrong time. */
        if ( (nptr != NULL) && 
             (nptr->seq_num  == ntohl(message->draw_cmd_num)) )
        {
            /* Action already exists. */
            free (message);
            ptr = nptr;
        }
        else
        {
            /* Insert the new action. */
            ptr = ActionMake (message, pptr, nptr);
            if (ptr == NULL)
            {
                perror ("failed to add an action to the history");
                return NULL;
            }
            if (pptr == NULL)
                page_ptr->actions = ptr;
            else
                pptr->next = ptr;
            if (nptr != NULL)
                nptr->prev = ptr;
        }
    }

    /* Update the writer summary. */
    if (writer_ptr->seq_num_max <= seq_num)
    {
        if (seq_num > (1 + writer_ptr->seq_num_max))
        {
            /* Fill in the gaps... */
            CheckCurrent();
        }

        writer_ptr->seq_num_max = seq_num;
    }

    return ptr;
}

/* Free all allocated memory. */
void TidyUp ()
{
    struct sender *sptr;
    struct sender *snptr;
    struct page   *pptr;

    /* Delete the sender list. */
    sptr = sender_head;
    while (sptr != NULL)
    {
        snptr = sptr;
        free (sptr);
        sptr = snptr;
    }

    /* Delete the action history. */
    pptr = page_root;
    while (pptr != NULL)
        pptr = PageDelete (pptr);
}


/* Return the highest sequence number by that writer on that page. */
uint32 SeqNumMax (uint32 page_num, uint32 page_ip, uint32 writer_ip)
{
    struct writer *wptr;

    wptr = WriterAdd (page_num, page_ip, writer_ip);

    /* This had better never happen. */
    if (wptr == NULL)
    {
        perror ("failed to add a writer to the history (SeqNumMax)");
        return 0;
    }

    return (wptr->seq_num_max);
}


uint32 PageIP (uint32 page_num)
{
    uint32  page_ip = 0;
    struct page   *ptr;

    ptr = page_root;
    while (ptr != NULL)
    {
        if (ptr->page_number >= page_num)
        {
            if (ptr->page_number == page_num)
            {
                page_ip = ptr->creator_ip;
                /*fprintf (stderr, "Found Page %d IP %08lX\n", 
                                           page_num, page_ip);*/
            }
            break;
        }
        ptr = ptr->next;
    }

    return (page_ip);
}


int SessionUpdate (struct pk_session *session)
{
    ;
}

/*************************************************************/
/*                                                           */
/* Page redraw routines.                                     */
/*                                                           */
/*************************************************************/

extern Tcl_Interp *tcl;


void PageScrollSet (struct page *pptr)
{
    char arg[LEN];
    long page_num;

    if (pptr == NULL)
        return;

    if (pptr->page_number < 1)
        return;

    page_num = PageIndex (pptr->page_number, pptr->creator_ip);
    if (page_num == 0)
        return;
    sprintf (arg, "set_page %ld %ld", page_num, PageCount());
    if (Tcl_Eval(tcl, arg) != TCL_OK)
    {
        printf ("set_page failed: %s\n", tcl->result);
    }
}


int PageDraw (struct page *pptr, int update)
{
    static unsigned long lastTime = 0;
    unsigned long nowTime;
    static int dirty = 0;
    struct action *aptr;
    int size;
    long page_num;

    /* Note that we need to do a draw. */
    if (update)
        dirty = 1;

    /* Don't bother if there is nothing new to draw. */
    if (!dirty)
        return 0;

    /* Don't do a PageDraw more than once per second. */
    nowTime = time (NULL);
    if (nowTime <= lastTime)
        return 0;

    /* Draw the page. */
    if (pptr == NULL)
        return 0;

    /* Update the scroll bar page index. */
    PageScrollSet (pptr);

    if (Tcl_Eval(tcl, "clear_page") != TCL_OK)
    {
        perror ("clear_page failed");
    }

    aptr = pptr->actions;
    while (aptr != NULL)
    {
        size = sizeof (struct pk_draw_msg) + 
                    ntohs(aptr->message->draw_cmd_length);
        ParseDrawMsg (aptr->message, size);

        aptr = aptr->next;
    }
    dirty = 0;
    lastTime = time (NULL);

    return 0;
}


struct page *PageCurrent (void)
{
    return CurrentPage;
}


void PageSet (uint32 page_num, uint32 page_ip)
{
    struct pk_ask_page *AskPage = NULL;
    struct mqueue      *q;
    uint32 size;

    CurrentPage = PageFind (page_num, page_ip);

    if (CurrentPage == NULL)
    {
        /* Don't know anything at all about this page. */
        AskPage = build_ask_page (page_num, page_ip, &size);
        if (AskPage != NULL)
        {
            /* Check for earlier, similar requests. */
            if ((q = QueueCheckAskPage (AskPage)) != NULL)
                QueueRemove(q);

            QueueAddTo (AskPage, size, Q_ASKPAGE);
        }

        return;
    }

    /* Check we have all the page data. */
    CheckCurrent();

    PageDraw (CurrentPage, 1);
}


uint32 PageNumMax (void)
{
    return (page_num_max);
}


void PageNumMaxSet (uint32 page_num)
{
    page_num_max = page_num;
}


/* Return the index of this page. */
uint32 PageIndex (uint32 page_num, uint32 page_ip)
{
    struct page *pptr = page_root;
    int count = 1;

    while (pptr != NULL)
    {
        if ( (pptr->page_number == page_num) &&
              (pptr->creator_ip == page_ip)  )
            return count;

        count++;
        pptr = pptr->next;
    }

    return 0;
}


/* Return the index of this page. */
uint32 PageIndexSet (uint32 page_num)
{
    struct page *pptr = page_root;
    int count = 1;

    while (pptr != NULL)
    {
        if (count == page_num)
        {
            PageSet (pptr->page_number, pptr->creator_ip);
            return 0;
        }

        count++;
        pptr = pptr->next;
    }

    return -1;
}


/* Return the page at this index. */
struct page *PageIndexFind (uint32 index)
{
    struct page *pptr = page_root;
    int count = 1;

    while (pptr != NULL)
    {
        if (count == index)
            return pptr;

        count++;
        pptr = pptr->next;
    }

    return NULL;
}


/* Return the number of pages in the database. */
uint32 PageCount (void)
{
    struct page *pptr = page_root;
    int count = 0;

    while (pptr != NULL)
    {
        count++;
        pptr = pptr->next;
    }

    return (count);
}


/* Takes an index into the page list, not a page number. */
int PageSetTcl (ClientData clientData, Tcl_Interp *tcl, 
                              int argc, char *argv[])
{
    uint32 page_num = -1;

    if (argc != 2)
    {
        sprintf (tcl->result, "wrong number of arguments");
        return TCL_ERROR;
    }

    sscanf (argv[1], "%ld", &page_num);

    if (page_num >= 0)
        PageIndexSet (page_num);

    return TCL_OK;
}


int PageNew (ClientData clientData, Tcl_Interp *tcl, 
                              int argc, char *argv[])
{
    uint32 page_new;
    char args[2][20] = {"Draw", "new_pg"};
    char *arg[2];

    arg[0] = args[0];
    arg[1] = args[1];

    if (argc != 1)
    {
        sprintf (tcl->result, "wrong number of arguments");
        return TCL_ERROR;
    }

    page_new = PageNumMax() + 1;
    PageAdd (page_new, this_ip());
    PageSet (page_new, this_ip());

    Draw (NULL, tcl, 2, arg);

    return TCL_OK;
}


/*************************************************************/
/*                                                           */
/* Check for missing data.                                   */
/*                                                           */
/*************************************************************/


int CheckCurrent (void)
{
    struct page        *page    = NULL;
    struct writer      *writer  = NULL;
    struct pk_req_draw *ReqDraw = NULL;
    struct mqueue      *q;
    uint32 seq_num_max, writer_ip;
    uint32 first_missing, last_missing;
    uint32 size;
    int i, dirty = 0;

    /* Only worry about the visible page. */
    page = PageCurrent();

    /* If we don't have a page. */
    if (page == NULL)
        return 0;

    /* Check each of the writers in turn. */
    writer = page->writers;
    while (writer != NULL)
    {
        seq_num_max = writer->seq_num_max;
        writer_ip   = writer->writer_ip;

        /* Find the first missing action. */
        first_missing = 0;
        for (i = 1; i <= seq_num_max; i++)
        {
            if (ActionFind(page->page_number, 
                  page->creator_ip, writer_ip, i) == NULL)
            {
                first_missing = i;

                break;
            }
        }

        if (first_missing != 0)
        {
            dirty = 1;

            /* Find the next present action. */
            last_missing = first_missing;
            for (i = first_missing; i <= seq_num_max; i++)
            {
                if (ActionFind(page->page_number, 
                      page->creator_ip, writer_ip, i) != NULL)
                {
                    break;
                }
            }
            last_missing = i - 1;

            /* Send the repair request. */
            ReqDraw = build_draw_request (page->page_number,
                             page->creator_ip, writer_ip, 
                             first_missing, last_missing, &size);
            if (ReqDraw != NULL)
            {
                /* Check to see if we already queued a   */
                /* similar request and if so deletes it. */
                if ((q = QueueCheckRequest (ReqDraw)) != NULL)
                    QueueRemove (q);

                QueueAddTo (ReqDraw, size, Q_REQUEST);
            }
        }

        writer = writer->next;
    }

    return dirty;
}


int CheckPages (void)
{
    struct pk_ask_page *AskPage;
    struct page        *page;
    struct mqueue      *q;
    uint32 size;
    int n;

    page = page_root;

    if (page == NULL)
    {
        /* Ask for any page information! */
        AskPage = build_ask_page (0, 0, &size);
        if (AskPage != NULL)
        {
            /* Check for earlier, similar requests. */
            if ((q = QueueCheckAskPage (AskPage)) != NULL)
                QueueRemove(q);

            QueueAddTo (AskPage, size, Q_ASKPAGE);
        }

        return 1;
    }

    n = 1;
    while (page != NULL)
    {
        if (page->page_number > n)
        {
            /* Ask about the missing pages.     */
            /* Don't know how to ask for a page */
            /* only by number! This might work. */
            AskPage = build_ask_page (n, 0, &size);
            if (AskPage != NULL)
            {
                /* Check for earlier, similar requests. */
                if ((q = QueueCheckAskPage (AskPage)) != NULL)
                    QueueRemove(q);

                QueueAddTo (AskPage, size, Q_ASKPAGE);
            }

            return 1;
        }

        /* Found page n, look for a page n+1. */
        if (page->page_number == n)
            n++;

        page = page->next;
    }

    return 0;
}


/*************************************************************/
/*                                                           */
/* Saving and loading entire sessions.                       */
/*                                                           */
/*************************************************************/

/* Save the entire history database to file. */
int WbDump (ClientData clientData, 
            Tcl_Interp *tcl, int argc, char *argv[])
{
    int pg, size, i;
    struct page   *page;
    struct action *aptr;
    uint8 *bptr;
    FILE *where;

    if (argc < 2)
    {
        sprintf (tcl->result, "too few arguments");
        return TCL_ERROR;
    }

    where = fopen (argv[1], "w");
    if (where == NULL)
    {
        sprintf (tcl->result, "couldn't open %s for write", argv[1]);
        return TCL_ERROR;
    }

    /* Dump each page, in the order they appear in wbd. */
    for (pg = 1; pg <= PageCount(); pg++)
    {
        page = PageIndexFind (pg);
        if (page != NULL)
        {
            aptr = page->actions;
            while (aptr != NULL)
            {
                /* Dump each action. */
                fprintf (where, "%8X ", aptr->writer_ip);
                fprintf (where, "%8X ", aptr->timestamp);
                fprintf (where, "%d ",  aptr->seq_num);

                /*fprintf (where, "%X ",  aptr->pk_type);*/
                fprintf (where, "%d ",  pg);

                /* Dump the draw data as hexadecimal. */
                size = (sizeof (uint32) + 2*sizeof(uint16)) +
                       htons (aptr->message->draw_cmd_length);
                fprintf (where, "%X\n",  size);

                bptr = (uint8 *) &(aptr->message->draw_timestamp);
                for (i = 0; i < size; i++)
                {
                    fprintf (where, "%02X", bptr[i]);

                    /* Keep the line lengths reasonable. */
                    if ((i / 128) > 0)
                        fprintf (where, "\n");
                }
                fprintf (where, "\n");

                aptr = aptr->next;
            }
        }
    }

    fclose (where);
}

/* All saved actions are read as though they had been written   */
/* by this author at the current time. This avoids problems     */
/* with looping timestamps (~1 year period) and data ownership. */
int WbReload (ClientData clientData, 
            Tcl_Interp *tcl, int argc, char *argv[])
{
    struct pk_draw_msg *mesg = NULL;
    struct page        *page = NULL;
    struct action      *aptr = NULL;
    uint8 *bptr;
    uint32 writer, time, type;
    uint32 seqnum, pgnum, size;
    uint32 value;
    int    result, i, offset;
    FILE  *where;

    if (argc < 2)
    {
        sprintf (tcl->result, "too few arguments");
        return TCL_ERROR;
    }

    where = fopen (argv[1], "r");
    if (where == NULL)
    {
        sprintf (tcl->result, "couldn't open %s for read", argv[1]);
        return TCL_ERROR;
    }

    /* Add the read pages after the existing ones. */
    offset = PageNumMax();

    while (!feof(where))
    {
        /* Read an action. */
        size = 0;
        result = fscanf (where, "%8X %8X %d %d %X\n", &writer,
                         &time, &seqnum, &pgnum, &size);

/*printf ("Reading mesage from %8X at %8X, page %d, seq %d, size %d\n",
  writer, time, pgnum, seqnum, size);*/


        if ((result == 5) && (size > 0))
        {
            mesg = malloc (sizeof(struct pk_draw_msg) - 
                            (sizeof (uint32) + 2*sizeof(uint16)) + 
                            size);
            if (mesg == NULL)
            {
                perror ("malloc failed");
                return -1;
            }

            /* Read in the body of the action. */
            bptr = (uint8 *) &mesg->draw_timestamp;
            for (i = 0; i < size; i++)
            {
                fscanf (where, "%02X", &value);
                bptr[i] = value;
            }

            /* Set the header to indicate we drew this action. */
            mesg->sender    = htonl(this_ip());
            mesg->timestamp = htonl(time_stamp());
            mesg->draw_timestamp = mesg->timestamp;
            mesg->pk_type   = htonl(PK_DRAW_MSG);
            mesg->page_ip   = htonl(this_ip());
            mesg->page_num  = pgnum + offset;

            mesg->draw_cmds_local = htonl(seqnum);
            mesg->draw_cmd_num    = htonl(seqnum);

            /* Add the action into the database. */
            ActionAdd (mesg);
        }
        else
        {
            break;
        }
    }

    fclose (where);
}


int Sender (ClientData clientData, Tcl_Interp *tcl, 
                              int argc, char *argv[])
{
    struct sender *ptr;
    struct writer *writer = NULL;
    char arg[SENDER_NAME_LENGTH + 16];
    int count = 0;
    int n;
    uint32 now;

    if (argc < 2)
    {
        sprintf (tcl->result, "Sender option ...");
        return TCL_ERROR;
    }

    if (strcmp (argv[1], "count") == 0)
    {
        /* Count the senders. */
        ptr = sender_head;
        while (ptr != NULL)
        {
            count++;
            ptr = ptr->next;
        }

        sprintf (tcl->result, "%d", count);
    }
    else if (strcmp (argv[1], "info") == 0)
    {
        /* Describe a sender. */
        if (argc < 3)
        {
            sprintf (tcl->result, "Sender info n");
            return TCL_ERROR;
        }
        n = -1;
        sscanf (argv[2], "%d", &n);
        if (n < 0)
        {
            sprintf (tcl->result, "");
            return TCL_OK;
        }

        /* Find the sender. */
        ptr = sender_head;
        count = 0;
        while (ptr != NULL)
        {
            if (count == n)
                break;

            count++;
            ptr = ptr->next;
        }

        /* Read out the data. */
        if ((count == n) && (ptr != NULL))
        {
            sprintf (arg, "%d.%d.%d.%d", 
                (ptr->sender_ip >> 24) & 0xff,
                (ptr->sender_ip >> 16) & 0xff,
                (ptr->sender_ip >> 8) & 0xff,
                (ptr->sender_ip & 0xff) );
            Tcl_AppendElement (tcl, arg);
            if (ptr->name[0] == '\0')
                sprintf (arg, "%8X", ptr->sender_ip);
            else
                sprintf (arg, "%s", ptr->name);
            Tcl_AppendElement (tcl, arg);
            sprintf (arg, "%lu0", (time_stamp() - ptr->last_sent));
            Tcl_AppendElement (tcl, arg);
            sprintf (arg, "%lu0", (ptr->last_seen - ptr->last_sent));
            Tcl_AppendElement (tcl, arg);
            sprintf (arg, "%lu0", (time_stamp() - ptr->last_drawn));
            Tcl_AppendElement (tcl, arg);
            if (PageCurrent() != NULL)
                writer = WriterFind (PageCurrent()->page_number,
                                     PageCurrent()->creator_ip, 
                                     ptr->sender_ip);
            if (writer == NULL)
                sprintf (arg, "0");
            else
                sprintf (arg, "%lu", writer->seq_num_max);
            Tcl_AppendElement (tcl, arg);
        }
        else
            sprintf (tcl->result, "");

    }
    else if (strcmp (argv[1], "active") == 0)
    {
        /* List the recently active senders. */
        if (argc < 3)
        {
            /* Active in the last second. */
            n = 100;
        }
        else
        {
            /* Active in the last n hundredths of a second. */
            n = -1;
            sscanf (argv[2], "%d", &n);
            if (n < 0)
                n = 100;
        }

        now = time_stamp();

        ptr = sender_head;
        while (ptr != NULL)
        {
            if ((now - ptr->last_seen) < ((uint32) n))
            {
                sprintf (arg, "%d", count);
                Tcl_AppendElement (tcl, arg);
            }

            count++;
            ptr = ptr->next;
        }
    }
    else if (strcmp (argv[1], "set") == 0)
    {
        /* Set descriptions of this wbd. */
        if (argc < 4)
        {
            sprintf (tcl->result, "Sender set option value");
            return TCL_ERROR;
        }

        if (strcmp (argv[2], "username") == 0)
        {
            /* Set the username. */
            if (strlen(argv[3]) > 0)
                ascii_source (argv[3]);
        }
        else if (strcmp (argv[2], "confname") == 0)
        {
            /* Not used anywhere. */
            ;
        }
        else
        {
            sprintf (tcl->result, "Sender set option value");
            return TCL_ERROR;
        }
    }

    return TCL_OK;
}



