/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#ifndef _LINEDIT_H
#define _LINEDIT_H

#include <wftk/font.h>
#include <wftk/widget.h>

namespace wftk {

/// LineEdit: a single-line text box for user input
/// 
/// Input size is limited graphically - when there isn't
/// enough room onscreen for the cursor, the input box is locked until
/// either enter or backspace is pressed.
///
/// Also plays the named sample "click" on keypress
class LineEdit : public Widget
{
 public:
  /// Constructor. 
  /// \param text Initial text for this LineEdit (defaults to empty string)
  /// \param font Initial font for this LineEdit (defaults to wftk-default)
  explicit LineEdit(const std::string& text = "", const Font& font = Font());
  ///
  virtual ~LineEdit();

  /// Return contents of this LineEdit
  const std::string& getText() {return text_;}
  /// Set text foreground color
  void setTextColor(const Color& fontCol);

  /// Set the contents of this LineEdit
  virtual void setText(const std::string&);
  /// Clear the contents of this LineEdit
  virtual void clearText() {setText("");}
  /// timer is used to blink the cursor
  virtual void timer();
  
  /// signal emitted when a key event is received within this widget
  SigC::Signal0<void> keyInput;
  /// signal emitted after the text has changed.
  SigC::Signal0<void> textChanged;
  /// signal emitted after enter is pressed
  SigC::Signal0<void> enterPressed;
  
  /// virtual function to handle keyboard events
  virtual bool keyEvent(const SDL_keysym&, bool pressed);
  /// virtual function to handle receipt of focus
  virtual void gainedFocus() {invalidate(textCursor_);}
  /// virtual function to handle loss of focus
  virtual void lostFocus() {invalidate(textCursor_);}

 protected:
  ///
  virtual void draw(Surface&, const Point&, const Region&);
  ///
  virtual void setPackingInfo();
  ///
  void textUpdate() {updateText(); invalidate();}
  ///
  std::string text_;
  ///
  Font textFont_;
 private: /* see .cpp file for documentation of these methods */
  /// 
  virtual Point drawText(Surface&, const Point&, const Region&);
  ///
  virtual void updateText();
  ///
  void drawCursor(Surface&, const Point&, const Region&);
  ///
  int cursorPos_;
  ///
  bool cursor_;
  ///
  bool drawCursor_;
  ///
  bool blockInput_;
  ///
  Rect textCursor_;
  ///
  int cursorCount_;
  ///
  Surface* textSurface_;
};

} // namespace wftk

#endif // _LINEDIT_H
