/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#include "rootwindow.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <sigc++/object_slot.h>

#include "debug.h"
#include "screensurface.h"
#include "application.h"
#include "pointer.h"

#include <assert.h>

#ifdef USE_SDL_OPENGL
  // shut up a redefinition warning
  #ifdef __WIN32__
  #undef NOMINMAX
  #endif
#include <SDL/SDL_opengl.h>
#endif

namespace wftk {

RootWindow* RootWindow::instance_ = 0;

RootWindow::RootWindow(int resX, int resY, int bpp, bool fullscreen,
	bool resizeable, const Surface& icon, Uint32 extra_flags, unsigned padding) :
  SingleContainer(padding, padding, padding, padding),
  iconified_(false),
  updateMouse_(true),
  resize_wait_(0)
{
  assert(!instance_);
  instance_ = this;

  Debug out(Debug::WIDGET_CREATE);

  // The logo must be set before the screen surface is created,
  // due to limitations in SDL

  out <<"setting window icon..."<<Debug::endl;
  if(!icon.empty()) //set the given surface as icon for our windows
    icon.setWMIcon();
  else {//otherwise put in a nice libwftk logo :)
    const Surface* logo = Surface::registry.find("wftk_logo");
    assert(logo);
    logo->setWMIcon();
  }
  out <<"ok."<<Debug::endl;

  out  << "creating RootWindow...";

  //grab the access to the rootwindow
  //the application may release it

  out << "Creating ScreenSurface" << Debug::endl;
  screen_ = new ScreenSurface(resX, resY, bpp, fullscreen, resizeable,
			extra_flags);
  out << "Resizing root window ScreenArea to match ScreenSurface" << Debug::endl;
  resize_ = screen_->rect();
  ScreenArea::resize(screen_->rect());

  out  << "done." << Debug::endl;

  // RootWindow needs an opaque background,
  // so invalidate()ing an area of the screen
  // will always result in whatever was there
  // before getting overdrawn
  setColor(Color(0, 0, 0));

	// Set up the GL context, if we are drawing to a GL window,
	// and if we have GL support enabled in configure.
#ifdef USE_SDL_OPENGL
	if (screen_->flags_ & SDL_OPENGL) {
		// Set up the context for drawing widgets
    glMatrixMode( GL_PROJECTION );
    glLoadIdentity();
    glOrtho(0, screen_->width(), screen_->height(), 0, -1.0, 1.0);
      
    glMatrixMode( GL_MODELVIEW );
    glLoadIdentity();
    glViewport(0,0, screen_->width(), screen_->height());
      
    glDrawBuffer(GL_BACK);
    glEnable(GL_DEPTH_TEST);
    glEnable(GL_TEXTURE_2D);
    glEnable(GL_BLEND);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);  
//    glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL);
    glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
  }	
#endif

  show();

  Application::instance()->draw.connect(SigC::slot(*this, &RootWindow::sync));
  Application::instance()->destroyed.connect(SigC::slot(*this, &RootWindow::destroy));
}

RootWindow::~RootWindow()
{
#ifdef USE_SDL_OPENGL
	// be nice and clean up the GL matrix stack
	if (screen_->flags_ & SDL_OPENGL) {
		// nothing to do here ... 
	  // (maybe free up loaded textures ?)

	}
#endif
  // do this first, in case any child (e.g., VideoWidget)
  // holds a pointer to the ScreenSurface
  removeChildren();

  delete screen_;

  assert(this == instance_);
  instance_ = 0;
}

int
RootWindow::modeAvailable(int resX, int resY, int bpp, bool fullscreen)
{ 
  Uint32 flags = SDL_ANYFORMAT | SDL_HWSURFACE | SDL_HWPALETTE;
 
  if(fullscreen)
    flags = flags | SDL_FULLSCREEN;

  return SDL_VideoModeOK(resX, resY, bpp, flags);
}

void RootWindow::sync()
{
  if(resize_wait_ && ! --resize_wait_) {
    screen_->mutex.grab();
    screen_->resize(resize_.w, resize_.h);
    screen_->mutex.release();
    ScreenArea::resize(resize_);
    resized.emit(resize_.w, resize_.h);
  }

  // put this here for people who have an old version
  // of the headers with the old version of updateMouse()
  if(updateMouse_) {
    mouseBuffer_.changed();
    updateMouse_ = false;
  }

  if(dirty()) {
    screen_->mutex.grab();
#ifdef USE_SDL_OPENGL
    if (screen_->flags_ & SDL_OPENGL) {
      invalidate();
    }
#endif
    blit(*screen_, Point(0, 0));
    screen_->mutex.release();
  }
  else
    mouseBuffer_.update(*screen_);

#ifdef USE_SDL_OPENGL
  // if we're using openGL, use this time to convert Surfaces to Textures.
  convertSurface();
#endif
}

void
RootWindow::setTitle(const std::string& title, const std::string& icon_name)
{
  SDL_WM_SetCaption(title.c_str(), icon_name.empty() ? 0 : icon_name.c_str());
}
 
bool 
RootWindow::handleEvent(const SDL_Event* event)
{
  Debug out(Debug::EVENTS);

  // This is where, for multiple root window support,
  // we would figure out which root window the event
  // should be handled by. At this point, it's easy.

  RootWindow* rw = instance();
  if(!rw)
    return false;

  switch(event->type) {
    case SDL_ACTIVEEVENT:
      assert(event->active.state & SDL_APPACTIVE);
      rw->iconified_ = !event->active.gain;
      if(rw->iconified_) {
        rw->hide();
        rw->iconified.emit();
      }
      else {
        rw->show();
        rw->restored.emit();
      }
      return true;
    case SDL_VIDEOEXPOSE: 
      rw->invalidate();
      return true;
    case SDL_VIDEORESIZE:
      out << "Got a resize event" << Debug::endl;
      rw->resize(event->resize.w, event->resize.h);
      return true;
    default:
      // unhandled event
      return false;
  }
}

void RootWindow::resize(Uint16 w, Uint16 h)
{
  if(!screen_->resizeable())
    return;
  // queue widget resize
  resize_ = Rect(0, 0, w, h);
  // !$%#%!$# SDL bug, and even this doesn't always fix it
  resize_wait_ = 2;
}

void RootWindow::resize()
{
  const PackingInfo& info = getPackingInfo();
  resize(info.x.pref > 0 ? info.x.pref : 1, info.y.pref > 0 ? info.y.pref : 1);
}

bool RootWindow::fullscreen() const
{
  return screen_->fullscreen();
}

void RootWindow::drawAfter(Surface& target, const Point& offset, const Region& r)
{
  assert(&target == screen_);
  assert(offset == Point(0, 0));

  Region to_update = r;
  to_update |= mouseBuffer_.update(*screen_, r);

#ifdef USE_SDL_OPENGL
  if (screen_->flags_ & SDL_OPENGL) {
    SDL_GL_SwapBuffers();
    return;
  }
#endif
  screen_->update(to_update);
}

void RootWindow::MouseBuffer::update(ScreenSurface& screen)
{
  if(!changed_)
    return;

  // This function is called when we don't need
  // to do any other drawing, so we need to take
  // care of locking the screen mutex ourselves.
  screen.mutex.grab();

  screen.update(update(screen, Region()));

  screen.mutex.release();
}

Region RootWindow::MouseBuffer::update(ScreenSurface& screen, const Region& dirty)
{
  const SoftPointer* new_ptr;

  if(!changed_)
    new_ptr = ptr_;
  else if(Mouse::instance()->hidden())
    new_ptr = 0;
  else
    new_ptr = dynamic_cast<const SoftPointer*>(Mouse::instance()->getPointer());

  if(!ptr_ && !new_ptr)
    return Region(); // hardware pointer or hidden, nothing to do

  // This function is called from RootWindow::drawAfter(),
  // inside the drawing code, so the screen's mutex is
  // already locked.

  // update the buffer and redraw the mouse in the dirty area
  if(!changed_) {
    assert(ptr_);
    Region dirty_copy(dirty);
    dirty_copy.offset(-screen_loc_.origin());
    screen.blit(buffer_, -screen_loc_.origin(), dirty_copy);
    ptr_->image().blit(screen, screen_loc_.origin(), dirty);
    return Region(); // don't need to update for mouse's sake, nothing changed
  }

  // do the full redraw

  Region updated;

  if(ptr_) { // only blit the buffer to non-dirty areas
    assert(!buffer_.empty());
    updated = screen_loc_ - dirty;
    buffer_.blit(screen, screen_loc_.origin(), updated);
  }

  ptr_ = new_ptr;

  if(ptr_) {
    const Surface& image = ptr_->image();

    Point target = Mouse::instance()->position() - ptr_->hotspot();
    screen_loc_ = Rect(target.x, target.y, image.width(), image.height());

    checkBufferSize(image, screen.pixelformat());

    screen.blit(buffer_, Rect(0, 0, 0, 0), screen_loc_);
    image.blit(screen, screen_loc_);
    updated |= screen_loc_;
  }

  changed_ = false;

  return updated;
}

void RootWindow::MouseBuffer::checkBufferSize(const Surface& image,
	const Pixelformat& format)
{
  bool need_resize = buffer_.empty();
  unsigned width, height;

  // grow to hold the largest of any mouse pointer used

  if(buffer_.width() < image.width()) {
    width = image.width();
    need_resize = true;
  }
  else
    width = buffer_.width();

  if(buffer_.height() < image.height()) {
    height = image.height();
    need_resize = true;
  }
  else
    height = buffer_.height();

  if(need_resize)
    buffer_.setSurface(width, height, format);
}

void RootWindow::convertSurface() {
#ifdef USE_SDL_OPENGL
	// one at a time
	std::set<Surface *>::iterator i = surfaces_.begin();
	if (i != surfaces_.end()) {
		Surface * surf = (*i);
		surf->makeGLTexture();
		surfaces_.erase(i);
	}
#endif
}

void RootWindow::addSurface(Surface * surf) {
#ifdef USE_SDL_OPENGL
	surfaces_.insert(surf);
#endif
}

} // namespace wftk
