/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux 

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#include "timer.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <SDL/SDL.h>
#include <limits.h>
#include <cassert>
#include <sigc++/hide.h>

#include "application.h" // to check for SDL initialization

namespace wftk {

std::set<wftk::Timer*> wftk::Timer::runningTimers_;
wftk::Time wftk::Timer::needUpdateBy_;

// This works as long as now() is called at least once per epoch.
// Since it's called from the application mainloop, this should
// be fine, as SDL will be polled more often than that.
Time
Time::now()
{
  static Uint32 old_ticks = 0;
  static Sint32 epoch = 0;

  if(!Application::instance()) // SDL not initialized yet
    return Time(0, 0);

  Uint32 ticks = SDL_GetTicks();
  if(ticks < old_ticks) // rollover
    ++epoch;
  old_ticks = ticks;

  return Time(epoch, ticks);
}

template<class C>
static inline Sint32
add_epoch_offset(C& val, Sint32 modulo, Sint32 amount)
{
  assert(modulo > 0);

  Sint32 tmp = (Sint32) val + amount;
  Sint32 tmp2 = tmp % modulo;
  if(tmp2 < 0)
    tmp2 += modulo;
  val = tmp2;
  return (tmp - tmp2) / modulo;
}

Time::Info
Time::info() const
{
  const Info epoch_time = {49, 17, 2, 47, 296};

  Info out;

  Uint32 running = ticks_;

  out.msecs = running % 1000;
  running /= 1000;
  out.seconds = running % 60;
  running /= 60;
  out.minutes = running % 60;
  running /= 60;
  out.hours = running % 24;
  running /= 24;
  out.days = running;

  // now add the epoch and handle rollover

  // this can't handle epoch_ > (2^31 - 1000)/ 296 == 7,225,008, or 
  // slightly less than a million years

  Sint32 tmp;
  tmp = add_epoch_offset(out.msecs, 1000,
	epoch_ * (Sint32) epoch_time.msecs);
  tmp = add_epoch_offset(out.seconds, 60,
	tmp + epoch_ * (Sint32) epoch_time.seconds);
  tmp = add_epoch_offset(out.minutes, 60,
	tmp + epoch_ * (Sint32) epoch_time.minutes);
  out.days += add_epoch_offset(out.hours, 24,
	tmp + epoch_ * (Sint32) epoch_time.hours);

  return out;
}

// essentially, this returns the 'ticks' part
// of the difference, dropping the high 'epoch'
// bits after subtraction
int
Time::operator-(const Time& t) const
{
  // assume int not larger than 32 bit FIXME
  const Uint32 max_val = 0xffffffff; // 2^32 - 1
  Uint32 diff;
  bool negative = false;
  switch(epoch_ - t.epoch_) {
    case 1:
      assert(ticks_ < t.ticks_);
      diff = ticks_ + (max_val - t.ticks_) + 1;
      negative = false;
      break;
    case 0:
      if(ticks_ >= t.ticks_) {
        diff = ticks_ - t.ticks_;
        negative = false;
      }
      else {
        diff = t.ticks_ - ticks_;
        negative = true;
      }
      break;
    case -1:
      assert(ticks_ > t.ticks_);
      diff = t.ticks_ + (max_val - ticks_) + 1;
      negative = true;
      break;
    default:
      assert(false);
  }

  // assume UINT_MAX >= INT_MAX, UINT_MAX >= -INT_MIN, which
  // should be valid on all platforms

#if (INT_MAX + INT_MIN) >= 0
  // -INT_MIN is a valid int
  const unsigned minus_int_min = -INT_MIN;
#else
  const unsigned minus_int_min = (unsigned)(-(INT_MAX + INT_MIN)) + (unsigned) INT_MAX;
#endif

  if(negative) {
    assert(diff <= minus_int_min);
#if (INT_MAX + INT_MIN) >= 0
    return -(int)diff;
#else
    if(diff <= (unsigned) INT_MAX)
      return -(int)diff;
    else
      return -(int)(diff - (unsigned) INT_MAX) - INT_MAX;
#endif
  }
  else {
    assert(diff <= INT_MAX);
    return diff;
  }
}

Time&
Time::operator+=(Uint32 val)
{
  if(val == 0) // problems calculating complement
    return *this;

  Uint32 complement = ~val + 1; // == 2^32 - val

  if(complement <= ticks_) {
    ++epoch_;
    ticks_ -= complement; // or, ticks_ += (val - 2^32)
  }
  else
    ticks_ += val;

  return *this;
}

Time&
Time::operator-=(Uint32 val)
{
  if(val <= ticks_)
    ticks_ -= val;
  else {
    Uint32 complement = ~val + 1; // == 2^32 - val
    --epoch_;
    ticks_ += complement; // or, ticks_ = 2^32 - (val - ticks_)
  }

  return *this;
}

unsigned
Time::negative_rectify(int val)
{
  const int int_offset = INT_MAX + INT_MIN;

  assert(val < 0);
  assert(sizeof(unsigned) >= sizeof(int)); // should be same size on all platforms

  if(int_offset >= 0) // -INT_MIN is a valid int
    return (unsigned) -val;

  if(val >= int_offset) // val is small, offsetting would push it positive
    return (unsigned) -val;

  return ((unsigned) -(val - int_offset)) + (unsigned) (-int_offset);
}

Timer::Timer(Uint32 mseconds, bool running) :
  running_(false),
  deltaT_(mseconds),
  currentEvent_(0)
{
  if(running)
    run();
}

Timer::~Timer()
{
  halt(); // pull out of runningTimers_, if necessary

  // don't emit a callback, we're being destroyed
  if(currentEvent_)
    currentEvent_->clear();
}

void
Timer::update(const Time& now)
{
  assert(running_);

  if(nextEvent_ <= now) {
    // overage is how much we went past the time we should have set off the timer
    unsigned overage = now - nextEvent_;
    Application::instance()->pushEvent(
      new Event(*this, overage + deltaT_));
    nextEvent_ = now + deltaT_;
  }
}

void
Timer::setNeedUpdateBy(bool force)
{
  if(force || nextEvent_ < needUpdateBy_)
    needUpdateBy_ = nextEvent_;
}

void
Timer::setInterval(unsigned int delta)
{
  deltaT_ = delta;

  setNeedUpdateBy();
}

void
Timer::run()
{
  if(running_)
    return;

  //enable timer and reset time
  running_ = true;
  nextEvent_ = Time::now() + deltaT_;

  // force the update if there were no running timers before
  setNeedUpdateBy(runningTimers_.empty());

  bool success = runningTimers_.insert(this).second;
  assert(success);
}

void
Timer::halt()
{
  if(!running_)
    return;

  running_ = false;

  bool success = runningTimers_.erase(this);
  assert(success);
}

void
Timer::processAllTimers()
{
  // Do this even if runningTimers_ is empty, to make
  // sure we keep bumping the epoch in Time::now().
  Time current = Time::now();

  if(runningTimers_.empty() || current < needUpdateBy_)
    return;

  std::set<Timer*>::iterator I = runningTimers_.begin();
  bool force = true; // for the first update of needUpdateBy_

  // and check if we need to set needUpdateBy_ with the rest
  while(I != runningTimers_.end()) {
    (*I)->update(current);
    (*I++)->setNeedUpdateBy(force);
    force = false;
  }
}

Uint32
Timer::limitWait(Uint32 max)
{
  if(runningTimers_.empty())
    return max;

  Time current = Time::now();

  if(current >= needUpdateBy_)
    return 0;

  if(current + max <= needUpdateBy_)
    return max;

  return needUpdateBy_ - current;
}

void
Timer::Event::operator()()
{
  // Check for data() every time,
  // in case it's destroyed in the callback

  if(data())
    data()->alarm(timePassed_);
}

SigC::Connection
Timer::Alarm::connect(const SigC::Slot0<void>& slot)
{
  return SigC::Signal1<void,unsigned int>::connect(SigC::hide<unsigned int>(slot));
}

SigC::Connection
Timer::Alarm::connect(const SigC::Slot1<void,unsigned int>& slot)
{
  return SigC::Signal1<void,unsigned int>::connect(slot);
}

} // namespace
