/* vim: set noet ts=4:
 * $Id: wmwork.c,v 1.27 2002/03/30 16:42:52 godisch Exp $
 *
 * Copyright (c) 2002 Martin A. Godisch <godisch@tcs.inf.tu-dresden.de>
 */

#include <ctype.h>
#include <errno.h>
#include <getopt.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/timeb.h>
#include <X11/xpm.h>

#include "wmgeneral.h"
#include "wmwork.h"
#include "wmwork.xpm"

#define PROGRAM_NAME     "wmwork"
#define PROGRAM_VERSION  "0.1.0"
#define DEFAULT_LOG_FILE ".wmworklog"

#define BUT_START (1)
#define BUT_PAUSE (2)
#define BUT_STOP  (3)
#define BUT_PREV  (4)
#define BUT_NEXT  (5)

#define CHAR_SRC_X1(N) ((N - 'A') * 5)
#define CHAR_SRC_Y1    (71)
#define CHAR_SRC_X2(N) ((N - ' ') * 5)
#define CHAR_SRC_Y2    (64)
#define CHAR_SRC_X3(N) ((N - '{') * 5 + 65)
#define CHAR_SRC_Y3    (57)
#define TIMER_SRC_X(N) (N * 5 + 80)
#define TIMER_SRC_Y    (64)

#define MAX_STRING (256)

char wmwork_mask_bits[64*64];
char logname[MAX_STRING];
struct timeb now;
struct Project
	*current = NULL,
	*first   = NULL;
long sess_time = 0;

int main(int argc, char *argv[])
{
	int n,
		show_days = 0,
		but_stat  = 0,
		millitm   = 0,
		running   = 0;
	long
		last_time = 0;
	static int signals[] =
		{SIGALRM, SIGHUP, SIGINT, SIGPIPE, SIGPOLL, SIGTERM, SIGUSR1, SIGUSR2, 0};
	XEvent Event;

	do_opts(argc, argv, &show_days);

	for (n = 0; signals[n]; n++) {
		if (signal(signals[n], handler) == SIG_ERR) {
			fprintf(stderr, "%s: cannot set handler for signal %d\n", PROGRAM_NAME, signals[n]);
			perror("signal");
		}
	}

	if (strlen(getenv("HOME")) < MAX_STRING - strlen(DEFAULT_LOG_FILE) - 5) {
		strcpy(logname, getenv("HOME"));
		strcat(logname, "/");
		strcat(logname, DEFAULT_LOG_FILE);
	} else {
		fprintf(stderr, "%s: $HOME is too long, using logfiles %s* in current directory\n", PROGRAM_NAME, DEFAULT_LOG_FILE);
		strcpy(logname, DEFAULT_LOG_FILE);
	}
	read_log();
	current = first;

	createXBMfromXPM(wmwork_mask_bits, wmwork_master_xpm, 64, 64);
	openXwindow(argc, argv, wmwork_master_xpm, wmwork_mask_bits, 64, 64);
	AddMouseRegion(BUT_START,  5, 48, 22, 58);
	AddMouseRegion(BUT_PAUSE, 23, 48, 40, 58);
	AddMouseRegion(BUT_STOP,  41, 48, 58, 58);
	AddMouseRegion(BUT_PREV,   5, 33, 16, 43);
	AddMouseRegion(BUT_NEXT,  47, 33, 58, 43);
	drawTime(current->time, sess_time, millitm, show_days);
	drawProject(current->name);

	while (1) {
		last_time = now.time;
		ftime(&now);
		if (running) {
			current->time += now.time - last_time;
			sess_time     += now.time - last_time;
			millitm        = now.millitm;
			drawTime(current->time, sess_time, millitm, show_days);
			RedrawWindow();
		}
		while (XPending(display)) {
			XNextEvent(display, &Event);
			switch (Event.type) {
			case Expose:
				RedrawWindow();
				break;
			case DestroyNotify:
				XCloseDisplay(display);
				exit(0);
			case ButtonPress:
				n = CheckMouseRegion(Event.xbutton.x, Event.xbutton.y);
				switch (n) {
				case BUT_START:
				case BUT_PAUSE:
				case BUT_STOP:
				case BUT_PREV:
				case BUT_NEXT:
					ButtonDown(n);
					break;
				}
				but_stat = n;
				RedrawWindow();
				break;
			case ButtonRelease:
				n = CheckMouseRegion(Event.xbutton.x, Event.xbutton.y);
				switch (but_stat) {
				case BUT_START:
				case BUT_PAUSE:
				case BUT_STOP:
				case BUT_PREV:
				case BUT_NEXT:
					ButtonUp(but_stat);
					break;
				}
				if (but_stat && n == but_stat) {
					switch (but_stat) {
					case BUT_START:
						running = 1;
						break;
					case BUT_PAUSE:
						running = 0;
						break;
					case BUT_STOP:
						write_log();
						write_record();
						running   = 0;
						sess_time = 0;
						break;
					case BUT_PREV:
						if (! running && sess_time == 0)
							current = current->prev;
						break;
					case BUT_NEXT:
						if (! running && sess_time == 0)
							current = current->next;
						break;
					}
					drawTime(current->time, sess_time, millitm, show_days);
					drawProject(current->name);
				}
				RedrawWindow();
				but_stat = 0;
				break;
			}
		}
		usleep(50000L);
	}

}

void ButtonDown(int button)
{
	switch (button) {
	case BUT_START:
		copyXPMArea( 65, 25, 18, 11,  5, 48);
		break;
	case BUT_PAUSE:
		copyXPMArea( 84, 25, 18, 11, 23, 48);
		break;
	case BUT_STOP:
		copyXPMArea(103, 25, 18, 11, 41, 48);
		break;
	case BUT_PREV:
		copyXPMArea(122, 25, 12, 11,  5, 33);
		break;
	case BUT_NEXT:
		copyXPMArea(135, 25, 12, 11, 47, 33);
		break;
	}
}

void ButtonUp(int button)
{
	switch (button) {
	case BUT_START:
		copyXPMArea( 65, 13, 18, 11,  5, 48);
		break;
	case BUT_PAUSE:
		copyXPMArea( 84, 13, 18, 11, 23, 48);
		break;
	case BUT_STOP:
		copyXPMArea(103, 13, 18, 11, 41, 48);
		break;
	case BUT_PREV:
		copyXPMArea(122, 13, 12, 11,  5, 33);
		break;
	case BUT_NEXT:
		copyXPMArea(135, 13, 12, 11, 47, 33);
		break;
	}
}

void ButtonEnable(int button)
{
	ButtonUp(button);
}

void ButtonDisable(int button)
{
	switch (button) {
	case BUT_START:
		copyXPMArea( 65,  1, 18, 11,  5, 48);
		break;
	case BUT_PAUSE:
		copyXPMArea( 84,  1, 18, 11, 23, 48);
		break;
	case BUT_STOP:
		copyXPMArea(103,  1, 18, 11, 41, 48);
		break;
	case BUT_PREV:
		copyXPMArea(122,  1, 12, 11,  5, 33);
		break;
	case BUT_NEXT:
		copyXPMArea(135,  1, 12, 11, 47, 33);
		break;
	}
}

void do_opts(int argc, char *argv[], int *show_days)
{
	static struct option long_opts[] = {
		{"days",    0, NULL, 'd'},
		{"help",    0, NULL, 'h'},
		{"version", 0, NULL, 'v'},
		{NULL,      0, NULL, 0  }};
	int i, opt_index = 0;

	while (1) {
		i = getopt_long(argc, argv, "dhv", long_opts, &opt_index);
		if (i == -1)
			break;
		switch (i) {
		case 'd':
			*show_days = 1;
			break;
		case 'h':
			printf("Usage: %s [-d|-h|-v]\n", argv[0]);
			printf("  -d, --days     displays time in ddd:hh:mm instead of hhh:mm:ss,\n");
			printf("  -h, --help     displays this command line summary,\n");
			printf("  -v, --version  displays the version number.\n");
			exit(0);
		case 'v':
			printf("%s version %s\n", PROGRAM_NAME, PROGRAM_VERSION);
			exit(0);
		case '?':
			exit(1);
		}
	}
}

void drawTime(long time1, long time2, int millitm, int show_days)
{
	long  d1 = 0, d2 = 0, h1 = 0, h2 = 0;
	short m1 = 0, m2 = 0, s1 = 0, s2 = 0;

	if (time1 >= 3600000 || time2 >= 3600000)
		show_days = 1;

	if (show_days) {
		d1 = time1 / 86400;
		d2 = time2 / 86400;
		time1 %=     86400;
		time2 %=     86400;
		if (d1 >= 1000 || d2 >= 1000) {
			d1 %= 1000;
			d2 %= 1000;
		}
	}
	h1 = time1 / 3600;
	h2 = time2 / 3600;
	time1 %=     3600;
	time2 %=     3600;
	m1 = time1 /   60;
	m2 = time2 /   60;
	s1 = time1 %   60;
	s2 = time2 %   60;

	if (show_days) {
		copyXPMArea(TIMER_SRC_X(d1 / 100), TIMER_SRC_Y, 5, 7,  8,  6);
		d1 %= 100;
		copyXPMArea(TIMER_SRC_X(d1 /  10), TIMER_SRC_Y, 5, 7, 14,  6);
		copyXPMArea(TIMER_SRC_X(d1 %  10), TIMER_SRC_Y, 5, 7, 20,  6);
		copyXPMArea(TIMER_SRC_X(h1 /  10), TIMER_SRC_Y, 5, 7, 30,  6);
		copyXPMArea(TIMER_SRC_X(h1 %  10), TIMER_SRC_Y, 5, 7, 36,  6);
		copyXPMArea(TIMER_SRC_X(m1 /  10), TIMER_SRC_Y, 5, 7, 46,  6);
		copyXPMArea(TIMER_SRC_X(m1 %  10), TIMER_SRC_Y, 5, 7, 52,  6);

		copyXPMArea(TIMER_SRC_X(d2 / 100), TIMER_SRC_Y, 5, 7,  8, 20);
		d2 %= 100;
		copyXPMArea(TIMER_SRC_X(d2 /  10), TIMER_SRC_Y, 5, 7, 14, 20);
		copyXPMArea(TIMER_SRC_X(d2 %  10), TIMER_SRC_Y, 5, 7, 20, 20);
		copyXPMArea(TIMER_SRC_X(h2 /  10), TIMER_SRC_Y, 5, 7, 30, 20);
		copyXPMArea(TIMER_SRC_X(h2 %  10), TIMER_SRC_Y, 5, 7, 36, 20);
		copyXPMArea(TIMER_SRC_X(m2 /  10), TIMER_SRC_Y, 5, 7, 46, 20);
		copyXPMArea(TIMER_SRC_X(m2 %  10), TIMER_SRC_Y, 5, 7, 52, 20);
	} else {
		copyXPMArea(TIMER_SRC_X(h1 / 100), TIMER_SRC_Y, 5, 7,  8,  6);
		h1 %= 100;
		copyXPMArea(TIMER_SRC_X(h1 /  10), TIMER_SRC_Y, 5, 7, 14,  6);
		copyXPMArea(TIMER_SRC_X(h1 %  10), TIMER_SRC_Y, 5, 7, 20,  6);
		copyXPMArea(TIMER_SRC_X(m1 /  10), TIMER_SRC_Y, 5, 7, 30,  6);
		copyXPMArea(TIMER_SRC_X(m1 %  10), TIMER_SRC_Y, 5, 7, 36,  6);
		copyXPMArea(TIMER_SRC_X(s1 /  10), TIMER_SRC_Y, 5, 7, 46,  6);
		copyXPMArea(TIMER_SRC_X(s1 %  10), TIMER_SRC_Y, 5, 7, 52,  6);

		copyXPMArea(TIMER_SRC_X(h2 / 100), TIMER_SRC_Y, 5, 7,  8, 20);
		h2 %= 100;
		copyXPMArea(TIMER_SRC_X(h2 /  10), TIMER_SRC_Y, 5, 7, 14, 20);
		copyXPMArea(TIMER_SRC_X(h2 %  10), TIMER_SRC_Y, 5, 7, 20, 20);
		copyXPMArea(TIMER_SRC_X(m2 /  10), TIMER_SRC_Y, 5, 7, 30, 20);
		copyXPMArea(TIMER_SRC_X(m2 %  10), TIMER_SRC_Y, 5, 7, 36, 20);
		copyXPMArea(TIMER_SRC_X(s2 /  10), TIMER_SRC_Y, 5, 7, 46, 20);
		copyXPMArea(TIMER_SRC_X(s2 %  10), TIMER_SRC_Y, 5, 7, 52, 20);
	}

	if (millitm < 500) {
		copyXPMArea(161, CHAR_SRC_Y1, 1, 7, 27,  6);
		copyXPMArea(161, CHAR_SRC_Y1, 1, 7, 27, 20);
		copyXPMArea(161, CHAR_SRC_Y1, 1, 7, 43,  6);
		copyXPMArea(161, CHAR_SRC_Y1, 1, 7, 43, 20);
	} else {
		copyXPMArea(163, CHAR_SRC_Y1, 1, 7, 27,  6);
		copyXPMArea(163, CHAR_SRC_Y1, 1, 7, 43,  6);
		copyXPMArea(163, CHAR_SRC_Y1, 1, 7, 27, 20);
		copyXPMArea(163, CHAR_SRC_Y1, 1, 7, 43, 20);
	}
}

void drawProject(const char *name)
{
	int i;

	for (i = 0; i < 3; i++) {
		copyXPMArea(CHAR_SRC_X1(' '), CHAR_SRC_Y1, 5, 7, 24 + i * 6, 35);
		if (i >= strlen(name))
			continue;
		if (name[i] >= 'A' && name[i] <= '`')
			copyXPMArea(CHAR_SRC_X1(name[i]), CHAR_SRC_Y1, 5, 7, 24 + i * 6, 35);
		else if (name[i] >= ' ' && name[i] <= '@')
			copyXPMArea(CHAR_SRC_X2(name[i]), CHAR_SRC_Y2, 5, 7, 24 + i * 6, 35);
		else if (name[i] >= '{' && name[i] <= '~')
			copyXPMArea(CHAR_SRC_X3(name[i]), CHAR_SRC_Y3, 5, 7, 24 + i * 6, 35);
		else
			copyXPMArea(CHAR_SRC_X2(' '),     CHAR_SRC_Y2, 5, 7, 24 + i * 6, 35);
	}
}

static void handler(int signo)
{
	write_log();
	write_record();
	exit(0);
}

int read_log(void)
{
	struct Project *p;
	FILE *F;
	char buffer[MAX_STRING], *colon, *s;
	int  i, line, n = 0;

	if (!(first = malloc(sizeof(struct Project)))) {
		fprintf(stderr, "%s: cannot allocate memory for element %d\n", PROGRAM_NAME, n);
		perror("malloc");
		exit(1);
	}
	strcpy(first->name, "NN\0");
	first->time = 0;
	first->prev = first;
	first->next = first;

	F = fopen(logname, "r");
	if (!F) {
		if (errno != ENOENT) {
			fprintf(stderr, "%s: cannot open logfile %s\n", PROGRAM_NAME, logname);
			perror("fopen");
		}
		return(1);
	}

	for (line = 1;; line++) {
		fgets(buffer, MAX_STRING, F);
		if (feof(F))
			break;

		if (n == 0) {
			p = first;
		} else if (!(p = malloc(sizeof(struct Project)))) {
			fprintf(stderr, "%s: cannot allocate memory for element %d\n", PROGRAM_NAME, n);
			perror("malloc");
			return(n);
		}

		if ((s = strchr(buffer, '\n')))
			*s = '\0';
		if (buffer[0] == '#' || buffer[0] == '\0') {
			if (n)
				free(p);
			continue;
		}
		colon = strchr(buffer, ':');
		if (!colon) {
			fprintf(stderr, "%s: missing `:' in %s line %d\n", PROGRAM_NAME, logname, line);
			if (n)
				free(p);
			continue;
		}
		i = colon - buffer < 3 ? colon - buffer : 3;
		p->name[i] = '\0';
		for (i--; i >= 0; i--) {
			p->name[i] = toupper(buffer[i]);
			if (p->name[i] == '/') {
				fprintf(stderr, "%s: `/' is fs delimiter in %s line %d\n", PROGRAM_NAME, logname, line);
				fprintf(stderr, "%s: converted forbidden `/' in project id into `|'\n", PROGRAM_NAME);
				p->name[i] = '|';
			}
		}
		p->time = strtol(++colon, &s, 10);
		if (*s && *s != ':' && *s != ' ') {
			fprintf(stderr, "%s: error converting timestamp `%s' in %s line %d\n", PROGRAM_NAME, colon, logname, line);
			if (n)
				free(p);
			continue;
		}

		p->prev = first->prev;
		p->next = first;
		p->prev->next = p;
		p->next->prev = p;
		n++;
	}

	fclose(F);
	return(n);
}

void write_log(void)
{
	struct Project *p;
	FILE   *F;

	umask(0077);
	F = fopen(logname, "w");
	if (!F) {
		fprintf(stderr, "%s: cannot open logfile %s\n", PROGRAM_NAME, logname);
		perror("fopen");
		return;
	}

	fprintf(F, "# wmwork configuration file\n");
	fprintf(F, "# do not edit while wmwork is running\n\n");
	p = first;
	do {
		fprintf(F, "%s:%li\n", p->name, p->time);
		p = p->next;
	} while (p != first);

	fclose(F);
}

void write_record(void)
{
	char filename[MAX_STRING + 4];
	char t_buffer[64];
	FILE *F;

	if (sess_time == 0)
		return;

	strcpy(filename, logname);
	strcat(filename, ".");
	strcat(filename, current->name);

	umask(0077);
	F = fopen(filename, "a");
	if (!F) {
		fprintf(stderr, "%s: cannot open logfile %s\n", PROGRAM_NAME, filename);
		perror("fopen");
		return;
	}
	strftime(t_buffer, 64, "%a %b %d %H:%M:%S %Z %Y", localtime(&now.time));
	fprintf(F, "%s %s %03li:%02li:%02li\n", t_buffer, current->name, sess_time / 3600, sess_time / 60 % 60, sess_time % 60);
	fclose(F);
}
