/* compregex.cc
 * This file belongs to Worker, a filemanager for UNIX/X11.
 * Copyright (C) 2004 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/* $Id: compregex.cc,v 1.2 2004/10/27 21:06:16 ralf Exp $ */

#include "compregex.h"
#include <aguix/lowlevelfunc.h>

CompRegEx::CompRegEx()
{
#ifdef HAVE_REGEX
  maxsize = 1024;
#endif
}

CompRegEx::~CompRegEx()
{
#ifdef HAVE_REGEX
  freeCompiled();
#endif
}

#ifdef HAVE_REGEX

void CompRegEx::freeCompiled()
{
  while ( compiled.size() > 0 ) {
    freeLast();
  }
  compiled.clear();
}

const struct CompRegEx::regExpComp_t &CompRegEx::getCompiled( const char *pattern )
{
  int erg;
  regExpComp_t *newre, **pe1;
  
  if ( pattern == NULL ) throw 1;

  pe1 = compiled.findToFront( pattern );
  if ( pe1 != NULL ) {
    return **pe1;
  }

  newre = new regExpComp_t;
  erg = regcomp( &(newre->patternReg), pattern, REG_NOSUB );
  if ( erg != 0 ) {
    // error
    throw 2;
  }
  newre->pattern = dupstring( pattern );
  
  if ( compiled.size() >= maxsize ) {
    freeLast();
  }
  compiled.insert( pattern, newre );
  return *newre;
}

void CompRegEx::freeLast()
{
  regExpComp_t **pe1, *e1;

  pe1 = compiled.getLast();
  if ( pe1 != NULL ) {
    e1 = *pe1;
    regfree( &(e1->patternReg) );
    _freesafe( e1->pattern );
    delete e1;
    compiled.removeLast();
  }
}

#endif

bool CompRegEx::match( const char *pattern, const char *str )
{
  bool matched = false;
#ifdef HAVE_REGEX
  int erg;
#endif

  if ( pattern == NULL ) return false;
  if ( strlen( pattern ) < 1 ) return false;
  if ( str == NULL ) return false;
  if ( strlen( str ) < 1 ) return false;
  
#ifdef HAVE_REGEX
  try {
    const struct regExpComp_t &x = getCompiled( pattern );
    erg = regexec( &(x.patternReg), str, 0, NULL, 0 );
    if ( erg == 0 ) matched = true;
  } catch( int err ) {
  }
#else
  if ( fnmatch( pattern, str, 0 ) == 0 ) matched = true;
#endif
  return matched;
}

