/*
 *   Copyright (C) 2003 by Jonathan Naylor G4KLX
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "JT6MLookups.h"
#include "JT6MDefs.h"
#include "JT6MFoldMessages.h"

#include "common/Exception.h"

CJT6MFoldMessages::CJT6MFoldMessages(int n) :
m_n(n),
m_pos(0),
m_alphabet(NULL)
{
	wxASSERT(n > 0 && (n % 2) == 0);

	m_alphabet = new CAverage[m_n * JT6M_ALPHABET_COUNT];
}

CJT6MFoldMessages::~CJT6MFoldMessages()
{
	delete[] m_alphabet;
}

void CJT6MFoldMessages::addBins(int pos, double* bins)
{
	wxASSERT(bins != NULL);
	wxASSERT(pos >= 0);

	if ((pos % 3) == 0)	// Sync data
		throw CException(wxT("pos % 3 == 0 !"));

	for (int i = 0; i < JT6M_ALPHABET_COUNT; i++) {
		int index = getIndex(m_pos, i);

		m_alphabet[index].addValue(bins[i]);
	}

	m_pos = (m_pos + 1) % m_n;
}

int CJT6MFoldMessages::getLength() const
{
	return m_n;
}

double CJT6MFoldMessages::getQuality() const
{
	CAverage total;
	CAverage best;

	for (int pos = 0; pos < m_n; pos++) {

		double bestVal = 0.0;

		for (int i = 0; i < JT6M_ALPHABET_COUNT; i++) {
			int index = getIndex(pos, i);

			double val = m_alphabet[index].getAverage();

			total.addValue(val);

			if (val > bestVal)
				bestVal = val;
		}

		best.addValue(bestVal);
	}

	return best.getAverage() / total.getAverage();
}

wxString CJT6MFoldMessages::getMessage() const
{
	wxString message;

	CJT6MLookups lookups;

	for (int pos = 0; pos < m_n; pos++) {
		int bestLetter = 0;
		int      index = getIndex(pos, 0);
		double bestVal = m_alphabet[index].getAverage();

		for (int letter = 1; letter < JT6M_ALPHABET_COUNT; letter++) {
			int  index = getIndex(pos, letter);
			double val = m_alphabet[index].getAverage();

			if (val > bestVal) {
				bestVal    = val;
				bestLetter = letter;
			}
		}

		wxChar c = lookups.lookupTone(bestLetter + 1);

		message.Append(c);
	}

	// Rotate the message until the text ends in a space
	int pos = message.Find(wxT(' '));
	if (pos != -1) {
		wxString first = message.BeforeFirst(wxT(' '));
		wxString  last = message.AfterFirst(wxT(' '));

		message.Clear();
		message.Append(last);
		message.Append(first);
	}

	return message;
}

int CJT6MFoldMessages::getIndex(int pos, int binNo) const
{
	wxASSERT(pos >= 0 && pos < m_n);
	wxASSERT(binNo >= 0 && binNo < JT6M_ALPHABET_COUNT);

	return pos * JT6M_ALPHABET_COUNT + binNo;
}
