/*  -*- C++ -*-
 *
 * Copyright (C) 2000 Frans Kaashoek (kaashoek@mit.edu)
 *                    Joshua MacDonald (jmacd@cs.berkeley.edu)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 *
 */

#include "xdfssrv.h"
#include "rxx.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <fcntl.h>
#include <assert.h>
#include <sys/time.h>

MESSAGE_TYPE  XNFS_ERROR_MESSAGE    ("XNFS", MCLASS_ERROR);

// @@ Not updating mtime, but don't care about atime

#ifndef MAINTAINER
enum { dumptrace = 0 };
#else /* MAINTAINER */
const bool dumptrace (getenv ("SFSRO_TRACE"));
#endif /* MAINTAINER */

#define NFS_MAXDATA 8192

static MKEY __xattr_mkey  ("xa");

#define ACCESS3_ALL (ACCESS3_READ   | ACCESS3_LOOKUP | ACCESS3_MODIFY | \
		     ACCESS3_EXTEND | ACCESS3_DELETE | ACCESS3_EXECUTE)

static const char* nfs3exp_to_string (int code)
{
    switch (code) {
    case NFSPROC3_NULL:     return "NFSPROC3_NULL";
    case NFSPROC3_GETATTR:  return "NFSPROC3_GETATTR";
    case NFSPROC3_FSINFO:   return "NFSPROC3_FSINFO";
    case NFSPROC3_FSSTAT:   return "NFSPROC3_FSSTAT";
    case NFSPROC3_ACCESS:   return "NFSPROC3_ACCESS";
    case NFSPROC3_LOOKUP:   return "NFSPROC3_LOOKUP";
    case NFSPROC3_READDIR:  return "NFSPROC3_READDIR";
    case NFSPROC3_READ:     return "NFSPROC3_READ";
    case NFSPROC3_CREATE:   return "NFSPROC3_CREATE";
    case NFSPROC3_WRITE:    return "NFSPROC3_WRITE";
    case NFSPROC3_COMMIT:   return "NFSPROC3_COMMIT";
    case NFSPROC3_REMOVE:   return "NFSPROC3_REMOVE";
    case NFSPROC3_RMDIR:    return "NFSPROC3_RMDIR";
    case NFSPROC3_RENAME:   return "NFSPROC3_RENAME";
    case NFSPROC3_LINK:     return "NFSPROC3_LINK";
    case NFSPROC3_MKDIR:    return "NFSPROC3_MKDIR";
    case NFSPROC3_SYMLINK:  return "NFSPROC3_SYMLINK";
    case NFSPROC3_READLINK: return "NFSPROC3_READLINK";
    case NFSPROC3_SETATTR:  return "NFSPROC3_SETATTR";
    default:                return "UNKNOWN";
    }
}

static nfsstat3
nfs_errno (int err)
{
    switch ((nfsstat3) err) {
    case NFS3ERR_PERM:
    case NFS3ERR_NOENT:
    case NFS3ERR_IO:
    case NFS3ERR_NXIO:
    case NFS3ERR_ACCES:
    case NFS3ERR_EXIST:
    case NFS3ERR_NODEV:
    case NFS3ERR_NOTDIR:
    case NFS3ERR_ISDIR:
    case NFS3ERR_INVAL:
    case NFS3ERR_FBIG:
    case NFS3ERR_NOSPC:
    case NFS3ERR_ROFS:
    case NFS3ERR_NAMETOOLONG:
    case NFS3ERR_NOTEMPTY:
    case NFS3ERR_STALE:
    case NFS3ERR_SERVERFAULT:
	return (nfsstat3) err;

    case DBFS_NOTFOUND:
	return NFS3ERR_NOENT;

    case DBFS_EXISTS:
	return NFS3ERR_EXIST;

    default:
	XNFS_ERROR (err) ("Unrecognized NFS errno");
	return NFS3ERR_SERVERFAULT;
    }
}

/**********************************************************************/
/*                          Client Proxy Object                       */
/**********************************************************************/

void
Proxy::sfs_getfsinfo (svccb *sbp)
{
    sfs_fsinfo fsinfo;

    fsinfo.set_prog      (ex_NFS_PROGRAM);
    fsinfo.nfs->set_vers (ex_NFS_V3);

    set_fh (fsinfo.nfs->v3->root, DBFS_ROOT_DIRECTORY.key ());

    // Formerly there was an if (_rwsrv), but I can't see it ever get
    // unassigned
    assert (_rwsrv);

    sbp->replyref (fsinfo);
}

ptr<rabin_priv>
Proxy::doconnect (const sfs_connectarg *ci, sfs_servinfo *si)
{
    _xsrv  = get_server ();

    (* si) = _xsrv->_servinfo;

    return _xsrv->_sk;
}

void
Proxy::nfs3dispatch (svccb *sbp)
{
    TXN txn;
    int ret;

    if (sbp == NULL) {
	XNFS_ERROR ("nfs3dispatch: dismount (sbp == null)");
	delete this;
	return;
    }

    if ((ret = txn.begin (get_server ()->_dbfs, DBFS_TXN_ASYNC))) {
	PROP_ERROR (ret) ("txn_begin");
    } else {

	switch (sbp->proc()) {
	case NFSPROC3_GETATTR:  ret = nfs3_getattr  (& txn, sbp); break;
	case NFSPROC3_FSINFO:   ret = nfs3_fsinfo   (& txn, sbp); break;
	case NFSPROC3_FSSTAT:   ret = nfs3_fsstat   (& txn, sbp); break;
	case NFSPROC3_ACCESS:   ret = nfs3_access   (& txn, sbp); break;
	case NFSPROC3_LOOKUP:   ret = nfs3_lookup   (& txn, sbp); break;
	case NFSPROC3_READDIR:  ret = nfs3_readdir  (& txn, sbp); break;
	case NFSPROC3_READ:     ret = nfs3_read     (& txn, sbp); break;
	case NFSPROC3_CREATE:   ret = nfs3_create   (& txn, sbp); break;
	case NFSPROC3_WRITE:    ret = nfs3_write    (& txn, sbp); break;
	case NFSPROC3_COMMIT:   ret = nfs3_commit   (& txn, sbp); break;
	case NFSPROC3_REMOVE:   ret = nfs3_remove   (& txn, sbp); break;
	case NFSPROC3_RMDIR:    ret = nfs3_remove   (& txn, sbp); break;
	case NFSPROC3_RENAME:   ret = nfs3_rename   (& txn, sbp); break;
	case NFSPROC3_LINK:     ret = nfs3_link     (& txn, sbp); break;
	case NFSPROC3_MKDIR:    ret = nfs3_mkdir    (& txn, sbp); break;
	case NFSPROC3_SETATTR:  ret = nfs3_setattr  (& txn, sbp); break;

	case NFSPROC3_NULL: {
	    // NULL is a special case, just reply NULL and return
	    XNFS_ERROR ("nfs3dispatch: NFSPROC3_NULL");
	    sbp->reply (NULL);
	    return;
	}

	// Unsupported procedures
	case NFSPROC3_SYMLINK:
	case NFSPROC3_READLINK:
	    ret = NFS3ERR_NOTSUPP;
	    break;
	// Unknown procedures
	default:
	    ret = NFS3ERR_INVAL;
	    break;
	}

	XNFS_ERROR (ret) ("nfs3dispatch: %s", nfs3exp_to_string (sbp->proc ()));
    }

    // Failure response
    if (ret != 0) {
	XNFS_ERROR (ret) ("nfs3dispatch");
	nfs3exp_err (sbp, nfs_errno (ret));
    }
}

//////////////////////////////////////////////////////////////////////
// Node handling
//////////////////////////////////////////////////////////////////////

int
Proxy::get_node (TXN *txn, svccb *sbp, guint64 inum, MAJORC &inop)
{
    int ret;

    if ((ret = txn->find_major (inop, XNUM (inum)))) {
	PROP_ERROR (ret) ("find_major");
	return ret;
    }

    return 0;
}

int
Proxy::get_dnode (TXN *txn, svccb *sbp, guint64 inum, MAJORC &inop)
{
    int ret;

    if ((ret = get_node (txn, sbp, inum, inop))) {
	PROP_ERROR (ret) ("get_node");
	return ret;
    }

    if (! inop.is_container ()) {
	XNFS_ERROR ("not a directory");
	return NFS3ERR_NOTDIR;
    }

    return 0;
}

int
Proxy::get_snode (TXN *txn, svccb *sbp, guint64 inum, MAJORC &inop)
{
    int ret;

    if ((ret = get_node (txn, sbp, inum, inop))) {
	PROP_ERROR (ret) ("get_node");
	return ret;
    }

    if (! inop.is_segment ()) {
	XNFS_ERROR ("not a segment");
	return NFS3ERR_INVAL;
    }

    return 0;
}

//////////////////////////////////////////////////////////////////////
// Attribute handling
//////////////////////////////////////////////////////////////////////

int
xattr_set (MAJORC &ino, const XNFS_STAT &xa)
{
    NODEC  att;
    int    ret;

    if ((ret = ino.node_mkey (__xattr_mkey, att))) {
	PROP_ERROR (ret) ("node_mkey");
	return ret;
    }

    if ((ret = att.put_segment ((guint8*) & xa, sizeof (xa), DBFS_OVERWRITE))) {
	PROP_ERROR (ret) ("put_segment");
	return ret;
    }

    return 0;
}

int
xattr_get (MAJORC &ino, XNFS_STAT &xa)
{
    NODEC  att;
    int    ret;

    if ((ret = ino.node_mkey (__xattr_mkey, att))) {
	PROP_ERROR (ret) ("node_mkey");
	return ret;
    }

    if ((ret = att.get_segment ((guint8*) & xa, 0))) {
	PROP_ERROR (ret) ("get_segment");
	return ret;
    }

    return 0;
}

void
xattr_update_time (const set_time &st, const nfstime3 &now, nfstime3 &val)
{
    switch (st.set) {
    case SET_TO_CLIENT_TIME:
	val = *st.time;
	//warnx << "set client time\n";
	break;
    case SET_TO_SERVER_TIME:
	val = now;
	//warnx << "set server time\n";
	break;
    case DONT_CHANGE:
	break;
    }
}

void
xattr_update (const sattr3 &attributes, const nfstime3 &now, XNFS_STAT &xa)
{
    if (attributes.mode.set) {
	xa.mode = *attributes.mode.val;
    }

    if (attributes.uid.set) {
	xa.uid = *attributes.uid.val;
	//warnx << "set UNIX uid " << xa.uid << "\n";
    }

    if (attributes.gid.set) {
	xa.gid = *attributes.gid.val;
	//warnx << "set UNIX gid " << xa.gid << "\n";
    }

    xattr_update_time (attributes.atime, now, xa.atime);
    xattr_update_time (attributes.mtime, now, xa.mtime);
}

void
Proxy::xattr_init (svccb *sbp, const sattr3 &attributes, XNFS_STAT &xa)
{
    nfstime3 now = nfs_now ();

    xa.mode  = 0;
    xa.uid   = 0;
    xa.gid   = 0;
    xa.atime = now;
    xa.mtime = now;
    xa.ctime = now;

    //warnx << "getaui " << sbp->getaui () << " " << authtab.size () << "\n";

    if (sbp->getaui () < authtab.size ()) {
	sfsauth_cred &cred = credtab[sbp->getaui ()];

	switch (cred.type) {
	case SFS_UNIXCRED:
	    xa.uid = cred.unixcred->uid;
	    xa.gid = cred.unixcred->gid;

	    //warnx << "create w/ UNIX uid " << xa.uid << " gid " << xa.gid << "\n";

	    break;
	default:
	    //warnx << "create w/o UNIX uid\n";
	    break;
	}
    }

    xattr_update (attributes, now, xa);
}

int
Proxy::get_attr (svccb *sbp, MAJORC &major, ex_post_op_attr &attr)
{
    ex_fattr3 fa;
    int ret;

    if ((ret = get_attr (sbp, major, fa))) {
	PROP_ERROR (ret) ("get_attr");
	return ret;
    }

    set_attr (attr, fa);
    return 0;
}

int
Proxy::get_attr (TXN *txn, svccb *sbp, guint64 inum, ex_fattr3 &attr)
{
    MAJORC ino;
    int ret;

    if ((ret = get_node (txn, sbp, inum, ino))) {
	PROP_ERROR (ret) ("get_node");
	return ret;
    }

    return get_attr (sbp, ino, attr);
}

int
Proxy::get_attr (svccb *sbp, MAJORC &ino, ex_fattr3 &attr)
{
    ftype3    t;
    guint64   s;
    XLNK      nlinks;
    XNFS_STAT xa;
    int ret;

    nlinks = ino.nlinks ();

    if ((ret = xattr_get (ino, xa))) {
	PROP_ERROR (ret) ("xattr_get");
	return ret;
    }

    switch (ino.type ()) {
    case FT_DirHash:
    case FT_DirBtree:

	t = NF3DIR;
	s = 512; // @@ No accounting
	break;

    case FT_LongSeg:
	t = NF3REG;
	s = ino.length ().key ();
	break;

    case FT_DirSeq:
    case FT_NotPresent:
    case FT_Reflink:
    case FT_ShortSeg:
    case FT_ViewSeg:
    default:
	DBFS_FATAL ("Unexpected node type: ") (dbfs_type_to_string (ino.type ()));
	abort ();
    }

    attr.fileid     = ino.number ().key ();
    attr.type       = t;
    attr.size       = s;
    attr.used       = s; // No accounting
    attr.nlink      = nlinks.key ();
    attr.expire     = 0; // Need attribute-cache invalidation first
    attr.rdev.major = 1;
    attr.rdev.minor = 1;
    attr.fsid       = 1; // No volume ID
    attr.uid        = xa.uid;
    attr.gid        = xa.gid;
    attr.mode       = xa.mode;
    attr.mtime      = xa.mtime;
    attr.ctime      = xa.ctime;
    attr.atime      = xa.atime;

    return 0;
}

int
Proxy::update_mtime (svccb *sbp, MAJORC &ino)
{
    int ret;
    XNFS_STAT xa;

    if ((ret = xattr_get (ino, xa))) {
	PROP_ERROR (ret) ("xattr_get");
	return ret;
    }

    xa.mtime = nfs_now ();

    if ((ret = xattr_set (ino, xa))) {
	PROP_ERROR (ret) ("xattr_set");
	return ret;
    }

    return 0;
}

//////////////////////////////////////////////////////////////////////
// Sub-Operations
//////////////////////////////////////////////////////////////////////

int
add_dots (MAJORC &dir, MAJORC &par)
{
    int ret;
    DKEY dot_dkey    (".");
    DKEY dotdot_dkey ("..");

    if ((ret = dir.dir_link_insert (dot_dkey, dir, DBFS_NOOVERWRITE))) {
	PROP_ERROR (ret) ("dir_link_insert (.)");
	return ret;
    }

    if ((ret = dir.dir_link_insert (dotdot_dkey, par, DBFS_NOOVERWRITE))) {
	PROP_ERROR (ret) ("dir_link_insert (..)");
	return ret;
    }

    return 0;
}

int
Proxy::dir_empty (svccb *sbp, MAJORC &ino, bool *empty)
{
    DIRC dirc;
    DKEY dkey;
    int  ret;

    // Note: This is a little clumsy...  open a cursor and check for
    // any entries that are not "." or "..".
    if ((ret = ino.dir_open (dirc))) {
	PROP_ERROR (ret) ("dir_open");
	return ret;
    }

    (* empty) = true;

    while (dirc.next ()) {

	ret = dirc.get_key (dkey);

	g_assert (ret == 0);

	if (dkey.size () == 1 && dkey.data ()[0] == '.') {
	    continue;
	}

	if (dkey.size () == 2 && dkey.data ()[0] == '.' && dkey.data ()[1] == '.') {
	    continue;
	}

	(* empty) = false;

	break;
    }

    if ((ret = dirc.close ())) {
	PROP_ERROR (ret) ("dirc_close");
	return ret;
    }

    return 0;
}

int
Proxy::create_link (svccb *sbp, MAJORC &dir_ino, MAJORC &linkto, DKEY &name, int flags)
{
    int ret;

    if ((ret = dir_ino.dir_link_insert (name, linkto, flags))) {
	// Note: DBFS_EXISTS is translated into NFS3ERR_EXISTS
	PROP_ERROR (ret) ("dir_link_insert");
	return ret;
    }

    if ((ret = update_mtime (sbp, dir_ino))) {
	PROP_ERROR (ret) ("update_mtime");
	return ret;
    }

    return 0;
}

int
Proxy::create_node (TXN          *txn,
		    svccb        *sbp,
		    diropargs3   *dirop,
		    XTYPE         type,
		    const sattr3 &attr,
		    MAJORC       &create_ino,
		    MAJORC       &dir_ino)
{
    guint64   dir_inum = get_inum (dirop->dir);
    DKEY      name (dirop->name);
    XNFS_STAT xa;
    int ret;

    if ((ret = get_dnode (txn, sbp, dir_inum, dir_ino))) {
	PROP_ERROR (ret) ("get_dnode");
	return ret;
    }

    if ((ret = dir_ino.sarea ().allocate (create_ino, 0))) {
	PROP_ERROR (ret) ("allocate node");
	return ret;
    }

    if (type == XTYPE_LONGSEG) {

	if ((ret = create_ino.unsafe_create (DBFS_NOOVERWRITE))) {
	    PROP_ERROR (ret) ("unsafe_create");
	    return ret;
	}
    } else if (type == XTYPE_DIRBTREE) {

	if ((ret = create_ino.mk_directory (type, DBFS_NOOVERWRITE))) {
	    PROP_ERROR (ret) ("mk_directory");
	    return ret;
	}
    } else {
	g_assert_not_reached ();
    }

    xattr_init (sbp, attr, xa);

    if ((ret = xattr_set (create_ino, xa))) {
	PROP_ERROR (ret) ("xattr_set");
	return ret;
    }

    if ((ret = create_link (sbp, dir_ino, create_ino, name, DBFS_NOOVERWRITE))) {
	PROP_ERROR (ret) ("create_link");
	return ret;
    }

    return 0;
}

int
Proxy::unlink (TXN *txn, svccb *sbp, diropargs3* dirop, MAJORC &del_ino)
{
    guint64  dir_inum = get_inum (dirop->dir);
    MAJORC    dir_ino;
    DIRC         dirc;
    DKEY         name (dirop->name);
    int           ret;

    if (dirop->name == ".." || dirop->name == ".") {
	XNFS_ERROR ("cannot delete . or ..");
	return NFS3ERR_INVAL;
    }

    if ((ret = get_dnode (txn, sbp, dir_inum, dir_ino))) {
	PROP_ERROR (ret) ("get_dnode");
	return ret;
    }

    if ((ret = dir_ino.dir_open (dirc))) {
	PROP_ERROR (ret) ("dir_open");
	return ret;
    }

    if ((ret = dirc.set_key (name, DBFS_LINK_RMW))) {
	// Note: DBFS_NOTFOUND is translated into NFS3ERR_NOENT
	PROP_ERROR (ret) ("set_key");
	return ret;
    }

    if ((ret = dirc.get_node (del_ino))) {
	PROP_ERROR (ret) ("get_node");
	return ret;
    }

    // This is called for rename(), remove(), and rmdir().  Check for
    // non-empty directory here.
    if (del_ino.is_container ()) {

	bool empty;

	if ((ret = dir_empty (sbp, del_ino, & empty))) {
	    PROP_ERROR (ret) ("dir_empty");
	    return ret;
	}

	if (! empty) {
	    XNFS_ERROR ("directory not empty");
	    return NFS3ERR_NOTEMPTY;
	}
    }

    // Its okay for the refcount to fall to zero here, since its not
    // collected until txn commit (and rename will raise the link
    // count).
    if ((ret = dirc.del_link ())) {
	PROP_ERROR (ret) ("del_link");
	return ret;
    }

    if ((ret = dirc.close ())) {
	PROP_ERROR (ret) ("dirc_close");
	return ret;
    }

    return 0;
}

//////////////////////////////////////////////////////////////////////
// Node Operations
//////////////////////////////////////////////////////////////////////

int
Proxy::nfs3_getattr (TXN *txn, svccb *sbp)
{
    int ret;
    ex_getattr3res  nfsres (NFS3_OK);
    guint64         inum = get_inum (sbp->template getarg<nfs_fh3> ());

    if ((ret = get_attr (txn, sbp, inum, *nfsres.attributes))) {
	PROP_ERROR (ret) ("get_attr");
	return ret;
    }

    sbp->reply (& nfsres);
    return 0;
}

int
Proxy::nfs3_access (TXN *txn, svccb *sbp)
{
    int ret;
    ex_access3res nfsres (NFS3_OK);
    ex_fattr3       attr;
    access3args      *aa = sbp->template getarg<access3args> ();
    guint64         inum = get_inum (aa->object);

    if ((ret = get_attr (txn, sbp, inum, attr))) {
	PROP_ERROR (ret) ("get_attr");
	return ret;
    }

    nfsres.resok->access = ACCESS3_ALL;

    set_attr (nfsres.resok->obj_attributes, attr);

    sbp->reply (& nfsres);
    return 0;
}

int
Proxy::nfs3_fsinfo (TXN *txn, svccb *sbp)
{
    ex_fsinfo3res res (NFS3_OK);

    res.resok->rtmax  = NFS_MAXDATA;
    res.resok->rtpref = NFS_MAXDATA;
    res.resok->rtmult = 512;
    res.resok->wtmax  = NFS_MAXDATA;
    res.resok->wtpref = NFS_MAXDATA;
    res.resok->wtmult = NFS_MAXDATA;
    res.resok->dtpref = NFS_MAXDATA;

    res.resok->maxfilesize = INT64 (0x7fffffffffffffff);

    res.resok->time_delta.seconds  = 0;
    res.resok->time_delta.nseconds = 1;

    // Skipping support for now: FSF3_SYMLINK
    res.resok->properties = (FSF3_HOMOGENEOUS | FSF3_LINK | FSF3_CANSETTIME);

    // Not returning obj_attributes
    sbp->reply (&res);
    return 0;
}

int
Proxy::nfs3_fsstat (TXN *txn, svccb *sbp)
{
    ex_fsstat3res res (NFS3_OK);

    res.resok->tbytes   = 0;
    res.resok->fbytes   = 0;
    res.resok->abytes   = 0;
    res.resok->tfiles   = 0;
    res.resok->ffiles   = 0;
    res.resok->afiles   = 0;
    res.resok->invarsec = 0;

    // Not returning obj_attributes
    sbp->reply (&res);
    return 0;
}

//////////////////////////////////////////////////////////////////////
// Directory Operations
//////////////////////////////////////////////////////////////////////

int
Proxy::nfs3_lookup (TXN *txn, svccb *sbp)
{
    diropargs3    *dirop = sbp->template getarg<diropargs3> ();
    guint64     dir_inum = get_inum (dirop->dir);
    MAJORC       dir_ino;
    MAJORC        lu_ino;
    ex_lookup3res nfsres (NFS3_OK);
    DKEY            name (dirop->name);
    int              ret;

    if ((ret = get_dnode (txn, sbp, dir_inum, dir_ino))) {
	PROP_ERROR (ret) ("get_dnode");
	return ret;
    }

    if ((ret = dir_ino.dir_link_lookup (name, lu_ino, 0))) {
	XNFS_ERROR (ret) ("dir_link");
	return ret;
    }

    if ((ret = get_attr (sbp, lu_ino, nfsres.resok->obj_attributes))) {
	PROP_ERROR (ret) ("get_attr");
	return ret;
    }

    set_fh (nfsres.resok->object, lu_ino.number ().key ());

    // Not returning dir_attributes
    sbp->reply (& nfsres);
    return 0;
}

int
Proxy::nfs3_readdir (TXN *txn, svccb *sbp)
{
    readdir3args *readdirop = sbp->template getarg<readdir3args> ();
    guint64        dir_inum = get_inum (readdirop->dir);
    MAJORC          dir_ino;
    uint64          ocookie;
    DIRC               dirc;
    DKEY            ent_nam;
    XNUM            ent_num;
    int                 ret;

    // Set up to approximate stopping after readdirop->count bytes.
    // Guess at size of fixed part of reply, taken from sfsusrv.
    ex_readdir3res      nfsres (NFS3_OK);
    rpc_ptr<entry3>    *oneent = & nfsres.resok->reply.entries;
    uint32          size_guess = 200;

    if ((ret = get_dnode (txn, sbp, dir_inum, dir_ino))) {
	PROP_ERROR (ret) ("get_dnode");
	return ret;
    }

    if ((ret = dir_ino.dir_open (dirc))) {
	PROP_ERROR (ret) ("dir_open");
	return ret;
    }

    if (readdirop->cookie != 0) {
	// The cookie is the 1-origin index of the last file
	// previously returned.

	for (ocookie = 1; ocookie <= readdirop->cookie; ocookie += 1) {

	    if (! dirc.next ()) {
		// Spec says return EOF when a cookie is invalid
		// Can't guarantee hash-stability anyway...
		XNFS_ERROR ("directory cookie eof (stale)");
		return NFS3ERR_STALE;
	    }
	}
    }

    nfsres.resok->reply.eof = true;

    while (dirc.next ()) {

	ret = dirc.get_key    (ent_nam);
	ret = dirc.get_number (ent_num);

	g_assert (ret == 0);

	str n ((const char*) ent_nam.data (), ent_nam.size ());

	size_guess += n.len() + 32; // Just a guess...

	if (size_guess > readdirop->count) {
	    nfsres.resok->reply.eof = false;
	    break;
	}

	(*oneent).alloc ();

	(*oneent)->name = n;
	(*oneent)->cookie = ocookie++;
	(*oneent)->fileid = ent_num.key ();

	oneent = &(*oneent)->nextentry;
    }

    if ((ret = dirc.close ())) {
	PROP_ERROR (ret) ("dirc_close");
	return ret;
    }

    // Not returning dir_attributes or cookieverf
    sbp->reply (& nfsres);
    return 0;
}

//////////////////////////////////////////////////////////////////////
// Create/Delete Operations
//////////////////////////////////////////////////////////////////////

int
Proxy::nfs3_mkdir (TXN *txn, svccb *sbp)
{
    mkdir3args       *mk = sbp->template getarg<mkdir3args> ();
    MAJORC       new_ino;
    MAJORC       dir_ino;
    ex_diropres3  nfsres (NFS3_OK);
    int ret;

    if ((ret = create_node (txn, sbp, & mk->where, XTYPE_DIRBTREE, mk->attributes, new_ino, dir_ino))) {
	PROP_ERROR (ret) ("create_node");
	return ret;
    }

    if ((ret = add_dots (new_ino, dir_ino))) {
	PROP_ERROR (ret) ("add_dots");
	return ret;
    }

    set_fh (nfsres.resok->obj, new_ino);

    // Not returning obj_attributes or dir_wcc
    sbp->reply (& nfsres);
    return 0;
}

int
Proxy::nfs3_create (TXN *txn, svccb *sbp)
{
    create3args       *ca = sbp->template getarg<create3args> ();
    MAJORC        new_ino;
    MAJORC        dir_ino;
    ex_diropres3   nfsres (NFS3_OK);
    int ret;

    if (ca->how.mode == EXCLUSIVE) {
	// This requires stable storage of the createverf3 token..
	XNFS_ERROR ("exclusive create not supported");
	return NFS3ERR_NOTSUPP;
    }

    if ((ret = create_node (txn, sbp, & ca->where, XTYPE_LONGSEG, *ca->how.obj_attributes, new_ino, dir_ino))) {
	PROP_ERROR (ret) ("create_node");
	return ret;
    }

    set_fh (nfsres.resok->obj, new_ino);

    // Not returning obj_attributes or dir_wcc
    sbp->reply (& nfsres);
    return 0;
}

int
Proxy::nfs3_remove (TXN *txn, svccb *sbp)
{
    diropargs3    *dirop = sbp->template getarg<diropargs3> ();
    ex_wccstat3   nfsres (NFS3_OK);
    MAJORC        ino;
    int ret;

    if ((ret = unlink (txn, sbp, dirop, ino))) {
	PROP_ERROR (ret) ("unlink");
	return ret;
    }

    // Not returning dir_wcc
    sbp->reply (& nfsres);
    return 0;
}

int
Proxy::nfs3_link (TXN *txn, svccb *sbp)
{
    link3args        *lk = sbp->template getarg<link3args> ();
    guint64    link_inum = get_inum (lk->file);
    guint64     dir_inum = get_inum (lk->link.dir);
    MAJORC      link_ino;
    MAJORC       dir_ino;
    DKEY            name (lk->link.name);
    ex_link3res   nfsres (NFS3_OK);
    int ret;

    // Don't need link_ino except to check that its a regular file
    // (prevent cycles)
    if ((ret = get_snode (txn, sbp, link_inum, link_ino))) {
	PROP_ERROR (ret) ("get_snode");
	return ret;
    }

    if ((ret = get_dnode (txn, sbp, dir_inum, dir_ino))) {
	PROP_ERROR (ret) ("get_dnode");
	return ret;
    }

    if ((ret = create_link (sbp, dir_ino, link_ino, name, DBFS_NOOVERWRITE))) {
	PROP_ERROR (ret) ("create_link");
	return ret;
    }

    // Not returning file_attributes or dir_wcc
    sbp->reply (&nfsres);
    return 0;
}

int
Proxy::nfs3_rename (TXN *txn, svccb *sbp)
{
    rename3args        *rn = sbp->template getarg<rename3args> ();
    guint64        to_inum = get_inum (rn->to.dir);
    MAJORC          to_dir;
    MAJORC         ren_ino;
    DKEY           to_name (rn->to.name);
    ex_rename3res   nfsres (NFS3_OK);
    int ret;

    if ((ret = get_dnode (txn, sbp, to_inum, to_dir))) {
	PROP_ERROR (ret) ("get_dnode");
	return ret;
    }

    if ((ret = unlink (txn, sbp, & rn->from, ren_ino))) {
	PROP_ERROR (ret) ("unlink");
	return ret;
    }

    if ((ret = create_link (sbp, to_dir, ren_ino, to_name, DBFS_OVERWRITE))) {
	PROP_ERROR (ret) ("create_link");
	return ret;
    }

    // Not returning fromdir_wcc or todir_wcc
    sbp->reply (& nfsres);
    return 0;
}

//////////////////////////////////////////////////////////////////////
// File Operations
//////////////////////////////////////////////////////////////////////

int
Proxy::nfs3_setattr (TXN *txn, svccb *sbp)
{
    setattr3args    *sa = sbp->template getarg<setattr3args> ();
    guint64        inum = get_inum (sa->object);
    ex_wccstat3  nfsres (NFS3_OK);
    MAJORC          ino;
    XNFS_STAT        xa;
    int ret;

    // Size modifies the default node, all the others modify
    // extended attributes.
    if (sa->new_attributes.size.set) {

	if ((ret = get_snode (txn, sbp, inum, ino))) {
	    PROP_ERROR (ret) ("get_snode");
	    return ret;
	}

	XSIZ new_size (*sa->new_attributes.size.val);
	int ret;

	if ((ret = ino.unsafe_truncate (new_size))) {
	    PROP_ERROR (ret) ("unsafe_truncate");
	    return ret;
	}

    } else if ((ret = get_node (txn, sbp, inum, ino))) {
	PROP_ERROR (ret) ("get_node");
	return ret;
    }

    if ((ret = xattr_get (ino, xa))) {
	PROP_ERROR (ret) ("xattr_get");
	return ret;
    }

    xattr_update (sa->new_attributes, nfs_now (), xa);

    if ((ret = xattr_set (ino, xa))) {
	PROP_ERROR (ret) ("xattr_set");
	return ret;
    }

    sbp->reply (& nfsres);
    return 0;
}

int
Proxy::nfs3_write (TXN *txn, svccb *sbp)
{
    write3args      *wa = sbp->template getarg<write3args> ();
    guint64        inum = get_inum (wa->file);
    MAJORC          ino;
    ex_write3res nfsres (NFS3_OK);
    int ret;

    assert (wa->count >= wa->data.size ());

    if ((ret = get_snode (txn, sbp, inum, ino))) {
	PROP_ERROR (ret) ("get_snode");
	return ret;
    }

    if ((ret = ino.unsafe_write ((guint8*) wa->data.base (), XSIZ (wa->count), XSIZ (wa->offset)))) {
	PROP_ERROR (ret) ("unsafe_write");
	return ret;
    }

    switch (wa->stable) {
    case FILE_SYNC:
    case DATA_SYNC:
	XNFS_ERROR (ret) ("Note: synchronous write");

	if ((ret = ino.unsafe_sync ())) {
	    PROP_ERROR (ret) ("unsafe_sync");
	    return ret;
	}

	if ((ret = update_mtime (sbp, ino))) {
	    PROP_ERROR (ret) ("update_mtime");
	    return ret;
	}

	break;
    case UNSTABLE:
	break;
    }

    nfsres.resok->count     = wa->count;
    nfsres.resok->committed = wa->stable;

    // Not returning file_wcc or verf
    sbp->reply (& nfsres);
    return 0;
}

int
Proxy::nfs3_commit (TXN *txn, svccb *sbp)
{
    commit3args      *ca = sbp->template getarg<commit3args> ();
    guint64         inum = get_inum (ca->file);
    MAJORC           ino;
    ex_commit3res nfsres (NFS3_OK);
    int ret;

    if ((ret = get_snode (txn, sbp, inum, ino))) {
	PROP_ERROR (ret) ("get_snode");
	return ret;
    }

    // Not returning file_wcc or verf
    // Ignoring offset, count
    if ((ret = ino.unsafe_sync ())) {
	PROP_ERROR (ret) ("unsafe_sync");
	return ret;
    }

    if ((ret = update_mtime (sbp, ino))) {
	PROP_ERROR (ret) ("update_mtime");
	return ret;
    }

    sbp->reply (& nfsres);
    return 0;
}

int
Proxy::nfs3_read (TXN *txn, svccb *sbp)
{
    read3args      *ra = sbp->template getarg<read3args> ();
    guint64       inum = get_inum (ra->file);
    MAJORC         ino;
    XSIZ         nread (MIN (NFS_MAXDATA, ra->count));
    XSIZ        offset (ra->offset);
    ex_read3res nfsres (NFS3_OK);
    int ret;

    if ((ret = get_snode (txn, sbp, inum, ino))) {
	PROP_ERROR (ret) ("get_snode");
	return ret;
    }

    if (offset >= ino.length ()) {
	nread = XSIZ (0);
    } else if ((ret = ino.unsafe_read ((guint8*) nfsres.resok->data.base (), nread, offset, nread))) {
	PROP_ERROR (ret) ("unsafe_read");
	return ret;
    }

    nfsres.resok->count = nread.key ();
    nfsres.resok->data.setsize (nread.key ());

    if (offset + nread >= ino.length ()) {
	nfsres.resok->eof = true;
    } else {
	nfsres.resok->eof = false;
    }

    // Not returning file_attributes
    sbp->reply (& nfsres);
    return 0;
}
