#include "xproxy.h"
#include <errno.h>
#include <sys/types.h>
#include <sys/wait.h>

char *peerHost = NULL;
int   peerPort = -1;
bool  clientProxyMode = false;

#define SCTX(s) (clientProxyMode ? ("client-" s) : ("server-" s))

DBFS *dbfs;
//pthread_mutex_t gethostbynameMutex = PTHREAD_MUTEX_INITIALIZER;
pthread_mutex_t HEADER_MUTEX = PTHREAD_MUTEX_INITIALIZER;
pthread_mutex_t THREAD_COUNT_MUTEX = PTHREAD_MUTEX_INITIALIZER;
int logEntryCounter = 1;
extern time_t timezone;
extern int daylight;
long timeOffset;
int  logHeaderLines   = 24;

int THREAD_COUNT_MAX  = 10;
int THREAD_COUNT      = 0;
int THREAD_WAIT_USECS = 2500;

// Maybe will get done:
/*** Can't cache any responses with Authorization header ***/

/* Can use delta-encoding for POST requests ... might have to take
 * entity body into account for making certain decisions, but it can't
 * think of a reason now why one would have to take the entity body
 * into account ... versus the cookie argument ***/

/* Persistent connections to source servers ... issues ... 1) not all
 * servers support persistent connections (www.yahoo.com); 2) will
 * connections be reused before they time out?; 3) when connection
 * times out, need to detect that and reset the socket; if it times
 * out after select returned but before request is sent, then the
 * request will be sent but will get lost ... is it worth implementing
 * persistent connections to source servers?  ... probably not ... if
 * persistent connections to source servers are not implemented, then
 * xProxy can become HTTP/1.0, doesn't need to handle chunking and can
 * eliminate checking for 100 responses and other HTTP/1.1 specific
 * responses !!! ***/

/*** Handle any number of concurrent connections ***/
/*** Need to cache HTTP version ??? ... otherwise xproxy will always return HTTP/1.1 responses ***/

// @@@ Mihut: This is not an xproxy error, but why
// does the client print code 226 if my browser sees
// "500 internal error"?
//  DBFS-ERROR: shared-fd still open?
//  PROP-ERROR: open_long: Invalid argument
//  PROP-ERROR: handle_drain
//  PROP-ERROR: xdfs_loc_insert_delta: Code needs work (-1)
//  XPROXY-ERROR: storage_insert_delta_from_buffer: Code needs work (-1)
//  DBFS-ERROR: transaction abort ID 10 (0 active)
//  LOG [xproxy] 226	8334	0	3036	http://slashdot.org/
//  LOG [xproxy] 226	38558	8334	66248	http://slashdot.org/

// @@@ File not found?
//  LOG [xproxy] -1	0	0	1301	http://www.eecs.harvard.edu/~nr/noweb/
//  LOG [xproxy] 502	0	0	1622	http://www.eecs.harvard.edu/~nr/noweb/

// Dependent on Josh:
/*** Handle list of MD5s in Delta-Base (JOSH WRITES: EASILY DONE) ***/
/*** Cache replacement policy (LFU + LRU) ***/

// Can be done now:
/*** Strange scenario ... when transferring a gziped ps file, the Content-Encoding is gzip ... xproxy should probably not decompress it ... xproxy should probably only decompress text files ??? ***/

void thread_count_incr (int change)
{
    MONITOR mon;

  again:

    mon.lock (& THREAD_COUNT_MUTEX);

    if ((THREAD_COUNT + change) > THREAD_COUNT_MAX) {

	mon.release ();

	usleep (THREAD_WAIT_USECS);

	goto again;
    }

    THREAD_COUNT += change;
}

void *connectionHandler(void *s) {
    Socket *connectionSocket = (Socket*) s;
    Socket *peerSocket = NULL;
    Request *requests = new Request[MAX_SIML_REQ];
    struct timeval timeoutValue;
    fd_set mainReadSet, mainWriteSet, tempReadSet, tempWriteSet;
    int maxSocketFD = connectionSocket->sockfd;
    int selectValue;
    int nextAvailable, nextToSendResponse; // used by both client proxy and server proxy
    int nextToSendRequest, nextToReceiveResponse; // used only by client proxy for the connection to server proxy
    RequestHeaderState requestHeaderState;
    ResponseHeaderState responseHeaderState;
    bool recvBytesCalled;

    DEBUG_VERBOSE ("Thread start: %d", (int) pthread_self());

    // Check to see if successful in allocating on the heap an array of requests.
    if (requests == NULL) {
	XPROXY_ERROR ("ERROR: thread %d could not allocate array of requests", (int) pthread_self());
	goto cleanup;
    }

    // Set timeout value for select.
    timeoutValue.tv_sec = (clientProxyMode ? 360 : 300); // set time out one minute higher for client proxy to avoid race conditions
    timeoutValue.tv_usec = 0;

    // Initialize read/write sets for select.
    FD_ZERO(&mainReadSet);
    FD_ZERO(&mainWriteSet);
    FD_ZERO(&tempReadSet);
    FD_ZERO(&tempWriteSet);

    // Set read bit for input connection socket.
    FD_SET(connectionSocket->sockfd, &mainReadSet);

    if (clientProxyMode) {
	// If client proxy, establish connection to server proxy.
	peerSocket = new Socket();
	if (peerSocket->sockfd == -1) {
	    XPROXY_ERROR ("ERROR: thread %d could not get socket for connection to server proxy", (int) pthread_self());
	    goto cleanup;
	}
	if (!peerSocket->connect(peerHost, peerPort)) {
	    XPROXY_ERROR ("ERROR: thread %d could not connect to server proxy", (int) pthread_self());
	    goto cleanup;
	}

	// If client proxy, all requests should go to the server proxy.
	for (int i = 0; i < MAX_SIML_REQ; i++) {
	    requests[i].out = peerSocket;
	}

	// Set socket non-blocking after connecting to server proxy.
	fcntl(peerSocket->sockfd, F_SETFL, (fcntl(peerSocket->sockfd, F_GETFL, 0) | O_NONBLOCK));

	// Set read bit for connection to server proxy.
	FD_SET(peerSocket->sockfd, &mainReadSet);
	if (peerSocket->sockfd > maxSocketFD) {
	    maxSocketFD = peerSocket->sockfd;
	}
    }

    // Set input connection socket non-blocking.
    fcntl(connectionSocket->sockfd, F_SETFL, (fcntl(connectionSocket->sockfd, F_GETFL, 0) | O_NONBLOCK));

    // Set input socket for requests.
    for (int i = 0; i < MAX_SIML_REQ; i++) {
	requests[i].in = connectionSocket;
    }

    nextAvailable = 0;
    nextToSendResponse = 0;
    nextToSendRequest = 0;
    nextToReceiveResponse = 0;

    while (1) {
	tempReadSet = mainReadSet;
	tempWriteSet = mainWriteSet;
	selectValue = select(maxSocketFD + 1, &tempReadSet, &tempWriteSet, NULL, &timeoutValue);

	if (selectValue == 0) {
	    DEBUG_PROXY (errno) ("ERROR: thread %lu client EOF", pthread_self ());
	    goto cleanup;
	} else if (selectValue < 0) {
	    XPROXY_ERROR (errno) ("ERROR: thread %lu select", pthread_self ());
	    goto cleanup;
	}

	// Read as many requests as possible from the connection socket.
	if (FD_ISSET(connectionSocket->sockfd, &tempReadSet)) {
	    // Receive as many bytes as possible from the input socket.
	    if (connectionSocket->recvBytes(NULL, MAX_SOCKET_BUFFER) <= 0) {
		// Client (proxy) closed the connection.
		DEBUG_VERBOSE ("NOTE: Client (proxy) closed the connection in thread %lu", pthread_self ());
		goto cleanup;
	    }

	    DEBUG_REQHEAD ("REQUEST(s) RECEIVED:\n%s", connectionSocket->start);

	    // Receive and parse as many requests as possible.
	    while (1) {
		requestHeaderState = parseRequest(&(requests[nextAvailable]));
		if (requestHeaderState == PARTIAL_REQUEST) {
		    DEBUG_REQHEAD ("PARTIAL_REQUEST:\n%s", connectionSocket->start);
		    break;
		}
		else {
		    if (requestHeaderState == FULL_REQUEST) {
			DEBUG_PROXY ("REQUEST thread %lu url: %s",
				     pthread_self (), requests[nextAvailable].requestFromClient.url);

#ifdef POST_DEBUG
			if (strstr(requests[nextAvailable].requestFromClient.url, ".jpg") == NULL &&
			    strstr(requests[nextAvailable].requestFromClient.url, ".gif") == NULL) {
			    DEBUG_POST ("%s", requests[nextAvailable].requestFromClient.url);
			}
#endif POST_DEBUG
			requests[nextAvailable].state = PROCESS_CLIENT_REQUEST;
		    }
		    else {
			prepareMessage(&(requests[nextAvailable]), (requestHeaderState == BAD_REQUEST ? STATUS_400 : STATUS_501), "");
			requests[nextAvailable].state = WAIT_TO_SEND_CLIENT_RESPONSE;
		    }
		    // Increment nextAvailable and make sure there is enough space in the array of requests.
		    nextAvailable = ((nextAvailable + 1) == MAX_SIML_REQ ? 0 : (nextAvailable + 1));
		    if (requests[nextAvailable].state != UNUSED) {
			XPROXY_ERROR ("ERROR: thread %d needs a larger array of requests", (int) pthread_self());
			goto cleanup;
		    }
		}
	    }
	}

	for (int i = nextToSendResponse; i != nextAvailable; i = ((i + 1) == MAX_SIML_REQ ? 0 : (i + 1))) {
	    if (requests[i].state == PROCESS_CLIENT_REQUEST) {
		// Process client (proxy) request.
		if (processClientRequest(&(requests[i]))) {
		    if (requests[i].responseFromServer.HTTPCode != -1) {
			// Local version is fresh.
			if (requests[i].responseFromServer.HTTPCode == 304) {
			    // Resource not modified, sending back 304 response.
			    prepareMessage(&(requests[i]), STATUS_304, "");
			}
			else {
			    // Sending back either 200 or 226 response, depending on whether the client provided an MD5 and whether there is a local version with that MD5.
			    prepareResponse(&(requests[i]));
			}
			requests[i].state = WAIT_TO_SEND_CLIENT_RESPONSE;
		    }
		    else {
			// Local version not existent or not fresh, need to send request to server.
			requests[i].state = WAIT_TO_SEND_SERVER_REQUEST;
		    }
		}
		else {
		    prepareMessage(&(requests[i]), STATUS_500, "Error in XDFS while processsing client request.");
		    requests[i].state = WAIT_TO_SEND_CLIENT_RESPONSE;
		}
	    }

	    if (requests[i].state == WAIT_TO_SEND_SERVER_REQUEST) {
		if (clientProxyMode) {
		    // This is a client proxy.
		    if (i == nextToSendRequest) {
			// Prepare request to be sent to the server proxy.
			prepareRequest(&(requests[i]));

			DEBUG_REQHEAD ("REQUEST BEING SENT:\n%s", requests[i].tempBufferStart);

			bool successful;
			if (!(successful = sendHTTPMessage(&(requests[i]), REQUEST))) {
			    // Connection to server proxy is down, try to reestablish connection to server proxy.
			    // No part of the header has been transmitted yet.
			    Socket *tempSocket = new Socket();
			    if (tempSocket->sockfd == -1) {
				XPROXY_ERROR ("ERROR: thread %d could not get socket for reestablishing connection to server proxy to transmit request", (int) pthread_self());
				goto cleanup;
			    }
			    if (!tempSocket->connect(peerHost, peerPort)) {
				XPROXY_ERROR ("ERROR: thread %d could not reconnect to server proxy to transmit request", (int) pthread_self());
				goto cleanup;
			    }
			    // Check if connection to server proxy already reset due to a timeout.
			    if (peerSocket->sockfd != -1) {
				FD_CLR(peerSocket->sockfd, &mainReadSet);
				FD_CLR(peerSocket->sockfd, &mainWriteSet);
				peerSocket->reset();
			    }
			    peerSocket->sockfd = tempSocket->sockfd;
			    if (peerSocket->sockfd > maxSocketFD) {
				maxSocketFD = peerSocket->sockfd;
			    }
			    FD_SET(peerSocket->sockfd, &mainReadSet); // sockets to peers always readable
			    // Set socket non-blocking after reconnecting to server proxy.
			    fcntl(peerSocket->sockfd, F_SETFL, (fcntl(peerSocket->sockfd, F_GETFL, 0) | O_NONBLOCK));
			    tempSocket->sockfd = -1; // avoid having the socket descriptor close the socket
			    delete tempSocket;
			}
			// Send request if it wasn't sent above (we reestablished connection to server proxy).
			if (!successful) {
			    if (!(successful = sendHTTPMessage(&(requests[i]), REQUEST))) {
				// This case would happen if we reestablished the connection to server proxy and still can't send the request (strange).
				XPROXY_ERROR ("ERROR: thread %d after reestablishing connection to server proxy still can't send the request", (int) pthread_self());
				goto cleanup;
			    }
			}
			if (successful) {
			    // Successfully transmitted all or part of the request.
			    if (requests[i].tempBufferLength == 0 && requests[i].bodyFromServerLength == 0) {
				// Done sending the entire request.
				// Reset bodyFromServer variables.
				if (strcmp(requests[i].requestFromClient.method, "POST") == 0) {
				    bzero(requests[i].bodyFromServer, requests[i].bodyFromServerAllocatedLength); // take out after system debugged ???
				    requests[i].bodyFromServerStart = requests[i].bodyFromServer;
				}
				nextToSendRequest = ((nextToSendRequest + 1) == MAX_SIML_REQ ? 0 : (nextToSendRequest + 1));
				requests[i].state = RECEIVE_SERVER_RESPONSE;
			    }
			    else {
				// Didn't send the entire request, need to check connection to server proxy for writability.
				FD_SET(peerSocket->sockfd, &mainWriteSet);
				requests[i].state = SEND_SERVER_REQUEST;
			    }
			}
		    }
		}
		else {
		    // This is a server proxy.
		    // Establish connection to the source web server; might want to reuse older connection to source web server later on ???
		    requests[i].out = new Socket();
		    if (requests[i].out->sockfd == -1) {
			XPROXY_ERROR ("ERROR: thread %d could not get socket for connection to server %s on port %d", (int) pthread_self(), requests[i].requestFromClient.server, requests[i].requestFromClient.port);
			goto cleanup;
		    }
		    if (!requests[i].out->connect(requests[i].requestFromClient.server, requests[i].requestFromClient.port)) {
			XPROXY_ERROR ("ERROR: thread %d could not connect to server %s on port %d", (int) pthread_self(), requests[i].requestFromClient.server, requests[i].requestFromClient.port);

			prepareMessage(&(requests[i]), STATUS_502, "Could not connect to server.");
			requests[i].state = WAIT_TO_SEND_CLIENT_RESPONSE;
		    }
		    else {
			// Set socket non-blocking after connecting to source web server.
			fcntl(requests[i].out->sockfd, F_SETFL, (fcntl(requests[i].out->sockfd, F_GETFL, 0) | O_NONBLOCK));
			// Prepare request to be sent to the server.
			prepareRequest(&(requests[i]));

			DEBUG_REQHEAD ("\nREQUEST BEING SENT:\n%s", requests[i].tempBufferStart);

			// We're trying to send the entire request here, possibly without
			// having to select on the writability of the server socket (we're
			// trying to avoid an extra loop and select call).
			bool successful;
			if (!(successful = sendHTTPMessage(&(requests[i]), REQUEST))) {
			    XPROXY_ERROR ("ERROR: thread %d could not send request to server %s on port %d",
					  (int) pthread_self(), requests[i].requestFromClient.server,
					  requests[i].requestFromClient.port);

			    prepareMessage(&(requests[i]), STATUS_502, "Connected to server, but could not send any part of request to it.");
			    requests[i].state = WAIT_TO_SEND_CLIENT_RESPONSE;
			}
			else {
			    if (requests[i].tempBufferLength == 0 && requests[i].bodyFromServerLength == 0) {
				// Done sending the entire request.
				// Reset bodyFromServer variables.
				if (strcmp(requests[i].requestFromClient.method, "POST") == 0) {
				    bzero(requests[i].bodyFromServer, requests[i].bodyFromServerAllocatedLength); // take out after system debugged ???
				    requests[i].bodyFromServerStart = requests[i].bodyFromServer;
				}
				FD_SET(requests[i].out->sockfd, &mainReadSet);
				requests[i].state = RECEIVE_SERVER_RESPONSE;
			    }
			    else {
				// Wasn't able to send the entire request, need to check server socket for writability on next select call.
				FD_SET(requests[i].out->sockfd, &mainWriteSet);
				requests[i].state = SEND_SERVER_REQUEST;
			    }
			    if (requests[i].out->sockfd > maxSocketFD) {
				maxSocketFD = requests[i].out->sockfd;
			    }
			}
		    }
		}
	    }

	    if (requests[i].state == SEND_SERVER_REQUEST && FD_ISSET(requests[i].out->sockfd, &tempWriteSet)) {

		DEBUG_PROXY ("Need to send the remaining part of the request");

		bool successful;
		if (!(successful = sendHTTPMessage(&(requests[i]), REQUEST))) {
		    if (clientProxyMode) {
			// Connection to server proxy went down in the middle of request header transmission.
			XPROXY_ERROR ("ERROR: thread %d can't reestablish connection to server proxy in the middle of request header transmission", (int) pthread_self());
			goto cleanup;
		    }
		    else {
			XPROXY_ERROR ("ERROR: thread %d could not send request to server %s on port %d", (int) pthread_self(), requests[i].requestFromClient.server, requests[i].requestFromClient.port);
			prepareMessage(&(requests[i]), STATUS_502, "Sent part of request to server, but unable to send entire request to it.");
			FD_CLR(requests[i].out->sockfd, &mainWriteSet);
			requests[i].state = WAIT_TO_SEND_CLIENT_RESPONSE;
		    }
		}
		else {
		    // Successfully transmitted all or part of the request.
		    if (requests[i].tempBufferLength == 0 && requests[i].bodyFromServerLength == 0) {
			// Done sending the entire request.
			// Reset bodyFromServer variables.
			if (strcmp(requests[i].requestFromClient.method, "POST") == 0) {
			    bzero(requests[i].bodyFromServer, requests[i].bodyFromServerAllocatedLength); // take out after system debugged ???
			    requests[i].bodyFromServerStart = requests[i].bodyFromServer;
			}
			if (clientProxyMode) {
			    nextToSendRequest = ((nextToSendRequest + 1) == MAX_SIML_REQ ? 0 : (nextToSendRequest + 1));
			}
			FD_CLR(requests[i].out->sockfd, &mainWriteSet);
			FD_SET(requests[i].out->sockfd, &mainReadSet); // If client proxy, this read bit is actually already set.
			requests[i].state = RECEIVE_SERVER_RESPONSE;
		    }
		}
	    }

	    if (requests[i].state == RECEIVE_SERVER_RESPONSE && FD_ISSET(requests[i].out->sockfd, &tempReadSet)) {
		if (clientProxyMode) {
		    // This is a client proxy.
		    // Receive and parse as many responses as possible.
		    // Sanity check, although responses should come in the order requests were sent.
		    if (i != nextToReceiveResponse) {
			XPROXY_ERROR ("ERROR: thread %d got nextToReceiveResponse mixed up", (int) pthread_self());
			goto cleanup;
		    }
		    recvBytesCalled = false;
		    while (1) {
			responseHeaderState = parseResponse(&(requests[nextToReceiveResponse]), &recvBytesCalled);

			DEBUG_VERBOSE ("bodyFromServerLength = %d", requests[i].bodyFromServerLength);

			if (responseHeaderState == FULL_RESPONSE) {
			    requests[nextToReceiveResponse].state = PROCESS_SERVER_RESPONSE;
			    nextToReceiveResponse = ((nextToReceiveResponse + 1) == MAX_SIML_REQ ? 0 : (nextToReceiveResponse + 1));
			}
			else if (responseHeaderState == PARTIAL_RESPONSE) {
			    break;
			}
			else {
			    // The response is malformed (BAD_RESPONSE) and it's really bad in case
			    // of persistent connections (let's say, there is no content-length and
			    // response is not chunked, so there is no way to tell where a response ends);
			    // thus, easy way out is to just cleanup the thread.
			    XPROXY_ERROR ("thread %lu in client proxy got bad response", pthread_self ());
			    goto cleanup;
			}
		    }
		    // Clear the read bit of peerSocket from tempReadSet to be able to check later (after looping through all active requests) if connection to server proxy is down (if no requests were waiting to read from the connection to the server proxy and this connection is readable, then it means the connection to server proxy is down).
		    FD_CLR(peerSocket->sockfd, &tempReadSet);
		}
		else {
		    // This is a server proxy.
		    recvBytesCalled = false;
		    responseHeaderState = parseResponse(&(requests[i]), &recvBytesCalled);

		    DEBUG_VERBOSE ("bodyFromServerLength = %d", requests[i].bodyFromServerLength);

		    if (responseHeaderState == FULL_RESPONSE) {
			// HTTP/1.0 clients don't understand 100 responses; if 100 response received, ignore it and wait for the next response.
			if (requests[i].responseFromServer.HTTPCode == 100) {
			    requests[i].responseFromServer.reset();
			}
			else {
			    FD_CLR(requests[i].out->sockfd, &mainReadSet);
			    requests[i].state = PROCESS_SERVER_RESPONSE;
			}
		    }
		    else if (responseHeaderState == BAD_RESPONSE) {
			prepareMessage(&(requests[i]), STATUS_502, "Bad server response.");
			FD_CLR(requests[i].out->sockfd, &mainReadSet);
			requests[i].state = WAIT_TO_SEND_CLIENT_RESPONSE;
		    }
		    // Otherwise it is a partial response.
		}
	    }

	    if (requests[i].state == PROCESS_SERVER_RESPONSE) {
		if (requests[i].responseFromServer.HTTPCode == 200 || requests[i].responseFromServer.HTTPCode == 226 || requests[i].responseFromServer.HTTPCode == 304) {
		    // Process server response.
		    if (processServerResponse(&(requests[i]))) {
			if (requests[i].responseFromServer.HTTPCode == 304) {
			    // Resource not modified, sending back 304 response.
			    prepareMessage(&(requests[i]), STATUS_304, "");
			}
			else {
			    // Sending back either 200 or 226.
			    prepareResponse(&(requests[i]));
			}
		    }
		    else {
			prepareMessage(&(requests[i]), STATUS_500, "Error in XDFS while processing server response.");
		    }
		}
		else {
		    // Just forward any other response different than 200, 226 or 304.
		    requests[i].bodyToClientStart = requests[i].bodyFromServerStart;
		    requests[i].bodyToClientLength = requests[i].bodyFromServerLength;
		    prepareResponse(&(requests[i]));
		}
		requests[i].state = WAIT_TO_SEND_CLIENT_RESPONSE;

		DEBUG_VERBOSE ("bodyToClientLength = %d", requests[i].bodyToClientLength);
		DEBUG_REQHEAD ("RESPONSE HEADER:\n%s", requests[i].tempBufferStart);
	    }

	    if (requests[i].state == WAIT_TO_SEND_CLIENT_RESPONSE && i == nextToSendResponse) {

		DEBUG_VERBOSE ("RESPONSE BEING SENT:\n%s\n\n%s", requests[i].tempBufferStart, requests[i].bodyToClientStart);
		FD_SET(connectionSocket->sockfd, &mainWriteSet);
		requests[i].state = SEND_CLIENT_RESPONSE;
	    }

	    if (requests[i].state == SEND_CLIENT_RESPONSE && FD_ISSET(connectionSocket->sockfd, &tempWriteSet)) {
		if (!sendHTTPMessage(&(requests[i]), RESPONSE)) {
		    goto cleanup;
		}
		// Check if we're done sending the entire response.
		if (requests[i].tempBufferLength == 0 && requests[i].bodyToClientLength == 0) {
		    // Done sending the response.
		    // Log timing information.

		    if (LOG_TAG (XPROXY)) {

			if (logHeaderLines > 0) {
			    MONITOR mon (& HEADER_MUTEX);
			    static int lines = 0;

			    if ((lines++ % logHeaderLines) == 0) {
				XPROXY_LOG ("RCode\tOrigSz\tCurSz\tTotTm\tURL");
			    }
			}

			XPROXY_LOG ("%d\t%d\t%d\t%d\t",
				    requests[i].responseFromServer.HTTPCode,
				    requests[i].originalSize,
				    requests[i].currentSize,
				    (requests[i].processClientRequestTime +
				     requests[i].parseResponseHeaderTime +
				     requests[i].processServerResponseTime)) (url_dkey (requests[i].requestFromClient.url));
		    }

		    if (requests[i].requestFromClient.connectionClose) {
			goto cleanup; // client doesn't want persistent connection
		    }
		    nextToSendResponse = ((nextToSendResponse + 1) == MAX_SIML_REQ ? 0 : (nextToSendResponse + 1));
		    FD_CLR(connectionSocket->sockfd, &mainWriteSet);
		    requests[i].reset();
		    if (!clientProxyMode) {
			// This is a server proxy and should deallocate connection to source web server for now ???
			delete requests[i].out;
			requests[i].out = NULL;
		    }
		}
	    }
	} // end of for loop going through all relevant requests

	if (clientProxyMode) {
	    // Check if connection to server proxy went down; if yes, reset socket.
	    if (FD_ISSET(peerSocket->sockfd, &tempReadSet)) {
		FD_CLR(peerSocket->sockfd, &mainReadSet);
		peerSocket->reset();
	    }
	}

    } // end of while(1) loop

  cleanup:
    if (connectionSocket) {
	delete connectionSocket;
    }
    if (peerSocket) {
	delete peerSocket;
    }
    if (!clientProxyMode) {
	for (int i = 0; i < MAX_SIML_REQ; i++) {
	    if (requests[i].out) {
		delete requests[i].out;
	    }
	}
    }

    if (requests) {
	delete [] requests;
    }

    thread_count_incr (-1);

    DEBUG_VERBOSE ("Thread end: %d", (int) pthread_self());

    return NULL;
}

static void usage ()
{
    NOPREFIX_OUT ("usage: xproxy client FSDIR -p port [-c -n -r -R -F -D] [-L LOGDIR] -s server:port");
    NOPREFIX_OUT ("usage: xproxy server FSDIR -p port [-c -n -r -R -F -D] [-L LOGDIR]");
    exit (2);
}

int main(int argc, char** argv)
{
    int   mainPort = -1;
    int   ret;
    int   dbfsFlag = 0;
    char *dbfsDir = NULL;
    char *dbfsLog = NULL;
    DBFS  dbfs_x (argc, argv);
    bool  debugMode = false;
    char *mode;
    char *lhl = getenv ("XPROXY_LOGHEADER");
    Socket *mainSocket = NULL, *newSocket = NULL;
    pthread_t tid;

    DKEY check1 (url_dkey ("http://foo.bar/stophere?dontseethis"));
    DKEY check2 (url_dkey ("http://foo.bar/stophere?whataboutthis"));

    g_assert (check1 == check2);

    dbfs = & dbfs_x;

    if (lhl) {
	logHeaderLines = atoi (lhl);
    }

    if (argc < 5) {
	usage ();
    }

    if (strcmp (argv[1], "client") == 0) {
	clientProxyMode = true;
	srandom (42);
    } else if (strcmp (argv[1], "server") == 0) {
	clientProxyMode = false;
	srandom (24);
    } else {
	usage ();
    }

    mode    = argv[1];
    dbfsDir = argv[2];

    for (argc -= 2, argv += 2; argc > 1; argc -= 1, argv += 1) {

	char* p = argv[1];
	char* a = argc > 2 ? argv[2] : NULL;

	if (strcmp (p, "-L") == 0) {
	    if (! a) usage ();
	    dbfsLog = a;
	    argc -=1; argv += 1;
	} else if (strcmp (p, "-p") == 0) {
	    if (! a) usage ();
	    mainPort = atoi (a);
	    argc -=1; argv += 1;
	} else if (strcmp (p, "-s") == 0) {
	    if (! a) usage ();
	    char *l = strchr (a, ':');
	    if (! l) usage ();
	    peerPort = atoi (l + 1);
	    *l = 0;
	    peerHost = a;
	    argc -=1; argv += 1;
	} else if (strcmp (p, "-c") == 0) {
	    dbfsFlag |= DBFS_CREATE;
	} else if (strcmp (p, "-n") == 0) {
	    dbfsFlag |= DBFS_CLEAR;
	} else if (strcmp (p, "-r") == 0) {
	    dbfsFlag |= DBFS_RECOVER;
	} else if (strcmp (p, "-R") == 0) {
	    dbfsFlag |= DBFS_RECOVER_ADMIN;
	} else if (strcmp (p, "-F") == 0) {
	    dbfsFlag |= DBFS_ASYNC_INIT;
	} else if (strcmp (p, "-D") == 0) {
	    debugMode = true;
	}
    }

    if (dbfsDir == NULL || mainPort < 0) {
	usage ();
    }

    if (clientProxyMode && (peerPort < 0 || peerHost == NULL)) {
	usage ();
    }

    // Done with the command line!
    if (! debugMode) {
	pid_t pid;

	if ((pid = vfork ()) < 0) {
	    SYS_ERROR (errno) ("vfork");
	    return 2;
	}

	// Exit the parent process
	if (pid > 0) {
	    INFO_XPROXY ("parent process exiting: PID %d", pid);
	    exit (0);
	}

	setsid ();

	while (1) {

	    if ((pid = fork ()) < 0) {
		SYS_ERROR (errno) ("fork");
		return 2;
	    }

	    // The second child goes on to run xProxy, the first child
	    // watches it and runs recovery
	    if (pid == 0) {
		break;
	    }

	    dbfsFlag &= ~DBFS_CLEAR;
	    dbfsFlag &= ~DBFS_CREATE;
	    dbfsFlag &= ~DBFS_RECOVER;
	    dbfsFlag &= ~DBFS_RECOVER_ADMIN;

	    int status;

	  again:

	    if ((ret = waitpid (pid, &status, 0)) != pid) {
		SYS_ERROR (errno) ("waitpid");
		return 2;
	    }

	    if (WIFEXITED (status)) {
		INFO_XPROXY ("%s exited with value %d", mode, WEXITSTATUS (status));
	    } else if (WIFSIGNALED (status)) {
		INFO_XPROXY ("%s terminated with signal %d", mode, WTERMSIG (status));
	    } else {
		g_assert (WIFSTOPPED (status));
		INFO_XPROXY ("%s stopped with signal %d -- continue", mode, WSTOPSIG (status));
		kill (pid, SIGCONT);
		goto again;
	    }

	    // Wait a bit...
	    INFO_XPROXY ("waiting for all threads to die");
	    usleep (500000);

	    if ((ret = storage_recover (dbfsDir))) {
		XPROXY_ERROR (ret) ("storage_recover");
		return 2;
	    }
	}
    }

    INFO_XPROXY ("child process starting: PID %d", getpid ());

    // Ignore broken pipe signal.
    signal(SIGPIPE, SIG_IGN);

    // Initialize timezone and daylight.
    tzset();
    timeOffset = timezone - 3600 * daylight;

    // Open XDFS.
    if ((ret = storage_init (dbfsDir, dbfsLog, dbfsFlag, dbfs))) {
	XPROXY_ERROR (ret) ("storage init failed");
	return 2;
    }

    // Get socket and bind it to the given local port.
    mainSocket = new Socket();
    if (mainSocket->sockfd == -1) {
	XPROXY_ERROR("could not get main socket");
	return 2;
    }
    if (!mainSocket->bind(mainPort)) {
	XPROXY_ERROR("could not bind socket to the given local port");
	return 2;
    }

    // Put socket into listening state.
    if (!mainSocket->listen(100)) {
	XPROXY_ERROR("could not listen for connections");
	return 2;
    }

    while (1) {
	if ((newSocket = mainSocket->accept()) == NULL) {
	    return 2; // fatal error
	}
	DEBUG_VERBOSE ("Accepted new socket: %d", newSocket->sockfd);

	thread_count_incr (1);

	pthread_create(&tid, NULL, &connectionHandler, (void*) newSocket);
	pthread_detach(tid);
    }

    return 1;
}

void prepareRequest(Request *request) {
    char date[32];

    // Reset tempBuffer (used for preparing request and response headers).
    request->tempBuffer[0] = '\0';
    request->tempBufferStart = request->tempBuffer;
    request->tempBufferLength = 0;

    // Prepare request headers.
    sprintf(request->tempBuffer, "%s %s HTTP/1.1\r\n", request->requestFromClient.method, (clientProxyMode ? request->requestFromClient.url : request->requestFromClient.resource));
    strcat(request->tempBuffer, "Accept-Encoding: gzip\r\n");
    if (request->requestFromClient.eTag[0] != '\0') {
	sprintf(request->tempBuffer + strlen(request->tempBuffer), "If-None-Match: %s\r\n", request->requestFromClient.eTag);
    }
    if (request->requestFromClient.ifModifiedSince != -1 && timeToStr(request->requestFromClient.ifModifiedSince - timeOffset, date)) {
	sprintf(request->tempBuffer + strlen(request->tempBuffer), "If-Modified-Since: %s\r\n", date);
    }
    if (!clientProxyMode) {
	// For now close all connections from the server proxy to the source web servers ???
	strcat(request->tempBuffer, "Connection: close\r\n");
    }
    if (request->haveLocalMD5) {
	// Only one MD5 included for now ??? ... include a list of MD5s later
	strcat(request->tempBuffer, "A-IM: xdelta\r\nDelta-Base: ");
	encodeMIME64(request->localMD5, request->tempBuffer + strlen(request->tempBuffer), 16);
	strcat(request->tempBuffer, "\r\n");
    }
    // If POST request, need to add content length and content type.
    if (strcmp(request->requestFromClient.method, "POST") == 0) {
	sprintf(request->tempBuffer + strlen(request->tempBuffer), "Content-Length: %d\r\n", request->bodyFromServerLength);
	sprintf(request->tempBuffer + strlen(request->tempBuffer), "Content-Type: %s\r\n", request->requestFromClient.contentType);
    }
    strcat(request->tempBuffer, request->requestFromClient.additionalHeaders);
    strcat(request->tempBuffer, "\r\n");
    request->tempBufferLength = strlen(request->tempBuffer);
}

void prepareResponse(Request *request) {
    char date[32];

    // Reset tempBuffer (used for preparing request and response headers).
    request->tempBuffer[0] = '\0';
    request->tempBufferStart = request->tempBuffer;
    request->tempBufferLength = 0;

    // Prepare response headers.
    sprintf(request->tempBuffer, "HTTP/1.1 %d %s\r\n", request->responseFromServer.HTTPCode, request->responseFromServer.reasonPhrase);
    // Disregard date sent by server because it could be inaccurate since there was some processing done by the proxy on the server response; thus, always generate a date.
    if (timeToStr(time(NULL), date)) {
	sprintf(request->tempBuffer + strlen(request->tempBuffer), "Date: %s\r\n", date);
    }
    if (request->responseFromServer.expires != -1 && timeToStr(request->responseFromServer.expires - timeOffset, date)) {
	sprintf(request->tempBuffer + strlen(request->tempBuffer), "Expires: %s\r\n", date);
    }
    if (request->responseFromServer.lastModified != -1 && timeToStr(request->responseFromServer.lastModified - timeOffset, date)) {
	sprintf(request->tempBuffer + strlen(request->tempBuffer), "Last-Modified: %s\r\n", date);
    }
    if (request->responseFromServer.eTag[0] != '\0') {
	sprintf(request->tempBuffer + strlen(request->tempBuffer), "ETag: %s\r\n", request->responseFromServer.eTag);
    }
    if (request->responseFromServer.server[0] != '\0') {
	sprintf(request->tempBuffer + strlen(request->tempBuffer), "Server: %s\r\n", request->responseFromServer.server);
    }
    // When server proxy, always include content length; when client proxy, include content length only if we have a response body.
    if (!clientProxyMode || request->bodyToClientLength > 0) {
	sprintf(request->tempBuffer + strlen(request->tempBuffer), "Content-Length: %d\r\n", request->bodyToClientLength);
    }
    // Include content type only if there is a response body.
    if (request->bodyToClientLength > 0) {
	sprintf(request->tempBuffer + strlen(request->tempBuffer), "Content-Type: %s\r\n", request->responseFromServer.contentType);
    }
    if (request->responseFromServer.haveDeltaBase) {
	strcat(request->tempBuffer, "IM: xdelta\r\nDelta-Base: ");
	encodeMIME64(request->responseFromServer.deltaBase, request->tempBuffer + strlen(request->tempBuffer), 16);
	strcat(request->tempBuffer, "\r\n");
    }
    if (request->responseFromServer.contentEncodingGZIP) {
	strcat(request->tempBuffer, "Content-Encoding: gzip\r\n");
    }
    strcat(request->tempBuffer, request->responseFromServer.additionalHeaders);
    strcat(request->tempBuffer, "\r\n");
    request->tempBufferLength = strlen(request->tempBuffer);
}

void prepareMessage(Request *request, const char *status, char *description) {
    char date[32];

    // Reset tempBuffer (used for preparing request and response headers).
    request->tempBuffer[0] = '\0';
    request->tempBufferStart = request->tempBuffer;
    request->tempBufferLength = 0;

    // Generate response body.
    if (strcmp(status, STATUS_304) == 0) {
	// Don't need to send a response body for a 304 response.
	request->bodyToClientLength = 0;
    }
    else {
	// Send a response body for all other statuses.
	strcpy(request->responseFromServer.contentType, "text/html");
	sprintf(request->bodyToClient, "<html><body><h2>%s</h2><h2>%s</h2></body></html>", status, description);
	request->bodyToClientLength = strlen(request->bodyToClient);
    }

    // Prepare response headers.
    sprintf(request->tempBuffer, "HTTP/1.1 %s\r\n", status);
    // Disregard date sent by server because it could be inaccurate since there was some processing done by the proxy on the server response; thus, always generate a date.
    if (timeToStr(time(NULL), date)) {
	sprintf(request->tempBuffer + strlen(request->tempBuffer), "Date: %s\r\n", date);
    }
    if (request->responseFromServer.server[0] != '\0') {
	sprintf(request->tempBuffer + strlen(request->tempBuffer), "Server: %s\r\n", request->responseFromServer.server);
    }
    // When server proxy, always include content length; when client proxy, include content length only if we have a response body.
    if (!clientProxyMode || request->bodyToClientLength > 0) {
	sprintf(request->tempBuffer + strlen(request->tempBuffer), "Content-Length: %d\r\n", request->bodyToClientLength);
    }
    // Include content type only if there is a response body.
    if (request->bodyToClientLength > 0) {
	sprintf(request->tempBuffer + strlen(request->tempBuffer), "Content-Type: %s\r\n", request->responseFromServer.contentType);
    }

    strcat(request->tempBuffer, "\r\n");
    request->tempBufferLength = strlen(request->tempBuffer);
}

bool processClientRequest(Request *request) {
    bool fresh = false;
    time_t storedExpires, storedLastModified, currentTime;
    char storedETag[MAX_HEADER_SIZE], storedServer[MAX_HEADER_SIZE], storedContentType[MAX_HEADER_SIZE];
    MAJORC curIno;
    int ret;
    int curInoStatus;
    StorageLocation xdfs (*dbfs, SCTX ("request"), request->requestFromClient.url);

    storedLastModified = -1;
    storedETag[0] = '\0';

    if ((ret = xdfs.open ())) {
	XPROXY_ERROR (ret) ("storage_xdfs_location");
	return false;
    }

    // Lookup the latest version of this URL.
    if ((curInoStatus = xdfs.get_last_version (& curIno)) &&
	(curInoStatus != DBFS_NOTFOUND)) {
	XPROXY_ERROR (curInoStatus) ("storage_get_last_version");
	return false;
    }

    // If the latest version is found, get attributes
    if (curInoStatus == 0) {

	if ((ret = xdfs.get_md5 (& curIno, (guint8*) request->localMD5))) {
	    XPROXY_ERROR (ret) ("storage_get_md5");
	    return false;
	}

	DEBUG_PROXY ("client-request latest version was found: have MD5");

	request->localMD5[16] = '\0';
	request->haveLocalMD5 = true;

	if ((ret = storage_get_attribute(&xdfs._txn, &curIno, __XPROXY_EXP_STCK, (guint8*) &storedExpires, sizeof(time_t), false)) ||
	    (ret = storage_get_attribute(&xdfs._txn, &curIno, __XPROXY_LAST_STCK, (guint8*) &storedLastModified, sizeof(time_t), false)) ||
	    (ret = storage_get_attribute(&xdfs._txn, &curIno, __XPROXY_ETAG_STCK, (guint8*) storedETag, MAX_HEADER_SIZE, true)) ||
	    (ret = storage_get_attribute(&xdfs._txn, &curIno, __XPROXY_SERVER_STCK, (guint8*) storedServer, MAX_HEADER_SIZE, true)) ||
	    (ret = storage_get_attribute(&xdfs._txn, &curIno, __XPROXY_TYPE_STCK, (guint8*) storedContentType, MAX_HEADER_SIZE, true))) {
	    XPROXY_ERROR (ret) ("storage_get_attribute");
	    return false;
	}

	time(&currentTime);
	if (storedExpires != -1 && storedExpires >= currentTime) {
	    DEBUG_PROXY ("client-request version has not expired");
	    fresh = true;
	}
    }

    if (fresh) {
	// Can fulfill request from the cache; will generate either 200, 226 or 304 response.
	if (request->requestFromClient.haveDeltaBase) {
	    // Client provided MD5.
	    if (memcmp(request->requestFromClient.deltaBase, request->localMD5, 16) == 0) {
		// Client MD5 matches local MD5 and version is fresh; thus, we can return a 304 response.
		request->responseFromServer.HTTPCode = 304;
		strcpy(request->responseFromServer.reasonPhrase, "Not Modified");
	    }
	    else {
		// MD5s do not match.
		int oldInoStatus;
		MAJORC oldIno;

		if ((oldInoStatus = xdfs.get_version_by_md5 ((guint8*) request->requestFromClient.deltaBase, &oldIno)) &&
		    (oldInoStatus != DBFS_NOTFOUND)) {
		    XPROXY_ERROR (oldInoStatus) ("storage_get_version_by_md5");
		    return false;
		}

		if (oldInoStatus == 0) {
		    // Server (proxy) has client (proxy) MD5.
		    request->bodyToClientLength = request->bodyToClientAllocatedLength;

		    if ((ret = xdfs.extract_delta_to_buffer (&oldIno, &curIno,
							     &(request->bodyToClient), &(request->bodyToClientLength)))) {
			XPROXY_ERROR (ret) ("storage_extract_delta_to_buffer");
			return false;
		    }

		    if (request->bodyToClientLength > request->bodyToClientAllocatedLength) {
			// bodyToClient has been allocated more space.
			request->bodyToClientStart = request->bodyToClient;
			request->bodyToClientAllocatedLength = request->bodyToClientLength;
		    }
		    memcpy(request->responseFromServer.deltaBase, request->requestFromClient.deltaBase, 16); // for now, just return the same delta base because we are not sending list of MD5s in the request ???
		    request->responseFromServer.deltaBase[16] = '\0';
		    request->responseFromServer.haveDeltaBase = true;
		    request->responseFromServer.HTTPCode = 226;
		    strcpy(request->responseFromServer.reasonPhrase, "IM Used");
		}
		else {
		    // Server (proxy) does not have client (proxy) MD5, send whole file back.
		    if ((request->bodyToClientLength =
			 storage_read_inode_into_buffer (&xdfs._txn, &curIno,
							 &(request->bodyToClient),
							 &(request->bodyToClientAllocatedLength))) == -1) {
			XPROXY_ERROR ("storage_read_inode_into_buffer");
			return false;
		    }

		    request->bodyToClientStart = request->bodyToClient; // just in case bodyToClient has been reallocated
		    request->responseFromServer.HTTPCode = 200;
		    strcpy(request->responseFromServer.reasonPhrase, "OK");
		}
	    }
	}
	else {
	    // Client didn't provide MD5; if client provided eTag or ifModifiedSince we might still be able to send a 304 response back; otherwise, send the whole file.
	    // Note that only one etag is accepted; if a list of etags is provided, then it will not match a (single) stored etag.
	    bool sendWholeFile = true;
	    if (storedETag[0] != '\0') {
		// Exact match on etag and modification time.
		if ((strcmp(request->requestFromClient.eTag, storedETag) == 0) && request->requestFromClient.ifModifiedSince == storedLastModified) {
		    DEBUG_PROXY ("PROCESS_CLIENT_REQUEST: SENDWHOLEFILE = FALSE: IMS and ETAGS match for %s\n", request->requestFromClient.url);
		    sendWholeFile = false;
		}
	    }
	    else {
		// Examine only modification time.
		if (storedLastModified != -1 && request->requestFromClient.ifModifiedSince >= storedLastModified) {
		    DEBUG_PROXY ("PROCESS_CLIENT_REQUEST: SENDWHOLEFILE = FALSE: IMS matches for %s\n", request->requestFromClient.url);
		    sendWholeFile = false;
		}
	    }
	    if (!sendWholeFile) {
		request->responseFromServer.HTTPCode = 304;
		strcpy(request->responseFromServer.reasonPhrase, "Not Modified");
	    }
	    else {
		// Send whole file back.
		if ((request->bodyToClientLength =
		     storage_read_inode_into_buffer(&xdfs._txn, &curIno,
						    &(request->bodyToClient),
						    &(request->bodyToClientAllocatedLength))) == -1) {
		    XPROXY_ERROR ("storage_read_inode_into_buffer");
		    return false;
		}

		request->bodyToClientStart = request->bodyToClient; // just in case bodyToClient has been reallocated
		request->responseFromServer.HTTPCode = 200;
		strcpy(request->responseFromServer.reasonPhrase, "OK");
	    }
	}

	// This is necessary for all three types of responses (200, 226, 304).
	strcpy(request->responseFromServer.server, storedServer);
	if (request->responseFromServer.HTTPCode != 304) {
	    request->responseFromServer.expires = storedExpires;
	    request->responseFromServer.lastModified = storedLastModified;
	    strcpy(request->responseFromServer.eTag, storedETag);
	    strcpy(request->responseFromServer.contentType, storedContentType);

	    // Compress only if server proxy and only text files.
	    if (!clientProxyMode && strstr(request->responseFromServer.contentType, "text/") == request->responseFromServer.contentType) {
		// If server proxy, compress result (full version or delta).
		int tempLength = (int) ((((request->bodyToClientLength + 12) * 1.01) > MAX_FILE_SIZE) ? ((request->bodyToClientLength + 12) * 1.01) : MAX_FILE_SIZE); // zlib says destination buffer should be 0.1% (use here 1%) more than source buffer length plus 12 bytes
		char *temp = new char[tempLength];

		request->bodyToClientAllocatedLength = tempLength;
		z_stream c_stream;
		c_stream.zalloc = (alloc_func) 0;
		c_stream.zfree = (free_func) 0;
		c_stream.opaque = (voidpf) 0;
		deflateInit2(&c_stream, Z_DEFAULT_COMPRESSION, Z_DEFLATED, -MAX_WBITS, MAX_MEM_LEVEL, Z_DEFAULT_STRATEGY);
		uLong tail[2];
		tail[0] = crc32(0L, Z_NULL, 0);
		int gz_magic[2] = {0x1f, 0x8b}; /* gzip magic header */
		sprintf(temp, "%c%c%c%c%c%c%c%c%c%c", gz_magic[0], gz_magic[1], Z_DEFLATED, 0 /*flags*/, 0,0,0,0 /*time*/, 0 /*xflags*/, OS_CODE);
		c_stream.next_in = (unsigned char*) request->bodyToClientStart;
		c_stream.avail_in = request->bodyToClientLength;
		c_stream.next_out = (unsigned char*) (temp + 10);
		c_stream.avail_out = (unsigned int) (tempLength - 10);
		deflate(&c_stream, Z_FINISH);
		tail[0] = convLong(crc32(tail[0], (unsigned char*) request->bodyToClientStart, request->bodyToClientLength));
		tail[1] = convLong(c_stream.total_in);
		memcpy(c_stream.next_out, tail, 8);
		deflateEnd(&c_stream);
		delete [] request->bodyToClient;
		request->bodyToClient = temp;
		request->bodyToClientStart = request->bodyToClient;
		request->bodyToClientLength = ((unsigned int) c_stream.next_out) + 8 - ((unsigned int) temp);
		request->responseFromServer.contentEncodingGZIP = true;

		DEBUG_PROXY ("COMPRESS: %s\n", request->requestFromClient.url);
	    }
	}
    }
    else {
	// If not fresh, send request to server (proxy).
	// Save ifModifiedSince and eTag sent in the client request.
	request->requestFromClient.originalIfModifiedSince = request->requestFromClient.ifModifiedSince;
	strcpy(request->requestFromClient.originalETag, request->requestFromClient.eTag);

	// Determine if conditional request should be made.
	bool sendConditionalRequest = false;
	if (request->requestFromClient.hasCookie) {
	    // If request contains cookies, make conditional request only if incoming request refers to the same version as the one cached (if any); compare versions, first, based on MD5s (if possible) and, second, based on modification dates and etags (if can't do MD5-based comparison).
	    if (request->requestFromClient.haveDeltaBase) {
		// Compare versions based on MD5s.
		if (request->haveLocalMD5 && (memcmp(request->requestFromClient.deltaBase, request->localMD5, 16) == 0)) {
		    DEBUG_PROXY ("CONDITIONAL REQUEST: MD5s match for %s", request->requestFromClient.url);
		    sendConditionalRequest = true;
		}
	    }
	    else {
		// Compare versions based on modification dates and etags.
		// NOTE: to support browsers (such as Netscape) that don't send etags (even though they're provided by the servers), I've had to decouple the two checks below because such browsers only send IMS in the request (no etag) and the client proxy has an etag, thus etags won't match, so we're trying to get a match on the modification time.
		if (strcmp(request->requestFromClient.eTag, storedETag) == 0) {
		    DEBUG_PROXY ("CONDITIONAL REQUEST: ETAGS match for %s", request->requestFromClient.url);
		    sendConditionalRequest = true;
		}
		if (request->requestFromClient.ifModifiedSince == storedLastModified) {
		    DEBUG_PROXY ("CONDITIONAL REQUEST: IMS matches for %s", request->requestFromClient.url);
		    sendConditionalRequest = true;
		}
	    }
	}
	else {
	    // If request doesn't contain cookies, make conditional request.
	    DEBUG_VERBOSE ("CONDITIONAL REQUEST: no cookies for %s", request->requestFromClient.url);
	    sendConditionalRequest = true;
	}
	if (sendConditionalRequest) {
	    request->requestFromClient.ifModifiedSince = storedLastModified;
	    strcpy(request->requestFromClient.eTag, storedETag);
	}
	else {
	    request->requestFromClient.ifModifiedSince = -1;
	    request->requestFromClient.eTag[0] = '\0';
	}
    }

    request->currentSize = request->bodyToClientLength;

    if ((ret = xdfs.commit ())) {
	XPROXY_ERROR (ret) ("txn_commit");
	return false;
    }

    GTIMER_STOP(request->processClientRequestTimer);
    request->processClientRequestTime = (int) (1000000 * GTIMER_ELAPSED(request->processClientRequestTimer));

    return true;
}

bool processServerResponse(Request *request) {
    MAJORC newIno, oldIno;
    char lastMD5[17];
    time_t storedExpires, storedLastModified;
    char storedETag[MAX_HEADER_SIZE], storedServer[MAX_HEADER_SIZE], storedContentType[MAX_HEADER_SIZE];
    char *compressedBodyFromServer = NULL;
    int compressedBodyFromServerLength, compressedBodyFromServerAllocatedLength;
    int ret;
    StorageLocation xdfs (*dbfs, SCTX ("response"), request->requestFromClient.url);

    GTIMER_START(request->processServerResponseTimer);

    if ((ret = xdfs.open ())) {
	XPROXY_ERROR (ret) ("storage_xdfs_location");
	return false;
    }

    request->originalSize = request->bodyFromServerLength;

    if (request->responseFromServer.HTTPCode == 200 || request->responseFromServer.HTTPCode == 226) {
	// Handle 200 and 226 responses.
	if (request->responseFromServer.contentEncodingGZIP) {
	    compressedBodyFromServer = request->bodyFromServer;
	    compressedBodyFromServerLength = request->bodyFromServerLength;
	    compressedBodyFromServerAllocatedLength = request->bodyFromServerAllocatedLength;

	    // Decompress page if compressed; be conservative and assume an output buffer 8 times bigger than compressed input or MAX_FILE_SIZE, whichever is larger.
	    int tempLength = (((request->bodyFromServerLength * 8) > MAX_FILE_SIZE) ? (request->bodyFromServerLength * 8) : MAX_FILE_SIZE);
	    char *temp = new char[tempLength];
	    z_stream c_stream;
	    c_stream.zalloc = (alloc_func) 0;
	    c_stream.zfree = (free_func) 0;
	    c_stream.opaque = (voidpf) 0;
	    c_stream.next_in= (unsigned char*) request->bodyFromServerStart + 10;
	    c_stream.avail_in = request->bodyFromServerLength - 10;

	    request->bodyFromServerAllocatedLength = tempLength;
	    request->responseFromServer.contentEncodingGZIP = false;
	    inflateInit2(&c_stream, -MAX_WBITS);
	    c_stream.next_out = (unsigned char*) temp;
	    c_stream.avail_out = (unsigned int) tempLength;
	    inflate(&c_stream, Z_FINISH);
	    //delete [] request->bodyFromServer;
	    request->bodyFromServerLength = ((unsigned int) c_stream.next_out) - ((unsigned int) temp);
	    request->bodyFromServer = temp;
	    request->bodyFromServerStart = request->bodyFromServer;
	    inflateEnd(&c_stream);

	    DEBUG_VERBOSE ("DECOMPRESS: %s", request->requestFromClient.url);
	}

	if(request->responseFromServer.haveDeltaBase) {
	    MAJORC againstIno;

	    g_assert (request->haveLocalMD5);

	    if ((ret = xdfs.get_version_by_md5 ((guint8*) request->localMD5, & againstIno))) {
		XPROXY_ERROR (ret) ("storage_get_version_by_md5--recently deleted?");
		return false;
	    }

	    // Use delta to reconstruct the page.
	    if ((ret = xdfs.insert_delta_from_buffer (& againstIno, request->bodyFromServerStart,
						      request->bodyFromServerLength, &newIno))) {
		XPROXY_ERROR (ret) ("storage_insert_delta_from_buffer");
		return false;
	    }
	}
	else {
	    // Insert whole version.

	    if ((ret = xdfs.insert_version_from_buffer (request->bodyFromServerStart,
							request->bodyFromServerLength, & newIno))) {
		XPROXY_ERROR (ret) ("storage_insert_version_from_buffer");
		return false;
	    }
	}
    }
    else {
	// Handle 304 response.
	if ((ret = xdfs.get_last_version(&newIno))) {
	    XPROXY_ERROR (ret) ("storage_get_last_version");
	    return false;
	}
    }

    if (request->responseFromServer.HTTPCode == 304) {
	// Retrieve stored attributes.
	if ((ret = storage_get_attribute(&xdfs._txn, &newIno, __XPROXY_LAST_STCK, (guint8*) &storedLastModified, sizeof(time_t), false)) ||
	    (ret = storage_get_attribute(&xdfs._txn, &newIno, __XPROXY_EXP_STCK, (guint8*) &storedExpires, sizeof(time_t), false)) ||
	    (ret = storage_get_attribute(&xdfs._txn, &newIno, __XPROXY_ETAG_STCK, (guint8*) storedETag, MAX_HEADER_SIZE, true)) ||
	    (ret = storage_get_attribute(&xdfs._txn, &newIno, __XPROXY_SERVER_STCK, (guint8*) storedServer, MAX_HEADER_SIZE, true)) ||
	    (ret = storage_get_attribute(&xdfs._txn, &newIno, __XPROXY_TYPE_STCK, (guint8*) storedContentType, MAX_HEADER_SIZE, true))) {
	    XPROXY_ERROR (ret) ("storage_get_attribute");
	    return false;
	}
    }
    else {
	// Set various attributes for the new version.
	// Always store a valid lastModified to make conditional requests later.
	storedLastModified = ((request->responseFromServer.lastModified == -1) ? time(NULL) : request->responseFromServer.lastModified);
	// Calculate expiration time.
	if (request->responseFromServer.maxAge != -1) {
	    // Give priority to max-age (if any).
	    if (request->responseFromServer.maxAge == 0) {
		storedExpires = 0; // "already expired", RFC-compliant caches should revalidate this response in the future
	    }
	    else {
		if (request->responseFromServer.date == -1) {
		    // This should never happen (but just in case); really strange to have a max-age, but not a Date in the response.
		    request->responseFromServer.date = time(NULL) + timeOffset; // now we have GMT time
		}
		storedExpires = request->responseFromServer.date + request->responseFromServer.maxAge;
	    }
	}
	else {
	    // Otherwise, use the Expires header (if any).
	    storedExpires = request->responseFromServer.expires;
	}
	strcpy(storedETag, request->responseFromServer.eTag);
	strcpy(storedServer, request->responseFromServer.server);
	strcpy(storedContentType, request->responseFromServer.contentType);
	if ((ret = storage_set_attribute(&xdfs._txn, &newIno, __XPROXY_LAST_STCK, (guint8*) &storedLastModified, sizeof(time_t))) ||
	    (ret = storage_set_attribute(&xdfs._txn, &newIno, __XPROXY_EXP_STCK, (guint8*) &storedExpires, sizeof(time_t))) ||
	    (ret = storage_set_attribute(&xdfs._txn, &newIno, __XPROXY_ETAG_STCK, (guint8*) storedETag, strlen (storedETag)+1)) ||
	    (ret = storage_set_attribute(&xdfs._txn, &newIno, __XPROXY_SERVER_STCK, (guint8*) storedServer, strlen (storedServer)+1)) ||
	    (ret = storage_set_attribute(&xdfs._txn, &newIno, __XPROXY_TYPE_STCK, (guint8*) storedContentType, strlen (storedContentType)+1))) {
	    XPROXY_ERROR (ret) ("storage_set_attribute");
	    return false;
	}
    }

    // By this point we have a complete page and with the correct attributes obtained from the response or from XDFS (if 304 response).

    if (request->requestFromClient.haveDeltaBase) {
	if ((ret = xdfs.get_md5 (&newIno, (guint8*) lastMD5))) {
	    XPROXY_ERROR (ret) ("storage_get_md5");
	    return false;
	}
	if (memcmp(request->requestFromClient.deltaBase, lastMD5, 16) == 0) {
	    // Client (proxy) has most recent version.
	    request->responseFromServer.HTTPCode = 304;
	    strcpy(request->responseFromServer.reasonPhrase, "Not Modified");
	}
	else {

	    if ((ret = xdfs.get_version_by_md5((guint8*) request->requestFromClient.deltaBase, &oldIno)) &&
		(ret != DBFS_NOTFOUND)) {
		XPROXY_ERROR (ret) ("storage_get_version_by_md5");
		return false;
	    }

	    if (ret == 0) {
		// Server proxy has client (proxy) MD5.
		request->bodyToClientLength = request->bodyToClientAllocatedLength;

		if ((ret = xdfs.extract_delta_to_buffer(&oldIno, &newIno,
							&(request->bodyToClient), &(request->bodyToClientLength)))) {
		    XPROXY_ERROR (ret) ("storage_extract_delta_to_buffer");
		    return false;
		}

		if (request->bodyToClientLength > request->bodyToClientAllocatedLength) {
		    // bodyToClient has been allocated more space.
		    request->bodyToClientStart = request->bodyToClient;
		    request->bodyToClientAllocatedLength = request->bodyToClientLength;
		}

		// for now, just return the same delta base because we
		// are not sending list of MD5s in the request ???
		memcpy(request->responseFromServer.deltaBase, request->requestFromClient.deltaBase, 16);
		request->responseFromServer.deltaBase[16] = '\0';
		request->responseFromServer.haveDeltaBase = true;
		request->responseFromServer.HTTPCode = 226;
		strcpy(request->responseFromServer.reasonPhrase, "IM Used");
	    }
	    else {
		// Server (proxy) does not have client (proxy) MD5, send entire file back.
		if (compressedBodyFromServer != NULL &&
		    request->responseFromServer.HTTPCode == 200 &&
		    ! clientProxyMode) {

		    // Send already compressed response from server if one is available.
		    delete [] request->bodyToClient;
		    request->bodyToClient = compressedBodyFromServer;
		    request->bodyToClientLength = compressedBodyFromServerLength;
		    request->bodyToClientAllocatedLength = compressedBodyFromServerAllocatedLength;
		    request->responseFromServer.contentEncodingGZIP = true;
		    compressedBodyFromServer = NULL;
		}
		else {

		    if ((request->bodyToClientLength = storage_read_inode_into_buffer(&xdfs._txn, &newIno, &(request->bodyToClient), &(request->bodyToClientAllocatedLength))) == -1) {
			XPROXY_ERROR ("storage_read_inode_into_buffer");
			return false;
		    }
		}
		request->bodyToClientStart = request->bodyToClient; // just in case bodyToClient has been reallocated
		request->responseFromServer.HTTPCode = 200;
		strcpy(request->responseFromServer.reasonPhrase, "OK");
	    }
	}
    }
    else {
	// Client (proxy) didn't provide an MD5, send entire file back; if client (proxy) provided eTag or ifModifiedSince, might be able to send 304 response.
	// Note that only one etag is accepted; if a list of etags is provided, then it will not match a (single) stored etag.
	bool sendWholeFile = true;
	if (storedETag[0] != '\0') {
	    // Exact match on etag and modification time.
	    if ((strcmp(request->requestFromClient.originalETag, storedETag) == 0) && request->requestFromClient.originalIfModifiedSince == storedLastModified) {
		DEBUG_PROXY ("PROCESS_SERVER_RESPONSE: SENDWHOLEFILE = FALSE: IMS and ETAGS match for %s\n", request->requestFromClient.url);
		sendWholeFile = false;
	    }
	}
	else {
	    // Examine only modification time.
	    if (storedLastModified != -1 && request->requestFromClient.originalIfModifiedSince >= storedLastModified) {
		DEBUG_PROXY ("PROCESS_SERVER_RESPONSE: SENDWHOLEFILE = FALSE: IMS matches for %s\n", request->requestFromClient.url);
		sendWholeFile = false;
	    }
	}
	if (!sendWholeFile) {
	    request->responseFromServer.HTTPCode = 304;
	    strcpy(request->responseFromServer.reasonPhrase, "Not Modified");
	}
	else {
	    // Send whole file back.
	    if (compressedBodyFromServer != NULL && request->responseFromServer.HTTPCode == 200 && !clientProxyMode) {
		// Send already compressed response from server if one is available.
		delete [] request->bodyToClient;
		request->bodyToClient = compressedBodyFromServer;
		request->bodyToClientLength = compressedBodyFromServerLength;
		request->bodyToClientAllocatedLength = compressedBodyFromServerAllocatedLength;
		request->responseFromServer.contentEncodingGZIP = true;
		compressedBodyFromServer = NULL;
	    }
	    else {
		if ((request->bodyToClientLength = storage_read_inode_into_buffer(&xdfs._txn, &newIno, &(request->bodyToClient), &(request->bodyToClientAllocatedLength))) == -1) {
		    XPROXY_ERROR ("storage_read_inode_into_buffer");
		    return false;
		}
	    }
	    request->bodyToClientStart = request->bodyToClient; // just in case bodyToClient has been reallocated
	    request->responseFromServer.HTTPCode = 200;
	    strcpy(request->responseFromServer.reasonPhrase, "OK");
	}
    }

    if (compressedBodyFromServer != NULL) {
	DEBUG_VERBOSE ("DELETE compressedBodyFromServer: %s\n", request->requestFromClient.url);
	delete [] compressedBodyFromServer;
    }

    // This is necessary for all three types of responses (200, 226, 304).
    strcpy(request->responseFromServer.server, storedServer);
    if (request->responseFromServer.HTTPCode != 304) {
	request->responseFromServer.expires = storedExpires;
	request->responseFromServer.lastModified = storedLastModified;
	strcpy(request->responseFromServer.eTag, storedETag);
	strcpy(request->responseFromServer.contentType, storedContentType);
	if (request->responseFromServer.HTTPCode == 200) {
	    request->responseFromServer.haveDeltaBase = false; // clear out previous haveDeltaBase
	}

	// Compress only if server proxy and only text files (and only if response is not already compressed).
	if (!clientProxyMode && (strstr(request->responseFromServer.contentType, "text/") == request->responseFromServer.contentType) && !request->responseFromServer.contentEncodingGZIP) {
	    // If server proxy, compress result (full version or delta).
	    int tempLength = (int) ((((request->bodyToClientLength + 12) * 1.01) > MAX_FILE_SIZE) ? ((request->bodyToClientLength + 12) * 1.01) : MAX_FILE_SIZE); // zlib says destination buffer should be 0.1% (use here 1%) more than source buffer length plus 12 bytes
	    char *temp = new char[tempLength];

	    request->bodyToClientAllocatedLength = tempLength;
	    z_stream c_stream;
	    c_stream.zalloc = (alloc_func) 0;
	    c_stream.zfree = (free_func) 0;
	    c_stream.opaque = (voidpf) 0;
	    deflateInit2(&c_stream, Z_DEFAULT_COMPRESSION, Z_DEFLATED, -MAX_WBITS, MAX_MEM_LEVEL, Z_DEFAULT_STRATEGY);
	    uLong tail[2];
	    tail[0] = crc32(0L, Z_NULL, 0);
	    int gz_magic[2] = {0x1f, 0x8b}; /* gzip magic header */
	    sprintf(temp, "%c%c%c%c%c%c%c%c%c%c", gz_magic[0], gz_magic[1], Z_DEFLATED, 0 /*flags*/, 0,0,0,0 /*time*/, 0 /*xflags*/, OS_CODE);
	    c_stream.next_in = (unsigned char*) request->bodyToClientStart;
	    c_stream.avail_in = request->bodyToClientLength;
	    c_stream.next_out = (unsigned char*) (temp + 10);
	    c_stream.avail_out = (unsigned int) (tempLength - 10);
	    deflate(&c_stream, Z_FINISH);
	    tail[0] = convLong(crc32(tail[0], (unsigned char*) request->bodyToClientStart, request->bodyToClientLength));
	    tail[1] = convLong(c_stream.total_in);
	    memcpy(c_stream.next_out, tail, 8);
	    deflateEnd(&c_stream);
	    delete [] request->bodyToClient;
	    request->bodyToClient = temp;
	    request->bodyToClientStart = request->bodyToClient;
	    request->bodyToClientLength = ((unsigned int) c_stream.next_out) + 8 - ((unsigned int) temp);
	    request->responseFromServer.contentEncodingGZIP = true;

	    DEBUG_VERBOSE ("COMPRESS: %s", request->requestFromClient.url);
	}
    }
    else {
	request->bodyToClientLength = 0;
    }

    request->currentSize = request->bodyToClientLength;

    if ((ret = xdfs.commit ())) {
	XPROXY_ERROR (ret) ("txn_commit");
	return false;
    }

    GTIMER_STOP(request->processServerResponseTimer);
    request->processServerResponseTime = (int) (1000000 * GTIMER_ELAPSED(request->processServerResponseTimer));

    return true;
}

bool sendHTTPMessage(Request *request, MessageType messageType) {
    Socket *s;
    char **bodyStart;
    int *bodyLength, n;

    if (messageType == REQUEST) {
	s = request->out;
	bodyStart = &(request->bodyFromServerStart);
	bodyLength = &(request->bodyFromServerLength);
    }
    else {
	s = request->in;
	bodyStart = &(request->bodyToClientStart);
	bodyLength = &(request->bodyToClientLength);
    }

    // Send headers first.
    if (request->tempBufferLength > 0) {
	// SIGPIPE is ignored, so will get -1 if the other end closed the connection.
	if ((n = s->sendBytes(request->tempBufferStart, request->tempBufferLength)) == -1) {
	    XPROXY_ERROR ("ERROR: thread %d could not send request/response headers\n", (int) pthread_self());
	    return false;
	}
	request->tempBufferStart += n;
	request->tempBufferLength -= n;
    }
    // Send the entity body.
    else if (*bodyLength > 0) {
	// SIGPIPE is ignored, so will get -1 if the other end closed the connection.
	if ((n = s->sendBytes(*bodyStart, *bodyLength)) == -1) {
	    XPROXY_ERROR ("ERROR: thread %d could not send request/response body\n", (int) pthread_self());
	    return false;
	}

	DEBUG_VERBOSE ("SEND: %d bytes\n", n);

	*bodyStart += n;
	*bodyLength -= n;
    }

    return true;
}

unsigned long convLong(uLong x) {
    unsigned long xL;
    char* xP = (char*) &xL;
    int n;

    for (n = 0; n < 4; n++) {
	xP[n] = (unsigned char) (x & 0xff);
	x >>= 8;
    }

    return xL;
}
