/*
 *	Xenophilia GTK+ Theme Engine
 * 
 *  xeno_theme_draw.c:
 *		Drawing routines.
 *
 *	Copyright  1999-2001 Johan Hanson <johan@tiq.com>.
 *	
 *	This library is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU Library General Public
 *	License as published by the Free Software Foundation; either
 *	version 2 of the License, or (at your option) any later version.
 *	
 *	This library is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *	Library General Public License for more details.
 *	
 *	You should have received a copy of the GNU Library General Public
 *	License along with this library; if not, write to the 
 *	Free Software Foundation, Inc., 59 Temple Place - Suite 330, 
 *	Boston, MA  02111-1307  USA.
 */

#include "xeno_theme_rc.h"
#include "xeno_theme_style.h"
#include "xeno_theme_draw.h"
#include "xeno_theme_color.h"
#include "xeno_theme_images.h"

#define SIGN(exp)	((exp)<0 ? -1 : 1)

#define XENO_RADIO_SIZE			13
#define XENO_RADIO_MENU_SIZE	6

#define DIRTY_GRADIENTS			1
#define COUNT_GRADIENT_STYLE	1

#if XENO_GTK2
#define RANGE_CLASS(widget)		GTK_RANGE_GET_CLASS(widget)
#else
#define RANGE_CLASS(widget)		GTK_RANGE_CLASS(((GtkObject *)(widget))->klass)
#endif

/* XenoShadowContext */
enum {
	XENO_TOP_GC	= 0,
	XENO_BOT_GC	= 1,
	XENO_MID_GC = 2,
};

typedef struct {
	GdkGC *	gc[3];
} XenoGCRing;

typedef struct {
	XenoGCRing	ring[3];
	gint		thickness;
} XenoShadowContext;

/* XenoGradient */
#if !DIRTY_GRADIENTS
typedef struct _XenoGradientApp	XenoGradientApp;
struct _XenoGradientApp {
	XenoGradientApp	**ptr, *next;
	XenoGradient	*gradient;
	GtkWidget		*widget;
}; /* 16 or 32 bytes */
#endif

struct _XenoGradient {
	XenoGradient	**ptr, *next;
	GtkStyle		*style;
	
	GdkPixmap		*pixmap[5];
  #if DIRTY_GRADIENTS
	gint			ref_count;
  #else
	XenoGradientApp	*apps;
  #endif
	guint16			width, height;
};

typedef struct _XenoGradientContext XenoGradientContext;
typedef void (* XenoGradientPixelFunc)(XenoGradientContext *, gfloat, gint, gint);

struct _XenoGradientContext {
	XenoGradientPixelFunc put_pixel;
	GdkImage	*image;
	GdkVisual	*visual;
	GdkColormap	*colormap;
	
	gfloat		alpha;
	gfloat		red_factor;
	gfloat		green_factor;
	gfloat		blue_factor;
	
	XenoColor	top;
	XenoColor	bot;
	XenoColor	bg;
	
	guint		top_pixel;
	guint		bot_pixel;
	guint		bg_pixel;
	
	gboolean	blend;
};


/*
 *	Prototypes
 */

/* Background */
static void	xeno_draw_bg_rectangle	(GtkStyle		*style,
									 GdkWindow		*window,
									 GtkStateType	state_type,
									 GdkRectangle	*area,
									 GtkWidget		*widget,
									 gint			x,
									 gint			y,
									 gint			width,
									 gint			height);

static void	xeno_draw_base_rectangle(GtkStyle		*style,
									 GdkWindow		*window,
									 GtkStateType	state_type,
									 GdkRectangle	*area,
									 GtkWidget		*widget,
									 gint			x,
									 gint			y,
									 gint			width,
									 gint			height);

/* Gradients */
static GdkPixmap *xeno_gradient_get	(GtkStyle		*style,
									 GdkWindow		*window,
									 GtkStateType 	state_type,
									 GtkWidget		*widget,
									 gint			width,
									 gint			height);

/* Images */
static void	xeno_draw_image			(GdkWindow		*window,
									 GtkWidget		*widget,
									 GdkRectangle	*area,
									 XenoImageType	image_type,
									 guint16		srcx,
									 guint16		srcy,
									 guint16		x,
									 guint16		y,
									 guint16		width,
									 guint16		height);

static void	xeno_draw_pixmap		(GdkWindow		*window,
									 GdkGC			*gc,
									 GdkRectangle	*area,
									 GdkPixmap		*pixmap,
									 GdkBitmap		*mask,
									 guint16		srcx,
									 guint16		srcy,
									 guint16		x,
									 guint16		y,
									 guint16		width,
									 guint16		height);

/* Decoration */
static void xeno_draw_lines			(GtkStyle		*style,
									 GdkWindow		*window,
									 GtkStateType	state_type,
									 GtkShadowType	shadow_type,
									 GdkRectangle	*area,
									 GtkWidget		*widget,
									 gint			x,
									 gint			y,
									 gint			width,
									 gint			height,
									 GtkOrientation	orientation);

static void xeno_draw_buds			(GtkStyle		*style,
									 GdkWindow		*window,
									 GtkStateType	state_type,
									 GtkShadowType	shadow_type,
									 GdkRectangle	*area,
									 GtkWidget		*widget,
									 gint			x,
									 gint			y,
									 gint			width,
									 gint			height,
									 GtkOrientation	orientation);

static void	xeno_draw_solid_arrow2	(GtkStyle		*style,
									 GdkWindow		*window,
									 GtkStateType	state_type,
									 GdkRectangle	*area,
									 GtkWidget		*widget,
									 GtkArrowType	arrow_type,
									 gint			x, 
									 gint			y, 
									 gint			width,
									 gint			height);


/*
 *	Static data
 */
#if !DIRTY_GRADIENTS
static GMemChunk *	xeno_gradient_app_chunk	= NULL;
#endif
static GMemChunk *	xeno_gradient_chunk		= NULL;
static GQuark		xeno_gradient_quark		= 0;
static gint			xeno_gradient_ref_count	= 0;
static gint			xeno_gradient_n			= 0;


/*** GtkStyleClass Methods ****************************************************/
void
xeno_draw_hline (GtkStyle     *style,
				 GdkWindow    *window,
				 GtkStateType  state_type,
				 GdkRectangle  *area,
				 GtkWidget     *widget,
				 xeno_char     *detail,
				 gint          x1,
				 gint          x2,
				 gint          y)
{
	gint thickness_light;
	gint thickness_dark;

	GtkWidget *cur;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);
	
	thickness_light = XENO_STYLE_YTHICKNESS(style) / 2;
	thickness_dark  = XENO_STYLE_YTHICKNESS(style) - thickness_light;

	if (detail) {
		if (!strcmp (detail, "label")) {
			if (area)
				gdk_gc_set_clip_rectangle (style->fg_gc[state_type], area);
			
			gdk_draw_line (window, style->fg_gc[state_type], x1, y, x2-1, y);
			
			if (area)
				gdk_gc_set_clip_rectangle (style->fg_gc[state_type], NULL);
			return;
		} else if (!strcmp (detail, "vpaned")) {
			return; /* do not draw */
		}
	}
	
	if (area) {
		gdk_gc_set_clip_rectangle (style->light_gc [state_type], area);
		gdk_gc_set_clip_rectangle (style->dark_gc [state_type], area);
		gdk_gc_set_clip_rectangle (style->mid_gc [state_type], area);
	}

	if ((widget)) {
		if (GTK_IS_TEAROFF_MENU_ITEM(widget)) {
			goto draw_normal;
		} else if ((cur = widget->parent)) {
			while (cur && GTK_CONTAINER(cur)->border_width == 0) {
				if (GTK_IS_BOX(cur) || GTK_IS_TABLE(cur)) {
					cur = cur->parent;
				} else if (GTK_IS_WINDOW(cur)) {
					if (XENO_STYLE_RC_DATA(style) && XENO_STYLE_RC_DATA(style)->flat_windows)
						goto draw_normal;
					goto draw_inframe;
				} else if (GTK_IS_FRAME(cur) || GTK_IS_MENU(cur)) {
					goto draw_ugly;
				} else {
					break;
				}
			}
		}
	}

  draw_normal:
	gdk_draw_line (window, style->dark_gc[state_type], x1,   y,   x2-2, y);
	gdk_draw_line (window, style->light_gc[state_type], x1+1, y+1, x2-1, y+1);
	gdk_draw_point (window, style->mid_gc[state_type], x1,   y+1);
	gdk_draw_point (window, style->mid_gc[state_type], x2-1, y);
	goto draw_done;
	
  draw_ugly:
	gdk_draw_line (window, style->dark_gc[state_type], x1, y,   x2-1, y);
	gdk_draw_line (window, style->light_gc[state_type], x1, y+1, x2-1, y+1);
	goto draw_done;
	
  draw_inframe:
	gdk_draw_line (window, style->dark_gc[state_type], x1+1, y,   x2-1, y);
	gdk_draw_line (window, style->light_gc[state_type], x1,   y+1, x2-2, y+1);
	gdk_draw_point (window, style->mid_gc[state_type], x1,   y);
	gdk_draw_point (window, style->mid_gc[state_type], x2-1, y+1);

  draw_done:
	if (area) {
		gdk_gc_set_clip_rectangle (style->light_gc [state_type], NULL);
		gdk_gc_set_clip_rectangle (style->dark_gc [state_type], NULL);
		gdk_gc_set_clip_rectangle (style->mid_gc [state_type], NULL);
	}
}


void
xeno_draw_vline (GtkStyle     *style,
				 GdkWindow    *window,
				 GtkStateType  state_type,
				 GdkRectangle  *area,
				 GtkWidget     *widget,
				 xeno_char     *detail,
				 gint          y1,
				 gint          y2,
				 gint          x)
{
	gint thickness_light;
	gint thickness_dark;

	GtkWidget *cur;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);
	
	thickness_light = XENO_STYLE_XTHICKNESS(style) / 2;
	thickness_dark  = XENO_STYLE_XTHICKNESS(style) - thickness_light;
	
	if (detail) {
		if (!strcmp (detail, "label")) {
			if (area) gdk_gc_set_clip_rectangle (style->fg_gc[state_type], area);
			gdk_draw_line (window, style->fg_gc[state_type], x, y1, x, y2-1);
			if (area) gdk_gc_set_clip_rectangle (style->fg_gc[state_type], NULL);
			return;
		} else if (!strcmp (detail, "hpaned")) {
			return; /* do not draw */
		}
	}
	
	if (area) {
		gdk_gc_set_clip_rectangle (style->light_gc [state_type], area);
		gdk_gc_set_clip_rectangle (style->dark_gc [state_type], area);
		gdk_gc_set_clip_rectangle (style->mid_gc [state_type], area);
	}

	if ((widget) && (cur = widget->parent)) {
		while (cur && GTK_CONTAINER(cur)->border_width == 0) {
			if (GTK_IS_BOX(cur) || GTK_IS_TABLE(cur)) {
				cur = cur->parent;
			} else if (GTK_IS_WINDOW(cur)) {
				if (XENO_STYLE_RC_DATA(style) && XENO_STYLE_RC_DATA(style)->flat_windows)
					goto draw_normal;
				goto draw_inframe;
			} else if (GTK_IS_FRAME(cur) || GTK_IS_MENU(cur)) {
				goto draw_ugly;
			} else {
				break;
			}
		}
	}
	
  draw_normal:
	gdk_draw_line (window, style->dark_gc [state_type], x,   y1+1, x,   y2-1);
	gdk_draw_line (window, style->light_gc[state_type], x+1, y1,   x+1, y2-2);
	gdk_draw_point (window, style->mid_gc[state_type], x,   y1);
	gdk_draw_point (window, style->mid_gc[state_type], x+1, y2-1);
	goto draw_done;
	
  draw_ugly:
	gdk_draw_line (window, style->dark_gc [state_type], x,   y1, x,   y2-1);
	gdk_draw_line (window, style->light_gc[state_type], x+1, y1, x+1, y2-1);
	goto draw_done;
	
  draw_inframe:
	gdk_draw_line (window, style->dark_gc [state_type], x,   y1,   x,   y2-2);
	gdk_draw_line (window, style->light_gc[state_type], x+1, y1+1, x+1, y2-1);
	gdk_draw_point (window, style->mid_gc[state_type], x+1, y1);
	gdk_draw_point (window, style->mid_gc[state_type], x,   y2-1);
	
  draw_done:
	if (area) {
		gdk_gc_set_clip_rectangle (style->light_gc [state_type], NULL);
		gdk_gc_set_clip_rectangle (style->dark_gc [state_type], NULL);
		gdk_gc_set_clip_rectangle (style->mid_gc [state_type], NULL);
	}
}


void
xeno_draw_shadow (GtkStyle      *style,
				  GdkWindow     *window,
				  GtkStateType   state_type,
				  GtkShadowType  shadow_type,
				  GdkRectangle  *area,
				  GtkWidget     *widget,
				  xeno_char     *detail,
				  gint           x,
				  gint           y,
				  gint           width,
				  gint           height)
{
	GdkPoint points[5];
	XenoRcData *rc_data;
	XenoShadowType newshadow;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);
	
	if ((width == -1) && (height == -1))
		gdk_window_get_size (window, &width, &height);
	else if (width == -1)
		gdk_window_get_size (window, &width, NULL);
	else if (height == -1)
		gdk_window_get_size (window, NULL, &height);
	
	rc_data = XENO_STYLE_RC_DATA(style);
	newshadow = (XenoShadowType)shadow_type;
	
	if (shadow_type == GTK_SHADOW_IN) {
		if (GTK_IS_RANGE(widget)) {
			if (   ((GtkRange *)widget)->in_child == RANGE_CLASS(widget)->step_forw
				|| ((GtkRange *)widget)->in_child == RANGE_CLASS(widget)->step_back)
			{
				newshadow = XENO_SHADOW_BUTTON_IN;
			} else if (state_type != GTK_STATE_INSENSITIVE) {
				state_type = GTK_STATE_NORMAL;
			}
		} else if (GTK_IS_TEXT(widget)) {
			if (GTK_WIDGET_HAS_FOCUS(widget)) {
				x--;
				y--;
				width+=2;
				height+=2;
			}
		} else if (   GTK_IS_ENTRY(widget)
				   && GTK_WIDGET_STATE(widget) == GTK_STATE_INSENSITIVE)
		{
			state_type = GTK_STATE_INSENSITIVE;
		}
		
		if (!rc_data || !(rc_data->config_shadows & (XENO_CONFIG_SHADOW_IN << state_type))) {
			if (GTK_IS_FRAME(widget) && !GTK_IS_PREVIEW(GTK_BIN(widget)->child)) {
				newshadow = XENO_SHADOW_THIN_IN;
				x++;		y++;
				width-=2;	height-=2;
			} else if (GTK_IS_PROGRESS(widget)) {
				newshadow = XENO_SHADOW_THIN_IN;
			} else if (   GTK_IS_BUTTON(widget)
					   && !GTK_IS_TOGGLE_BUTTON(widget)
					   && state_type == GTK_STATE_ACTIVE)
			{
				newshadow = XENO_SHADOW_BUTTON_IN;
			}
		}
	} else if (shadow_type == GTK_SHADOW_OUT) {
		if (   GTK_IS_BUTTON(widget)
			&& ((GtkButton *)widget)->relief != GTK_RELIEF_NORMAL
			&& state_type == GTK_STATE_PRELIGHT)
		{
			newshadow = (rc_data) ? rc_data->shadow_button_relief : XENO_SHADOW_BUTTON_RELIEF;
		} else if (GTK_IS_MENU_ITEM(widget)) {
			newshadow = (rc_data) ? rc_data->shadow_menu_item : XENO_SHADOW_IN;
		} else if (!rc_data || !(rc_data->config_shadows & (XENO_CONFIG_SHADOW_OUT << state_type))) {
			if (   (GTK_IS_FRAME(widget) && !GTK_IS_EVENT_BOX(widget->parent))
					   || GTK_IS_RULER(widget)
					   || (detail && (   !strcmp(detail, "handlebox_bin")
									  || !strcmp(detail, "dockitem_bin")))
					   )
			{
				newshadow = XENO_SHADOW_THIN_OUT;
			} else if (GTK_IS_MENU(widget)) {
				if (   GTK_IS_MENU(widget)
				   && ((GtkMenu *)widget)->torn_off)
				{
					newshadow = (rc_data && rc_data->flat_windows) ? XENO_SHADOW_NONE : XENO_SHADOW_THIN_OUT;
				}
			} else if (GTK_IS_MENU_BAR(widget)) {
				if (widget->parent && GTK_IS_HANDLE_BOX(widget->parent))
					return;
				
				if (rc_data && rc_data->flat_windows) {
					xeno_draw_hline (style, window, state_type, area, widget, detail,
									 x, x+width, y+height-2);
					return;
				} else if (XENO_STYLE_YTHICKNESS(style) > 1) {
					newshadow = XENO_SHADOW_ETCHED_IN;
					x--;
					y--;
					height++;
					width+=2;
				}
			}
		}
	}
	
	if (newshadow == XENO_SHADOW_NONE)
		return;
	
	points[0].x = x;			points[0].y = y;
	points[1].x = x;			points[1].y = y+height-1;
	points[2].x = x+width-1;	points[2].y = y+height-1;
	points[3].x = x+width-1;	points[3].y = y;
	points[4].x = x;			points[4].y = y;
	
	xeno_draw_polygon (style, window, state_type, (GtkShadowType)newshadow, area,
					   widget, NULL, points, 5, FALSE);
}

static void
xeno_shadow_init (XenoShadowContext	*ctx,
				  GtkStyle			*style,
				  GdkWindow			*window,
				  GtkStateType		state_type,
				  XenoShadowType	shadow_type,
				  GdkRectangle		*area,
				  GtkWidget			*widget,
				  xeno_char			*detail)
{
	const XenoShadowStyle	*shadow_style;
	const XenoShadowRing	*ring;
	XenoStyleData	*style_data;
	XenoRcData		*rc_data;
	GtkStyle		*style2, *style_use;
	GdkGC			*gc;
	XenoShadowType	xeno_shadow;
	GtkStateType	state2, state_use;
	gint			i, j, pen, thickness;
	
	style2 = style;
	state2 = GTK_STATE_NORMAL;
	if (widget != NULL && widget->parent != NULL) {
		style2 = widget->parent->style;
		state2 = widget->parent->state;
	}
	
	if (state_type == GTK_STATE_INSENSITIVE || state2 == GTK_STATE_INSENSITIVE)
		state_type = state2 = GTK_STATE_INSENSITIVE;
	
	thickness = MIN(XENO_STYLE_XTHICKNESS(style), XENO_STYLE_YTHICKNESS(style));
	
	xeno_shadow = shadow_type;
	if (shadow_type <= XENO_SHADOW_OUT) {
		if (shadow_type != XENO_SHADOW_NONE && (rc_data = XENO_STYLE_RC_DATA(style)) != NULL)
			xeno_shadow = ((shadow_type == XENO_SHADOW_IN) ? rc_data->shadow_in : rc_data->shadow_out)[state_type];
	}
	if (xeno_shadow == XENO_SHADOW_NONE)
		thickness = 0;
	
	if (thickness > 0) {
		style_data = XENO_STYLE_DATA(style);
		shadow_style = &xeno_shadows[xeno_shadow - XENO_SHADOW_ETCHED_IN];
		thickness = MIN(shadow_style->thickness, thickness);
		
		if (thickness > 0) {
			if (thickness == 1) {
				ring = &shadow_style->thin_ring[0];
			} else if (thickness == 2) {
				ring = &shadow_style->medium_ring[0];
			} else if (thickness == 3) {
				ring = &shadow_style->thick_ring[0];
			}
			
			for (i = 0; i < thickness; ++i) {
				for (j = 0; j < 3; ++j) {
					pen		  = ring[i][j][0];
					state_use = ring[i][j][1];
					style_use = style;
					if ((state_use & XENO_PARENT) != 0) {
						style_use = style2;
						state_use &= XENO_STATE_MASK;
					}
					if (state_use == (GtkStateType)XENO_CURRENT) {
						state_use = state_type;
					} else if (state_type == GTK_STATE_INSENSITIVE || state2 == GTK_STATE_INSENSITIVE) {
						state_use = GTK_STATE_INSENSITIVE;
					} else if (state_use == (GtkStateType)XENO_PARENT_CURRENT) {
						state_use = state2;
					}
					state_use &= XENO_GTK_STATE_MASK;
					
					switch (pen) {
					  case XENO_WHITE:
						if ((style_data = XENO_STYLE_DATA(style_use)) != NULL) {
							gc = style_data->white_gc[state_use];
							break;
						}
					  case XENO_PURE_WHITE:
						gc = style_use->white_gc;
						break;
						
					  case XENO_BLACK:
						if ((style_data = XENO_STYLE_DATA(style_use)) != NULL) {
							gc = style_data->black_gc[state_use];
							break;
						}
					  case XENO_PURE_BLACK:
						gc = style_use->black_gc;
						break;
						
					  case XENO_FG:
						gc = style_use->fg_gc[state_use];
						break;
					  case XENO_BASE:
						gc = style_use->base_gc[state_use];
						break;
					  case XENO_TEXT:
						gc = style_use->text_gc[state_use];
						break;
					  case XENO_LIGHT:
						gc = style_use->light_gc[state_use];
						break;
					  case XENO_DARK:
						gc = style_use->dark_gc[state_use];
						break;
					  case XENO_MID:
						gc = style_use->mid_gc[state_use];
						break;
					  case XENO_BG:
						gc = style_use->bg_gc[state_use];
						break;
					  default:
						gc = NULL;
					}
					ctx->ring[i].gc[j] = gc;
				}
			}
			
			if (area) {
				gdk_gc_set_clip_rectangle (ctx->ring[0].gc[XENO_TOP_GC], area);
				gdk_gc_set_clip_rectangle (ctx->ring[0].gc[XENO_BOT_GC], area);
				gdk_gc_set_clip_rectangle (ctx->ring[0].gc[XENO_MID_GC], area);
				if (thickness > 1) {
					gdk_gc_set_clip_rectangle (ctx->ring[1].gc[XENO_TOP_GC], area);
					gdk_gc_set_clip_rectangle (ctx->ring[1].gc[XENO_BOT_GC], area);
					gdk_gc_set_clip_rectangle (ctx->ring[1].gc[XENO_MID_GC], area);
					if (thickness > 2) {
						gdk_gc_set_clip_rectangle (ctx->ring[2].gc[XENO_TOP_GC], area);
						gdk_gc_set_clip_rectangle (ctx->ring[2].gc[XENO_BOT_GC], area);
						gdk_gc_set_clip_rectangle (ctx->ring[2].gc[XENO_MID_GC], area);
					}
				}
			}
		}
	}
	ctx->thickness = thickness;
}

static void
xeno_shadow_done (XenoShadowContext *ctx, GdkRectangle *area)
{
	g_return_if_fail (ctx != NULL);
	
	if (area && ctx->thickness > 0) {
		if (ctx->thickness > 1) {
			if (ctx->thickness > 2) {
				gdk_gc_set_clip_rectangle (ctx->ring[2].gc[XENO_MID_GC], NULL);
				gdk_gc_set_clip_rectangle (ctx->ring[2].gc[XENO_BOT_GC], NULL);
				gdk_gc_set_clip_rectangle (ctx->ring[2].gc[XENO_TOP_GC], NULL);
			}
			gdk_gc_set_clip_rectangle (ctx->ring[1].gc[XENO_MID_GC], NULL);
			gdk_gc_set_clip_rectangle (ctx->ring[1].gc[XENO_BOT_GC], NULL);
			gdk_gc_set_clip_rectangle (ctx->ring[1].gc[XENO_TOP_GC], NULL);
		}
		gdk_gc_set_clip_rectangle (ctx->ring[0].gc[XENO_MID_GC], NULL);
		gdk_gc_set_clip_rectangle (ctx->ring[0].gc[XENO_BOT_GC], NULL);
		gdk_gc_set_clip_rectangle (ctx->ring[0].gc[XENO_TOP_GC], NULL);
	}
}

void
xeno_draw_polygon (GtkStyle      *style,
				   GdkWindow     *window,
				   GtkStateType   state_type,
				   GtkShadowType  shadow_type,
				   GdkRectangle  *area,
				   GtkWidget     *widget,
				   xeno_char     *detail,
				   GdkPoint      *points,
				   gint           npoints,
				   gboolean       fill)
{
	static const gdouble pi_over_4	 = M_PI_4;
	static const gdouble pi_3_over_4 = M_PI_4 * 3;
	
	XenoShadowContext ctx;
	XenoGCRing	*ring;
	GdkGC		*gc, *mid_gc;
	gdouble		angle;
	gint		i, j, x,y, x2,y2, mx, sign;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);
	g_return_if_fail (points != NULL);
	
	if (fill) {
		GdkGC *bg = style->bg_gc[state_type];
		
		if (area) gdk_gc_set_clip_rectangle (bg, area);
		gdk_draw_polygon (window, bg, TRUE, points, npoints);
		
		if (shadow_type == GTK_SHADOW_NONE)
			gdk_draw_polygon (window, bg, FALSE, points, npoints);
		
		if (area) gdk_gc_set_clip_rectangle (bg, NULL);
	}
	
	if (shadow_type == GTK_SHADOW_NONE)
		return;
	
	xeno_shadow_init (&ctx, style, window, state_type, (XenoShadowType)shadow_type, area, widget, detail);
	if (ctx.thickness == 0)
		return;
	
	for (i = ctx.thickness - 1; i >= 0; --i) {
		ring = &ctx.ring[i];
		for (j = 0; j < npoints - 1; ++j) {
			x	= points[j].x;
			y	= points[j].y;
			x2	= points[j+1].x;
			y2	= points[j+1].y;
			
			if ((x == x2) && (y == y2)) {
				angle = 0;
			} else {
				angle = atan2 (y2 - y, x2 - x);
			}
			
			if ((angle > -pi_3_over_4 - 0.0625) && (angle < pi_over_4 - 0.0625)) {
				if (i > 0) {
					if (angle > -pi_over_4) {
						y  -= i;
						y2 -= i;
					} else {
						x  -= i;
						x2 -= i;
					}
				}
				mid_gc = gc = ring->gc[XENO_BOT_GC];
				if (sign != 0) {
					mid_gc = ring->gc[XENO_MID_GC];
					mx = x + i;
					sign = 0;
				}
			} else {
				if (i > 0) {
					if ((angle < -pi_3_over_4) || (angle > pi_3_over_4)) {
						y  += i;
						y2 += i;
					} else {
						x  += i;
						x2 += i;
					}
				}
				mid_gc = gc = ring->gc[XENO_TOP_GC];
				if (sign != 1) {
					mid_gc = ring->gc[XENO_MID_GC];
					mx = x - i;
					sign = 1;
				}
			}
			if (gc)
				gdk_draw_line (window, gc, x, y, x2, y2);
			
			if (j > 0 && mid_gc != NULL && mid_gc != gc)
				gdk_draw_point (window, mid_gc, mx, y);
		}
	}
	xeno_shadow_done(&ctx, area);
}


static void
xeno_draw_solid_arrow2 (GtkStyle		*style,
						GdkWindow		*window,
						GtkStateType	state_type,
						GdkRectangle	*area,
						GtkWidget		*widget,
						GtkArrowType	arrow_type,
						gint			x, 
						gint			y, 
						gint			width,
						gint			height)
{
	XenoStyleData *style_data;
	GdkGC	*black_gc;
	GdkGC	*dark_gc;
	gint	d, i, s;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);
	
	black_gc = style->fg_gc[state_type];
	dark_gc = NULL;
	if (   !xeno_pseudocolor
		&& ((style_data = XENO_STYLE_DATA(style)) != NULL))
	{
		if ((dark_gc = style_data->fgmid_gc[state_type]) == NULL)
		{
			GdkGCValues gc_values;
			
			gc_values.foreground.red   = (style->bg[state_type].red   + style->fg[state_type].red  ) / 2;
			gc_values.foreground.green = (style->bg[state_type].green + style->fg[state_type].green) / 2;
			gc_values.foreground.blue  = (style->bg[state_type].blue  + style->fg[state_type].blue ) / 2;
			gc_values.font = style->font;
			
			if (gdk_colormap_alloc_color (style->colormap, &gc_values.foreground, FALSE, FALSE)) {
				dark_gc = gtk_gc_get (style->depth, style->colormap, &gc_values,
									  GDK_GC_FOREGROUND | GDK_GC_FONT);
				style_data->fgmid_gc[state_type] = dark_gc;
			}
		}
	}
	if (dark_gc == NULL)
		dark_gc = style->fg_gc[GTK_STATE_INSENSITIVE];
	
	if (area) {
		gdk_gc_set_clip_rectangle (black_gc, area);
		gdk_gc_set_clip_rectangle (dark_gc, area);
	}
	
	if (height < 3)
		height = 3;
	if (width < 3)
		width = 3;
	
	if (arrow_type == GTK_ARROW_UP || arrow_type == GTK_ARROW_DOWN) {
		d = (width << 14) / height;
		x += width >> 2;
	} else {
		d = (height << 14) / width;
		y += height >> 2;
	}
	if (arrow_type == GTK_ARROW_UP || arrow_type == GTK_ARROW_LEFT) {
		s = d;
	} else {
		s = (height>>1) * d;
		d = -d;
	}

	if (arrow_type == GTK_ARROW_UP || arrow_type == GTK_ARROW_DOWN) {
		for (i = 0; i < (height>>1); i++) {
			gdk_draw_line (window, black_gc,
						   (unsigned) (x - ((s-0x00003fff)>>15)), y + i,
						   (unsigned) (x + ((s-0x00003fff)>>15)), y + i);
			if ((s & 0x00004000)) {
				gdk_draw_point (window, dark_gc, (unsigned) (x + (s>>15)), y + i);
				gdk_draw_point (window, dark_gc, (unsigned) (x - (s>>15)), y + i);
			}
			s += d;
		}
	} else {
		for (i = 0; i < (width>>1); i++) {
			gdk_draw_line (window, black_gc,
						   x + i, (unsigned) (y - ((s-0x00003fff)>>15)),
						   x + i, (unsigned) (y + ((s-0x00003fff)>>15)));
			if ((s & 0x00004000)) {
				gdk_draw_point (window, dark_gc, x + i, (unsigned) (y + (s>>15)));
				gdk_draw_point (window, dark_gc, x + i, (unsigned) (y - (s>>15)));
			}
			s += d;
		}
	}
	
	if (area) {
		gdk_gc_set_clip_rectangle (black_gc, NULL);
		gdk_gc_set_clip_rectangle (dark_gc, NULL);
	}
}


void
xeno_draw_arrow (GtkStyle      *style,
				 GdkWindow     *window,
				 GtkStateType   state_type,
				 GtkShadowType  shadow_type,
				 GdkRectangle  *area,
				 GtkWidget     *widget,
				 xeno_char     *detail,
				 GtkArrowType   arrow_type,
				 gboolean       fill,
				 gint           x,
				 gint           y,
				 gint           width,
				 gint           height)
{
	gint	t;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);
	
	if ((width == -1) && (height == -1))
		gdk_window_get_size (window, &width, &height);
	else if (width == -1)
		gdk_window_get_size (window, &width, NULL);
	else if (height == -1)
		gdk_window_get_size (window, NULL, &height);

/*	} else if (GTK_IS_SPIN_BUTTON(widget)) {
		gtk_paint_box (style, window, state_type,
					   shadow_type, area, widget, detail, x, y, width, height);
		x      += 2;
		y      += 2;
		width  -= 5;
		height -= 5;
		goto draw_black;
*/

	if (   widget
		&& GTK_IS_RANGE(widget)
		&& XENO_STYLE_IS_XENO(style)
		&& XENO_STYLE_RC_DATA(style))
	{
		if (widget->state == GTK_STATE_INSENSITIVE)
			state_type = GTK_STATE_INSENSITIVE;
		
		if (XENO_STYLE_RC_DATA(style)->stepper_box) {
			GtkAdjustment *adjust;
			gint xt, yt;
			
			adjust = GTK_RANGE(widget)->adjustment;
			xt = XENO_STYLE_XTHICKNESS(style);
			yt = XENO_STYLE_YTHICKNESS(style);

			gtk_paint_box (style, window, state_type, shadow_type,
						   area, widget, detail, x, y, width, height);
			
			x += xt;
			y += yt;
			width -= xt + xt;
			height -= yt + yt;
			
			if (XENO_STYLE_RC_DATA(style)->stepper_arrows) {
				shadow_type = GTK_SHADOW_OUT;
				
				if (state_type == GTK_STATE_NORMAL || state_type == GTK_STATE_ACTIVE)
					state_type = GTK_STATE_SELECTED;
				goto draw_normal;
			}
		} else {
			xeno_draw_flat_box (style, window,
								state_type == GTK_STATE_INSENSITIVE ? GTK_STATE_INSENSITIVE : GTK_STATE_ACTIVE,
								GTK_SHADOW_NONE, area, widget, detail, x, y, width, height);
		}
		
		if (XENO_STYLE_RC_DATA(style)->stepper_arrows) {
			goto draw_normal;
		} else {
			x++;
			y++;
			width = (width - 3) << 1;
			height = (height - 3) << 1;
			goto draw_black;
		}
	}

	if (   widget
		&& GTK_IS_ARROW(widget)
		&& widget->parent
		&& widget->parent->parent
		&& GTK_IS_COMBO(widget->parent->parent))
	{
		fill = FALSE;
		/*
		x		+= 2;
		y		+= 2;
		width	-= 4;
		height	-= 4;
		*/
		x += (width >> 1) - 4;
		y += (height >> 1) - 2;
		
		width	= 0x12;
		height	= 0x09;
		
		goto draw_black;
	
	} else if (   (detail && !strcmp ("menuitem", detail))
			   || (widget && widget->parent && widget->parent && GTK_IS_BUTTON(widget->parent)))
	{
		x		+= 1;
		y		+= 1;
		width	-= 2;
		height	-= 2;
		
		t = style->font->ascent - height;
		if (t < 0) {
				x -= t>>1;
				y -= t>>1;
				height += t;
				width += t;
		}

		width  = width << 1;
		height = height << 1;
		state_type = widget->state;
		
	  draw_black:
		xeno_draw_solid_arrow2 (style, window, state_type, area, widget, arrow_type,
								x, y, width, height);
	} else {
		GdkPoint points[4];
		gint half_width, half_height, w;

	  draw_normal:

		if (GTK_IS_RANGE(widget)) {
			x++;		y++;
			width-=2;	height-=2;
		} else if (   widget == NULL
				   && width == 10
				   && height == 10
				   && arrow_type == GTK_ARROW_LEFT)
		{
			GtkWidget	*widget;
			
			gdk_window_get_user_data (window, (gpointer)(&widget));
			if (GTK_IS_TEAROFF_MENU_ITEM(widget)) {
				y = GTK_CONTAINER(widget)->border_width;
				x = y + (GTK_MENU_ITEM(widget)->toggle_size > 10)
						? GTK_MENU_ITEM(widget)->toggle_size + 3
						: 20;
				y += ((widget->allocation.height - y * 2) - XENO_STYLE_YTHICKNESS(style)) / 2;
				
				while (x-5 > 0) {
					xeno_draw_hline (style, window, GTK_STATE_NORMAL, NULL, NULL, NULL,
									 x-10, x-5, y);
					x -= 10;
				}
				return;
			}
		}
		
		/* draw boxed arrows */
		if (   GTK_IS_SPIN_BUTTON(widget)
			|| GTK_IS_NOTEBOOK(widget)
			|| GTK_IS_RANGE(widget))
		{
			if (shadow_type == GTK_SHADOW_IN) {
				shadow_type = (GtkShadowType)XENO_SHADOW_BUTTON_IN;
			} else if (shadow_type == GTK_SHADOW_ETCHED_IN) {
				state_type = GTK_STATE_INSENSITIVE;
				shadow_type = GTK_SHADOW_OUT;
			}
		}
		
		width &= 0xfffe;
		height &= 0xfffe;

		w = MIN(width, height);
		x += (width-w)/2;
		y += (height-w)/2;
		width = height = w;
		
		half_width = width / 2;
		half_height = height / 2;

		switch (arrow_type) {
		  case GTK_ARROW_UP:
			points[0].x = x + half_width - 1;
			points[0].y = y;
			points[1].x = x;
			points[1].y = y + height - 1;
			points[2].x = x + width  - 1;
			points[2].y = y + height - 1;
			points[3].x = x + half_width;
			points[3].y = y;
			break;

		  case GTK_ARROW_DOWN:
			points[0].x = x + half_width ;
			points[0].y = y + height     - 1;
			points[1].x = x + width - 1;
			points[1].y = y;
			points[2].x = x;
			points[2].y = y;
			points[3].x = x + half_width - 1;
			points[3].y = y + height     - 1;
			break;

		  case GTK_ARROW_LEFT:
			points[0].x = x;
			points[0].y = y + half_height;
			points[1].x = x + width - 1;
			points[1].y = y + height - 1;
			points[2].x = x + width - 1;
			points[2].y = y;
			points[3].x = x;
			points[3].y = y + half_height - 1;
			break;

		  case GTK_ARROW_RIGHT:
			points[0].x = x + width - 1;
			points[0].y = y + half_height - 1;
			points[1].x = x;
			points[1].y = y;
			points[2].x = x;
			points[2].y = y + height - 1;
			points[3].x = x + width - 1;
			points[3].y = y + half_height;
			break;

		  default:
			return;
		}
		
		xeno_draw_polygon (style, window, state_type, XENO_SHADOW_USER(shadow_type), area, widget, detail,
						   points, 4, fill);
	}
}


void
xeno_draw_diamond (GtkStyle      *style,
				   GdkWindow     *window,
				   GtkStateType   state_type,
				   GtkShadowType  shadow_type,
				   GdkRectangle  *area,
				   GtkWidget     *widget,
				   xeno_char     *detail,
				   gint           x,
				   gint           y,
				   gint           width,
				   gint           height)
{
	GdkPoint	points[6];
	gint		half_width;
	gint		half_height;

	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);

	if ((width == -1) && (height == -1))
		gdk_window_get_size (window, &width, &height);
	else if (width == -1)
		gdk_window_get_size (window, &width, NULL);
	else if (height == -1)
		gdk_window_get_size (window, NULL, &height);

	half_width	= width / 2;
	half_height	= height / 2;

	width = half_width * 2;
	height = half_height * 2 - 1;

	points[0].x = x+half_width-1;	points[0].y = y;
	points[1].x = x;				points[1].y = y+half_height-1;
	points[2].x = x+half_width-1;	points[2].y = y+height-1;
	points[3].x = x+half_width;		points[3].y = y+height-1;
	points[4].x = x+width-1;		points[4].y = y+half_height-1;
	points[5].x = x+half_width;		points[5].y = y;

	xeno_draw_polygon (style, window, state_type, XENO_SHADOW_USER(shadow_type), area, widget, detail,
					   points, 3, FALSE);
	xeno_draw_polygon (style, window, state_type, XENO_SHADOW_USER(shadow_type), area, widget, detail,
					   points+3, 3, FALSE);
}

void
xeno_draw_oval (GtkStyle      *style,
				GdkWindow     *window,
				GtkStateType   state_type,
				GtkShadowType  shadow_type,
				GdkRectangle  *area,
				GtkWidget     *widget,
				xeno_char     *detail,
				gint           x,
				gint           y,
				gint           width,
				gint           height)
{
	XenoShadowContext ctx;
	GdkGC *bg_gc;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);
	
	bg_gc = style->bg_gc[state_type];
	if (area)
		gdk_gc_set_clip_rectangle(bg_gc, area);
	
	shadow_type = XENO_SHADOW_USER(shadow_type);
	xeno_shadow_init (&ctx, style, window, state_type, (XenoShadowType)shadow_type, area, widget, detail);
	
	if (ctx.thickness < 2) {
		gdk_draw_arc (window, bg_gc, TRUE, x, y, width, height, 0, 360*64);
		gdk_draw_arc (window, ctx.ring[0].gc[XENO_MID_GC], FALSE, x, y, width, height, 15*64, 60*64);
		gdk_draw_arc (window, ctx.ring[0].gc[XENO_MID_GC], FALSE, x, y, width, height, -105*64, -60*64);
	} else {
		gdk_draw_arc (window, ctx.ring[1].gc[XENO_TOP_GC], TRUE, x, y, width, height, 50*64, 150*64);
		gdk_draw_arc (window, ctx.ring[0].gc[XENO_MID_GC], TRUE, x, y, width, height, 20*64, 30*64);
		gdk_draw_arc (window, ctx.ring[1].gc[XENO_BOT_GC], TRUE, x, y, width, height, 20*64, -150*64);
		gdk_draw_arc (window, ctx.ring[0].gc[XENO_MID_GC], TRUE, x, y, width, height, -130*64, -30*64);
		
		gdk_draw_arc (window, ctx.ring[1].gc[XENO_TOP_GC], FALSE, x, y, width, height, 55*64, 20*64);
		gdk_draw_arc (window, ctx.ring[0].gc[XENO_MID_GC], FALSE, x, y, width, height, 35*64, 20*64);
		gdk_draw_arc (window, ctx.ring[1].gc[XENO_BOT_GC], FALSE, x, y, width, height, 15*64, 20*64);
		
		gdk_draw_arc (window, ctx.ring[1].gc[XENO_BOT_GC], FALSE, x, y, width, height, -105*64, -20*64);
		gdk_draw_arc (window, ctx.ring[0].gc[XENO_MID_GC], FALSE, x, y, width, height, -125*64, -20*64);
		gdk_draw_arc (window, ctx.ring[1].gc[XENO_TOP_GC], FALSE, x, y, width, height, -145*64, -20*64);
		
		gdk_draw_arc (window, bg_gc, TRUE, x+2, y+2, width-4, height-4, 0, 360*64);
	}
	
	gdk_draw_arc (window, ctx.ring[0].gc[XENO_TOP_GC], FALSE, x, y, width, height, 75*64, 120*64);
	gdk_draw_arc (window, ctx.ring[0].gc[XENO_BOT_GC], FALSE, x, y, width, height, 15*64, -120*64);
	
	xeno_shadow_done (&ctx, area);
	
	if (area)
		gdk_gc_set_clip_rectangle(bg_gc, NULL);
}

void
xeno_draw_string (GtkStyle      *style,
				  GdkWindow     *window,
				  GtkStateType   state_type,
				  GdkRectangle  *area,
				  GtkWidget     *widget,
				  xeno_char     *detail,
				  gint           x,
				  gint           y,
				  const gchar   *string)
{
	XenoRcData *rc_data;
	XenoStyleData *style_data;
	GdkGC *fg_gc, *bg_gc, *tmp_gc;
	XenoShadowType shadow_type;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);
	
	shadow_type = (detail && strcmp(detail, "frame") == 0) ? XENO_SHADOW_XENO_OUT : XENO_SHADOW_NONE;
	
	if ((rc_data = XENO_STYLE_RC_DATA(style)) != NULL) {
		if ((rc_data->config_shadows & (XENO_CONFIG_SHADOW_TEXT << state_type)) != 0)
			shadow_type = rc_data->shadow_text[state_type];
	}
	
	switch (shadow_type) {
	  case XENO_SHADOW_XENO_IN:
	  case XENO_SHADOW_XENO_OUT:
		if ((style_data = XENO_STYLE_DATA(style)) != NULL) {
			fg_gc = style_data->white_gc[state_type];
			bg_gc = style_data->black_gc[state_type];
		} else {
			fg_gc = style->white_gc;
			bg_gc = style->black_gc;
		}
		break;
		
	  case XENO_SHADOW_THIN_IN:
	  case XENO_SHADOW_THIN_OUT:
	  case XENO_SHADOW_ETCHED_IN:
	  case XENO_SHADOW_ETCHED_OUT:
		fg_gc = style->light_gc[state_type];
		bg_gc = style->dark_gc[state_type];
		break;
		
	  default:
		fg_gc = style->fg_gc[state_type];
		if (area)
			gdk_gc_set_clip_rectangle(fg_gc, area);
		
		gdk_draw_string (window, style->font, fg_gc, x, y, string);
		
		if (area)
			gdk_gc_set_clip_rectangle(fg_gc, NULL);
		return;
	}
	
	if (XENO_SHADOW_IS_IN(shadow_type)) {
		tmp_gc = fg_gc;
		fg_gc = bg_gc;
		bg_gc = tmp_gc;
	}
	
	if (area) {
		gdk_gc_set_clip_rectangle (fg_gc, area);
		gdk_gc_set_clip_rectangle (bg_gc, area);
	}
	gdk_draw_string (window, style->font, bg_gc, x+1, y+1, string);
	gdk_draw_string (window, style->font, fg_gc, x, y, string);
	if (area) {
		gdk_gc_set_clip_rectangle (bg_gc, NULL);
		gdk_gc_set_clip_rectangle (fg_gc, NULL);
	}
}

void 
xeno_draw_box  (GtkStyle      *style,
				GdkWindow     *window,
				GtkStateType   state_type,
				GtkShadowType  shadow_type,
				GdkRectangle  *area,
				GtkWidget     *widget,
				xeno_char     *detail,
				gint           x,
				gint           y,
				gint           width,
				gint           height)
{
	GtkStyle		*bg_style;
	XenoKnobType	knob_style;
	GtkOrientation	line_orientation = GTK_ORIENTATION_HORIZONTAL;
	GtkOrientation	knob_orientation = GTK_ORIENTATION_HORIZONTAL;
	gint			mouse_x, mouse_y;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);
	
	if ((width == -1) && (height == -1))
		gdk_window_get_size (window, &width, &height);
	else if (width == -1)
		gdk_window_get_size (window, &width, NULL);
	else if (height == -1)
		gdk_window_get_size (window, NULL, &height);
	
	bg_style = style;
	knob_style = XENO_KNOB_NONE;
	if (widget) {
		if (GTK_IS_BUTTON(widget)) {
			if ((detail) && !strcmp ("buttondefault", detail)) {
				x++; y++;
				width--; height--;
				
				if (shadow_type == GTK_SHADOW_IN) {
					if (XENO_STYLE_RC_DATA(style))
						shadow_type = XENO_STYLE_RC_DATA(style)->shadow_button_default;
				}
				
				if (state_type != GTK_STATE_INSENSITIVE)
					state_type = GTK_STATE_NORMAL;
				
				if (widget && widget->parent)
					bg_style = gtk_widget_get_style(widget->parent);
			} else if (   GTK_IS_TOGGLE_BUTTON(widget)
					   && ((GtkToggleButton *)widget)->active)
			{
				if (   ((GtkButton *)widget)->button_down
					&& ((GtkButton *)widget)->in_button)
				{
					state_type = GTK_STATE_ACTIVE;
				} else if (   XENO_STYLE_RC_DATA(style)
						   && !XENO_SHADOW_IS_IN(XENO_STYLE_RC_DATA(style)->shadow_in[GTK_STATE_NORMAL]))
				{
					state_type = GTK_STATE_SELECTED;
				} else {
					xeno_draw_base_rectangle (style, window, state_type, area, widget,
											  x, y, width, height);
					goto draw_shadow;
				}
			} else if (GTK_IS_OPTION_MENU(widget) && detail && strcmp(detail, "optionmenu")==0) {
				y++;
				height -= 2;
			} else if (   ((GtkButton *)widget)->relief != GTK_RELIEF_NORMAL
					   && widget->parent)
			{
				bg_style = gtk_widget_get_style(widget->parent);
			}
		} else if (GTK_IS_PANED(widget)) {
			gdk_window_get_size (window, &width, &height);
			x = 0;
			y = 0;
			
			state_type = widget->state;
			gdk_window_get_pointer (window, &mouse_x, &mouse_y, NULL);
			if (mouse_x >= 0 && mouse_x < width && mouse_y >= 0 && mouse_y < height)
				state_type = GTK_STATE_PRELIGHT;
			
			if (XENO_STYLE_RC_DATA(style))
				knob_style = XENO_STYLE_RC_DATA(style)->paned_knobs[state_type];
			
			area = NULL;
			
			if (knob_style == XENO_KNOB_NONE) {
				shadow_type = GTK_SHADOW_NONE;
			} else if (knob_style == XENO_KNOB_BOX) {
				shadow_type = GTK_SHADOW_ETCHED_IN;
			} else {
				knob_orientation = GTK_IS_VPANED(widget) ? GTK_ORIENTATION_HORIZONTAL
														 : GTK_ORIENTATION_VERTICAL;
				line_orientation = knob_orientation;
				shadow_type = GTK_SHADOW_NONE;
			}
		} else if (   GTK_IS_SCROLLBAR(widget)
				   && (detail && strcmp(detail, "slider")==0)
				   && XENO_STYLE_RC_DATA(style)
				   && (knob_style = XENO_STYLE_RC_DATA(style)->scrollbar_knobs[state_type]) != XENO_KNOB_NONE)
		{
			/* state_type = widget->state; */
			knob_orientation = GTK_IS_VSCROLLBAR(widget) ? GTK_ORIENTATION_VERTICAL
														 : GTK_ORIENTATION_HORIZONTAL;
			line_orientation = knob_orientation ^ 1;
		} else if (GTK_IS_PROGRESS(widget) && shadow_type == GTK_SHADOW_OUT) {
			GtkProgressBarOrientation or = ((GtkProgressBar *)widget)->orientation;
			GtkProgressBarStyle		  st = ((GtkProgressBar *)widget)->bar_style;
			
			xeno_draw_bg_rectangle (style, window, GTK_STATE_SELECTED, area, widget,
				x, y,
				width  - (st==GTK_PROGRESS_DISCRETE && (or==GTK_PROGRESS_RIGHT_TO_LEFT || or==GTK_PROGRESS_LEFT_TO_RIGHT)),
				height - (st==GTK_PROGRESS_DISCRETE && (or==GTK_PROGRESS_BOTTOM_TO_TOP || or==GTK_PROGRESS_TOP_TO_BOTTOM))
				);
			return;
		}
	}
	
	/* draw already */
	xeno_draw_bg_rectangle (bg_style, window, state_type, area, widget, x, y, width, height);
	
  draw_shadow:
	xeno_draw_shadow (bg_style, window, state_type, shadow_type, area,
					  widget, detail, x, y, width, height);
	
	if (knob_style != XENO_KNOB_NONE && knob_style != XENO_KNOB_BOX) {
		if (knob_style == XENO_KNOB_DIMPLE) {
			xeno_draw_image (window, widget, area,
							 ((state_type == GTK_STATE_PRELIGHT)
							  ? XENO_IMAGE_DIMPLE_PRELIGHT
							  : ((state_type == GTK_STATE_INSENSITIVE)
							     ? XENO_IMAGE_DIMPLE_INSENSITIVE
							     : XENO_IMAGE_DIMPLE_NORMAL)),
							 0, 0, x+width/2-3, y+height/2-3, 6, 6);
		} else {
			gint w, h, xt, yt;
			
			if (GTK_IS_PANED(widget)) {
				xt = 0;
				yt = 0;
			} else {
				xt = XENO_STYLE_XTHICKNESS(style) + 1;
				yt = XENO_STYLE_YTHICKNESS(style); + 1;
			}
			
			if (knob_orientation == GTK_ORIENTATION_VERTICAL) {
				h = MIN (height - xt - xt, width + height/7);
				y = y + height/2 - h/2;
				x = x + xt;
				w = width - xt - xt;
			  /*
				if (GTK_IS_HPANED(widget)) {
					xeno_draw_vline (style, window, state_type, area, widget, NULL, 0, y-2, 2);
					xeno_draw_vline (style, window, state_type, area, widget, NULL, y+h+2, height, 2);
				}
			  */
			} else {
				w = MIN (width - xt - xt, height + width/7);
				x = x + width/2 - w/2;
				y = y + yt;
				h = height - yt - yt;
			  /*
				if (GTK_IS_VPANED(widget)) {
					xeno_draw_hline (style, window, state_type, area, widget, NULL, 0, x-2, 2);
					xeno_draw_hline (style, window, state_type, area, widget, NULL, x+w-2, width, 2);
				}
			  */
			}
			
			if (knob_style == XENO_KNOB_BUDS) {
				xeno_draw_buds (style, window, state_type, GTK_SHADOW_ETCHED_OUT,
								area, widget, x, y, w, h, line_orientation);
			} else if (knob_style == XENO_KNOB_HOLES) {
				xeno_draw_buds (style, window, state_type, GTK_SHADOW_ETCHED_IN,
								area, widget, x, y, w, h, line_orientation);
			} else  {
				xeno_draw_lines (style, window, state_type, GTK_SHADOW_OUT,
								 area, widget, x, y, w, h,
								 line_orientation);
			}
		}
	}
}


void 
xeno_draw_flat_box (GtkStyle      *style,
				    GdkWindow     *window,
				    GtkStateType   state_type,
				    GtkShadowType  shadow_type,
				    GdkRectangle  *area,
				    GtkWidget     *widget,
				    xeno_char     *detail,
				    gint           x,
				    gint           y,
				    gint           width,
				    gint           height)
{
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);

	if ((width == -1) && (height == -1))
		gdk_window_get_size (window, &width, &height);
	else if (width == -1)
		gdk_window_get_size (window, &width, NULL);
	else if (height == -1)
		gdk_window_get_size (window, NULL, &height);

	if (GTK_IS_RADIO_BUTTON(widget) || GTK_IS_CHECK_BUTTON(widget)) {
		return;
	  /*	
		if ((widget = GTK_BIN(widget)->child) == NULL)
			return;
		
		x = widget->allocation.x - 1;
		y = widget->allocation.y + (widget->allocation.height - widget->requisition.height) / 2;
		width  = widget->requisition.width+2;
		height = widget->requisition.height;
		if (GTK_IS_MISC(widget) && GTK_MISC(widget)->xalign > 0.0)
			width = widget->parent->allocation.width - x;
	  */
	}
	
	if (GTK_IS_EDITABLE(widget)) {
	  #if !XENO_GTK2
		if (!GTK_EDITABLE(widget)->editable && state_type != GTK_STATE_SELECTED)
			state_type = GTK_STATE_INSENSITIVE;
	  #endif
		
		xeno_draw_base_rectangle (style, window, state_type, area, widget, x, y, width, height);
	} else {
		xeno_draw_bg_rectangle (style, window, state_type, area, widget, x, y, width, height);
		
		if (detail) {
			if (   GTK_IS_WINDOW(widget)
				&& !strcmp("base", detail)
				&& !(XENO_STYLE_RC_DATA(style) && XENO_STYLE_RC_DATA(style)->flat_windows))
			{
				xeno_draw_shadow (style, window, GTK_STATE_NORMAL,
								  (GTK_WINDOW(widget)->type == GTK_WINDOW_POPUP)
								  ? XENO_SHADOW_OUT : XENO_SHADOW_THIN_OUT,
								  area, widget, NULL, x, y, width, height);
			} else if (!strcmp("tooltip", detail)) {
				xeno_draw_shadow (style, window, GTK_STATE_NORMAL, XENO_SHADOW_NOTE, area, widget,
								  NULL, x, y, width, height);
			}
		}
	}
}

void 
xeno_draw_check   (GtkStyle      *style,
				   GdkWindow     *window,
				   GtkStateType   state_type,
				   GtkShadowType  shadow_type,
				   GdkRectangle  *area,
				   GtkWidget     *widget,
				   xeno_char     *detail,
				   gint           x,
				   gint           y,
				   gint           width,
				   gint           height)
{
	GdkGC		*bg_gc;
	GdkPixmap	*pixmap, *mask;
	XenoImageType	image_type;
	XenoMaskType	mask_type;
	GtkStateType	shadow_state;
	gint		ascent, xt, yt, bw, t;
	gboolean	down=FALSE;
	gboolean	active=FALSE;
	
	bg_gc	= style->bg_gc[state_type];
	xt		= XENO_STYLE_XTHICKNESS(style);
	yt		= XENO_STYLE_YTHICKNESS(style);
	bw		= GTK_CONTAINER(widget)->border_width;
	
	ascent	= style->font->ascent;
	if (widget && GTK_IS_BIN(widget) && GTK_BIN(widget)->child)
		ascent = GTK_BIN(widget)->child->style->font->ascent;
	
	if ((detail) && ((!strcmp ("checkbutton", detail)))) {
		height	= (ascent + yt + yt) | 1;
		if (height + 2 > widget->allocation.height -  bw - bw)
			height = widget->allocation.height - bw - bw - 2;
		width = height - yt - yt + xt + xt;
		
		x		= widget->allocation.x + bw + xt + 1;
		y		= widget->allocation.y + bw + ((widget->allocation.height - height)>>1);
		
		state_type = GTK_WIDGET_STATE(widget);
		
		if (shadow_type == GTK_SHADOW_IN) {
			down = TRUE;
			if (state_type == GTK_STATE_ACTIVE)
				state_type = GTK_STATE_NORMAL;
		}
		
		if (   GTK_IS_BUTTON(widget)
			&& ((GtkButton *)widget)->button_down
			&& ((GtkButton *)widget)->in_button)
		{
			active = TRUE;
			state_type = GTK_STATE_ACTIVE;
			shadow_type = GTK_SHADOW_IN;
		}
		
		xeno_draw_box (style, window, state_type, shadow_type,
					   area, widget, detail, x, y, width, height);
		
		x += xt+1;
		y += yt+1;
		width -= xt+xt+2;
		height -= yt+yt+2;
		
		if (down) {
			if (width < 9) {
				image_type = XENO_IMAGE_CHECK_BUTTON_7_NORMAL;
				width = 7;
			} else if (width < 11) {
				image_type = XENO_IMAGE_CHECK_BUTTON_9_NORMAL;
				width = 9;
			} else {
				image_type = XENO_IMAGE_CHECK_BUTTON_11_NORMAL;
				width = 11;
			}
			
			if (active) {
				pixmap = xeno_pixmap_get(window, style, style, image_type + 3);
			} else if (state_type == GTK_STATE_INSENSITIVE) {
				pixmap = xeno_pixmap_get(window, style, style, image_type + 2);
			} else if (state_type == GTK_STATE_PRELIGHT) {
				pixmap = xeno_pixmap_get(window, style, style, image_type + 1);
			} else {
				pixmap = xeno_pixmap_get(window, style, style, image_type);
			}
			mask = xeno_image_mask (image_type);
			
			if (pixmap && mask)
				xeno_draw_pixmap (window, bg_gc, area, pixmap, mask, 0, 0, x, y, width, width);
		}
	} else if ((detail) && ((!strcmp ("check", detail)))) {
		if (widget && GTK_IS_CHECK_MENU_ITEM(widget)) {
			height = ascent - 2;
			/*
			if (height + bw*2 + yt*2 > widget->allocation.height)
				height = widget->allocation.height - bw*2 - yt*2;
			*/
			width = height;
			
			x = bw + xt + 1;
			y = (widget->allocation.height - height) >> 1;
		}
		/*
		xeno_draw_bg_rectangle (style, window, state_type, area, widget, x, y, width, height);
		*/
		if (widget && GTK_IS_CHECK_MENU_ITEM(widget)) {
			down = GTK_CHECK_MENU_ITEM(widget)->active;
		} else {
			if (   (shadow_type == GTK_SHADOW_IN  && state_type != GTK_STATE_PRELIGHT)
				|| (shadow_type == GTK_SHADOW_OUT && state_type == GTK_STATE_PRELIGHT))
				down = TRUE;
		}
		
		if (down) {
			t = MIN(width, height);
			y += (height - t) >> 1;
			x += (width - t) >> 1;
			width = height = t;
			
			if (width < 9) {
				image_type = XENO_IMAGE_CHECK_MENU_7_NORMAL;
				width = 7;
			} else if (width < 11) {
				image_type = XENO_IMAGE_CHECK_MENU_9_NORMAL;
				width = 9;
			} else {
				image_type = XENO_IMAGE_CHECK_MENU_11_NORMAL;
				width = 11;
			}

			if (state_type == GTK_STATE_INSENSITIVE) {
				pixmap = xeno_pixmap_get(window, style, style, image_type + 2);
			} else if (state_type == GTK_STATE_PRELIGHT) {
				pixmap = xeno_pixmap_get(window, style, style, image_type + 1);
			} else {
				pixmap = xeno_pixmap_get(window, style, style, image_type);
			}
			mask = xeno_image_mask (image_type);
			
			if (pixmap)
				xeno_draw_pixmap (window, bg_gc, area, pixmap, mask, 0, 0, x, y, width, width);
		}
	} else {
		gtk_paint_box (style, window, state_type, shadow_type, area, widget, detail,
					   x, y, width, height);
	}
}


void 
xeno_draw_option  (GtkStyle      *style,
				   GdkWindow     *window,
				   GtkStateType   state_type,
				   GtkShadowType  shadow_type,
				   GdkRectangle  *area,
				   GtkWidget     *widget,
				   xeno_char     *detail,
				   gint           x,
				   gint           y,
				   gint           width,
				   gint           height)
{
	GdkGC *gc = style->bg_gc[state_type];
	if ((detail) && ((!strcmp ("radiobutton", detail))))
	{
		GdkPixmap *pixmap[] = {NULL, NULL, NULL, NULL, NULL, NULL, NULL};
		GdkBitmap *mask;
		gint index;
		
		GtkStyle *style2 = style;

		if (widget->parent && widget->parent->style)
			style2 = widget->parent->style;
		else
			g_warning ("widget does not have any parent\n");

		pixmap[0] = xeno_pixmap_get(window, style, style2, XENO_IMAGE_RADIO_NORMAL_OUT);
		pixmap[1] = xeno_pixmap_get(window, style, style2, XENO_IMAGE_RADIO_PRELIGHT_OUT);
		pixmap[2] = xeno_pixmap_get(window, style, style2, XENO_IMAGE_RADIO_INSENSITIVE_OUT);

		pixmap[5] = xeno_pixmap_get(window, style, style2, XENO_IMAGE_RADIO_INSENSITIVE_IN);
		pixmap[6] = xeno_pixmap_get(window, style, style2, XENO_IMAGE_RADIO_ACTIVE_IN);
		pixmap[3] = xeno_pixmap_get(window, style, style2, XENO_IMAGE_RADIO_NORMAL_IN);
		pixmap[4] = xeno_pixmap_get(window, style, style2, XENO_IMAGE_RADIO_PRELIGHT_IN);
		mask = xeno_image_mask (XENO_IMAGE_RADIO_NORMAL_OUT);
		
		if (shadow_type == GTK_SHADOW_IN) {
			index = 3;
		} else {
			index = 0;
		}
		if (state_type == GTK_STATE_INSENSITIVE) {
			index += 2;
		} else if (GTK_BUTTON(widget)->button_down && GTK_BUTTON(widget)->in_button) {
			index = 6;
		} else if (state_type == GTK_STATE_PRELIGHT) {
			index += 1;
		}

		if (pixmap[0]) {
			y += (height-XENO_RADIO_SIZE)/2;
			xeno_draw_pixmap (window, gc, area, pixmap[index], mask, 0, 0, x, y,
							  XENO_RADIO_SIZE, XENO_RADIO_SIZE);
		}
	} else if ((detail) && ((!strcmp ("option", detail)))) {
		GdkPixmap	*pixmaps[] = {NULL, NULL, NULL};
		GdkBitmap	*mask;
		GdkPixmap	*pixmap;
		GtkStyle	*style2 = style;
		gint		y2;
		
		style2 = (widget->parent && widget->parent->style) ? widget->parent->style : style;

		pixmaps[0] = xeno_pixmap_get(window, style, style2, XENO_IMAGE_RADIO_MENU_NORMAL);
		pixmaps[1] = xeno_pixmap_get(window, style, style2, XENO_IMAGE_RADIO_MENU_PRELIGHT);
		pixmaps[2] = xeno_pixmap_get(window, style, style2, XENO_IMAGE_RADIO_MENU_INSENSITIVE);
		
		y  = (widget->allocation.height - XENO_RADIO_MENU_SIZE)/2;
		y2 = GTK_CONTAINER(widget)->border_width
			 + XENO_STYLE_YTHICKNESS(style)
			 + 1
			 + style->font->ascent
			 - XENO_RADIO_MENU_SIZE;
		y = MIN(y, y2);
		
		x = x + (width - XENO_RADIO_MENU_SIZE)/2;
		
		width  = XENO_RADIO_MENU_SIZE;
		height = XENO_RADIO_MENU_SIZE;
		/*
		xeno_draw_bg_rectangle (style, window, state_type, area, widget, x, y, width, height);
		*/
		if (shadow_type == GTK_SHADOW_IN) {
			switch (state_type) {
			  case GTK_STATE_INSENSITIVE:	pixmap = pixmaps[2]; break;
			  case GTK_STATE_PRELIGHT:		pixmap = pixmaps[1]; break;
			  default:						pixmap = pixmaps[0];
			}
			mask = xeno_image_mask (XENO_IMAGE_RADIO_MENU_NORMAL);
			
			if (pixmap)
				xeno_draw_pixmap(window, gc, area, pixmap, mask, 0, 0, x, y, width, height);
		}
	} else
		gtk_paint_diamond (style, window, state_type, shadow_type, area, widget, 
						   detail, x, y, width, height);
}


void 
xeno_draw_cross   (GtkStyle      *style,
				   GdkWindow     *window,
				   GtkStateType   state_type,
				   GtkShadowType  shadow_type,
				   GdkRectangle  *area,
				   GtkWidget     *widget,
				   xeno_char     *detail,
				   gint           x,
				   gint           y,
				   gint           width,
				   gint           height)
{
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);
	
	g_warning ("xeno_draw_cross(): FIXME, this functionality is not implemented in GTK+.");
}

void 
xeno_draw_ramp    (GtkStyle      *style,
				   GdkWindow     *window,
				   GtkStateType   state_type,
				   GtkShadowType  shadow_type,
				   GdkRectangle  *area,
				   GtkWidget     *widget,
				   xeno_char     *detail,
				   GtkArrowType   arrow_type,
				   gint           x,
				   gint           y,
				   gint           width,
				   gint           height)
{
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);

	g_warning ("xeno_draw_ramp(): FIXME, this functionality is not implemented in GTK+.");
}

void
xeno_draw_tab (GtkStyle      *style,
			   GdkWindow     *window,
			   GtkStateType   state_type,
			   GtkShadowType  shadow_type,
			   GdkRectangle  *area,
			   GtkWidget     *widget,
			   xeno_char     *detail,
			   gint           x,
			   gint           y,
			   gint           width,
			   gint           height)
{
	gint h, w;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);

	if ((detail) && (!strcmp ("optionmenutab", detail))) {
		if (XENO_STYLE_RC_DATA(style) && XENO_STYLE_RC_DATA(style)->popup_arrows) {
			y = (widget->allocation.height >> 1) - 1 + (widget->allocation.height & 1);
			h = y/2 - XENO_STYLE_YTHICKNESS(style);
			w = h*2;
			x = widget->allocation.width - XENO_STYLE_XTHICKNESS(style) - w * 2;
			
			xeno_draw_solid_arrow2 (style, window, state_type, area, widget, GTK_ARROW_UP,
									x, y-h, w+w, h+h);
			
			xeno_draw_solid_arrow2 (style, window, state_type, area, widget, GTK_ARROW_DOWN,
									x, widget->allocation.height-y, w+w, h+h);
			
			xeno_draw_vline (style, window, state_type, area, widget, NULL,
							 XENO_STYLE_YTHICKNESS(style) + 1,
							 widget->allocation.height - XENO_STYLE_YTHICKNESS(style) - 1,
							 x-w);
			return;
		}
		
		h = MIN(height, (width/2 + 0x0001) & 0xfffe);
		y += (height - h)/2;
		width += h & 0x0001;
		height = h;
	}
	gtk_paint_shadow (style, window, state_type, XENO_SHADOW_USER(shadow_type), area, widget, detail,
					  x, y, width, height);
}

void 
xeno_draw_shadow_gap (GtkStyle       *style,
				      GdkWindow      *window,
				      GtkStateType    state_type,
				      GtkShadowType   shadow_type,
				      GdkRectangle   *area,
				      GtkWidget      *widget,
				      xeno_char      *detail,
				      gint            x,
				      gint            y,
				      gint            width,
				      gint            height,
				      GtkPositionType gap_side,
				      gint            gap_x,
				      gint            gap_width)
{
	GdkPoint points[7];
	int i=0, n=4;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);

	if ((width == -1) && (height == -1))
		gdk_window_get_size (window, &width, &height);
	else if (width == -1)
		gdk_window_get_size (window, &width, NULL);
	else if (height == -1)
		gdk_window_get_size (window, NULL, &height);

	switch (gap_side) {
	  case GTK_POS_TOP:
		if (gap_x > 0) {
			n++;
			points[i].x   = x+gap_x;
			points[i++].y = y;
		}
		goto cont_top;
		
	  case GTK_POS_BOTTOM:
	    if (gap_x+gap_width < width) {
			n++;
			points[i].x   = x + gap_x + gap_width;
			points[i++].y = y + height-1;
		}
		goto cont_bottom;
	
	  case GTK_POS_LEFT:
		if (gap_x+gap_width < height) {
			n++;
			points[i].x   = x;
			points[i++].y = y + gap_x + gap_width;
		}
		goto cont_left;
	
	  case GTK_POS_RIGHT:
	    if (gap_x > 0) {
			n++;
			points[i].x   = x+width-1;
			points[i++].y = y+gap_x;
		}
		goto cont_right;
	}

	while(1) {
	  cont_top:
		points[i].x	  = x;
		points[i++].y = y;
		if (i==n) {
			if (gap_x > 0) {
				n++;
				points[i].x = x;
				points[i].y = y + gap_x;
			}
			goto done;
		}

	  cont_left:
		points[i].x	  = x;
		points[i++].y = y+height-1;
		if (i==n) {
			if (gap_x > 0) {
				n++;
				points[i].x = x + gap_x;
				points[i].y = y + height - 1;
			}
			goto done;
		}

	  cont_bottom:
		points[i].x	  = x+width-1;
		points[i++].y = y+height-1;
		if (i==n) {
			if (gap_x + gap_width < width) {
				n++;
				points[i].x = x + width - 1;
				points[i].y = y + gap_x + gap_width;
			}
			goto done;
		}

	  cont_right:
		points[i].x	  = x+width-1;
		points[i++].y = y;
		if (i==n) {
			if (gap_x + gap_width < width) {
				n++;
				points[i].x = x + gap_x + gap_width;;
				points[i].y = y;
			}
			goto done;
		}
	};
  done:
	xeno_draw_polygon (style, window, state_type, XENO_SHADOW_USER(shadow_type), area,
					   widget, detail, points, n, FALSE);
	
	return;
}


void 
xeno_draw_box_gap (GtkStyle       *style,
				   GdkWindow      *window,
				   GtkStateType    state_type,
				   GtkShadowType   shadow_type,
				   GdkRectangle   *area,
				   GtkWidget      *widget,
				   xeno_char      *detail,
				   gint            x,
				   gint            y,
				   gint            width,
				   gint            height,
				   GtkPositionType gap_side,
				   gint            gap_x,
				   gint            gap_width)
{
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);
	
	if ((width == -1) && (height == -1))
		gdk_window_get_size (window, &width, &height);
	else if (width == -1)
		gdk_window_get_size (window, &width, NULL);
	else if (height == -1)
		gdk_window_get_size (window, NULL, &height);
	
	xeno_draw_bg_rectangle (style, window, state_type, area, widget, x, y, width, height);
	
	xeno_draw_shadow_gap(style, window, state_type, shadow_type, area, widget, detail,
						 x, y, width, height, gap_side, gap_x, gap_width);

	return;
}

void 
xeno_draw_extension (GtkStyle       *style,
				     GdkWindow      *window,
				     GtkStateType    state_type,
				     GtkShadowType   shadow_type,
				     GdkRectangle   *area,
				     GtkWidget      *widget,
				     xeno_char      *detail,
				     gint            x,
				     gint            y,
				     gint            width,
				     gint            height,
				     GtkPositionType gap_side)
{
	GtkStyle	*style2;
	GtkStateType state2;
	GdkPoint	points[8];
	gint		x2, y2;
	gint		bx, by, bx2, by2;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);
	
	if ((width == -1) && (height == -1))
		gdk_window_get_size (window, &width, &height);
	else if (width == -1)
		gdk_window_get_size (window, &width, NULL);
	else if (height == -1)
		gdk_window_get_size (window, NULL, &height);
	
	if (widget->state == GTK_STATE_INSENSITIVE)
		state_type = GTK_STATE_INSENSITIVE;
	
	x2 = x + width - 1;
	y2 = y + height - 1;
	
	switch (gap_side) {
	  case GTK_POS_BOTTOM:
		points[0].x = x2;	points[0].y = y2;
		points[1].x = x2;	points[1].y = y+5;
		points[2].x = x2-2;	points[2].y = y+2;
		points[3].x = x2-5;	points[3].y = y;
		points[4].x = x+5;	points[4].y = y;
		points[5].x = x+2;	points[5].y = y+2;
		points[6].x = x;	points[6].y = y+5;
		points[7].x = x;	points[7].y = y2;
		bx	= x;	by	= y;
		bx2 = x2-4;	by2	= y;
		break;
		
	  case GTK_POS_RIGHT:
		points[0].y = y;	points[0].x = x2;
		points[1].y = y;	points[1].x = x+5;
		points[2].y = y+2;	points[2].x = x+2;
		points[3].y = y+5;	points[3].x = x;
		points[4].y = y2-5;	points[4].x = x;
		points[5].y = y2-2;	points[5].x = x+2;
		points[6].y = y2;	points[6].x = x+5;
		points[7].y = y2;	points[7].x = x2;
		bx	= x;	by	= y;
		bx2 = x;	by2	= y2-4;
		break;
		
	  case GTK_POS_LEFT:
		points[0].y = y2;	points[0].x = x;
		points[1].y = y2;	points[1].x = x2-5;
		points[2].y = y2-2;	points[2].x = x2-2;
		points[3].y = y2-5;	points[3].x = x2;
		points[4].y = y+5;	points[4].x = x2;
		points[5].y = y+2;	points[5].x = x2-2;
		points[6].y = y;	points[6].x = x2-5;
		points[7].y = y;	points[7].x = x;
		x2 -= 4;
		bx	= x2;	by	= y;
		bx2 = x2;	by2	= y2-4;
		break;
		
	  case GTK_POS_TOP:
		points[0].x = x;	points[0].y = y;
		points[1].x = x;	points[1].y = y2-5;
		points[2].x = x+2;	points[2].y = y2-2;
		points[3].x = x+5;	points[3].y = y2;
		points[4].x = x2-5;	points[4].y = y2;
		points[5].x = x2-2;	points[5].y = y2-2;
		points[6].x = x2;	points[6].y = y2-5;
		points[7].x = x2;	points[7].y = y;
		y2 -= 4;
		bx	= x;	by	= y2;
		bx2 = x2-4;	by2	= y2;
		break;
		
	  default:
		return;
	}
	
	style2 = (widget->parent) ? widget->parent->style : style;
	state2 = (widget->parent) ? widget->parent->state : GTK_STATE_NORMAL;
	
	xeno_draw_bg_rectangle (style2, window, state2, area, widget, bx, by, 5, 5);
	xeno_draw_bg_rectangle (style2, window, state2, area, widget, bx2, by2, 5, 5);
	
	xeno_draw_polygon (style, window, state_type, XENO_SHADOW_USER(shadow_type),
				   area, widget, detail, points, 8, TRUE);

	if (!xeno_pseudocolor) {
		GdkPixmap	*pixmap;
		GdkBitmap	*mask;
		
		switch (state_type) {
		  case GTK_STATE_ACTIVE:
			pixmap = xeno_pixmap_get (window, style, style2, XENO_IMAGE_NOTEBOOK_CORNER_ACTIVE);
			break;
		  case GTK_STATE_INSENSITIVE:
			pixmap = xeno_pixmap_get (window, style, style2, XENO_IMAGE_NOTEBOOK_CORNER_INSENSITIVE);
			break;
		  default:
			pixmap = xeno_pixmap_get (window, style, style2, XENO_IMAGE_NOTEBOOK_CORNER_NORMAL);
		}
		mask = xeno_image_mask (XENO_IMAGE_NOTEBOOK_CORNER_ACTIVE);
	
		if (pixmap) {
			if (gap_side == GTK_POS_BOTTOM || gap_side == GTK_POS_RIGHT) {
				xeno_draw_pixmap (window, style->bg_gc[state_type], area,
								  pixmap, mask, 0, 0, x, y, 7, 7);
			}
			if (gap_side == GTK_POS_BOTTOM || gap_side == GTK_POS_LEFT) {
				xeno_draw_pixmap (window, style->bg_gc[state_type], area,
								  pixmap, mask, 6, 0, x+width-7, y, 7, 7);
			}
			if (gap_side == GTK_POS_TOP || gap_side == GTK_POS_RIGHT) {
				xeno_draw_pixmap (window, style->bg_gc[state_type], area,
								  pixmap, mask, 0, 6, x, y+height-7, 7, 7);
			}
			if (gap_side == GTK_POS_TOP || gap_side == GTK_POS_LEFT) {
				xeno_draw_pixmap (window, style->bg_gc[state_type], area,
								  pixmap, mask, 6, 6, x+width-7, y+height-7, 7, 7);
			}
		} else {
			g_warning("no pixmap\n");
		}
	}
}


void 
xeno_draw_focus (GtkStyle      *style,
				 GdkWindow     *window,
				 GdkRectangle  *area,
				 GtkWidget     *widget,
				 xeno_char     *detail,
				 gint           x,
				 gint           y,
				 gint           width,
				 gint           height)
{
	GdkGC		*gc;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);
	
	if (GTK_WIDGET_STATE(widget) == GTK_STATE_INSENSITIVE)
		return;

	if ((width == -1) && (height == -1)) {
		gdk_window_get_size (window, &width, &height);
		width -= 1;
		height -= 1;
	} else if (width == -1) {
		gdk_window_get_size (window, &width, NULL);
		width -= 1;
	} else if (height == -1) {
		gdk_window_get_size (window, NULL, &height);
		height -= 1;
	}
	
	gc = style->black_gc;	/* hardcoded by X servers */
	if (area)
		gdk_gc_set_clip_rectangle (gc, area);
	
	if ((detail) && !strcmp (detail, "add-mode")) {
		gdk_gc_set_line_attributes (gc, 1, GDK_LINE_ON_OFF_DASH, 0, 0);
		gdk_gc_set_dashes (gc, 0, "\4\4", 2);
		
		gdk_draw_rectangle (window, gc, FALSE, x, y, width, height);
		
		gdk_gc_set_line_attributes (gc, 1, GDK_LINE_SOLID, 0, 0);
	} else {
		if (GTK_IS_OPTION_MENU(widget)) {
			y++;
			height-=2;
		}
		if (GTK_IS_CHECK_BUTTON(widget) || GTK_IS_RADIO_BUTTON(widget) || 
			(detail && !strcmp("tab", detail)))
		{
			static const gchar data[] = { 0x01, 0x02 };
			static GdkBitmap *bm = NULL;
			
			if (!bm)
				bm = gdk_bitmap_create_from_data(window, data, 2, 2);

			if (bm) {
				gdk_gc_set_stipple(gc, bm);
				gdk_gc_set_fill(gc, GDK_STIPPLED);
				gdk_gc_set_ts_origin(gc, x, y);
				
				gdk_draw_rectangle (window, gc, FALSE, x, y, width & 0xfffe, height & 0xfffe);
				
				gdk_gc_set_fill(gc, GDK_SOLID);
				goto done;
			}
		} else if (GTK_IS_EDITABLE(widget)) {
			int t;
			xeno_draw_shadow (style, window, widget->state, XENO_SHADOW_IN, area, widget, detail,
							  x,y, width+1,height+1);
			
			t = (GTK_IS_TEXT(widget)) ? 2 : 1;
			x += t;
			y += t;
			t <<= 1;
			width -= t;
			height -= t;
		} else if (   GTK_IS_TOGGLE_BUTTON(widget)
				   && (   GTK_TOGGLE_BUTTON(widget)->active
					   || widget->state == GTK_STATE_ACTIVE))
		{
			xeno_draw_shadow (style, window, widget->state, GTK_SHADOW_IN, area, widget, detail,
							  x,y, width+1,height+1);
			x += 2;
			y += 2;
			width -= 4;
			height -= 4;
		}
		
		gdk_draw_rectangle (window, gc, FALSE, x, y, width, height);
	}
	
  done:
	if (area)
		gdk_gc_set_clip_rectangle (gc, NULL);
}


void 
xeno_draw_slider (GtkStyle      *style,
				  GdkWindow     *window,
				  GtkStateType   state_type,
				  GtkShadowType  shadow_type,
				  GdkRectangle  *area,
				  GtkWidget     *widget,
				  xeno_char     *detail,
				  gint           x,
				  gint           y,
				  gint           width,
				  gint           height,
				  GtkOrientation orientation)
{
	GdkGC *lgc, *dgc, *mgc;
	gint i, w, t, xt, yt;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);
	
	if ((width == -1) && (height == -1))
		gdk_window_get_size (window, &width, &height);
	else if (width == -1)
		gdk_window_get_size (window, &width, NULL);
	else if (height == -1)
		gdk_window_get_size (window, NULL, &height);
	
	xt = MAX(0, XENO_STYLE_XTHICKNESS(style) - 1);
	yt = MAX(0, XENO_STYLE_YTHICKNESS(style) - 1);
	
	t = 0;
	if (GTK_WIDGET_HAS_FOCUS(widget)) {
		t = 1;
		xt++;
		yt++;
		xeno_draw_focus (style, window, area, widget, detail, x, y, width-1, height-1);
	}
	
	state_type = widget->state;
	
	gtk_paint_box (style, window, state_type, shadow_type,
				   area, widget, detail, x+t, y+t, width-t-t, height-t-t);
	
	lgc = style->light_gc[state_type];
	dgc = style->dark_gc[state_type];
	mgc = style->mid_gc[state_type];
	
	if (area) {
		gdk_gc_set_clip_rectangle (lgc, area);
		gdk_gc_set_clip_rectangle (dgc, area);
		gdk_gc_set_clip_rectangle (mgc, area);
	}
	
	if (orientation == GTK_ORIENTATION_HORIZONTAL) {
		w = MIN(width, height + width/6);
		x = x + width/2 - w/2;
		y += yt;
		height -= yt + yt;
		for (i=x; i<x+w; i+=3) {
			gdk_draw_line (window, dgc,		i,  	y+1,	i,   	y+height-1);
			gdk_draw_line (window, lgc,		i+1,	y,		i+1,	y+height-2);
			gdk_draw_point (window, mgc,	i,		y);
			gdk_draw_point (window, mgc,	i+1,	y+height-1);
		}
	} else {
		w = MIN(height, width + height/6);
		y = y + height/2 - w/2;
		x += xt;
		width -= xt + xt;
		for (i=y; i<x+w; i+=3) {
			gdk_draw_line (window, dgc,		x+1,	i,		x+width-2,	i  );
			gdk_draw_line (window, lgc,		x+1,	i+1,	x+width-2,	i+1);
			gdk_draw_point (window, mgc,	x,		i);
			gdk_draw_point (window, mgc,	x+width-1, i+1);
		}
	}
	if (area) {
		gdk_gc_set_clip_rectangle (lgc, NULL);
		gdk_gc_set_clip_rectangle (dgc, NULL);
		gdk_gc_set_clip_rectangle (mgc, NULL);
	}
}

void
xeno_draw_handle  (GtkStyle      *style,
				   GdkWindow     *window,
				   GtkStateType   state_type,
				   GtkShadowType  shadow_type,
				   GdkRectangle  *area,
				   GtkWidget     *widget,
				   xeno_char     *detail,
				   gint           x,
				   gint           y,
				   gint           width,
				   gint           height,
				   GtkOrientation orientation)
{
	GdkGC *light_gc, *dark_gc, *mid_gc, *white_gc;
	XenoKnobType knob_type;
	gint	xt, yt;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);

	if ((width == -1) && (height == -1))
		gdk_window_get_size (window, &width, &height);
	else if (width == -1)
		gdk_window_get_size (window, &width, NULL);
	else if (height == -1)
		gdk_window_get_size (window, NULL, &height);
	
	light_gc	= style->light_gc[state_type];
	dark_gc		= style->dark_gc[state_type];
	mid_gc		= style->mid_gc[state_type];
	white_gc	= style->white_gc;
	
	xt = XENO_STYLE_XTHICKNESS(style);
	yt = XENO_STYLE_YTHICKNESS(style);
	
	knob_type = XENO_KNOB_NONE;
	if (XENO_STYLE_RC_DATA(style))
		knob_type = XENO_STYLE_RC_DATA(style)->handle_knob;
	
	if (knob_type >= XENO_KNOB_LINES)
		shadow_type = GTK_SHADOW_NONE;
	
	if (shadow_type != GTK_SHADOW_NONE) {
		switch (shadow_type) {
		  case GTK_SHADOW_IN:
			shadow_type = (GtkShadowType)XENO_SHADOW_THIN_IN;
			break;
	
		  case GTK_SHADOW_OUT:
			shadow_type = (GtkShadowType)XENO_SHADOW_THIN_OUT;
			break;
	
		  case GTK_SHADOW_ETCHED_IN:
			shadow_type = (GtkShadowType)XENO_SHADOW_THIN_OUT;
			state_type = GTK_STATE_INSENSITIVE;
			break;
		
		  default:
			shadow_type = GTK_SHADOW_NONE;
		}
		xeno_draw_box (style, window, state_type, shadow_type, area, widget, NULL, x, y,
					   width  - ((orientation==GTK_ORIENTATION_VERTICAL)?1:0),
					   height - ((orientation==GTK_ORIENTATION_HORIZONTAL)?1:0));
	}
	
	if (   shadow_type != GTK_SHADOW_NONE
		&& detail
		&& (   !strcmp(detail, "handlebox")
			|| !strcmp(detail, "dockitem")))
	{
		gdk_gc_set_clip_rectangle (mid_gc, area);
		gdk_gc_set_clip_rectangle (light_gc, area);
		if (orientation == GTK_ORIENTATION_VERTICAL) {
			gdk_draw_line(window, light_gc, x+width-1, y, x+width-1, y+height-2);
			gdk_draw_point(window, mid_gc, x+width-1, y+height-1);
			width--;
		} else {
			gdk_draw_line(window, light_gc, x+1, y+height-1, x+width-2, y+height-1);
			gdk_draw_point(window, mid_gc, x+width-1, y+height-1);
			height--;
		}
		gdk_gc_set_clip_rectangle (light_gc, NULL);
		gdk_gc_set_clip_rectangle (mid_gc, NULL);
	}
	
	if (knob_type == XENO_KNOB_BOX) {
		switch (state_type) {
		  case GTK_STATE_NORMAL:
		  case GTK_STATE_ACTIVE:
			state_type = GTK_STATE_PRELIGHT;
			break;
		  default:
			;
		}
		xeno_draw_box (style, window, state_type, GTK_SHADOW_ETCHED_IN, area, widget, NULL,
					   x+2, y+2, width-4, height-4);
	} else if (knob_type == XENO_KNOB_LINES) {
		xeno_draw_lines (style, window, state_type, GTK_SHADOW_OUT, area, widget,
						 x+2, y+2, width-4, height-4, orientation);
	} else {
		if (knob_type == XENO_KNOB_HOLES) {
			shadow_type = GTK_SHADOW_ETCHED_IN;
		} else if (knob_type == XENO_KNOB_BUDS) {
			shadow_type = GTK_SHADOW_ETCHED_OUT;
		} else {
			shadow_type = GTK_SHADOW_OUT;
		}	
		
		xeno_draw_buds (style, window, state_type, shadow_type, area, widget,
						x+2, y+2, width-4, height-4, orientation);
	}
}


/*** Background ***************************************************************/
static void
xeno_paint_background	(GtkStyle		*style,
						 GdkDrawable	*drawable,
						 GdkGC			*bg_gc,
						 GdkPixmap		*bg_pixmap,
						 GtkStateType	state_type,
						 GdkRectangle	*area,
						 gint			tile_x,
						 gint			tile_y,
						 gint			window_width,
						 gint			window_height)
{
	GdkRectangle	rect;
	XenoRcData		*rc_data;
	XenoRcOrigin	*rc_origin;
	XenoOriginType	origin_type;
	gint			pixmap_width, pixmap_height;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (drawable != NULL);
	g_return_if_fail (area != NULL);
	g_return_if_fail (bg_gc != NULL);
	
	rc_data = XENO_STYLE_RC_DATA(style);
	origin_type = XENO_ORIGIN_DEFAULT;
	if (rc_data) {
		rc_origin = &rc_data->origin[state_type];
		if (origin_type == XENO_ORIGIN_UNKNOWN)
			origin_type = rc_origin->type;
	}
	
	if (area == NULL) {
		rect.x = rect.y = 0;
		rect.width = window_width;
		rect.height = window_height;
		area = &rect;
	}
	
	if (bg_pixmap != NULL && bg_pixmap != (GdkPixmap *)GDK_PARENT_RELATIVE) {
		if (origin_type == XENO_ORIGIN_ALIGNMENT) {
			gdk_window_get_size (bg_pixmap, &pixmap_width, &pixmap_height);
			tile_x += rc_origin->x * (window_width  % pixmap_width  + (((window_width  / pixmap_width ) & 1) ? 0 : pixmap_width));
			tile_y += rc_origin->y * (window_height % pixmap_height + (((window_height / pixmap_height) & 1) ? 0 : pixmap_height));
		}
		gdk_gc_set_fill (bg_gc, GDK_TILED);
		gdk_gc_set_tile (bg_gc, bg_pixmap);
		gdk_gc_set_ts_origin (bg_gc, tile_x, tile_y);
	}
	gdk_draw_rectangle (drawable, bg_gc, TRUE, area->x, area->y, area->width, area->height);
	if (bg_pixmap)
		gdk_gc_set_fill (bg_gc, GDK_SOLID);
}

static void
xeno_draw_bg_rectangle	(GtkStyle		*style,
						 GdkDrawable	*drawable,
						 GtkStateType	state_type,
						 GdkRectangle	*area,
						 GtkWidget		*widget,
						 gint			x,
						 gint			y,
						 gint			width,
						 gint			height)
{
	GdkRectangle	rect;
	GdkPixmap		*bg_pixmap;
	XenoRcData		*rc_data;
	XenoOriginType	origin_type;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (drawable != NULL);
	/*
	if (width < 0 || height < 0) {
		gint window_width, window_height;
		gdk_window_get_size (drawable, &window_width, &window_height);
		if (width < 0) width = window_width - x;
		if (height < 0) height = window_height - y;
	}
	*/
	rect.x		= x;
	rect.y		= y;
	rect.width	= width;
	rect.height	= height;
	if (area) {
		GdkRectangle rect2;
		if (!gdk_rectangle_intersect (&rect, area, &rect2))
			return;
		
		rect = rect2;
	}
	
	bg_pixmap	= style->bg_pixmap[state_type];
	origin_type	= XENO_ORIGIN_DEFAULT;
	rc_data		= XENO_STYLE_RC_DATA(style);
	if (rc_data) {
		origin_type = rc_data->origin[state_type].type;
		if (rc_data->gradient[state_type].type != XENO_GRADIENT_NONE && widget) {
			bg_pixmap = xeno_gradient_get (style, drawable, state_type, widget, width, height);
			if (origin_type == XENO_ORIGIN_DEFAULT && (x | y) != 0)
				origin_type = XENO_ORIGIN_UNKNOWN;
		}
	}
	
	if (   widget && !GTK_WIDGET_NO_WINDOW(widget)
	  #if XENO_GTK2
		&& GDK_IS_WINDOW(drawable)
	  #else
		&& gdk_window_get_type (drawable) != GDK_WINDOW_PIXMAP
	  #endif
		&& (bg_pixmap == NULL || origin_type == XENO_ORIGIN_DEFAULT))
	{
		if (bg_pixmap) {
			if (bg_pixmap == (GdkPixmap *)GDK_PARENT_RELATIVE)
				bg_pixmap = NULL;
			
			gdk_window_set_back_pixmap (drawable, bg_pixmap, bg_pixmap == NULL);
		} else {
			gdk_window_set_background (drawable, &style->bg[state_type]);
		}
		gdk_window_clear_area (drawable, rect.x, rect.y, rect.width, rect.height);
	} else {
		xeno_paint_background (style, drawable, style->bg_gc[state_type], bg_pixmap, state_type,
							   &rect, x, y, width, height);
	}
}

static void
xeno_draw_base_rectangle(GtkStyle		*style,
						 GdkWindow		*window,
						 GtkStateType	state_type,
						 GdkRectangle	*area,
						 GtkWidget		*widget,
						 gint			x,
						 gint			y,
						 gint			width,
						 gint			height)
{
	GdkGC			*gc;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);
	
	if ((width | height) < 0)
		gdk_window_get_size (window, (width < 0) ? &width : NULL, (height < 0) ? &height : NULL);
	
	gc = style->base_gc[state_type];
	
	if (area)
		gdk_gc_set_clip_rectangle (gc, area);
	
	gdk_gc_set_fill(gc, GDK_SOLID);
	gdk_draw_rectangle(window, gc, TRUE, x, y, width, height);
	
	if (area)
		gdk_gc_set_clip_rectangle (gc, NULL);
}


/*
 *	XenoGradient
 */

/* color functions */
static void
xeno_gradient_put_pseudo_color (XenoGradientContext *ctx, gfloat f, gint x, gint y)
{
	guint	pixel;
	guint16	t;
	
	pixel = ctx->bot_pixel;
	f = f - 1.0;
	if (f < 0.0) {
		pixel = ctx->top_pixel;
		f = -f;
	}
	if (f <= xeno_dither(x, y)) {
		if (ctx->blend)
			return;
		
		pixel = ctx->bg_pixel;
	}
	gdk_image_put_pixel (ctx->image, x, y, pixel);
}

static void
xeno_gradient_put_true_color (XenoGradientContext *ctx, gfloat f, gint x, gint y)
{
	XenoColor	tmp, *ptr;
	GdkColor	color;
	
	ptr = &ctx->bot;
	f = f - 1.0;
	if (f < 0.0) {
		ptr = &ctx->top;
		f = -f;
	}
	xeno_color_blend (&ctx->bg, ptr, f * ctx->alpha, &tmp);
	xeno_color_dither (&tmp, xeno_visual, x, y, &color);
	gdk_colormap_alloc_color (xeno_colormap, &color, FALSE, TRUE);
	gdk_image_put_pixel (ctx->image, x, y, color.pixel);
}

static void
xeno_gradient_blend_true_color (XenoGradientContext *ctx, gfloat f, gint x, gint y)
{
	guint32 pixel;
	
	pixel = gdk_image_get_pixel (ctx->image, x, y);
	xeno_color_init (&ctx->bg,
					 (pixel & ctx->visual->red_mask)   * ctx->red_factor,
					 (pixel & ctx->visual->green_mask) * ctx->green_factor,
					 (pixel & ctx->visual->blue_mask)  * ctx->blue_factor);
	xeno_gradient_put_true_color (ctx, f, x, y);
}

static void
xeno_gradient_context_init (XenoGradientContext *ctx, const GtkStyle *style, GtkStateType state_type,
							GdkVisual *visual, GdkColormap *colormap, GdkImage *image,
							gboolean blend, gboolean realize)
{
	GdkColor			 color;
	XenoColor			 tmp;
	const XenoColor		 *top_color;
	const XenoStyleData	 *style_data;
	const XenoRcData	 *rc_data;
	const XenoRcGradient *rc_gradient;
	gfloat alpha;
	
	style_data	= XENO_STYLE_DATA(style);
	rc_data		= XENO_STYLE_RC_DATA(style);
	rc_gradient	= &rc_data->gradient[state_type];
	
	ctx->visual	= visual;
	ctx->colormap = colormap;
	ctx->image	= image;
	ctx->blend	= blend;
	
	top_color = style_data->white;
	if (rc_gradient->type == XENO_GRADIENT_NONE) {
		alpha = 0.0;
	} else {
		alpha = rc_gradient->factor;
		if (alpha < 1.0) {
			alpha = 1.0 - MAX(alpha, rc_data->shade[state_type]);
			top_color = style_data->black;
		} else {
			alpha = MIN(alpha, rc_data->shine[state_type]) - 1.0;
		}
	}
	ctx->alpha = alpha;
	top_color = &top_color[state_type];
	ctx->top = *top_color;
	
	xeno_color_from_gdk (&ctx->bg, &style->bg[state_type]);
	xeno_color_flip (&ctx->bg, top_color, &ctx->bg, &ctx->bot);
	
	if (xeno_pseudocolor) {
		ctx->put_pixel = xeno_gradient_put_pseudo_color;
		if (realize) {
			ctx->bg_pixel = style->bg[state_type].pixel;
			
			xeno_color_blend (&ctx->bg, &ctx->top, alpha, &tmp);
			xeno_color_to_gdk (&tmp, &color);
			gdk_colormap_alloc_color (colormap, &color, FALSE, TRUE);
			ctx->top_pixel = color.pixel;
			
			xeno_color_blend (&ctx->bg, &ctx->bot, alpha, &tmp);
			xeno_color_to_gdk (&tmp, &color);
			gdk_colormap_alloc_color (colormap, &color, FALSE, TRUE);
			ctx->bot_pixel = color.pixel;
		}
	} else {
		ctx->put_pixel = xeno_gradient_put_true_color;
		if (visual->type == GDK_VISUAL_TRUE_COLOR && blend) {
			ctx->put_pixel    = xeno_gradient_blend_true_color;
			
			ctx->red_factor   = 1.0 / visual->red_mask;
			ctx->green_factor = 1.0 / visual->green_mask;
			ctx->blue_factor  = 1.0 / visual->blue_mask;
		}
	}
}

void
xeno_gradient_rgb (const GtkStyle *style, GtkStateType state_type, gfloat f, XenoColor *dst)
{
	XenoGradientContext ctx;
	XenoColor	*ptr;
	
	if (   XENO_STYLE_RC_DATA(style)->gradient[state_type].type == XENO_GRADIENT_NONE
		|| xeno_pseudocolor)
	{
		xeno_color_from_gdk (dst, &style->bg[state_type]);
	} else {
		xeno_gradient_context_init (&ctx, style, state_type, xeno_visual, style->colormap,
									NULL, FALSE, FALSE);
		ptr = &ctx.bot;
		f -= 1.0;
		if (f < 0.0) {
			ptr = &ctx.top;
			f = -f;
		}
		xeno_color_blend (&ctx.bg, ptr, f * ctx.alpha, dst);
	}
}

/* realize */
static GdkPixmap *
xeno_gradient_realize  (GtkStyle		 *style,
						GdkWindow		 *window,
						GtkStateType 	 state_type,
						XenoGradient	 *gradient)
{
	GdkColor			color;
	GdkVisual			*visual;
	GdkPixmap			*pixmap;
	GdkPixmap			*bg_pixmap;
	GdkGC				*gc;
	GdkImage			*image;
	
	XenoGradientContext	ctx;
	XenoRcGradient		*rc_gradient;
	XenoGradientType	gradient_type;
	XenoGradientSet		*gradient_set;
	
	gint	x, y, width, height, i;
	gint	min_width, min_height;
	gint	pixmap_width, pixmap_height;
	gfloat	f, g, t;
	
	g_return_val_if_fail (style != NULL, NULL);
	g_return_val_if_fail (window != NULL, NULL);
	g_return_val_if_fail (gradient != NULL, NULL);
	
	min_width = 8;
	if (xeno_pseudocolor)
		min_width = 16;
	min_height = min_width;
	bg_pixmap = style->bg_pixmap[state_type];
	if (bg_pixmap) {
		if (bg_pixmap == (GdkPixmap *)GDK_PARENT_RELATIVE) {
			bg_pixmap = NULL;
		} else {
			gdk_window_get_size (bg_pixmap, &pixmap_width, &pixmap_height);
			min_width  = MAX(min_width, pixmap_width);
			min_height = MAX(min_height, pixmap_height);
		}
	}
	
	rc_gradient   = &XENO_STYLE_RC_DATA(style)->gradient[state_type];
	gradient_type = rc_gradient->type;
	switch (gradient_type) {
	  case XENO_GRADIENT_VERTICAL:
		width  = min_width;
		height = gradient->height;
		break;
		
	  case XENO_GRADIENT_HORIZONTAL:
		width  = gradient->width;
		height = min_height;
		if (rc_gradient->interlaced && height == 1)
			height = 2;
		break;
		
	  case XENO_GRADIENT_DIAGONAL:
		width  = gradient->width;
		height = gradient->height;
		break;
		
	  case XENO_GRADIENT_NONE:
	  default:
		return bg_pixmap;
	}
	
	/* Create pixmap and draw */
	visual = gdk_window_get_visual (window);
	if (!bg_pixmap) {
		if ((image = gdk_image_new (GDK_IMAGE_NORMAL, visual, width, height)) == NULL)
			return NULL;
	}
	if ((pixmap = gdk_pixmap_new (window, width, height, visual->depth)) == NULL)
		return bg_pixmap;
	
	if (bg_pixmap) {
		GdkRectangle area;
		area.x = area.y = 0;
		area.width = width;
		area.height = height;
		xeno_paint_background (style, pixmap, style->bg_gc[state_type], bg_pixmap,
							   state_type, &area, 0, 0, width, height);
		if ((image = gdk_image_get (pixmap, 0, 0, width, height)) == NULL) {
			gdk_pixmap_unref (pixmap);
			return bg_pixmap;
		}
	}
	
	xeno_gradient_context_init (&ctx, style, state_type, visual, gdk_window_get_colormap(window),
								image, bg_pixmap != NULL, TRUE);

	if (!rc_gradient->interlaced) {
		if (gradient_type == XENO_GRADIENT_VERTICAL) {
			f = 2.0 / (height - 1);
			for (y = 0; y < height; ++y) {
				t = f * y;
				for (x = 0; x < width; ++x)
					ctx.put_pixel (&ctx, t, x, y);
			}
		} else if (gradient_type == XENO_GRADIENT_HORIZONTAL) {
			f = 2.0 / (width - 1);
			for (y = 0; y < height; ++y) {
				for (x = 0; x < width; ++x)
					ctx.put_pixel (&ctx, f * x, x, y);
			}
		} else {
			f = 1.0 / (width - 1);
			g = 1.0 / (height - 1);
			for (y = 0; y < height; ++y) {
				for (x = 0; x < width; ++x)
					ctx.put_pixel (&ctx, f*x + g*y, x, y);
			}
		}
	} else {
		if (gradient_type == XENO_GRADIENT_VERTICAL) {
			f = 1.0 / (height - 1);
			for (y = 0; y < height; ++y) {
				t = f * y + (y & 0x01);
				for (x = 0; x < width; ++x)
					ctx.put_pixel (&ctx, t, x, y);
			}
		} else if (gradient_type == XENO_GRADIENT_HORIZONTAL) {
			f = 1.0 / (width - 1);
			for (y = 0; y < height; ++y) {
				t = y & 0x01;
				for (x = 0; x < width; ++x) {
					ctx.put_pixel (&ctx, f*x + t, x, y);
				}
			}
		} else {
			f = 0.5 / (width - 1);
			g = 0.5 / (height - 1);
			for (y = 0; y < height; ++y) {
				t = y & 0x01;
				for (x = 0; x < width; ++x)
					ctx.put_pixel (&ctx, f*x + g*y + t, x, y);
			}
		}
	}
	gdk_draw_image (pixmap, style->bg_gc[state_type], image, 0, 0, 0, 0, width, height);
	gdk_image_destroy (image);
	
	/* make pixmap available in more than one state if possible */
	gradient_set = &XENO_STYLE_DATA(style)->gradient_set;
	state_type = gradient_set->redirect[state_type];
	for (i = 0; i < 5; ++i) {
		if (gradient_set->redirect[i] == state_type) {
			if (gradient->pixmap[i] != NULL)
				gdk_pixmap_unref (gradient->pixmap[i]);
			
			gradient->pixmap[i] = pixmap;
			if (state_type != i)
				gdk_pixmap_ref (pixmap);
		}
	}
	return pixmap;
}

static void
xeno_gradient_destroy (XenoGradient *gradient)
{
	XenoGradient **ptr, *next;
	GdkPixmap	 *pixmap;
	gint i;
	
	g_return_if_fail (gradient != NULL);
  #if DIRTY_GRADIENTS
	g_return_if_fail (gradient->ref_count == 0);
  #else
	g_return_if_fail (gradient->apps == NULL);
  #endif
  #if COUNT_GRADIENT_STYLE
	gtk_style_unref (gradient->style);
  #endif
	
	if ((ptr = gradient->ptr) != NULL) {
		if ((*ptr = next = gradient->next) != NULL)
			next->ptr = ptr;
	}
	for (i = 0; i < 5; ++i) {
		if ((pixmap = gradient->pixmap[i]) != NULL) {
			gdk_pixmap_unref(pixmap);
		}
	}
	
	g_mem_chunk_free (xeno_gradient_chunk, gradient);
	if (--xeno_gradient_ref_count == 0) {
		g_mem_chunk_destroy (xeno_gradient_chunk);
		xeno_gradient_chunk = NULL;
	}
}

#if DIRTY_GRADIENTS
static void
xeno_gradient_unref (XenoGradient *gradient)
{
	g_return_if_fail (gradient != NULL);
	
	if (--gradient->ref_count == 0)
		xeno_gradient_destroy (gradient);
}
#else
static void
xeno_gradient_app_destroy (XenoGradientApp *app)
{
	g_return_if_fail (app != NULL);
	
	if ((*app->ptr = app->next) != NULL) {
		app->next->ptr = app->ptr;
	} else if (app->ptr == &app->gradient->apps) {
		xeno_gradient_destroy (app->gradient);
	}
	g_mem_chunk_free (xeno_gradient_app_chunk, app);
}
#endif

static GdkPixmap *
xeno_gradient_get	(GtkStyle		*style,
					 GdkWindow		*window,
					 GtkStateType 	state_type,
					 GtkWidget		*widget,
					 gint			width,
					 gint			height)
{
  #if !DIRTY_GRADIENTS
	XenoGradientApp	 *app;
  #endif
	XenoGradient	 *gradient;
	XenoGradientSet	 *gradient_set;
	XenoRcData		 *rc_data;
	XenoGradientType gradient_type;
	GdkGC		*gc;
	GdkPixmap	*pixmap;
	gint		i;
	
	g_return_val_if_fail (style != NULL, NULL);
	g_return_val_if_fail (window != NULL, NULL);
	g_return_val_if_fail (widget != NULL, NULL);
	g_return_val_if_fail (XENO_STYLE_RC_DATA(style), NULL);
	
	rc_data = XENO_STYLE_RC_DATA(style);
	gradient_type = rc_data->gradient[state_type].type;
	if (gradient_type == XENO_GRADIENT_NONE)
		return NULL;
	
	/* init the gradient system */
	if (xeno_gradient_quark == 0)
		xeno_gradient_quark = g_quark_from_string ("XenoGradientWidgetMapping");
	
  #if !DIRTY_GRADIENTS
	if (xeno_gradient_app_chunk == NULL)
		xeno_gradient_app_chunk = g_mem_chunk_create (XenoGradientApp, 32, G_ALLOC_AND_FREE);
	
  #endif
	if (xeno_gradient_chunk == NULL)
		xeno_gradient_chunk = g_mem_chunk_create (XenoGradient, 8, G_ALLOC_AND_FREE);
	
	/* Lookup gradient attached to widget */
  #if DIRTY_GRADIENTS
	if ((gradient = gtk_object_get_data_by_id (GTK_OBJECT(widget), xeno_gradient_quark)) != NULL) {
  #else
	if ((app = gtk_object_get_data_by_id (GTK_OBJECT(widget), xeno_gradient_quark)) != NULL) {
		gradient = app->gradient;
  #endif
		if (   gradient->ptr != NULL
			&& gradient->style == style
			&& (   (gradient_type == XENO_GRADIENT_VERTICAL   && gradient->height == height)
				|| (gradient_type == XENO_GRADIENT_HORIZONTAL && gradient->width == width)
				|| (gradient->width == width && gradient->height == height)))
		{
			goto match;
		}
  #if DIRTY_GRADIENTS
		gtk_object_remove_data_by_id (GTK_OBJECT(widget), xeno_gradient_quark);
  #else
		if ((*app->ptr = app->next) != NULL) {
			app->next->ptr = app->ptr;
		} else if (gradient->apps == NULL) {
			xeno_gradient_destroy (gradient);
		}
	} else {
		if ((app = g_mem_chunk_alloc (xeno_gradient_app_chunk)) == NULL)
			return NULL;
		
		app->widget = widget;
		gtk_object_set_data_by_id_full (GTK_OBJECT(widget), xeno_gradient_quark,
										app, (GtkDestroyNotify)xeno_gradient_app_destroy);
  #endif
	}
	
	/* Find gradient with the right size */
	gradient_set = &XENO_STYLE_DATA(style)->gradient_set;
	for (gradient = gradient_set->gradients; gradient != NULL; gradient = gradient->next) {
		if (   (gradient_type == XENO_GRADIENT_VERTICAL   && gradient->height == height)
			|| (gradient_type == XENO_GRADIENT_HORIZONTAL && gradient->width == width)
			|| (gradient->width == width && gradient->height == height))
		{
			goto attach;
		}
	}
	
	/* Create new gradient */
	if ((gradient = g_mem_chunk_alloc (xeno_gradient_chunk)) == NULL) {
	  #if !DIRTY_GRADIENTS
		g_mem_chunk_free (xeno_gradient_app_chunk, app);
	  #endif
		return NULL;
	}
	
	gradient->style = style;
  #if COUNT_GRADIENT_STYLE
	gtk_style_ref (style);
  #endif
	
	if ((gradient->next = gradient_set->gradients) != NULL)
		gradient->next->ptr = &gradient->next;
	gradient->ptr = &gradient_set->gradients;
	gradient_set->gradients = gradient;
	
	gradient->width  = width;
	gradient->height = height;
	for (i = 0; i < 5; ++i)
		gradient->pixmap[i] = NULL;
	
  #if DIRTY_GRADIENTS
	gradient->ref_count = 0;
	xeno_gradient_ref_count += 1;
  #else
	gradient->apps = NULL;
  #endif
	
  attach:
	/* Attach gradient to widget */
  #if DIRTY_GRADIENTS
	gtk_object_set_data_by_id_full (GTK_OBJECT(widget), xeno_gradient_quark,
									gradient, (GtkDestroyNotify)xeno_gradient_unref);
	gradient->ref_count += 1;
  #else
	if ((app->next = gradient->apps) != NULL)
		app->next->ptr = &app->next;
	app->ptr = &gradient->apps;
	gradient->apps = app;
	app->gradient = gradient;
  #endif
	
  match:
	/* Create gradient pixmap if necessary */
	if ((pixmap = gradient->pixmap[state_type]) == NULL)
		pixmap = xeno_gradient_realize (style, window, state_type, gradient);
	
	return pixmap;
}

void xeno_gradient_set_realize (XenoGradientSet *gradient_set, GtkStyle *style)
{
	XenoRcData		*rc_data;
	XenoRcGradient	*rc_gradient;
	XenoGradient	*gradient;
	gint			i, j;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (gradient_set != NULL);
	
	rc_data = XENO_STYLE_RC_DATA(style);
	if (rc_data == NULL)
		return;
	
	/* clear existing gradients, reusing existing structures but forcing the pixmaps
	   to be re-realized
	*/
	for (gradient = gradient_set->gradients; gradient != NULL; gradient = gradient->next) {
		for (i = 0; i < 5; ++i) {
			if (gradient->pixmap[i] != NULL) {
				gdk_pixmap_unref (gradient->pixmap[i]);
				gradient->pixmap[i] = NULL;
			}
		}
	}
	
	/* determine which pixmaps we can use for more than one state */
	for (i = 0; i < 5; ++i) {
		rc_gradient	= &rc_data->gradient[i];
		if (rc_gradient->type != XENO_GRADIENT_NONE) {
			for (j = 0; j < i; ++j) {
				if (   rc_gradient[i].type == rc_data->gradient[j].type
					&& rc_gradient[i].interlaced == rc_data->gradient[i].interlaced
					&& rc_gradient[i].factor == rc_data->gradient[j].factor
					&& rc_data->white[i] == rc_data->white[j]
					&& rc_data->black[i] == rc_data->black[j]
					&& style->bg_pixmap[i] == style->bg_pixmap[j]
					&& gdk_color_equal (&style->bg[i], &style->bg[j]))
				{
					break;
				}
			}
			gradient_set->redirect[i] = j;
		}
	}
}

void xeno_gradient_set_unrealize (XenoGradientSet *gradient_set) {
  #if !DIRTY_GRADIENTS
	XenoGradientApp	*app, *next_app;
  #endif
	XenoGradient	*gradient, *next_gradient;
	gint i, n;
	
	g_return_if_fail (gradient_set != NULL);
	
	for (gradient = gradient_set->gradients; gradient != NULL; gradient = next_gradient) {
		next_gradient = gradient->next;
	  #if DIRTY_GRADIENTS
		/* dissolve the list and unrealize */
		*gradient->ptr = NULL;
		gradient->ptr = NULL;
	  #if !COUNT_GRADIENT_STYLE
		gradient->style = NULL;
	  #endif
		for (i = 0; i < 5; ++i) {
			if (gradient->pixmap[i]) {
				gdk_pixmap_unref (gradient->pixmap[i]);
				gradient->pixmap[i] = NULL;
			}
		}
	  #else
		for (app = gradient->apps; app != NULL; app = next_app) {
			next_app = app->next;
			gtk_object_remove_data_by_id (GTK_OBJECT(app->widget), xeno_gradient_quark);
		}
	  #endif
	}
}

/*** Decoration ***************************************************************/
/*
static void
xeno_draw_knob	(GtkStyle		*style,
				 GdkWindow		*window,
				 GtkStateType	state_type,
				 GtkShadowType	shadow_type,
				 GdkRectangle	*area,
				 GtkWidget		*widget,
				 gint			x,
				 gint			y,
				 gint			width,
				 gint			height,
				 GtkOrientation	orientation,
				 gboolean		center)
{
}
*/


static void
xeno_draw_lines	(GtkStyle		*style,
				 GdkWindow		*window,
				 GtkStateType	state_type,
				 GtkShadowType	shadow_type,
				 GdkRectangle	*area,
				 GtkWidget		*widget,
				 gint			x,
				 gint			y,
				 gint			width,
				 gint			height,
				 GtkOrientation	orientation)
{
	GdkGC	*light_gc, *dark_gc, *mid_gc;
	gint	i;

	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);

	if (shadow_type == GTK_SHADOW_IN) {
		dark_gc  = style->dark_gc[state_type];
		light_gc = style->light_gc[state_type];
	} else {
		light_gc = style->dark_gc[state_type];
		dark_gc	 = style->light_gc[state_type];
	}
	mid_gc = style->mid_gc[state_type];
	
	if (area) {
		gdk_gc_set_clip_rectangle (dark_gc, area);
		gdk_gc_set_clip_rectangle (light_gc, area);
		gdk_gc_set_clip_rectangle (mid_gc, area);
	}
	
	if (orientation == GTK_ORIENTATION_HORIZONTAL) {
		for (i = y + ((height % 3) & 0x01); i < y + height; i += 3) {
			gdk_draw_line  (window, dark_gc,	x,			i,		x+width-2,	i  );
			gdk_draw_line  (window, light_gc,	x+1, 		i+1,	x+width-1,	i+1);
			gdk_draw_point (window, mid_gc,		x,			i+1);
			gdk_draw_point (window, mid_gc,		x+width-1,	i);
		}
	} else {
		for (i = x + ((width % 3) & 0x01); i < x + width; i += 3) {
			gdk_draw_line  (window, dark_gc,	i,		y,   	i,   y+height-2);
			gdk_draw_line  (window, light_gc,	i+1,	y+1, 	i+1, y+height-1);
			gdk_draw_point (window, mid_gc,		i+1,	y);
			gdk_draw_point (window, mid_gc,		i,		y+height-1);
		}
	}
	
	if (area) {
		gdk_gc_set_clip_rectangle (mid_gc, NULL);
		gdk_gc_set_clip_rectangle (light_gc, NULL);
		gdk_gc_set_clip_rectangle (dark_gc, NULL);
	}
}

static void
xeno_draw_buds (GtkStyle		*style,
				GdkWindow		*window,
				GtkStateType	state_type,
				GtkShadowType	shadow_type,
				GdkRectangle	*area,
				GtkWidget		*widget,
				gint			x,
				gint			y,
				gint			width,
				gint			height,
				GtkOrientation	orientation)
{
	GdkGC	*light_gc, *dark_gc, *mid_gc;
	gint	x2, y2;
	
	g_return_if_fail (style != NULL);
	g_return_if_fail (window != NULL);
	
	if (shadow_type == GTK_SHADOW_OUT || shadow_type == GTK_SHADOW_ETCHED_OUT) {
		light_gc = style->white_gc;
		mid_gc	 = style->mid_gc[state_type];
		dark_gc  = style->dark_gc[state_type];
	} else {
		light_gc = style->dark_gc[state_type];;
		mid_gc = style->mid_gc[state_type];
		dark_gc	 = style->light_gc[state_type];
	}
	
	if (area) {
		gdk_gc_set_clip_rectangle (dark_gc, area);
		gdk_gc_set_clip_rectangle (light_gc, area);
		gdk_gc_set_clip_rectangle (mid_gc, area);
	}

	if (   (shadow_type == GTK_SHADOW_OUT || shadow_type == GTK_SHADOW_IN)
		&& orientation == GTK_ORIENTATION_VERTICAL)
	{
		/* netscape - style */
		for (y2 = y+1; y2 < y+height-1; y2 += 3) {
			for (x2 = x; x2 < x+width-1; x2 += 6) {
				gdk_draw_point (window, light_gc,	x2,		y2);
				gdk_draw_point (window, dark_gc,	x2 + 1, y2 + 1);
				gdk_draw_point (window, mid_gc,		x2 + 1,	y2);
				gdk_draw_point (window, mid_gc,		x2,		y2 + 1);
			}
		}
		for (y2 = y; y2 < y+height-1; y2 += 3) {
			for (x2 = x+3; x2 < x+width-1; x2 += 6) {
				gdk_draw_point (window, light_gc,	x2,		y2    );
				gdk_draw_point (window, dark_gc,	x2 + 1, y2 + 1);
				gdk_draw_point (window, mid_gc,		x2 + 1,	y2    );
				gdk_draw_point (window, mid_gc,		x2,		y2 + 1);
			}
		}
	} else {
		/* mac - style */
		x += (width % 3) & 0x01;
		y += (height% 3) & 0x01;
		
		for (y2 = y; y2 < y+height-1; y2 += 3) {
			for (x2 = x; x2 < x+width-1; x2 += 3) {
				gdk_draw_point (window, light_gc,	x2,		y2	);
				gdk_draw_point (window, mid_gc,		x2+1,	y2	);
				gdk_draw_point (window, mid_gc,		x2,		y2+1);
				gdk_draw_point (window, dark_gc,	x2+1,	y2+1);
			}
		}
	}
	
	if (area) {
		gdk_gc_set_clip_rectangle (mid_gc, NULL);
		gdk_gc_set_clip_rectangle (light_gc, NULL);
		gdk_gc_set_clip_rectangle (dark_gc, NULL);
	}
}


/*** Images *******************************************************************/
static void
xeno_draw_pixmap (GdkWindow *window, GdkGC *gc, GdkRectangle *area, GdkPixmap *pixmap, GdkBitmap *mask,
				  guint16 srcx, guint16 srcy,
				  guint16 x, guint16 y, guint16 width, guint16 height)
{
	gint			xd, yd, wd, hd;

	if (area) {
		xd = MAX(area->x, x) - x; if (xd < 0 || xd >= width) return;
		yd = MAX(area->y, y) - y; if (yd < 0 || yd >= height) return;

		wd = width  - (MIN(area->x+area->width,  x+width)  -x -xd); if (wd < 0) return;
		hd = height - (MIN(area->y+area->height, y+height) -y -yd); if (hd < 0) return;
	} else {
		xd = yd = wd = hd = 0;
	}

	gdk_gc_set_clip_mask (gc, mask);
	gdk_gc_set_clip_origin (gc, x-srcx, y-srcy);
	gdk_draw_pixmap (window, gc, pixmap, srcx+xd, srcy+yd, x+xd, y+yd, width-wd, height-hd);
	gdk_gc_set_clip_mask (gc, NULL);
}


static void
xeno_draw_image (GdkWindow *window, GtkWidget *widget, GdkRectangle *area, XenoImageType image_type,
				 guint16 srcx, guint16 srcy,
				 guint16 x, guint16 y, guint16 width, guint16 height)
{
	GdkPixmap	*pixmap;
	GtkStyle	*style, *style2;
	
	style = widget->style;
	style2 = (widget->parent) ? widget->parent->style : style;
	if (!window)
		window = widget->window;
	
	pixmap = xeno_pixmap_get(window, style, style2, image_type);
	if (pixmap)
		xeno_draw_pixmap (window, style->bg_gc[widget->state], area,
						  pixmap, xeno_image_mask(image_type), srcx, srcy, x, y, width, height);
}


