/*
 *	Xenophilia GTK+ Theme Engine
 *
 *  xeno_theme_main.c:
 *		Theme engine
 *
 *	Copyright  1999-2001 Johan Hanson <johan@tiq.com>.
 *	
 *	This library is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU Library General Public
 *	License as published by the Free Software Foundation; either
 *	version 2 of the License, or (at your option) any later version.
 *	
 *	This library is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *	Library General Public License for more details.
 *	
 *	You should have received a copy of the GNU Library General Public
 *	License along with this library; if not, write to the 
 *	Free Software Foundation, Inc., 59 Temple Place - Suite 330, 
 *	Boston, MA  02111-1307  USA.
 */

#include <stdlib.h>

#include "xeno_theme_main.h"
#include "xeno_theme_rc.h"
#include "xeno_theme_style.h"
#include "xeno_theme_patches.h"

#if !XENO_GTK2
#include "xeno_theme_style.h"
#endif

#include <gmodule.h>
#include <gtk/gtkrc.h>


/* Theme functions to export */
#if XENO_GTK2
void		theme_init				(GTypeModule *module);
GtkRcStyle *theme_create_rc_style	(void);
#else
void		theme_init				(GtkThemeEngine *engine);
#endif

void		theme_exit				(void);

/* GtkThemeEngine */
#if !XENO_GTK2
static guint	xeno_parse_rc_style		(GScanner *scanner, GtkRcStyle *rc_style);
static void		xeno_merge_rc_style		(GtkRcStyle *dest, GtkRcStyle *src);
static void		xeno_rc_style_to_style	(GtkStyle *style, GtkRcStyle *rc_style);
static void		xeno_destroy_rc_style	(GtkRcStyle *rc_style);

static void		xeno_duplicate_style	(GtkStyle *dest, GtkStyle	*src);
static void		xeno_destroy_style		(GtkStyle *style);

#define xeno_realize_style				xeno_style_realize
#define xeno_unrealize_style			xeno_style_unrealize

#endif

/* Exported variables */
gboolean		xeno_pseudocolor	= FALSE;
GdkVisual *		xeno_visual			= NULL;
GdkColormap *	xeno_colormap		= NULL;


/*
 *	GtkThemeEngine
 */

#if XENO_GTK2

GtkRcStyle * theme_create_rc_style (void) {
	return (GtkRcStyle *)xeno_rc_style_new();
}

#else
static guint
xeno_parse_rc_style (GScanner *scanner, GtkRcStyle *rc_style)
{
	XenoRcData	*new_data, *old_data;
	
	old_data = rc_style->engine_data;
	if (old_data) {
		new_data = xeno_rc_data_clone (old_data);
		xeno_rc_data_unref (old_data);
	} else {
		new_data = xeno_rc_data_new ();
	}
	
	rc_style->engine_data = new_data;
	return xeno_rc_data_parse (new_data, scanner);
}

static void
xeno_merge_rc_style (GtkRcStyle *dest, GtkRcStyle *src)
{
	if (src->engine_data) {
		if (dest->engine_data) {
			xeno_rc_data_merge (dest->engine_data, src->engine_data);
		} else {
			dest->engine_data = xeno_rc_data_clone (src->engine_data);
		}
	}
}

static void
xeno_rc_style_to_style (GtkStyle *style, GtkRcStyle *rc_style) {
	gint thickness;
	
	g_return_if_fail (style->engine_data == NULL);
	
	thickness = 2;
	if (rc_style->engine_data) {
		style->engine_data = xeno_style_data_new ();
		thickness = ((XenoRcData *)rc_style->engine_data)->thickness;
		style->klass = (GtkStyleClass *)&xeno_style_classes[CLAMP(thickness, 0, 3)];
	}
	/* style->engine_data = NULL; */
}

static void
xeno_destroy_rc_style (GtkRcStyle *rc_style) {
	if (rc_style->engine_data) 
		xeno_rc_data_unref (rc_style->engine_data);
	
	rc_style->engine_data = NULL;
}

static void
xeno_duplicate_style (GtkStyle *dest, GtkStyle *src) {
	XenoStyleData *style_data;
	
	g_return_if_fail (XENO_STYLE_IS_XENO(src));
	g_return_if_fail (XENO_STYLE_IS_XENO(dest));
	
	style_data = NULL;
	if (src->engine_data)
		style_data = xeno_style_data_clone (src->engine_data);
	
	dest->engine_data = style_data;
}

static void
xeno_destroy_style (GtkStyle *style) {
	g_return_if_fail (XENO_STYLE_IS_XENO(style));
	
	if (style->engine_data)
		xeno_style_data_unref (style->engine_data);
	
	style->engine_data = NULL;
}
#endif

#if XENO_GTK2
void theme_init (GTypeModule *module) {
#else
void theme_init (GtkThemeEngine *engine) {
#endif
	const char	*env_str;
	GdkVisual	*visual;
	
  #if !XENO_GTK2
	/* Initialize theme engine structure */
	engine->parse_rc_style    = xeno_parse_rc_style;
	engine->merge_rc_style    = xeno_merge_rc_style;
	engine->rc_style_to_style = xeno_rc_style_to_style;
	engine->duplicate_style   = xeno_duplicate_style;
	engine->realize_style     = xeno_realize_style;
	engine->unrealize_style   = xeno_unrealize_style;
	engine->destroy_rc_style  = xeno_destroy_rc_style;
	engine->destroy_style     = xeno_destroy_style;
	engine->set_background    = NULL;
  #endif
	
	/* Test display and see if we need to reduce color usage */
	env_str = getenv ("XENO_THEME_CMAP");
	visual = gtk_widget_get_default_visual ();
	
	xeno_pseudocolor = FALSE;
	if (env_str && (strcasecmp(env_str, "YES") == 0 || strcasecmp(env_str, "TRUE") == 0)) {
		xeno_pseudocolor = TRUE;
	} else if (env_str == NULL || strcasecmp(env_str, "TEST") == 0) {
		if (visual != NULL) {
			if (visual->type == GDK_VISUAL_PSEUDO_COLOR || visual->type == GDK_VISUAL_DIRECT_COLOR)
				xeno_pseudocolor = TRUE;
		}
	}
	
	xeno_visual = visual;
	xeno_colormap = gtk_widget_get_default_colormap();
	
	/* Install patches */
	xeno_patches_install();
}

void theme_exit (void)
{
  #if !XENO_GTK2
	if (xeno_style_data_chunk)
		g_mem_chunk_destroy (xeno_style_data_chunk);
	
	if (xeno_rc_data_chunk)
		g_mem_chunk_destroy (xeno_rc_data_chunk);
  #endif
	
	xeno_patches_remove();
}


/*
 * The following function will be called by GTK+ when the module
 * is loaded and checks to see if we are compatible with the
 * version of GTK+ that loads us.
 */
G_MODULE_EXPORT const gchar* g_module_check_init (GModule *module);

const gchar* g_module_check_init (GModule *module) {
	return gtk_check_version (GTK_MAJOR_VERSION,
							  GTK_MINOR_VERSION,
							  GTK_MICRO_VERSION - GTK_INTERFACE_AGE);
}


/* end */

