/*
 *  xfmedia - simple gtk2 media player based on xine
 *
 *  Copyright (c) 2004-2005 Brian Tarricone, <bjt23@cornell.edu>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License ONLY.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <xfmedia/xfmedia-interface.h>

#include <libxfce4util/libxfce4util.h>

#include "xfmedia-common.h"
#include "mainwin.h"
#include <xfmedia/xfmedia-playlist.h>
#include "xfmedia-xine.h"

gint
xfmedia_interface_get_session_number(XfmediaPlugin *plugin)
{
    XfmediaMainwin *mwin;
    
    g_return_val_if_fail(plugin, -1);
    
    mwin = g_object_get_data(G_OBJECT(plugin), "__xfmedia-mainwin");
    
    return mwin->session_id;
}

gboolean
xfmedia_engine_open(XfmediaPlugin *plugin, const gchar *location)
{
    XfmediaMainwin *mwin;
    
    g_return_val_if_fail(plugin && location, FALSE);
    
    mwin = g_object_get_data(G_OBJECT(plugin), "__xfmedia-mainwin");
    
    return xfmedia_xine_open(mwin->xfx, location);
}

gboolean
xfmedia_engine_play(XfmediaPlugin *plugin)
{
    XfmediaMainwin *mwin;
    
    g_return_val_if_fail(plugin, FALSE);
    
    mwin = g_object_get_data(G_OBJECT(plugin), "__xfmedia-mainwin");
    
    if(xfmedia_xine_get_status(mwin->xfx) == XINE_STATUS_PLAY
            && xfmedia_xine_get_param(mwin->xfx, XINE_PARAM_SPEED) != XINE_SPEED_NORMAL)
    {
        if(xfmedia_xine_get_param(mwin->xfx, XINE_PARAM_SPEED) == XINE_SPEED_PAUSE)
            xfmedia_xine_set_param(mwin->xfx, XINE_PARAM_AUDIO_CLOSE_DEVICE, 0);
        xfmedia_xine_set_param(mwin->xfx, XINE_PARAM_SPEED, XINE_SPEED_NORMAL);
        return TRUE;
    } else
        return xfmedia_xine_play(mwin->xfx, 0, 0);
}

void
xfmedia_engine_pause(XfmediaPlugin *plugin)
{
    XfmediaMainwin *mwin;
    
    g_return_if_fail(plugin);
    
    mwin = g_object_get_data(G_OBJECT(plugin), "__xfmedia-mainwin");
    
    if(xfmedia_xine_get_status(mwin->xfx) == XINE_STATUS_PLAY) {
        xfmedia_xine_set_param(mwin->xfx, XINE_PARAM_AUDIO_CLOSE_DEVICE, 1);
        xfmedia_xine_set_param(mwin->xfx, XINE_PARAM_SPEED, XINE_SPEED_PAUSE);
    }
}

void
xfmedia_engine_stop(XfmediaPlugin *plugin)
{
    XfmediaMainwin *mwin;
    
    g_return_if_fail(plugin);
    
    mwin = g_object_get_data(G_OBJECT(plugin), "__xfmedia-mainwin");
    
    xfmedia_xine_stop(mwin->xfx);
}

gboolean
xfmedia_engine_seek(XfmediaPlugin *plugin, gint time)
{
    XfmediaMainwin *mwin;
    
    g_return_val_if_fail(plugin && time >= 0, FALSE);
    
    mwin = g_object_get_data(G_OBJECT(plugin), "__xfmedia-mainwin");
    
    xfmedia_xine_set_param(mwin->xfx, XINE_PARAM_AUDIO_CLOSE_DEVICE, 0);
    return xfmedia_xine_play(mwin->xfx, 0, time);
}

gboolean
xfmedia_engine_seek_stream_pos(XfmediaPlugin *plugin, gint position)
{
    XfmediaMainwin *mwin;
    
    g_return_val_if_fail(plugin && position >= 0, FALSE);
    
    mwin = g_object_get_data(G_OBJECT(plugin), "__xfmedia-mainwin");
    
    xfmedia_xine_set_param(mwin->xfx, XINE_PARAM_AUDIO_CLOSE_DEVICE, 0);
    return xfmedia_xine_play(mwin->xfx, position, 0);
}

gint
xfmedia_engine_get_current_time(XfmediaPlugin *plugin)
{
    XfmediaMainwin *mwin;
    gint pos_stream, pos_time, length_time;
    
    g_return_val_if_fail(plugin, 0);
    
    mwin = g_object_get_data(G_OBJECT(plugin), "__xfmedia-mainwin");
    
    if(xfmedia_xine_get_pos_length(mwin->xfx, &pos_stream, &pos_time, &length_time))
        return pos_time;
    else
        return 0;
}

gint
xfmedia_engine_get_current_stream_pos(XfmediaPlugin *plugin)
{
    XfmediaMainwin *mwin;
    gint pos_stream, pos_time, length_time;
    
    g_return_val_if_fail(plugin, 0);
    
    mwin = g_object_get_data(G_OBJECT(plugin), "__xfmedia-mainwin");
    
    if(xfmedia_xine_get_pos_length(mwin->xfx, &pos_stream, &pos_time, &length_time))
        return pos_stream;
    else
        return 0;
}

gint
xfmedia_engine_get_total_time(XfmediaPlugin *plugin)
{
    XfmediaMainwin *mwin;
    gint pos_stream, pos_time, length_time;
    
    g_return_val_if_fail(plugin, 0);
    
    mwin = g_object_get_data(G_OBJECT(plugin), "__xfmedia-mainwin");
    
    if(xfmedia_xine_get_pos_length(mwin->xfx, &pos_stream, &pos_time, &length_time))
        return length_time;
    else
        return 0;
}

XfmediaStatus
xfmedia_engine_get_status(XfmediaPlugin *plugin)
{
    XfmediaMainwin *mwin;
    
    g_return_val_if_fail(plugin, XFMEDIA_STATUS_UNAVAILABLE);
    
    mwin = g_object_get_data(G_OBJECT(plugin), "__xfmedia-mainwin");
    
    return (XfmediaStatus)xfmedia_xine_get_status(mwin->xfx);
}

XfmediaSpeed
xfmedia_engine_get_speed(XfmediaPlugin *plugin)
{
    XfmediaMainwin *mwin;
    
    g_return_val_if_fail(plugin, XFMEDIA_SPEED_NORMAL);
    
    mwin = g_object_get_data(G_OBJECT(plugin), "__xfmedia-mainwin");
    
    return (XfmediaSpeed)xfmedia_xine_get_param(mwin->xfx, XINE_PARAM_SPEED);
}

void
xfmedia_engine_set_speed(XfmediaPlugin *plugin, XfmediaSpeed speed)
{
    XfmediaMainwin *mwin;
    
    g_return_if_fail(plugin && speed >= XFMEDIA_SPEED_PAUSED
            && speed <= XFMEDIA_SPEED_FASTER);
    
    mwin = g_object_get_data(G_OBJECT(plugin), "__xfmedia-mainwin");
    
    if(xfmedia_xine_get_param(mwin->xfx, XINE_PARAM_SPEED) == XINE_SPEED_PAUSE
             && speed != XFMEDIA_SPEED_PAUSED)
    {
        xfmedia_xine_set_param(mwin->xfx, XINE_PARAM_AUDIO_CLOSE_DEVICE, 0);
    } else if(speed == XFMEDIA_SPEED_PAUSED)
        xfmedia_xine_set_param(mwin->xfx, XINE_PARAM_AUDIO_CLOSE_DEVICE, 1);
    
    xfmedia_xine_set_param(mwin->xfx, XINE_PARAM_SPEED, (gint)speed);
}

gboolean
xfmedia_get_fullscreen(XfmediaPlugin *plugin)
{
    XfmediaMainwin *mwin;
    
    g_return_val_if_fail(plugin, FALSE);
    
    mwin = g_object_get_data(G_OBJECT(plugin), "__xfmedia-mainwin");
    
    return xfmedia_xine_is_fullscreen(mwin->xfx);
}

void
xfmedia_set_fullscreen(XfmediaPlugin *plugin, gboolean setting)
{
    XfmediaMainwin *mwin;
    
    g_return_if_fail(plugin);
    
    mwin = g_object_get_data(G_OBJECT(plugin), "__xfmedia-mainwin");
    
    xfmedia_xine_set_fullscreen(mwin->xfx, setting);
}

gboolean
xfmedia_get_video_window_visible(XfmediaPlugin *plugin)
{
    XfmediaMainwin *mwin;
    
    g_return_val_if_fail(plugin, FALSE);
    
    mwin = g_object_get_data(G_OBJECT(plugin), "__xfmedia-mainwin");
    
    return (xfmedia_xine_is_fullscreen(mwin->xfx)
            || GTK_WIDGET_VISIBLE(mwin->video_window));
}

void
xfmedia_set_video_window_visible(XfmediaPlugin *plugin, gboolean setting)
{
    XfmediaMainwin *mwin;
    
    g_return_if_fail(plugin);
    
    mwin = g_object_get_data(G_OBJECT(plugin), "__xfmedia-mainwin");
    
    if(!setting && xfmedia_xine_is_fullscreen(mwin->xfx))
        xfmedia_xine_set_fullscreen(mwin->xfx, FALSE);
    else if(!xfmedia_xine_is_fullscreen(mwin->xfx)) {
        if(setting)
            gtk_widget_show(mwin->video_window);
        else
            gtk_widget_hide(mwin->video_window);
    }
}

G_CONST_RETURN gchar *
xfmedia_info_get_location(XfmediaPlugin *plugin)
{
    XfmediaMainwin *mwin;
    
    g_return_val_if_fail(plugin, NULL);
    
    mwin = g_object_get_data(G_OBJECT(plugin), "__xfmedia-mainwin");
    
    return xfmedia_xine_get_current_mrl(mwin->xfx);
}


gchar *
xfmedia_info_get_name(XfmediaPlugin *plugin)
{
    XfmediaMainwin *mwin;
    gint idx;
    gchar *name = NULL;
    
    g_return_val_if_fail(plugin, NULL);
    
    mwin = g_object_get_data(G_OBJECT(plugin), "__xfmedia-mainwin");
    
    if(!mwin->cur_playing)
        return NULL;
    
    idx = xfmedia_playlist_entry_ref_get_index(mwin->cur_playing);
    if(idx != -1)
        xfmedia_playlist_get(mwin->plist, idx, &name, NULL, NULL);
    
    return name;
}

gboolean
xfmedia_info_get_has_video(XfmediaPlugin *plugin)
{
    XfmediaMainwin *mwin;
    
    g_return_val_if_fail(plugin, FALSE);
    
    mwin = g_object_get_data(G_OBJECT(plugin), "__xfmedia-mainwin");
    
    return xfmedia_xine_get_stream_info(mwin->xfx, XINE_STREAM_INFO_HAS_VIDEO);
}

gboolean
xfmedia_info_get_has_audio(XfmediaPlugin *plugin)
{
    XfmediaMainwin *mwin;
    
    g_return_val_if_fail(plugin, FALSE);
    
    mwin = g_object_get_data(G_OBJECT(plugin), "__xfmedia-mainwin");
    
    return xfmedia_xine_get_stream_info(mwin->xfx, XINE_STREAM_INFO_HAS_AUDIO);
}

gboolean
xfmedia_info_get_audio_info(XfmediaPlugin *plugin, gchar **codec, gint *bitrate,
        gint *sample_rate, gint *channels, gint *bits_per_sample)
{
    XfmediaMainwin *mwin;
    gint status;
    
    g_return_val_if_fail(plugin, FALSE);
    
    mwin = g_object_get_data(G_OBJECT(plugin), "__xfmedia-mainwin");
    
    status = xfmedia_xine_get_status(mwin->xfx);
    if(status == XINE_STATUS_IDLE || status == XINE_STATUS_QUIT
            || !xfmedia_xine_get_stream_info(mwin->xfx, XINE_STREAM_INFO_HAS_AUDIO))
    {
        return FALSE;
    }
    
    if(codec)
        *codec = g_strdup(xfmedia_xine_get_meta_info(mwin->xfx, XINE_META_INFO_AUDIOCODEC));
    if(bitrate)
        *bitrate = xfmedia_xine_get_stream_info(mwin->xfx, XINE_STREAM_INFO_AUDIO_BITRATE);
    if(sample_rate)
        *sample_rate = xfmedia_xine_get_stream_info(mwin->xfx, XINE_STREAM_INFO_AUDIO_SAMPLERATE);
    if(channels)
        *channels = xfmedia_xine_get_stream_info(mwin->xfx, XINE_STREAM_INFO_AUDIO_CHANNELS);
    if(bits_per_sample)
        *bits_per_sample = xfmedia_xine_get_stream_info(mwin->xfx, XINE_STREAM_INFO_AUDIO_BITS);
    
    return TRUE;
}

gboolean
xfmedia_info_get_video_info(XfmediaPlugin *plugin, gchar **codec, gint *bitrate,
        gdouble *fps, gint *width, gint *height, gdouble *aspect)
{
    XfmediaMainwin *mwin;
    gint status;
    
    g_return_val_if_fail(plugin, FALSE);
    
    mwin = g_object_get_data(G_OBJECT(plugin), "__xfmedia-mainwin");
    
    status = xfmedia_xine_get_status(mwin->xfx);
    if(status == XINE_STATUS_IDLE || status == XINE_STATUS_QUIT
            || !xfmedia_xine_get_stream_info(mwin->xfx, XINE_STREAM_INFO_HAS_VIDEO))
    {
        return FALSE;
    }
    
    if(codec)
        *codec = g_strdup(xfmedia_xine_get_meta_info(mwin->xfx, XINE_META_INFO_VIDEOCODEC));
    if(bitrate)
        *bitrate = xfmedia_xine_get_stream_info(mwin->xfx, XINE_STREAM_INFO_BITRATE);
    if(fps) {
        gint frame_duration = xfmedia_xine_get_stream_info(mwin->xfx, XINE_STREAM_INFO_FRAME_DURATION);
        if(frame_duration == 0)
            *fps = 0;
        else
            *fps = 90000.0 / frame_duration;
    }
    if(width)
        *width = xfmedia_xine_get_stream_info(mwin->xfx, XINE_STREAM_INFO_VIDEO_WIDTH);
    if(height)
        *height = xfmedia_xine_get_stream_info(mwin->xfx, XINE_STREAM_INFO_VIDEO_HEIGHT);
    if(aspect)
        *aspect = xfmedia_xine_get_stream_info(mwin->xfx, XINE_STREAM_INFO_VIDEO_RATIO) / 10000.0;
    
    return TRUE;
}
