<?xml version="1.0" encoding="UTF-8"?>
<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
                xmlns:xgf="http://www.engl.virginia.edu/OE/xgridfit-data"
                version="1.0">

  <!-- This file is part of xgridfit.
       It is distributed under the GNU Public License, version 2.
       Copyright (c) 2006-7 by Peter S. Baker
  -->
  
  <!--

      push-num

      Pushes a single number, using the appropriate command.  If
      add-mode=true the push instruction is not output, but only a
      separator and the number; this adds the command onto a
      previously output push instruction. If add-mode=false then a
      PUSH command is output according to the size param, or if none,
      PUSHB for numbers between 0 and 255, PUSHW otherwise.
  -->
  <xsl:template name="push-num">
    <xsl:param name="num"/>
    <xsl:param name="expect" select="1"/>
    <xsl:param name="add-mode" select="false()"/>
    <xsl:param name="size"/>
    <xsl:param name="with-leading-newline" select="true()"/>
    <xsl:variable name="size-s">
      <xsl:choose>
        <xsl:when test="$size">
          <xsl:value-of select="$size"/>
        </xsl:when>
        <xsl:when test="number($num) &gt;= 0 and number($num) &lt; 256">
          <xsl:text>B</xsl:text>
        </xsl:when>
        <xsl:otherwise>
          <xsl:text>W</xsl:text>
        </xsl:otherwise>
      </xsl:choose>
    </xsl:variable>
    <xsl:choose>
      <xsl:when test="$add-mode">
        <xsl:value-of select="$push-num-separator"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:call-template name="push-command">
          <xsl:with-param name="size" select="$size-s"/>
          <xsl:with-param name="count" select="$expect"/>
          <xsl:with-param name="with-leading-newline"
			  select="$with-leading-newline"/>
        </xsl:call-template>
        <xsl:value-of select="$inst-newline"/>
      </xsl:otherwise>
    </xsl:choose>
    <xsl:value-of select="$num"/>
  </xsl:template>
  
  <!--

      number-command

      For commands that take a single number argument.
  -->
  <xsl:template name="number-command">
    <xsl:param name="num"/>
    <xsl:param name="cmd"/>
    <xsl:param name="with-leading-newline" select="true()"/>
    <xsl:call-template name="push-num">
      <xsl:with-param name="num" select="$num"/>
      <xsl:with-param name="with-leading-newline"
		      select="$with-leading-newline"/>
    </xsl:call-template>
    <xsl:call-template name="simple-command">
      <xsl:with-param name="cmd" select="$cmd"/>
    </xsl:call-template>
  </xsl:template>
  
  
  <!--

      get-number-literal

      Any number or identifier that can be resolved to a number
      literal at compile-time can be fetched with this template.

      If $val cannot be resolved to a number literal, this returns
      'NaN' (a string). This is detected by <push-value> (the main
      client for this template), which then does further testing of
      the identifier.
  -->
  <xsl:template name="get-number-literal">
    <xsl:param name="val"/>
    <xsl:param name="offset" select="0"/>
    <xsl:param name="cvt-mode" select="'index'"/>
    <xsl:param name="force-to-index" select="false()"/>
    <xsl:param name="ancestor-glyph" select="ancestor::glyph"/>
    <xsl:param name="permitted" select="'12xrnc'"/>
    <xsl:param name="call-macro-param-set"/>
    <xsl:param name="all-macro-params"
	       select="ancestor::macro/param|ancestor::macro/params/param"/>
    <xsl:param name="all-constants"
	       select="$ancestor-glyph/constant|$ancestor-glyph/declarations/constant"/>
    <xsl:param name="all-global-constants"
	       select="/xgridfit/constant|/xgridfit/profile/constant"/>
    <!-- Get a normalized version of the $val parameter. -->
    <xsl:variable name="valn">
      <xsl:call-template name="normalize-expression">
        <xsl:with-param name="s" select="$val"/>
      </xsl:call-template>
    </xsl:variable>
    <xsl:choose>
      <!--
	  If we're in a macro and this is a macro parameter, recurse
	  substituting the value. I don't think this will work for a
	  call-macro inside a macro. This would be a very desirable
	  thing, so investigate further.
      -->
      <xsl:when test="$all-macro-params[@name=$valn]">
	<!--
	    We find out the value to substitute for the
	    parameter. Normally it will be the @value of the
	    with-param element being evaluated; but it can be the
	    @value of the param element - i.e. a default value.
	-->
	<xsl:variable name="sub-value">
	  <xsl:choose>
	    <xsl:when test="$call-macro-param-set[@name=$valn or @param-id=$valn]/@value">
	      <xsl:value-of select="$call-macro-param-set[@name=$valn or @param-id=$valn]/@value"/>
	    </xsl:when>
	    <xsl:when test="$all-macro-params[@name=$valn]/@value">
	      <xsl:value-of select="$all-macro-params[@name=$valn]/@value"/>
	    </xsl:when>
	    <xsl:otherwise>
	      <xsl:call-template name="error-message">
		<xsl:with-param name="msg">
		  <xsl:text>Can't find value for param </xsl:text>
		  <xsl:value-of select="$valn"/>
		  <xsl:text> in macro </xsl:text>
		  <xsl:choose>
		    <xsl:when test="ancestor::macro/@name">
		      <xsl:value-of select="ancestor::macro/@name"/>
		    </xsl:when>
		    <xsl:otherwise>
		      <xsl:value-of select="ancestor::macro/@id"/>
		    </xsl:otherwise>
		  </xsl:choose>
		</xsl:with-param>
	      </xsl:call-template>
	    </xsl:otherwise>
	  </xsl:choose>
	</xsl:variable>
	<!--
	    Now call get-number-literal recursively. We'll very likely
	    need a reference to a glyph program so we can pull values
	    from the declaration. If we already have an
	    $ancestor-glyph, fine; if not, try to find one by looking
	    along the ancestor axis of our with-param element.
	-->
	<xsl:choose>
	  <xsl:when test="$ancestor-glyph">
	    <xsl:call-template name="get-number-literal">
	      <xsl:with-param name="val" select="$sub-value"/>
	      <xsl:with-param name="offset" select="$offset"/>
	      <xsl:with-param name="cvt-mode" select="$cvt-mode"/>
	      <xsl:with-param name="force-to-index" select="$force-to-index"/>
	      <xsl:with-param name="ancestor-glyph" select="$ancestor-glyph"/>
	      <xsl:with-param name="permitted" select="$permitted"/>
	    </xsl:call-template>
	  </xsl:when>
	  <xsl:otherwise>
	    <xsl:call-template name="get-number-literal">
	      <xsl:with-param name="val" select="$sub-value"/>
	      <xsl:with-param name="offset" select="$offset"/>
	      <xsl:with-param name="cvt-mode" select="$cvt-mode"/>
	      <xsl:with-param name="force-to-index" select="$force-to-index"/>
	      <xsl:with-param name="ancestor-glyph"
			      select="$call-macro-param-set/ancestor::glyph"/>
	      <xsl:with-param name="permitted" select="$permitted"/>
	      <xsl:with-param name="call-macro-param-set"
			      select="$call-macro-param-set"/>
	    </xsl:call-template>	    
	  </xsl:otherwise>
	</xsl:choose>
      </xsl:when>
      <xsl:otherwise>
	<xsl:variable name="off">
	  <xsl:choose>
	    <xsl:when test="$offset and string-length($offset) &gt; 0">
	      <xsl:choose>
		<xsl:when test="number($offset) or number($offset) = 0">
		  <xsl:value-of select="$offset"/>
		</xsl:when>
		<xsl:otherwise>
		  <xsl:call-template name="get-number-literal">
		    <xsl:with-param name="val" select="$offset"/>
		    <xsl:with-param name="ancestor-glyph" select="$ancestor-glyph"/>
		    <xsl:with-param name="permitted" select="'1n'"/>
		    <xsl:with-param name="call-macro-param-set"
				    select="$call-macro-param-set"/>
		  </xsl:call-template>
		</xsl:otherwise>
	      </xsl:choose>
	    </xsl:when>
	    <xsl:otherwise>
	      <xsl:text>0</xsl:text>
	    </xsl:otherwise>
	  </xsl:choose>
	</xsl:variable>
	<xsl:variable name="all-but-last-char">
	  <xsl:choose>
	    <xsl:when test="string-length($valn) &gt; 1">
	      <xsl:value-of select="substring($valn,1,string-length($valn)-1)"/>
	    </xsl:when>
	    <xsl:otherwise>
	      <xsl:value-of select="$valn"/>
	    </xsl:otherwise>
	  </xsl:choose>
	</xsl:variable>
	<xsl:variable name="last-char">
	  <xsl:value-of select="substring($valn,string-length($valn))"/>
	</xsl:variable>
	<!-- This variable optimizes a bit for programs in which constants declared
	     in the glyph element are very common. -->
	<xsl:variable name="v-name">
	  <xsl:choose>
	    <xsl:when test="$all-constants[@name=$valn]/@value">
	      <xsl:value-of select="$all-constants[@name=$valn]/@value"/>
	    </xsl:when>
	    <xsl:when test="$all-constants[@name=$valn]/@num">
	      <xsl:value-of select="$all-constants[@name=$valn]/@num"/>
	    </xsl:when>
	  </xsl:choose>
	</xsl:variable>
	<xsl:choose>
	  <xsl:when test="string-length($v-name) &gt; 0 and contains($permitted,'n')">
	    <!-- It's a locally declared constant. If it's an integer, just return it.
		 Otherwise, call this recursively to find out what it is. -->
	    <xsl:choose>
	      <xsl:when test="(number($v-name) or number($v-name) = 0) and
			      not(contains($v-name,'.'))">
		<xsl:value-of select="number($v-name) + number($off)"/>
	      </xsl:when>
	      <xsl:otherwise>
		<xsl:call-template name="get-number-literal">
		  <xsl:with-param name="val" select="$v-name"/>
		  <xsl:with-param name="offset" select="$off"/>
		  <xsl:with-param name="cvt-mode" select="$cvt-mode"/>
		  <xsl:with-param name="ancestor-glyph" select="$ancestor-glyph"/>
		  <xsl:with-param name="permitted" select="$permitted"/>
		  <xsl:with-param name="call-macro-param-set"
				  select="$call-macro-param-set"/>
		</xsl:call-template>
	      </xsl:otherwise>
	    </xsl:choose>
	  </xsl:when>
	  <xsl:when test="contains($permitted,'c') and
			  ($cvt-mode = 'index' or $force-to-index) and
			  key('cvt',$valn)">
	    <!-- It's the index of a control value.  $force-to-index is
		 only true when we have had an expression in the form
		 index(val). This overrides any default interpretation of
		 a cv. -->
	    <xsl:value-of select="count(key('cvt',$valn)/preceding-sibling::control-value)"/>
	  </xsl:when>
	  <xsl:when test="contains($permitted,'x') and contains($valn,'.') and
			  (number($valn) = 0 or number($valn))">
	    <!-- It's a number literal with a decimal point, understood to
		 represent a distance on the raster grid (F26dot6). -->
	    <xsl:value-of select="round(number($valn) * 64)"/>
	  </xsl:when>
	  <xsl:when test="contains($permitted,'1') and (number($valn) = 0 or
			  number($valn))">
	    <!-- It's a number literal, an integer. -->
	    <xsl:value-of select="number($valn) + number($off)"/>
	  </xsl:when>
	  <xsl:when test="contains($permitted,'x') and
			  (number($all-but-last-char) or number($all-but-last-char) = 0) and
			  $last-char = 'p'">
	    <!-- It's a number literal with suffix p, a distance on the
		 raster grid (F26dot6). -->
	    <xsl:value-of select="round(number($all-but-last-char) * 64)"/>
	  </xsl:when>
	  <xsl:when test="contains($permitted,'n') and contains($valn,'/') and
			  not(contains($valn,' / '))">
	    <!-- It's a number in glyph/constant format, i.e. most likely a reference
		 to a point in another glyph. Recurse to resolve the reference. -->
	    <xsl:call-template name="get-number-literal">
	      <xsl:with-param name="val"
			      select="normalize-space(substring-after($valn,'/'))"/>
	      <xsl:with-param name="offset" select="$off"/>
	      <xsl:with-param name="cvt-mode" select="$cvt-mode"/>
	      <xsl:with-param name="ancestor-glyph"
			      select="/xgridfit/glyph[@ps-name =
				      normalize-space(substring-before($valn,'/'))]"/>
	      <xsl:with-param name="permitted" select="$permitted"/>
	      <xsl:with-param name="call-macro-param-set"
			      select="$call-macro-param-set"/>
	    </xsl:call-template>
	  </xsl:when>
	  <xsl:when test="contains($permitted,'r') and
			  document('xgfdata.xml')/*/xgf:round-states/xgf:round[@name = $valn]">
	    <!-- It's a round state, referenced by name. -->
	    <xsl:value-of select="document('xgfdata.xml')/*/xgf:round-states/xgf:round[@name
				  = $valn]/@num"/>
	  </xsl:when>
	  <xsl:when test="contains($permitted,'n') and
			  $all-global-constants[@name=$valn]">
	    <!-- It's a global constant. -->
	    <xsl:variable name="nn">
	      <xsl:choose>
		<xsl:when test="$all-global-constants[@name = $valn]/@value">
		  <xsl:value-of select="$all-global-constants[@name = $valn]/@value"/>
		</xsl:when>
		<xsl:when test="$all-global-constants[@name = $valn]/@num">
		  <xsl:value-of select="$all-global-constants[@name = $valn]/@num"/>
		</xsl:when>
	      </xsl:choose>
	    </xsl:variable>
	    <xsl:choose>
	      <xsl:when test="(number($nn) or number($nn) = 0) and
			      not(contains($nn,'.'))">
		<xsl:value-of select="number($nn) + number($off)"/>
	      </xsl:when>
	      <xsl:otherwise>
		<xsl:call-template name="get-number-literal">
		  <xsl:with-param name="val" select="$nn"/>
		  <xsl:with-param name="offset" select="$off"/>
		  <xsl:with-param name="cvt-mode" select="$cvt-mode"/>
		  <xsl:with-param name="ancestor-glyph" select="$ancestor-glyph"/>
		  <xsl:with-param name="permitted" select="$permitted"/>
		  <xsl:with-param name="call-macro-param-set"
				  select="$call-macro-param-set"/>
		</xsl:call-template>
	      </xsl:otherwise>
	    </xsl:choose>
	  </xsl:when>
	  <xsl:when test="contains($permitted,'2') and $last-char = 'v' and
			  number($all-but-last-char) &gt;= -1 and
			  number($all-but-last-char) &lt;= 1">
	    <!-- It's an F2dot14 number, for setting a vector. -->
	    <xsl:value-of select="round(number($all-but-last-char) * 16384)"/>
	  </xsl:when>
	  <xsl:otherwise>
	    <xsl:variable name="op">
	      <xsl:call-template name="find-operator">
		<xsl:with-param name="s" select="$valn"/>
	      </xsl:call-template>
	    </xsl:variable>
	    <xsl:choose>
	      <xsl:when test="$op != 'NaN'">
		<!-- The string contains an operator: it's an expression. -->
		<!-- What we got from find-operator was the length of a substring
		     beginning with the operator and extending to the end of the
		     expression. Starting with this info we can break down the
		     expression into two or three parts, depending on whether
		     the operator is unary or binary. First get the length of
		     the first token (if any). -->
		<xsl:variable name="tok-one-len"
			      select="string-length($valn) - number($op)"/>
		<!-- Next get the first token itself (empty if there is none) -->
		<xsl:variable name="first-tok">
		  <xsl:choose>
		    <xsl:when test="$tok-one-len &gt; 0">
		      <xsl:value-of select="normalize-space(substring($valn,
					    1,$tok-one-len))"/>
		    </xsl:when>
		    <xsl:otherwise>
		      <xsl:value-of select="''"/>
		    </xsl:otherwise>
		  </xsl:choose>
		</xsl:variable>
		<!-- An intermediate string containing the operator and the rest
		     of the expression (if any). -->
		<xsl:variable name="ts">
		  <xsl:choose>
		    <xsl:when test="$tok-one-len &gt; 0">
		      <xsl:value-of select="substring($valn,$tok-one-len + 1)"/>
		    </xsl:when>
		    <xsl:otherwise>
		      <xsl:value-of select="$valn"/>
		    </xsl:otherwise>
		  </xsl:choose>
		</xsl:variable>
		<!-- Get the operator -->
		<xsl:variable name="operator">
		  <xsl:call-template name="get-first-token">
		    <xsl:with-param name="s" select="$ts"/>
		  </xsl:call-template>
		</xsl:variable>
		<xsl:variable name="index-only"
			      select="boolean($operator = 'index')"/>
		<!-- Get the operator direction indicator (left, right, both) -->
		<xsl:variable
		    name="op-dir"
		    select="document('xgfdata.xml')/*/xgf:operators/xgf:operator[@symbol =
			    $operator]/@dir"/>
		<!-- Get the last token (if any). -->
		<xsl:variable name="last-tok">
		  <xsl:call-template name="get-remaining-tokens">
		    <xsl:with-param name="s" select="$ts"/>
		  </xsl:call-template>
		</xsl:variable>
		<!-- Now try to resolve the values to numbers by calling this
		     template recursively. -->
		<xsl:variable name="first-num">
		  <xsl:if test="$op-dir = 'b'">
		    <xsl:call-template name="get-number-literal">
		      <xsl:with-param  name="ancestor-glyph"
				       select="$ancestor-glyph"/>
		      <xsl:with-param  name="val"
				       select="normalize-space($first-tok)"/>
		      <xsl:with-param name="call-macro-param-set"
				      select="$call-macro-param-set"/>
		    </xsl:call-template>
		  </xsl:if>
		</xsl:variable>
		<xsl:variable name="last-num">
		  <xsl:call-template name="get-number-literal">
		    <xsl:with-param  name="ancestor-glyph"
				     select="$ancestor-glyph"/>
		    <xsl:with-param  name="val"
				     select="normalize-space($last-tok)"/>
		    <xsl:with-param  name="force-to-index" select="index-only"/>
		    <xsl:with-param  name="permitted">
		      <xsl:choose>
			<xsl:when test="index-only">
			  <xsl:value-of select="'c'"/>
			</xsl:when>
			<xsl:otherwise>
			  <xsl:value-of select="'12xrnc'"/>
			</xsl:otherwise>
		      </xsl:choose>
		    </xsl:with-param>
		    <xsl:with-param name="call-macro-param-set"
				    select="$call-macro-param-set"/>
		  </xsl:call-template>
		</xsl:variable>
		<!-- If we've gotten the number(s), do the calculation. -->
		<xsl:choose>
		  <!--
		      When we get the operator "nan" (not a number),
		      we're no longer interested in what the number of
		      the argument is, but only whether it is a number
		      or not.
		  -->
		  <xsl:when test="$operator = 'nan'">
		    <xsl:choose>
		      <xsl:when test="$last-num = 'NaN'">
			<xsl:value-of select="1"/>
		      </xsl:when>
		      <xsl:otherwise>
			<xsl:value-of select="0"/>
		      </xsl:otherwise>
		    </xsl:choose>
		  </xsl:when>
		  <xsl:when test="(not($force-to-index) and $first-num = 'NaN') or
				  $last-num = 'NaN'">
		    <xsl:text>NaN</xsl:text>
		  </xsl:when>
		  <xsl:otherwise>
		    <xsl:choose>
		      <xsl:when test="$operator = 'index'">
			<xsl:value-of select="number($last-num)"/>
		      </xsl:when>
		      <xsl:when test="$operator = 'not'">
			<xsl:choose>
			  <xsl:when test="number($last-num)">
			    <xsl:value-of select="0"/>
			  </xsl:when>
			  <xsl:otherwise>
			    <xsl:value-of select="1"/>
			  </xsl:otherwise>
			</xsl:choose>
		      </xsl:when>
		      <xsl:when test="$operator = '+'">
			<xsl:value-of select="number($first-num) + number($last-num)"/>
		      </xsl:when>
		      <xsl:when test="$operator = '-'">
			<xsl:value-of select="number($first-num) - number($last-num)"/>
		      </xsl:when>
		      <xsl:when test="$operator = '='">
			<xsl:choose>
			  <xsl:when test="number($first-num) = number($last-num)">
			    <xsl:value-of select="1"/>
			  </xsl:when>
			  <xsl:otherwise>
			    <xsl:value-of select="0"/>
			  </xsl:otherwise>
			</xsl:choose>
		      </xsl:when>
		      <xsl:when test="$operator = '!='">
			<xsl:choose>
			  <xsl:when test="number($first-num) != number($last-num)">
			    <xsl:value-of select="1"/>
			  </xsl:when>
			  <xsl:otherwise>
			    <xsl:value-of select="0"/>
			  </xsl:otherwise>
			</xsl:choose>
		      </xsl:when>
		      <xsl:when test="$operator = '&lt;'">
			<xsl:choose>
			  <xsl:when test="number($first-num) &lt; number($last-num)">
			    <xsl:value-of select="1"/>
			  </xsl:when>
			  <xsl:otherwise>
			    <xsl:value-of select="0"/>
			  </xsl:otherwise>
			</xsl:choose>
		      </xsl:when>
		      <xsl:when test="$operator = '&gt;'">
			<xsl:choose>
			  <xsl:when test="number($first-num) &gt; number($last-num)">
			    <xsl:value-of select="1"/>
			  </xsl:when>
			  <xsl:otherwise>
			    <xsl:value-of select="0"/>
			  </xsl:otherwise>
			</xsl:choose>
		      </xsl:when>
		      <xsl:when test="$operator = '&lt;='">
			<xsl:choose>
			  <xsl:when test="number($first-num) &lt;= number($last-num)">
			    <xsl:value-of select="1"/>
			  </xsl:when>
			  <xsl:otherwise>
			    <xsl:value-of select="0"/>
			  </xsl:otherwise>
			</xsl:choose>
		      </xsl:when>
		      <xsl:when test="$operator = '&gt;='">
			<xsl:choose>
			  <xsl:when test="number($first-num) &gt;= number($last-num)">
			    <xsl:value-of select="1"/>
			  </xsl:when>
			  <xsl:otherwise>
			    <xsl:value-of select="0"/>
			  </xsl:otherwise>
			</xsl:choose>
		      </xsl:when>
		      <xsl:when test="$operator = 'and'">
			<xsl:choose>
			  <xsl:when test="number($first-num) and number($last-num)">
			    <xsl:value-of select="1"/>
			  </xsl:when>
			  <xsl:otherwise>
			    <xsl:value-of select="0"/>
			  </xsl:otherwise>
			</xsl:choose>
		      </xsl:when>
		      <xsl:when test="$operator = 'or'">
			<xsl:choose>
			  <xsl:when test="number($first-num) or number($last-num)">
			    <xsl:value-of select="1"/>
			  </xsl:when>
			  <xsl:otherwise>
			    <xsl:value-of select="0"/>
			  </xsl:otherwise>
			</xsl:choose>
		      </xsl:when>
		      <!-- We won't deal with any other operators here. -->
		      <xsl:otherwise>
			<xsl:text>NaN</xsl:text>
		      </xsl:otherwise>
		    </xsl:choose>
		  </xsl:otherwise>
		</xsl:choose>
	      </xsl:when>
	      <xsl:otherwise>
		<!-- It can't be resolved to a number at compile time. -->
		<xsl:value-of select="'NaN'"/>
	      </xsl:otherwise>
	    </xsl:choose>
	  </xsl:otherwise>
	</xsl:choose>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>
  
  <!--

      push-value

      Converts a value to a number (if necessary) and pushes it onto
      the stack. "Values" include:

        1. Variables (named locations in the Storge Area), including
           graphics-state variables
        2. CVT entries
        3. Function parameters
        4. Special read-only values: "pixels-per-em," "point-size," one of
           the round-states (same names as the "round" attribute values)
        5. Constants from the <declarations> or <profile> element
        6. Number literals
        7. Expressions

      There is one required parameter:

      val
         The value to be pushed. It may be any of the value types that
         xgridfit recognizes, subject to restrictions imposed by the
         "permitted" parameter.

      There are several optional parameters:

      cvt-mode
         If the identifier is that of a control value, determines whether
         to push the index of value's position in the cvt table or the
         actual value. This may be 'index' or 'value'; the default is
         'index'.
      ancestor-glyph
         The <glyph> that is the ancestor of the caller to this template.
         This and the next two parameters can usually be determined
         automatically, but they are supplied when this template is called
         from <call-function>, for there the context is unexpected.
      ancestor-function
         The <function> (if any) that is the ancestor of the caller to
         this template.
      ancestor-prep
         The <pre-program> element (if any) that is the ancestor of the caller
         to this template.
      add-mode
         Whether to output a PUSH command or add a number after an existing
         one. The default is false: a PUSH command is output.
      offset
         Deprecated. A number to add to val. The number may be positive or
         negative.  An offset may be added only to variables, function
         parameters and constants. If val is a constant, val+offset is
         resolved at compile time; otherwise, code is generated to do
         it at run time. The default is zero (and when offset is zero,
         no run-time code is generated.
      permitted
         A string indicating what kinds of values may be pushed on this
         call to push-value.
         1   a number
         2   an F2dot14 fixed-point number (e.g. 0.23456v)
         x   an F26dot6 fixed-point number (e.g. 1p, 4.35)
         m   pixels-per-em
         p   point-size
         l   a graphics state variable
         r   a round state
         f   function parameter
         v   variable
         n   named value (constant)
         c   control value (either index or value)
         The default is '12xmplrfvnc' - that is, any kind of value at all.
      expect
         How many numbers are to be pushed with a single PUSH instruction.
         This is looked at only if the present call is for the first number
         in the list.
      call-macro-param-set
         If we're compiling a macro, this should contain the <param-set>
	 containing the parameters for this call to the macro. From it we
	 can also recover lost contexts: glyph, function or pre-program.
      -->
  <xsl:template name="push-value">
    <xsl:param name="val"/>
    <xsl:param name="cvt-mode" select="'index'"/>
    <xsl:param name="force-to-index" select="false()"/>
    <xsl:param name="ancestor-glyph" select="ancestor::glyph"/>
    <xsl:param name="ancestor-function" select="ancestor::function"/>
    <xsl:param name="ancestor-prep" select="ancestor::pre-program"/>
    <xsl:param name="add-mode" select="false()"/>
    <xsl:param name="offset" select="0"/>
    <xsl:param name="permitted" select="'12xmplrfvnc'"/>
    <xsl:param name="expect" select="1"/>
    <xsl:param name="with-leading-newline" select="true()"/>
    <xsl:param name="call-macro-param-set"/>
    <xsl:param name="all-function-params"
	       select="$ancestor-function/param|$ancestor-function/params/param"/>
    <xsl:param name="all-macro-params"
	       select="ancestor::macro/param|ancestor::macro/params/param"/>
    <xsl:param name="all-variables"
	       select="$ancestor-glyph/variable|$ancestor-glyph/variables/variable|
		       $ancestor-function/variable|$ancestor-function/variables/variable|
		       $ancestor-prep/variable|$ancestor-prep/variables/variable"/>
    <xsl:param name="all-global-variables"
	       select="/xgridfit/variable|/xgridfit/profile/variable"/>
    <xsl:variable name="val-normed">
      <xsl:call-template name="normalize-expression">
        <xsl:with-param name="s" select="$val"/>
      </xsl:call-template>
    </xsl:variable>
    <xsl:choose>
      <xsl:when test="$all-macro-params[@name=$val-normed]">
	<xsl:variable name="sub-value">
	  <xsl:choose>
	    <xsl:when test="$call-macro-param-set[@name=$val-normed or
			    @param-id=$val-normed]/@value">
	      <xsl:value-of select="$call-macro-param-set[@name=$val-normed or
				    @param-id=$val-normed]/@value"/>
	    </xsl:when>
	    <xsl:when test="$all-macro-params[@name=$val-normed]/@value">
	      <xsl:value-of select="$all-macro-params[@name=$val-normed]/@value"/>
	    </xsl:when>
	    <xsl:otherwise>
	      <xsl:call-template name="error-message">
		<xsl:with-param name="msg">
		  <xsl:text>Can't find value for param </xsl:text>
		  <xsl:value-of select="$val-normed"/>
		  <xsl:text> in macro </xsl:text>
		  <xsl:choose>
		    <xsl:when test="ancestor::macro/@name">
		      <xsl:value-of select="ancestor::macro/@name"/>
		    </xsl:when>
		    <xsl:otherwise>
		      <xsl:value-of select="ancestor::macro/@id"/>
		    </xsl:otherwise>
		  </xsl:choose>
		</xsl:with-param>
	      </xsl:call-template>
	    </xsl:otherwise>
	  </xsl:choose>
	</xsl:variable>
	<xsl:call-template name="push-value">
	  <xsl:with-param
	      name="val"
	      select="$call-macro-param-set[@name=$val-normed or @param-id=$val-normed]/@value"/>
	  <xsl:with-param name="cvt-mode" select="$cvt-mode"/>
	  <xsl:with-param name="force-to-index" select="$force-to-index"/>
	  <xsl:with-param name="ancestor-glyph"
			  select="$call-macro-param-set/ancestor::glyph"/>
	  <xsl:with-param name="ancestor-function"
			  select="$call-macro-param-set/ancestor::function"/>
	  <xsl:with-param name="ancestor-prep"
			  select="$call-macro-param-set/ancestor::pre-program"/>
	  <xsl:with-param name="add-mode" select="$add-mode"/>
	  <xsl:with-param name="offset" select="$offset"/>
	  <xsl:with-param name="permitted" select="$permitted"/>
	  <xsl:with-param name="expect" select="$expect"/>
	  <xsl:with-param name="with-leading-newline" select="$with-leading-newline"/>
	  <xsl:with-param name="call-macro-param-set" select="$call-macro-param-set"/>
	</xsl:call-template>
      </xsl:when>
      <xsl:otherwise>
	<!-- Try to fetch the number as a literal. We'll get "NaN" if it is not. -->
	<xsl:variable name="n">
	  <xsl:call-template name="get-number-literal">
	    <xsl:with-param name="val" select="$val-normed"/>
	    <xsl:with-param name="ancestor-glyph" select="$ancestor-glyph"/>
	    <xsl:with-param name="permitted" select="$permitted"/>
	    <xsl:with-param name="cvt-mode" select="$cvt-mode"/>
	    <xsl:with-param name="offset" select="$offset"/>
	    <xsl:with-param name="call-macro-param-set" select="$call-macro-param-set"/>
	  </xsl:call-template>
	</xsl:variable>
	<!-- We must have an offset number. Zero if none has been specified. -->
	<xsl:variable name="off">
	  <xsl:choose>
	    <xsl:when test="$offset and string-length($offset) &gt; 0">
	      <xsl:call-template name="get-number-literal">
		<xsl:with-param name="val" select="$offset"/>
		<xsl:with-param name="ancestor-glyph" select="$ancestor-glyph"/>
		<xsl:with-param name="permitted" select="'1n'"/>
	      </xsl:call-template>
	    </xsl:when>
	    <xsl:otherwise>
	      <xsl:text>0</xsl:text>
	    </xsl:otherwise>
	  </xsl:choose>
	</xsl:variable>
	<xsl:choose>
	  <xsl:when test="number($n) or number($n) = 0">
	    <!-- It's a number literal: push it. -->
	    <xsl:call-template name="push-num">
	      <xsl:with-param name="num" select="$n"/>
	      <xsl:with-param name="add-mode" select="$add-mode"/>
	      <xsl:with-param name="expect" select="$expect"/>
	      <xsl:with-param name="with-leading-newline"
			      select="$with-leading-newline"/>
	    </xsl:call-template>
	  </xsl:when>
	  <xsl:when test="contains($permitted,'m') and $val-normed='pixels-per-em'">
	    <xsl:call-template name="simple-command">
	      <xsl:with-param name="cmd" select="'MPPEM'"/>
	    </xsl:call-template>
	  </xsl:when>
	  <xsl:when test="contains($permitted,'p') and $val-normed='point-size'">
	    <xsl:call-template name="simple-command">
	      <xsl:with-param name="cmd" select="'MPS'"/>
	    </xsl:call-template>
	  </xsl:when>
	  <xsl:when test="contains($permitted,'l') and
			  document('xgfdata.xml')/*/xgf:var-locations/xgf:loc[@name =
			  $val-normed]">
	    <!-- It's one of the predefined variables. -->
	    <xsl:call-template name="push-num">
	      <xsl:with-param
		  name="num"
		  select="document('xgfdata.xml')/*/xgf:var-locations/xgf:loc[@name =
			  $val-normed]/@val"/>
	      <xsl:with-param name="with-leading-newline" select="$with-leading-newline"/>
	    </xsl:call-template>
	    <xsl:call-template name="simple-command">
	      <xsl:with-param name="cmd" select="'RS'"/>
	    </xsl:call-template>
	  </xsl:when>
	  <xsl:when test="contains($permitted,'v') and
			  $all-variables[@name=$val-normed or @id=$val-normed]">
	    <!-- It's a variable declared in a glyph, function or pre-program. -->
	    <xsl:call-template name="push-variable">
	      <xsl:with-param name="val"
			      select="$all-variables[@name=$val-normed or @id=$val-normed]"/>
	      <xsl:with-param name="add-mode" select="$add-mode"/>
	      <xsl:with-param name="index-only" select="$force-to-index"/>
	    </xsl:call-template>
	    <xsl:if test="$off != 0">
	      <xsl:call-template name="number-command">
		<xsl:with-param name="num" select="$off"/>
		<xsl:with-param name="cmd" select="'ADD'"/>
	      </xsl:call-template>
	    </xsl:if>
	  </xsl:when>
	  <xsl:when test="contains($permitted,'v') and
			  $all-global-variables[@name=$val-normed or
			  @id=$val-normed]">
	    <!-- It's a global variable, declared in the profile. -->
	    <xsl:call-template name="push-global-variable">
	      <xsl:with-param
		  name="val"
		  select="$all-global-variables[@name=$val-normed or
			  @id=$val-normed]"/>
	      <xsl:with-param name="index-only" select="$force-to-index"/>
	    </xsl:call-template>
	    <xsl:if test="$off != 0">
	      <xsl:call-template name="number-command">
		<xsl:with-param name="num" select="$off"/>
		<xsl:with-param name="cmd" select="'ADD'"/>
	      </xsl:call-template>
	    </xsl:if>
	  </xsl:when>
	  <xsl:when test="contains($permitted,'f') and
			  $all-function-params[@name=$val-normed]">
	    <!-- It's a function parameter. -->
	    <xsl:call-template name="push-function-parameter">
	      <xsl:with-param name="val" select="$val-normed"/>
	      <xsl:with-param name="ancestor-function" select="$ancestor-function"/>
	    </xsl:call-template>
	    <xsl:if test="$off != 0">
	      <xsl:call-template name="number-command">
		<xsl:with-param name="num" select="$off"/>
		<xsl:with-param name="cmd" select="'ADD'"/>
	      </xsl:call-template>
	    </xsl:if>
	  </xsl:when>
	  <xsl:when test="contains($permitted,'c') and
			  key('cvt',$val-normed) and $cvt-mode = 'value'">
	    <!-- It's a control-value index, and we want the value itself. -->
	    <xsl:call-template name="push-num">
	      <xsl:with-param name="num">
		<xsl:call-template name="get-number-literal">
		  <xsl:with-param name="val" select="$val-normed"/>
		  <xsl:with-param name="permitted" select="'c'"/>
		  <xsl:with-param name="cvt-mode" select="'index'"/>
		</xsl:call-template>
	      </xsl:with-param>
	      <xsl:with-param name="with-leading-newline" select="$with-leading-newline"/>
	    </xsl:call-template>
	    <xsl:call-template name="simple-command">
	      <xsl:with-param name="cmd" select="'RCVT'"/>
	    </xsl:call-template>
	  </xsl:when>
	  <xsl:otherwise>
	    <xsl:variable name="op">
	      <xsl:call-template name="find-operator">
		<xsl:with-param name="s" select="$val-normed"/>
	      </xsl:call-template>
	    </xsl:variable>
	    <xsl:choose>
	      <xsl:when test="$op != 'NaN'">
		<!-- It's an expression. -->
		<xsl:variable name="tok-one-len"
			      select="string-length($val-normed) - number($op)"/>
		<xsl:variable name="first-tok">
		  <xsl:choose>
		    <xsl:when test="$tok-one-len &gt; 0">
		      <xsl:value-of
			  select="normalize-space(substring($val-normed,1,$tok-one-len))"/>
		    </xsl:when>
		    <xsl:otherwise>
		      <xsl:value-of select="''"/>
		    </xsl:otherwise>
		  </xsl:choose>
		</xsl:variable>
		<!-- An intermediate string containing the operator and the rest
		     of the expression (if any). -->
		<xsl:variable name="ts">
		  <xsl:choose>
		    <xsl:when test="$tok-one-len &gt; 0">
		      <xsl:value-of select="substring($val,$tok-one-len + 1)"/>
		    </xsl:when>
		    <xsl:otherwise>
		      <xsl:value-of select="$val"/>
		    </xsl:otherwise>
		  </xsl:choose>
		</xsl:variable>
		<!-- Get the operator -->
		<xsl:variable name="operator">
		  <xsl:call-template name="get-first-token">
		    <xsl:with-param name="s" select="$ts"/>
		  </xsl:call-template>
		</xsl:variable>
		<xsl:variable name="index-only"
			      select="boolean($operator = 'index')"/>
		<!-- Get the operator direction indicator (left, right, both) -->
		<xsl:variable
		    name="op-dir"
		    select="document('xgfdata.xml')/*/xgf:operators/xgf:operator[@symbol =
			    $operator]/@dir"/>
		<!-- Get the last token (if any). -->
		<xsl:variable name="last-tok">
		  <xsl:call-template name="get-remaining-tokens">
		    <xsl:with-param name="s" select="$ts"/>
		  </xsl:call-template>
		</xsl:variable>
		<xsl:if test="$op-dir = 'b'">
		  <xsl:call-template name="push-value">
		    <xsl:with-param name="val" select="$first-tok"/>
		    <xsl:with-param name="ancestor-glyph" select="$ancestor-glyph"/>
		    <xsl:with-param name="ancestor-function" select="$ancestor-function"/>
		    <xsl:with-param name="ancestor-prep" select="$ancestor-prep"/>
		  </xsl:call-template>
		</xsl:if>
		<xsl:call-template name="push-value">
		  <xsl:with-param name="val" select="$last-tok"/>
		  <xsl:with-param name="ancestor-glyph" select="$ancestor-glyph"/>
		  <xsl:with-param name="ancestor-function" select="$ancestor-function"/>
		  <xsl:with-param name="ancestor-prep" select="$ancestor-prep"/>
		  <xsl:with-param name="force-to-index" select="$index-only"/>
		  <xsl:with-param name="permitted">
		    <xsl:choose>
		      <xsl:when test="$index-only">
			<xsl:value-of select="'v'"/>
		      </xsl:when>
		      <xsl:otherwise>
			<xsl:value-of select="'12xmplrfvnc'"/>
		      </xsl:otherwise>
		    </xsl:choose>
		  </xsl:with-param>
		</xsl:call-template>
		<xsl:choose>
		  <xsl:when test="$operator = 'round'">
		    <xsl:call-template name="simple-command">
		      <xsl:with-param name="cmd" select="'ROUND'"/>
		      <xsl:with-param name="modifier">
			<xsl:call-template name="color-bits">
			  <xsl:with-param name="color" select="'gray'"/>
			</xsl:call-template>
		      </xsl:with-param>
		    </xsl:call-template>
		  </xsl:when>
		  <xsl:when test="$operator != 'index'">
		    <xsl:call-template name="simple-command">
		      <xsl:with-param
			  name="cmd"
			  select="document('xgfdata.xml')/*/xgf:operators/xgf:operator[@symbol =
				  $operator]"/>
		    </xsl:call-template>
		  </xsl:when>
		</xsl:choose>
	      </xsl:when>
	      <xsl:otherwise>
		<!-- It's a mistake. -->
		<xsl:call-template name="error-message">
		  <xsl:with-param name="msg">
		    <xsl:text>Cannot resolve value or variable </xsl:text>
		    <xsl:value-of select="$val-normed"/>
		    <xsl:text>.</xsl:text>
		  </xsl:with-param>
		</xsl:call-template>
	      </xsl:otherwise>
	    </xsl:choose>
	  </xsl:otherwise>
	</xsl:choose>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>
  
  
  <!--

      store-value

      Values can be stored either in variables (= the Storage Area) or
      in the CVT table. Variables include not only those that the
      program declares, but also those that store that graphics state,
      of which several can be written to: minimum-distance,
      control-value-cut-in, single-width, single-width-cut-in,
      delta-base, delta-shift. Two others, round-state and
      custom-round-state, can be read, but can be written to only via
      <set-round-state> and <with-round-state>.

      This template finds a value on top of the stack and stores it in
      the specified location. If it fails, it issues a warning and
      leaves the value on the stack.
  -->
  <xsl:template name="store-value">
    <xsl:param name="vname"/>
    <xsl:param name="call-macro-param-set"/>
    <xsl:param name="all-variables"
	       select="ancestor::glyph/variable|ancestor::glyph/variables/variable|
		       ancestor::function/variable|ancestor::function/variables/variable|
		       ancestor::pre-program/variable|ancestor::pre-program/variables/variable"/>
    <xsl:param name="all-global-variables"
	       select="/xgridfit/variable|/xgridfit/profile/variable"/>
    <xsl:choose>
      <xsl:when test="$vname = 'return' and ancestor::function[@return = 'yes']">
        <xsl:call-template name="stack-top-to-storage">
          <xsl:with-param name="loc" select="$var-return-value"/>
        </xsl:call-template>
      </xsl:when>
      <xsl:when test="$all-variables[@name=$vname or @id=$vname]">
        <xsl:call-template name="store-variable">
          <xsl:with-param name="val"
            select="$all-variables[@name=$vname or @id=$vname]"/>
        </xsl:call-template>
      </xsl:when>
      <xsl:when test="key('cvt',$vname)">
        <xsl:call-template name="store-control-value">
          <xsl:with-param name="cvtname" select="$vname"/>
	  <xsl:with-param name="call-macro-param-set"
			  select="$call-macro-param-set"/>
        </xsl:call-template>
      </xsl:when>
      <xsl:when test="$vname = 'custom-round-state'">
        <xsl:call-template name="warning">
          <xsl:with-param name="msg">
            <xsl:text>custom-round-state can be written to only via </xsl:text>
            <xsl:text>&lt;set-round-state&gt;</xsl:text>
            <xsl:value-of select="$newline"/>
            <xsl:text>and &lt;with-round-state&gt;. </xsl:text>
            <xsl:text>The value this program has attempted</xsl:text>
            <xsl:value-of select="$newline"/>
            <xsl:text>to write to it is being discarded.</xsl:text>
          </xsl:with-param>
          <xsl:with-param name="with-pop" select="true()"/>
        </xsl:call-template>
      </xsl:when>
      <xsl:when test="$vname = 'round-state'">
        <xsl:call-template name="warning">
          <xsl:with-param name="msg">
            <xsl:text>round-state can be written to only via </xsl:text>
            <xsl:text>&lt;set-round-state&gt;</xsl:text>
            <xsl:value-of select="$newline"/>
            <xsl:text>and &lt;with-round-state&gt;. </xsl:text>
            <xsl:text>The value this program has attempted</xsl:text>
            <xsl:value-of select="$newline"/>
            <xsl:text>to write to it is being discarded.</xsl:text>
          </xsl:with-param>
          <xsl:with-param name="with-pop" select="true()"/>
        </xsl:call-template>
      </xsl:when>
      <!-- Any graphics variable except for round-state and custom-round-state can be
           written to, and the graphics state is automatically changed. -->
      <xsl:when test="document('xgfdata.xml')/*/xgf:var-locations/xgf:loc[@name = $vname]">
        <xsl:choose>
          <xsl:when test="contains($vname,'default')">
            <xsl:choose>
              <xsl:when test="ancestor::pre-program">
                <xsl:call-template name="set-simple-graphics-var">
                  <xsl:with-param name="loc"
                    select="document('xgfdata.xml')/*/xgf:var-locations/xgf:loc[@name =
                    $vname]/@vice"/>
                  <xsl:with-param name="default-loc"
                    select="document('xgfdata.xml')/*/xgf:var-locations/xgf:loc[@name =
                    $vname]/@val"/>
                  <xsl:with-param name="cmd"
                    select="document('xgfdata.xml')/*/xgf:var-locations/xgf:loc[@name =
                    $vname]"/>
                </xsl:call-template>
              </xsl:when>
              <xsl:otherwise>
                <xsl:call-template name="warning">
                  <xsl:with-param name="msg">
                    <xsl:text>It is not permitted to write to the </xsl:text>
                    <xsl:text>default graphics variable </xsl:text>
                    <xsl:value-of select="$vname"/>
                    <xsl:value-of select="$newline"/>
                    <xsl:text>except in the &lt;pre-program&gt;. </xsl:text>
                    <xsl:text>The value this program has attempted</xsl:text>
                    <xsl:value-of select="$newline"/>
                    <xsl:text>to write to it is being discarded.</xsl:text>
                  </xsl:with-param>
                  <xsl:with-param name="with-pop" select="true()"/>
                </xsl:call-template>
              </xsl:otherwise>
            </xsl:choose>
          </xsl:when>
          <xsl:otherwise>
            <xsl:call-template name="set-simple-graphics-var">
              <xsl:with-param name="loc"
                select="document('xgfdata.xml')/*/xgf:var-locations/xgf:loc[@name =
                $vname]/@val"/>
              <xsl:with-param name="default-loc"
                select="document('xgfdata.xml')/*/xgf:var-locations/xgf:loc[@name =
                $vname]/@vice"/>
              <xsl:with-param name="cmd"
                select="document('xgfdata.xml')/*/xgf:var-locations/xgf:loc[@name =
                $vname]"/>
            </xsl:call-template>
          </xsl:otherwise>
        </xsl:choose>
      </xsl:when>
      <xsl:when test="$all-global-variables[@name=$vname or @id=$vname]">
        <xsl:call-template name="store-global-variable">
          <xsl:with-param name="val"
            select="$all-global-variables[@name=$vname or @id=$vname]"/>
        </xsl:call-template>
      </xsl:when>
      <xsl:otherwise>
        <xsl:call-template name="warning">
          <xsl:with-param name="msg">
            <xsl:text>Could not store value </xsl:text>
            <xsl:value-of select="$vname"/>
            <xsl:text>. It will be left on the stack.</xsl:text>
          </xsl:with-param>
        </xsl:call-template>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:template>
  
  
  <!-- 2. Handling function parameters -->
  
  <!--

      push-function-parameter

      This is always called from inside a function; it is the way
      functions read the parameters that have been passed to them.

      Start with a name; look up corresponding index; use that index
      to copy the proper value from wherever it is in the stack to the
      top of the stack.

      The complication is that something may have been put on top of
      the stack since parameters were pushed on.  To take care of
      that, we started the function by recording the stack depth, and
      now a little arithmetic will get us to where the parameters are
      stored.
  -->
  <xsl:template name="push-function-parameter">
    <xsl:param name="val"/>
    <xsl:param name="ancestor-function" select="ancestor::function"/>
    <xsl:param name="all-params"
	       select="$ancestor-function/param|$ancestor-function/params/param"/>
    <!-- We recorded the depth of the stack after we pushed parameters on.
         Now get the present depth for comparison. -->
    <xsl:call-template name="simple-command">
      <xsl:with-param name="cmd" select="'DEPTH'"/>
    </xsl:call-template>
    <!-- The depth of the stack when the function started is stored in
         var-function-stack-count. -->
    <xsl:call-template name="number-command">
      <xsl:with-param name="num" select="$var-function-stack-count"/>
      <xsl:with-param name="cmd" select="'RS'"/>
    </xsl:call-template>
    <!-- Subtract that from the current stack depth. -->
    <xsl:call-template name="simple-command">
      <xsl:with-param name="cmd" select="'SUB'"/>
    </xsl:call-template>
    <!-- Get the position of the relevant param element and add it to the
         number we just calculated. -->
    <xsl:call-template name="push-num">
      <xsl:with-param name="num"
        select="count($all-params[@name = $val]/preceding-sibling::param) + 1"/>
    </xsl:call-template>
    <xsl:call-template name="simple-command">
      <xsl:with-param name="cmd" select="'ADD'"/>
    </xsl:call-template>
    <!-- Now we know the location on the stack of the parameter we want.
         Copy it to the top. -->
    <xsl:call-template name="simple-command">
      <xsl:with-param name="cmd" select="'CINDEX'"/>
    </xsl:call-template>
  </xsl:template>
  
  <!-- 3. Variables (Storage Locations) -->
  
  <!-- "Variables" are really locations in the TT Storage Area. We
       adopt a more familiar term. Several locations at the bottom of
       the Storage Area are reserved by xgridfit for legacy storage
       and for tracking state info; above that is a block of global
       variables; above that is a "variable frame" (defined by
       var-frame-bottom and var-frame-top), which moves up and down as
       the scope changes. Result is that every glyph program and every
       function (also the pre-program) gets its own private space in
       the Storage Area. If you call a function recursively, each
       iteration gets its own set of variables. -->
  
  <!-- Call at the beginning of each glyph program and the pre-program
       to set up the variable frame. Code is inserted only if the
       program contains variables or calls a function (we don't try to
       predict whether the function will contain variables). -->
  <xsl:template name="set-up-variable-frame">
    <xsl:param name="with-leading-newline" select="true()"/>
    <xsl:call-template name="push-num">
      <xsl:with-param name="num" select="$var-frame-bottom"/>
      <xsl:with-param name="expect" select="2"/>
      <xsl:with-param name="with-leading-newline" select="$with-leading-newline"/>
    </xsl:call-template>
    <xsl:call-template name="push-num">
      <xsl:with-param name="num" select="$variable-frame-base"/>
      <xsl:with-param name="add-mode" select="true()"/>
    </xsl:call-template>
    <xsl:call-template name="simple-command">
      <xsl:with-param name="cmd" select="'WS'"/>
    </xsl:call-template>
    <xsl:call-template name="push-num">
      <xsl:with-param name="num" select="$var-frame-top"/>
    </xsl:call-template>
    <xsl:choose>
      <xsl:when test="variable or variables/variable">
        <xsl:call-template name="push-num">
          <xsl:with-param name="num"
            select="$variable-frame-base +
            count(variable|variables/variable)"/>
        </xsl:call-template>
      </xsl:when>
      <xsl:otherwise>
        <xsl:call-template name="push-num">
          <xsl:with-param name="num" select="$variable-frame-base"/>
        </xsl:call-template>
      </xsl:otherwise>
    </xsl:choose>
    <xsl:call-template name="simple-command">
      <xsl:with-param name="cmd" select="'WS'"/>
    </xsl:call-template>
  </xsl:template>
  
  <!--

      push-variable

      Start with a variable element; get its position in the variables
      list; add reserved-storage to that index to get the location to
      read from in Storage.  Read from Storage and push the number to
      the top of the stack.
  -->
  <xsl:template name="push-variable">
    <xsl:param name="val"/>
    <xsl:param name="index-only" select="false()"/>
    <xsl:call-template name="number-command">
      <xsl:with-param name="num" select="$var-frame-bottom"/>
      <xsl:with-param name="cmd" select="'RS'"/>
    </xsl:call-template>
    <xsl:call-template name="number-command">
      <xsl:with-param name="num" select="count($val/preceding-sibling::variable) + 1"/>
      <xsl:with-param name="cmd" select="'ADD'"/>
    </xsl:call-template>
    <xsl:if test="not($index-only)">
      <xsl:call-template name="simple-command">
	<xsl:with-param name="cmd" select="'RS'"/>
      </xsl:call-template>
    </xsl:if>
  </xsl:template>
  
  <xsl:template name="push-global-variable">
    <xsl:param name="val"/>
    <xsl:param name="index-only" select="false()"/>
    <xsl:call-template name="push-num">
      <xsl:with-param name="num"
		      select="count($val/preceding-sibling::variable)
			      + $global-variable-base + 1"/>
    </xsl:call-template>
    <xsl:if test="not($index-only)">
      <xsl:call-template name="simple-command">
	<xsl:with-param name="cmd" select="'RS'"/>
      </xsl:call-template>
    </xsl:if>
  </xsl:template>
  
  
  <!--

      store-variable

      The value to be stored has got to be on top of the stack.
  -->
  <xsl:template name="store-variable">
    <xsl:param name="val"/>
    <xsl:call-template name="number-command">
      <xsl:with-param name="num" select="$var-frame-bottom"/>
      <xsl:with-param name="cmd" select="'RS'"/>
    </xsl:call-template>
    <xsl:call-template name="number-command">
      <xsl:with-param name="num" select="count($val/preceding-sibling::variable) + 1"/>
      <xsl:with-param name="cmd" select="'ADD'"/>
    </xsl:call-template>
    <xsl:call-template name="simple-command">
      <xsl:with-param name="cmd" select="'SWAP'"/>
    </xsl:call-template>
    <xsl:call-template name="simple-command">
      <xsl:with-param name="cmd" select="'WS'"/>
    </xsl:call-template>
  </xsl:template>
  
  <xsl:template name="store-global-variable">
    <xsl:param name="val"/>
    <xsl:call-template name="push-num">
      <xsl:with-param name="num" select="$global-variable-base + 1 +
        count($val/preceding-sibling::variable)"/>
    </xsl:call-template>
    <xsl:call-template name="simple-command">
      <xsl:with-param name="cmd" select="'SWAP'"/>
    </xsl:call-template>
    <xsl:call-template name="simple-command">
      <xsl:with-param name="cmd" select="'WS'"/>
    </xsl:call-template>
  </xsl:template>

  <xsl:template match="variable" mode="initialize">
    <xsl:if test="@value">
      <xsl:call-template name="push-value">
	<xsl:with-param name="val" select="@value"/>
      </xsl:call-template>
      <xsl:call-template name="store-value">
	<xsl:with-param name="vname" select="@name"/>
      </xsl:call-template>
    </xsl:if>
  </xsl:template>
  
  <!--

      stack-top-to-storage

      Pops a number off the stack and puts it in the specified
      location in the Storage Area. $loc must be a number.
  -->
  <xsl:template name="stack-top-to-storage">
    <xsl:param name="loc"/>
    <xsl:call-template name="push-num">
      <xsl:with-param name="num" select="$loc"/>
    </xsl:call-template>
    <xsl:call-template name="simple-command">
      <xsl:with-param name="cmd" select="'SWAP'"/>
    </xsl:call-template>
    <xsl:call-template name="simple-command">
      <xsl:with-param name="cmd" select="'WS'"/>
    </xsl:call-template>
  </xsl:template>
  
  <!-- Copies a value from one storage location to another.
       Numbers only: no constants, vars. Location indexes must
       be less than 256. For internal use only. -->
  <xsl:template name="storage-to-storage">
    <xsl:param name="src"/>
    <xsl:param name="dest"/>
    <xsl:call-template name="push-num">
      <xsl:with-param name="num" select="$dest"/>
      <xsl:with-param name="expect" select="2"/>
    </xsl:call-template>
    <xsl:call-template name="push-num">
      <xsl:with-param name="num" select="$src"/>
      <xsl:with-param name="add-mode" select="true()"/>
    </xsl:call-template>
    <xsl:call-template name="simple-command">
      <xsl:with-param name="cmd" select="'RS'"/>
    </xsl:call-template>
    <xsl:call-template name="simple-command">
      <xsl:with-param name="cmd" select="'WS'"/>
    </xsl:call-template>
  </xsl:template>
  
  <!-- 4. Control Values (the CVT table) -->
  
  <!--

      store-control-value

      We can put anything at all into the CVT. Up to the user to make
      sure it makes sense. This version works only with pixel units.
      I'll have to think of some method of working with FUnits, though
      I'm not sure why one would want to write FUnits to the CVT.
  -->
  <xsl:template name="store-control-value">
    <xsl:param name="cvtname"/>
    <xsl:param name="call-macro-param-set"/>
    <xsl:call-template name="push-value">
      <xsl:with-param name="val" select="$cvtname"/>
      <xsl:with-param name="permitted" select="'c'"/>
      <xsl:with-param name="call-macro-param-set"
		      select="$call-macro-param-set"/>
    </xsl:call-template>
    <xsl:call-template name="simple-command">
      <xsl:with-param name="cmd" select="'SWAP'"/>
    </xsl:call-template>
    <xsl:call-template name="simple-command">
      <xsl:with-param name="cmd" select="'WCVTP'"/>
    </xsl:call-template>
  </xsl:template>

</xsl:stylesheet>
