/***********************************************************
 * XMMS InfoPipe plugin
 * Distributed under GNU GPL. See the COPYING file for details.
 *
 * Written by Weyfour WWWWolf (Urpo Lankinen),
 * 2000-12-09 and onward
 *
 * 2001-01-10: Edited some stuff, and added some stuff ;)
 *                - Shogun (peter@pcswebdesign.nl)
 *
 * 2001-01-11: Hacked a bit further by WWWWolf.
 *
 * 2001-02-12: Listened to the feedback even more, and changed it...
 *             Time for rel 1.1!
 *
 * 2002-02-08: Year since the last release... well, it is time to
 *             apply the patches and get going =)
 *
 * $Id: infopipe.c,v 1.3 2002/02/07 23:18:20 wwwwolf Exp $
 *
 ***********************************************************/

/***********************************************************/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <fcntl.h>
#include <pthread.h>
#include <time.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/time.h>
#include <xmms/configfile.h>
#include <xmms/plugin.h>
#include <xmms/util.h>
#include <xmms/xmmsctrl.h>

#include "infopipe.h"

/***********************************************************/
/*
  Name and version
 */
char infopipe_ver[40];

/*
  The data structure that holds the information about this plugin...
 */
GeneralPlugin infopipe_gp =
{
  NULL,                          // Handle (Filled by XMMS)
  NULL,                          // Filename (Filled by XMMS)
  0,                             // Session ID
  infopipe_ver,                  // Description
  init_plugin,                   // Init function
  show_about,                    // About box
  NULL,                          // Configure
  finalize_plugin                // Cleanup
};


/***********************************************************/

/*
  XMMS hookup function. This routine is called by XMMS after it dlopen()s
  this library, and it's supposed to return a pointer to GeneralPlugin
  struct.
 */
GeneralPlugin *get_gplugin_info(void) {
  strcpy(infopipe_ver, "InfoPipe ");
  strcat(infopipe_ver, INFOPIPE_VERSION);

  return &infopipe_gp;
}

/* Pipe handling thread. */
pthread_t pipehandler;

/* FIFO file name. */
char *fifo_file;

/*
  The function that's called when the plugin is initialized, as specified
  in infopipe_gp structure...
  This will create the pipe special file, then start a thread to handle
  pipe requests.
 */
void init_plugin(void) {


  /* Handle multiple session names.
     (Changed by WWWWolf - snprintf vs. sprintf, eliminated Yet Another
     Possible Buffer Overflow Sploit =)
   */

  /* Allocate space */
  fifo_file = malloc(FIFONAMEBUFSIZE);
  snprintf(fifo_file, FIFONAMEBUFSIZE, "%s_%s.%d",
	   FIFO_FILE_PFX, cuserid(NULL), infopipe_gp.xmms_session);

  /* See if the file exists... */
  if(access(fifo_file,F_OK)==0) {
    /* Let's nuke the old pipe that we found. Probably leftovers from old
       session.
    */
    if(unlink(fifo_file)!=0) {
      perror("Unable to remove the old pipe.");
      xmms_remote_quit(infopipe_gp.xmms_session);
      return;
    }
  }
  
  /* Create the pipe. */
  /* Edited by Shogun, made it readable for other users too,
     handy for PHP/Perl scripts running under a different user ;) */
  /* Ok, cool, I was about to do that change myself, too, due to
     Popular Demand =) - WWWWolf */
  if(mkfifo(fifo_file,0644) != 0) {
    perror("Unable to create a pipe");
    xmms_remote_quit(infopipe_gp.xmms_session);
  }

  /* Remove old symlink if it exists, and create a new symlink
     that points to most recent pipe. Yes, this means that the symlink
     always points to the most recently created pipe.

     We handle all errors except
       - "not found", because, duh, if it wasn't there at first place,
       - "permission denied" - if we can't yank the damn link from
         there, someone else probably has a good
         explanation for that, so let's leave it at that...
  */

  if(unlink(FIFO_FILE_PFX)!=0 &&
     (errno != ENOENT && errno != EPERM && errno != EACCES) ) {
    /* Oops, we did something wrong. */
    perror("Unable to reasonably remove the symbolic link");
    xmms_remote_quit(infopipe_gp.xmms_session);
    return;
  } else {
    /* We succeeded. Let's try to create the link. */
    if(symlink(fifo_file,FIFO_FILE_PFX)!=0) {
      perror("Unable to create symbolic link");
      xmms_remote_quit(infopipe_gp.xmms_session);
      return;    
    }
  }

  /* Create the thread that handles pipe stuff. */
  if(pthread_create(&pipehandler,NULL,
		    (void *) &request_handler_thread,
		    NULL) != 0) {
    perror("Unable the create new thread (ie, no InfoPipe for you!)");
    xmms_remote_quit(infopipe_gp.xmms_session);
  }
}

/*
  The function that's called when XMMS quits. It will get rid of the thread
  and the special file
*/
void finalize_plugin(void) {

  /* Wait for it to finish... even thought it never does that. */
  if(pthread_cancel(pipehandler) != 0) {
    perror("Thread cancel failed");
    xmms_remote_quit(infopipe_gp.xmms_session);
  }

  /* ...remove the pipe ... even when we never do this either... */
  if(unlink(fifo_file)!=0) {
    perror("Unable to remove the pipe in the cleanup phase");
    xmms_remote_quit(infopipe_gp.xmms_session);
  }
}

/*
  This is the thread that runs in background, waiting for reader at the
  InfoPipe and then sending information.

  In 1.1, we use fcntl() instead of waiting 'round, because this seems to be
  the Right Way and apparently more cooler anyway if you get millions
  of hits...
  Joel Yliluoma <bisqwit@iki.fi> helped me with this in the newsgroup. Praise!
*/
void request_handler_thread(void) {

  fd_set fds;
  FILE *p;  /* the pipe */
  int fd; /* File descriptor for pipe, and its flags. */

  for(;;) {
    /* Open the pipe as file descriptor. */
    fd = open(fifo_file, O_WRONLY);
    
    if(fd == -1) {
      perror("xmms_infopipe: Pipe open failed");
      xmms_remote_quit(infopipe_gp.xmms_session);
    }

    /* Set the file handle to use non-blocking I/O */
    fcntl(fd, F_SETFL, fcntl(fd, F_GETFL) | O_NONBLOCK);

    /* (Thanks to jre for pointing out this can be put to
       the open() call, but regrettably that caused just a segmentation fault...
       This form should work!)
    */

    /* Wait here until we have a reader... */
    FD_ZERO(&fds);
    FD_SET(fd, &fds);
    if(select(fd+1, NULL, &fds, NULL, NULL) <= 0)
      break;

    p = fdopen(fd, "w");
    blast_info(p);

    fclose(p);
    close(fd);
    /* Changed to 1 second after request... report if you have problems.
       FIXME: Should use XMMS configfile facility & config dialog???
    */
    sleep(1); /* Umm, or non-blockingness still doesn't work without this!
		 Is there some nicer way of saying this, like "wait
		 until no reader?" select()? */
  }
}

/*
  This will get the XMMS information and print them out to the pipe.
 */
void blast_info(FILE *pipe) {
  char play_status[30];
  gint tunes = xmms_remote_get_playlist_length(infopipe_gp.xmms_session);
  gint current = xmms_remote_get_playlist_pos(infopipe_gp.xmms_session);

  /* Edited by Shogun, added totalTime for a bit of optimizing.
     Now there is 1 xmms call less ;) */
  gint totalTime,seconds,minutes;

  fprintf(pipe, "XMMS protocol version: %d\n",
	  xmms_remote_get_version(infopipe_gp.xmms_session));
  fprintf(pipe, "InfoPipe Plugin version: %s\n",
	  INFOPIPE_VERSION);

  if(xmms_remote_is_playing(infopipe_gp.xmms_session)) {
    if(xmms_remote_is_paused(infopipe_gp.xmms_session)) {
      strcpy(play_status,"Paused");
    } else
      strcpy(play_status,"Playing");
  } else {
    strcpy(play_status,"Stopped");
  }

  fprintf(pipe, "Status: %s\n", play_status);

  fprintf(pipe, "Tunes in playlist: %d\n", tunes);
  fprintf(pipe, "Currently playing: %d\n", (current+1));

  /* Edited by Shogun, First get the count of milliseconds that the
     current song is playing */
  totalTime = (gint)xmms_remote_get_output_time(infopipe_gp.xmms_session);
  fprintf(pipe, "uSecPosition: %d\n",totalTime);

  /* Than calculate the seconds and minutes */
  seconds = (gint)((gfloat)totalTime / (gfloat)1000);
  minutes = (gint)((gfloat)seconds / (gfloat)60); 
  seconds -= minutes * 60;
  fprintf(pipe, "Position: %d:%02d\n",minutes,seconds);
  
  /* Edited by Shogun, add info about the total time of the
     current song, would be nice to know ;) */
  totalTime = 
	(gint)xmms_remote_get_playlist_time(infopipe_gp.xmms_session,current);
  fprintf(pipe, "uSecTime: %d\n",totalTime);
  
  seconds = (gint)((gfloat)totalTime / (gfloat)1000);
  minutes = (gint)((gfloat)seconds / (gfloat)60);
  seconds -= minutes * 60;
  fprintf(pipe, "Time: %d:%02d\n",minutes,seconds);

  fprintf(pipe, "Title: %s\n",
	  xmms_remote_get_playlist_title(infopipe_gp.xmms_session,current));
  fprintf(pipe, "File: %s\n",
	  xmms_remote_get_playlist_file(infopipe_gp.xmms_session,current));

}
