/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2002 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* Based on xmms visualization plugin mechanism */


#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <unistd.h>
#include <dirent.h>
#include <errno.h>

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include <math.h>

#include <xmms/plugin.h>

#include <gdk/gdk.h>

#ifdef HPUX
# include <dl.h>
#else
# include <dlfcn.h>
#endif

#ifdef HPUX
# define SHARED_LIB_EXT ".sl"
#else
# define SHARED_LIB_EXT ".so"
#endif

#ifndef RTLD_NOW
# define RTLD_NOW 0
#endif

#include "singit_macros.h"
#include "singit_macros_private.h"
#include "singit_config_private.h"
#include "singit_main.h"
#include "singit_plugin_data.h"
#include "singit_plugin_scanner.h"
#include "dlg_singit_config.h"

DisplayerPluginData *dp_data = NULL;

#define PLUGINSUBDIR "xmms-singit"

extern VisPlugin singit_vp;
extern SingitStatus singit_status;

void scan_dis_plugins(gchar *dirname);
void add_plugin(gchar * filename);
void dis_plugin_disable(DisplayerPlugin *dp);

gint dislist_compare_func(const void *a, const void *b)
{
	return strcasecmp(((DisplayerPlugin *) a)->description, ((DisplayerPlugin *) b)->description);
}

void plugins_init()
{
	gchar *dir;

	#ifdef CODEDEBUG
	DEBUG(("singit_plugin_scanner.c [plugins_init]\n"), 9);
	#endif

	if (dp_data_attach(dp_data)) { return; }

	dp_data = dp_data_new();

#ifndef DISABLE_USER_PLUGIN_DIR
	dir = g_strconcat(g_get_home_dir(), "/.xmms/Plugins/Visualization", NULL);
	scan_dis_plugins(dir);
	g_free(dir);
	/*
	 * This is in a separate loop so if the user puts them in the
	 * wrong dir we'll still get them in the right order (home dir
	 * first                                                - Zinx
	 */
	dir = g_strconcat(g_get_home_dir(), "/.xmms/Plugins/Visualization/", PLUGINSUBDIR, NULL);
	scan_dis_plugins(dir);
	g_free(dir);
#endif

	dir = g_strconcat(XMMS_VISUALIZATION_PLUGIN_DIR, "/", PLUGINSUBDIR, NULL);
	scan_dis_plugins(dir);
	g_free(dir);

	dp_data->dis_list = g_list_sort(dp_data->dis_list, dislist_compare_func);
	if (singit_config_attach(singit_config)) {
		dis_plugin_enable_from_stringified_list(getSCD(singit_config)->enabled_dplugins);
		singit_config_detach(TRUE);
	}
}

void* open_dynamic_lib(char *filename)
{
#ifdef HPUX
	/*
	 * Use shl_load family of functions on HP-UX. HP-UX does not
	 * support dlopen on 32-bit PA-RISC executables
	 */
	return shl_load(filename, BIND_DEFERRED, 0);
#else
	return dlopen(filename, RTLD_NOW);
#endif
}

void close_dynamic_lib(void* handle)
{
#ifdef HPUX
	shl_t h = handle;
	shl_unload(h);
#else
	dlclose(handle);
#endif
}

void* find_dynamic_symbol(void *handle, char *symbol)
{
#ifdef HPUX
	void *gpi;
	shl_t h = handle;
	if ((shl_findsym(&h, symbol, TYPE_PROCEDURE, &gpi)) != 0)
		gpi = NULL;
	return gpi;
#else
#ifdef SYMBOL_PREFIX
	char *sym = g_strconcat(SYMBOL_PREFIX, symbol, NULL);
	void *symh = dlsym(handle, sym);
	g_free(sym);
	return symh;
#else
	return dlsym(handle, symbol);
#endif /* SYMBOL_PREFIX */
#endif /* HPUX */
}

void dynamic_lib_error(void)
{
#ifdef HPUX
	perror("Error loading plugin!");
#else
	fprintf(stderr, "%s\n", dlerror());
#endif
}

int dis_plugin_check_duplicate(char *filename)
{
	GList *l;
	gchar *base_filename = g_basename(filename);
	/*
	 * erg.. gotta check 'em all, surely there's a better way
	 *                                                 - Zinx
	 */

	for (l = dp_data->dis_list; l; l = l->next)
		if (!strcmp(base_filename,
			    g_basename(((InputPlugin*)l->data)->filename)))
			return 1;

	return 0;
}

void add_dis_plugin(gchar * filename)
{
	void *h;
	void *(*gpi) (void);

	if (dis_plugin_check_duplicate(filename))
		{ return; }

	if ((h = open_dynamic_lib(filename)) == NULL)
	{
		dynamic_lib_error();
		return;
	}

	if ((gpi = find_dynamic_symbol(h, "get_dplugin_info")) != NULL)
	{
		DisplayerPlugin *p;

		p = (DisplayerPlugin *) gpi();
		p->handle = h;
		p->filename = g_strdup(filename);
		p->xmms_session = singit_vp.xmms_session;
		p->disable = dis_plugin_disable;
		dp_data->dis_list = g_list_prepend(dp_data->dis_list, p);
	}
	else
		{ close_dynamic_lib(h); }
}

void scan_dis_plugins(gchar *dirname)
{
	gchar *filename, *ext;
	DIR *dir;
	struct dirent *ent;
	struct stat statbuf;

	dir = opendir(dirname);
	if (!dir)
		return;

	while ((ent = readdir(dir)) != NULL)
	{
		filename = g_strdup_printf("%s/%s", dirname, ent->d_name);
		if (!stat(filename, &statbuf) && S_ISREG(statbuf.st_mode) &&
		    ((ext = strrchr(ent->d_name, '.')) != NULL))
			if (!strcmp(ext, SHARED_LIB_EXT))
				add_dis_plugin(filename);
		g_free(filename);
	}
	closedir(dir);
}

void plugins_finish(void)
{
	DisplayerPlugin *dp;
	GList *node;

	#ifdef CODEDEBUG
	DEBUG(("singit_plugin_scanner.c [plugins_finish] : "), 9);
	#endif

	if (!dp_data_detach(dp_data, FALSE)) {
		#ifdef CODEDEBUG
		DEBUG(("Just detached\n"), 9);
		#endif
		return;
	}
	#ifdef CODEDEBUG
	DEBUG(("Real finished\n"), 9);
	#endif

	if (singit_config_attach(singit_config)) {
		g_free(getSCD(singit_config)->enabled_dplugins);
		getSCD(singit_config)->enabled_dplugins = dis_plugin_stringify_enabled_list();
		singit_config_save_plugins();
		singit_config_detach(TRUE);
	}

	if (dp_data && dp_data->initialized) {
		singit_config_save_positions();
		plugins_finalize();
	}

	node = get_dis_plugin_list();
	while (node)
	{
		dp = (DisplayerPlugin *) node->data;
		close_dynamic_lib(dp->handle);
		node = node->next;
	}

	dp_data_free(dp_data);
	dp_data = NULL;
}

void plugins_set_time(gint time, LSong *cur_song, GList *real_next)
{
	DisplayerPlugin *dp;
	GList *node, *next;

	if (!dp_data) { return; }

	node = get_dis_plugin_enabled_list(TRUE);
	while (node)
	{
		dp = (DisplayerPlugin *) node->data;
		next = node->next;
		if (node && node->data && ((DisplayerPlugin *) node->data)->set_time) {
			((DisplayerPlugin *) node->data)->set_time(time, cur_song, real_next);
		}
		node = next;
	}
}

typedef enum {

	dpc_show = 0,
	dpc_hide,
	dpc_toggle,
	dpc_update,
	dpc_playback_start,
	dpc_playback_stop,
	dpc_about,
	dpc_configure
}
DPCallback;

inline void real_emitter(GList *node, DPCallback callback, gboolean enabled)
{
	if (node && node->data) {
		if (dp_data && dp_data->initialized && enabled) {
			switch (callback) {
			case dpc_show:
				if (((DisplayerPlugin *) node->data)->show)
					((DisplayerPlugin *) node->data)->show();
				break;
			case dpc_hide:
				if (((DisplayerPlugin *) node->data)->hide)
					((DisplayerPlugin *) node->data)->hide();
				break;
			case dpc_toggle:
					if (((DisplayerPlugin *) node->data)->toggle)
					((DisplayerPlugin *) node->data)->toggle();
				break;
			case dpc_update:
				if (((DisplayerPlugin *) node->data)->update)
					((DisplayerPlugin *) node->data)->update(getSCD(singit_config));
				break;
			case dpc_playback_start:
				if (((DisplayerPlugin *) node->data)->playback_start)
					((DisplayerPlugin *) node->data)->playback_start();
				break;
			case dpc_playback_stop:
				if (((DisplayerPlugin *) node->data)->playback_stop)
					((DisplayerPlugin *) node->data)->playback_stop();
				break;
			default: // To suppress compiler warnings
				break;
			}
		}
		switch (callback) {
		case dpc_about:
			if (((DisplayerPlugin *) node->data)->about)
				((DisplayerPlugin *) node->data)->about();
			break;
		case dpc_configure:
			if (((DisplayerPlugin *) node->data)->configure)
				((DisplayerPlugin *) node->data)->configure();
			break;
		default: // To suppress compiler warnings
			break;
		}
	}
}

/*
	if (plugin < 0) ==> for all enabled
*/
void emit_callback(DPCallback callback, gint plugin)
{
	GList *node;

	if (plugin < 0) {
		node = get_dis_plugin_enabled_list(FALSE);
		while (node)
		{
			real_emitter(node, callback, TRUE);
			node = node->next;
		}
	}
	else {
		if (plugin >= g_list_length(dp_data->dis_list)) { return; }
		node = g_list_nth(dp_data->dis_list, plugin);
		g_return_if_fail(node);
		real_emitter(node, callback, is_dis_plugin_enabled(plugin));
	}
}

void dis_plugin_show(gint plugin)
{
	#ifdef CODEDEBUG
	DEBUG(("singit_plugin_scanner.c [dis_plugin_show]\n"), 9);
	#endif

	emit_callback(dpc_show, plugin);
}

void dis_plugin_hide(gint plugin)
{
	#ifdef CODEDEBUG
	DEBUG(("singit_plugin_scanner.c [dis_plugin_hide]\n"), 9);
	#endif

	emit_callback(dpc_hide, plugin);
}

void dis_plugin_toggle(gint plugin)
{
	#ifdef CODEDEBUG
	DEBUG(("singit_plugin_scanner.c [dis_plugin_toggle]\n"), 9);
	#endif

	emit_callback(dpc_toggle, plugin);
}

void dis_plugin_update(gint plugin)
{
	#ifdef CODEDEBUG
	DEBUG(("singit_plugin_scanner.c [dis_plugin_update]\n"), 9);
	#endif

	emit_callback(dpc_update, plugin);
}

void dis_plugin_playback_start(gint plugin)
{
	#ifdef CODEDEBUG
	DEBUG(("singit_plugin_scanner.c [dis_plugin_playback_start]\n"), 9);
	#endif

	if (dp_data->playback_started) { return; }
	emit_callback(dpc_playback_start, plugin);
	dp_data->playback_started = TRUE;
}

void dis_plugin_playback_stop(gint plugin)
{
	#ifdef CODEDEBUG
	DEBUG(("singit_plugin_scanner.c [dis_plugin_playback_stop]\n"), 9);
	#endif

	if (!dp_data->playback_started) { return; }
	emit_callback(dpc_playback_stop, plugin);
	dp_data->playback_started = FALSE;
}

void dis_plugin_about(gint plugin)
{
	#ifdef CODEDEBUG
	DEBUG(("singit_plugin_scanner.c [dis_plugin_about]\n"), 9);
	#endif

	emit_callback(dpc_about, plugin);
}

void dis_plugin_configure(gint plugin)
{
	#ifdef CODEDEBUG
	DEBUG(("singit_plugin_scanner.c [dis_plugin_configure]\n"), 9);
	#endif

	emit_callback(dpc_configure, plugin);
}

void dis_plugin_render_freq(gint16 freq_data[2][256])
{
	GList *node;
	node = get_dis_plugin_enabled_list(TRUE);
	while (node)
	{
		if (((DisplayerPlugin *) node->data)->render_freq)
			((DisplayerPlugin *) node->data)->render_freq(freq_data);
		node = node->next;
	}
}

void dis_plugin_render_pcm(gint16 pcm_data[2][512])
{
	GList *node;
	node = get_dis_plugin_enabled_list(TRUE);
	while (node)
	{
		if (((DisplayerPlugin *) node->data)->render_pcm)
			((DisplayerPlugin *) node->data)->render_pcm(pcm_data);
		node = node->next;
	}
}

GList *get_dis_plugin_list(void)
{
	if (!dp_data) { return NULL; }
	return dp_data->dis_list;
}

GList *get_dis_plugin_enabled_list(gboolean initialized)
{
	if (!dp_data) { return NULL; }
	if (initialized && !dp_data->initialized) { return NULL; }
	return dp_data->enabled_list;
}

void dis_plugin_disable(DisplayerPlugin *dp)
{
	gint i = g_list_index(dp_data->dis_list,dp);
	set_dis_plugin_status_by_number(i, FALSE);
	config_dis_plugins_rescan();
}

void set_dis_plugin_status_by_pointer(DisplayerPlugin *dp, gboolean enable)
{
	gint i = g_list_index(dp_data->dis_list, dp);

	#ifdef CODEDEBUG
	DEBUG(("singit_plugin_scanner.c [set_dis_plugin_status_by_pointer] : "), 9);
	if (enable) { DEBUG(("Enable\n"), 9); }
	else { DEBUG(("Disable\n"), 9); }
	#endif

	set_dis_plugin_status_by_number(i, enable);
}

void set_dis_plugin_status_by_number(gint plugin, gboolean enable)
{
	GList *node;
	DisplayerPlugin *dp;

	#ifdef CODEDEBUG
	DEBUG(("singit_plugin_scanner.c [set_dis_plugin_status_by_number] : "), 9);
	if (enable) { DEBUG(("Enable\n"), 9); }
	else { DEBUG(("Disable\n"), 9); }
	#endif

	if (!dp_data) { return; }
	if (plugin >= g_list_length(dp_data->dis_list)) { return; }
	node = g_list_nth(dp_data->dis_list, plugin);
	g_return_if_fail(node);

	dp = (DisplayerPlugin *) node->data;
	g_return_if_fail(dp);

	if (enable) {
		if (!g_list_find(dp_data->enabled_list, dp)) {
			dp_data->enabled_list = g_list_append(dp_data->enabled_list, dp);
			if (dp->init && dp_data->initialized) {
				dp->init();
				if (dp->update) {
					dp->update(getSCD(singit_config));
				}
			}
			singit_status.initialize_plugins = TRUE;
		}
	}
	else {
		if (g_list_find(dp_data->enabled_list, dp)) {
			dp_data->enabled_list = g_list_remove(dp_data->enabled_list, dp);

			if (dp->finish) {
				dp->finish();
			}
			singit_status.initialize_plugins = TRUE;
			singit_config_save_positions();
			if (!dp_data->enabled_list) { singit_vp.disable_plugin(&singit_vp); }
		}
	}
	config_dis_plugins_rescan();
}

gboolean is_dis_plugin_enabled(gint plugin)
{
	return (g_list_find(dp_data->enabled_list,
		(DisplayerPlugin *) g_list_nth(dp_data->dis_list, plugin)->data) ? TRUE : FALSE);
}

gchar *dis_plugin_stringify_enabled_list(void)
{
	gchar *enalist = NULL, *temp, *temp2;
	GList *node;

	g_return_val_if_fail(dp_data != NULL, NULL);

	#ifdef CODEDEBUG
	DEBUG(("singit_plugin_scanner.c [dis_plugin_stringify_enabled_list]\n"), 9);
	#endif

	node = dp_data->enabled_list;
	while (node) {
		temp = enalist;
		temp2 = g_strdup(g_basename(((DisplayerPlugin *) node->data)->filename));
		if (enalist) {
			enalist = g_strconcat(temp, ",", temp2, NULL);
			g_free(temp2);
		}
		else {
			enalist = temp2;
		}
		g_free(temp);
		node = node->next;
	}
	return enalist;
}

void dis_plugin_enable_from_stringified_list(gchar * list)
{
	gchar **plugins, *base;
	GList *node;
	gint i;
	DisplayerPlugin *dp;

	#ifdef CODEDEBUG
	DEBUG(("singit_plugin_scanner.c [dis_plugin_enable_from_stringified_list]\n"), 9);
	#endif

	if (!list || !strcmp(list, ""))
		return;
	plugins = g_strsplit(list, ",", 0);
	i = 0;
	while (plugins[i])
	{
		node = dp_data->dis_list;
 		while (node)
		{
			base = g_basename(((DisplayerPlugin *) node->data)->filename);
			if (!strcmp(plugins[i], base))
			{
				dp = node->data;
				dp_data->enabled_list = g_list_append
					(dp_data->enabled_list, (DisplayerPlugin *) dp);
			}
			node = node->next;
		}
		i++;
	}
	g_strfreev(plugins);
}

gboolean plugins_initialize()
{
	GList *node;
	DisplayerPlugin *dp;

	if (!((dp_data) && !dp_data->initialized)) { return FALSE; }

	#ifdef CODEDEBUG
	DEBUG(("singit_plugin_scanner.c [plugins_initialize]\n"), 9);
	#endif

	node = dp_data->enabled_list;
	while (node)
	{
		dp = node->data;
		if(dp->init) {
			dp->init();
			if (dp->update) {
				dp->update(getSCD(singit_config));
			}
			GDK_THREADS_LEAVE();
			while (g_main_iteration(FALSE));
			GDK_THREADS_ENTER();
		}
		node = node->next;
	}

	dp_data->initialized = TRUE;

	return TRUE;
}

gboolean plugins_finalize()
{
	GList *node;
	DisplayerPlugin *dp;

	if (!((dp_data) && dp_data->initialized)) { return FALSE; }

	#ifdef CODEDEBUG
	DEBUG(("singit_plugin_scanner.c [plugins_finalize]\n"), 9);
	#endif

	node = dp_data->enabled_list;
	while (node)
	{
		dp = node->data;
		if (dp->finish) {
			dp->finish();
		}
		node = node->next;
	}

	GDK_THREADS_LEAVE();
	while (g_main_iteration(FALSE));
	GDK_THREADS_ENTER();

	dp_data->initialized = FALSE;

	return TRUE;
}
