/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2003 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <string.h>
#include <stdio.h>
#include <gtk/gtkbin.h>
#include <gtk/gtkclist.h>
#include <gtk/gtkscrolledwindow.h>
#include <gdk/gdkkeysyms.h>

#include <xmms/plugin.h>
#include <xmms/xmmsctrl.h>

#include "singit/song.h"

#include "singit_debug.h"
#include "singit_song_private.h"
#include "editor_singit_main.h"
#include "editor_config.h"
#include "editor_status.h"

#include "editor_clist_timestamps.h"

static SingitEditorViewClass *parent_class = NULL;

extern VisPlugin singit_vp;

static gboolean editor_clist_timestamps_keypress(GtkWidget *widget, GdkEventKey *event, gpointer data);
static gboolean editor_clist_clicked_event(GtkWidget *widget, GdkEventButton *event, gpointer user_data);
static void editor_clist_select_row_event(GtkWidget *widget, gint row, gint column, GdkEventButton *event, gpointer user_data);

static void editor_clist_timestamps_destroy (GtkObject *object);
static gchar *editor_clist_timestamps_get_text(SingitEditorView *sev);
static void editor_clist_timestamps_set_text(SingitEditorView *sev, const gchar *text);

static void editor_clist_timestamps_class_init (EditorClistTimestampsClass *klass)
{
	GtkObjectClass *object_klass;
	SingitEditorViewClass *sev_klass;

	object_klass = (GtkObjectClass*) klass;
	sev_klass = (SingitEditorViewClass*) klass;

	parent_class = gtk_type_class(TYPE_SINGIT_EDITOR_VIEW);

	object_klass->destroy = editor_clist_timestamps_destroy;
	sev_klass->get_text = editor_clist_timestamps_get_text;
	sev_klass->set_text = editor_clist_timestamps_set_text;
}

static void editor_clist_timestamps_init (EditorClistTimestamps *ect)
{
	ect->scrolled_window = gtk_scrolled_window_new(NULL, NULL);

	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(ect->scrolled_window),
		GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
	gtk_container_add(GTK_CONTAINER(ect), ect->scrolled_window);
	gtk_widget_show(GTK_WIDGET(ect->scrolled_window));

	ect->clist = gtk_clist_new(1);
	gtk_clist_set_selection_mode(GTK_CLIST(ect->clist), GTK_SELECTION_SINGLE);
	gtk_clist_set_reorderable(GTK_CLIST(ect->clist), FALSE);
	gtk_container_add(GTK_CONTAINER(ect->scrolled_window), ect->clist);
	gtk_clist_set_column_width(GTK_CLIST(ect->clist), 0, 1);

	gtk_signal_connect(GTK_OBJECT(ect->clist),
		"key-press-event", GTK_SIGNAL_FUNC(editor_clist_timestamps_keypress), ect);
	gtk_signal_connect(GTK_OBJECT(ect->clist),
		"button_press_event", GTK_SIGNAL_FUNC(editor_clist_clicked_event), ect);
	gtk_signal_connect(GTK_OBJECT(ect->clist),
		"select_row", GTK_SIGNAL_FUNC(editor_clist_select_row_event), ect);

	gtk_widget_show(ect->clist);
}

GtkType editor_clist_timestamps_get_type (void)
{
	static guint ect_type = 0;

	if (!ect_type) {
		GtkTypeInfo ect_info = {
			(gchar*) "EditorClistTimestamps",
			sizeof (EditorClistTimestamps),
			sizeof (EditorClistTimestampsClass),
			(GtkClassInitFunc) editor_clist_timestamps_class_init,
			(GtkObjectInitFunc) editor_clist_timestamps_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL
		};

		ect_type = gtk_type_unique (TYPE_SINGIT_EDITOR_VIEW, &ect_info);
	}

	return ect_type;
}

static void editor_clist_timestamps_destroy (GtkObject *object)
{
	EditorClistTimestamps *ect;

	g_return_if_fail (ect = EDITOR_CLIST_TIMESTAMPS (object));

	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

static gchar *editor_clist_timestamps_get_text(SingitEditorView *sev)
{
	gint chars = 0, cur_row = 1;
	gchar *result = NULL, *temp, *concat_str;
	EditorClistTimestamps *ect;

#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("editor_clist_timestamps.c [editor_clist_timestamps_get_text]\n"));
#endif

	g_return_val_if_fail(IS_SINGIT_EDITOR_VIEW(sev), NULL);

	ect = EDITOR_CLIST_TIMESTAMPS(sev);

	if (GTK_CLIST(ect->clist)->rows > 0) {
		gtk_clist_get_text(GTK_CLIST(ect->clist), 0, 0, &concat_str);
		result = g_strdup(concat_str);
		if (cur_row < (gint) ect->active_line)
			{ chars = strlen(concat_str) + strlen("\n"); }
		while (cur_row < GTK_CLIST(ect->clist)->rows) {
			gtk_clist_get_text(GTK_CLIST(ect->clist), cur_row, 0, &concat_str);
			if (cur_row < (gint) ect->active_line)
				{ chars += strlen(concat_str) + strlen("\n"); }
			temp = result;
			result = g_strconcat(temp, "\n", concat_str, NULL);
			g_free(temp);
			cur_row++;
		}
		sev->cursor_text_offset = chars;
	}
	else { sev->cursor_text_offset = 0; }

	return result;
}

static void editor_clist_timestamps_set_text(SingitEditorView *sev, const gchar *text)
{
	gchar **lines, *lbr[1][1];
	gint i = 0, line = 0;
	guint chars = 0;
	EditorClistTimestamps *ect;

#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("editor_clist_timestamps.c [editor_clist_timestamps_set_text]\n"));
#endif

	g_return_if_fail(IS_SINGIT_EDITOR_VIEW(sev));

	ect = EDITOR_CLIST_TIMESTAMPS(sev);

	gtk_clist_freeze(GTK_CLIST(ect->clist));
	gtk_clist_clear(GTK_CLIST(ect->clist));
	if (text) {
		lines = g_strsplit(text, "\n", 1000);
		while (lines[i]) {
			lbr[0][0] = lines[i];
			gtk_clist_append(GTK_CLIST(ect->clist), lbr[0]);
			if ((chars < sev->cursor_text_offset) && (i > 0)) {
				chars += strlen(lines[i-1]) + strlen("\n");
				line++;
			}
			i++;
		}
		g_strfreev(lines);
	}
	gtk_clist_thaw(GTK_CLIST(ect->clist));
	if (sev->cursor_text_offset < 0)
		{ line = GTK_CLIST(ect->clist)->rows - 1; }
	ect->active_line = line;
	gtk_clist_moveto(GTK_CLIST(ect->clist), line, 0, 0.5, 0);

#ifdef CODEDEBUG
	if (!gtk_clist_row_is_visible(GTK_CLIST(ect->clist), line)) {
		DEBUG(DLV_ALL, ("Error: line %i not visible!\n", line));
	}
#endif
}

static gchar* get_time_token_string()
{
	gint time;
	gchar *time_stamp = g_new(gchar, 12);

	time = xmms_remote_get_output_time(singit_vp.xmms_session);
	if (time > GET_SCD->reactionTime) { time -= GET_SCD->reactionTime; }
	if (GET_ECD->extendedLyrics) {
		time_stamp[11] = '\0';
		sprintf(time_stamp, "[%.2i:%.2i:%.3i]", time / 60000, time / 1000 % 60, time % 1000);
	}
	else {
		time_stamp[7] = '\0';
		sprintf(time_stamp, "[%.2i:%.2i]", time / 60000, time / 1000 % 60);
	}

	return &time_stamp[0];
}

#ifdef CODEDEBUG
static gchar *print_modifiers(gint state, gboolean compact)
{
	gchar *result = NULL;
	if (compact == FALSE) {
		gchar *lines[15];
		lines[0] = g_strdup_printf("GDK_SHIFT_MASK:   %s\n", (state & GDK_SHIFT_MASK) ? "on" : "off");
		lines[1] = g_strdup_printf("GDK_LOCK_MASK:    %s\n", (state & GDK_LOCK_MASK) ? "on" : "off");
		lines[2] = g_strdup_printf("GDK_CONTROL_MASK: %s\n", (state & GDK_CONTROL_MASK) ? "on" : "off");
		lines[3] = g_strdup_printf("GDK_MOD1_MASK:    %s\n", (state & GDK_MOD1_MASK) ? "on" : "off");
		lines[4] = g_strdup_printf("GDK_MOD2_MASK:    %s\n", (state & GDK_MOD2_MASK) ? "on" : "off");
		lines[5] = g_strdup_printf("GDK_MOD3_MASK:    %s\n", (state & GDK_MOD3_MASK) ? "on" : "off");
		lines[6] = g_strdup_printf("GDK_MOD4_MASK:    %s\n", (state & GDK_MOD4_MASK) ? "on" : "off");
		lines[7] = g_strdup_printf("GDK_MOD5_MASK:    %s\n", (state & GDK_MOD5_MASK) ? "on" : "off");
		lines[8] = g_strdup_printf("GDK_BUTTON1_MASK: %s\n", (state & GDK_BUTTON1_MASK) ? "on" : "off");
		lines[9] = g_strdup_printf("GDK_BUTTON2_MASK: %s\n", (state & GDK_BUTTON2_MASK) ? "on" : "off");
		lines[10] = g_strdup_printf("GDK_BUTTON3_MASK: %s\n", (state & GDK_BUTTON3_MASK) ? "on" : "off");
		lines[11] = g_strdup_printf("GDK_BUTTON4_MASK: %s\n", (state & GDK_BUTTON4_MASK) ? "on" : "off");
		lines[12] = g_strdup_printf("GDK_BUTTON5_MASK: %s\n", (state & GDK_BUTTON5_MASK) ? "on" : "off");
		lines[13] = g_strdup_printf("GDK_RELEASE_MASK: %s\n", (state & GDK_RELEASE_MASK) ? "on" : "off");
		lines[14] = 0;
		result = g_strjoinv("", lines);
		g_strfreev(lines);
	}
	else {
		result = g_strconcat("        SLCM    B    R\nStates: ",
			(state & GDK_SHIFT_MASK) ? "o" : ".",
			(state & GDK_LOCK_MASK) ? "o" : ".",
			(state & GDK_CONTROL_MASK) ? "o" : ".",
			(state & GDK_MOD1_MASK) ? "o" : ".",
			(state & GDK_MOD2_MASK) ? "o" : ".",
			(state & GDK_MOD3_MASK) ? "o" : ".",
			(state & GDK_MOD4_MASK) ? "o" : ".",
			(state & GDK_MOD5_MASK) ? "o" : ".",
			(state & GDK_BUTTON1_MASK) ? "o" : ".",
			(state & GDK_BUTTON2_MASK) ? "o" : ".",
			(state & GDK_BUTTON3_MASK) ? "o" : ".",
			(state & GDK_BUTTON4_MASK) ? "o" : ".",
			(state & GDK_BUTTON5_MASK) ? "o" : ".",
			(state & GDK_RELEASE_MASK) ? "o" : ".",
			"\n", 0);
	}
	return result;
}
#endif

static gboolean editor_clist_clicked_event(GtkWidget *widget, GdkEventButton *event, gpointer user_data)
{
	gchar *oldline = NULL, *newline = NULL;
#ifdef CODEDEBUG
	gchar *modifiers;
#endif
	guint tag_type = 0;
	gint str_len;
	gchar *time_stamp;
	gint row, column;

#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("editor_clist_timestamps.c [insert_remove_timestamp_event]\n"));
#endif

	g_return_val_if_fail(IS_SINGIT_EDITOR_VIEW(user_data), FALSE);

	if (!event) { return TRUE; }

	if (event->type != GDK_BUTTON_PRESS) // skip double and triple click events
		{ return TRUE; }

	if (!gtk_clist_get_selection_info(GTK_CLIST(widget), event->x, event->y, &row, &column))
		{ return TRUE; }

	gtk_clist_get_text(GTK_CLIST(widget), row, column, &oldline);
	switch (event->button) {
	case 1: // Left button
#ifdef CODEDEBUG
		DEBUG_AREA(DLV_ALL,
			debug("%s\n", "Left button"); // To calm the compiler
			modifiers = print_modifiers(event->state, TRUE);
			debug(modifiers);
			g_free(modifiers);
		);
#endif
		if (event->state & GDK_CONTROL_MASK) {
			tag_type = singit_song_get_timetag_type(oldline);
			switch (tag_type) {
			case 1:
				if (strlen(oldline) == 7) { newline = g_strdup("\0"); }
				else { newline = g_strdup(&oldline[7]); }
				gtk_clist_set_text(GTK_CLIST(widget), row, column, newline);
				return FALSE;
			case 2:
				if (strlen(oldline) == 11) { newline = g_strdup("\0"); }
				else { newline = g_strdup(&oldline[11]); }
				gtk_clist_set_text(GTK_CLIST(widget), row, column, newline);
				return FALSE;
			}
		}
		else {
			EDITOR_CLIST_TIMESTAMPS(user_data)->active_line = row;
			time_stamp = get_time_token_string();
			newline = g_strconcat(time_stamp, oldline, NULL);
			gtk_clist_set_text(GTK_CLIST(widget), row, column, newline);
			if (((row+1) < GTK_CLIST(widget)->rows) &&
				!(event->state & (GDK_SHIFT_MASK | GDK_LOCK_MASK)))
			{
				gtk_clist_moveto(GTK_CLIST(widget), (row+1), -1, 0.5, 0);
			}
			g_free(time_stamp);
		}
		break;
	case 3: // Right button
#ifdef CODEDEBUG
		DEBUG_AREA(DLV_ALL,
			debug("%s\n", "Right button"); // To calm the compiler
			modifiers = print_modifiers(event->state, TRUE);
			debug(modifiers);
			g_free(modifiers);
		);
#endif
		if (event->state & GDK_CONTROL_MASK) {
			str_len = strlen(oldline);
			if (str_len >= 7) {
				if ((tag_type = singit_song_get_timetag_type(&oldline[str_len - 7])) != 0) {
					if (strlen(oldline) == 7) { newline = g_strdup("\0"); }
					else {
						oldline[str_len - 7] = '\0';
						newline = g_strdup(oldline);
					}
					gtk_clist_set_text(GTK_CLIST(widget), row, column, newline);
					return FALSE;
				}
				else if ((str_len >= 11) && ((tag_type = singit_song_get_timetag_type(&oldline[str_len - 11])) != 0)) {
					if (strlen(oldline) == 11) { newline = g_strdup("\0"); }
					else {
						oldline[str_len - 11] = '\0';
						newline = g_strdup(oldline);
					}
					gtk_clist_set_text(GTK_CLIST(widget), row, column, newline);
					return FALSE;
				}
			}
		}
		else {
			EDITOR_CLIST_TIMESTAMPS(user_data)->active_line = row;
			time_stamp = get_time_token_string();
			newline = g_strconcat(oldline, time_stamp, NULL);
			gtk_clist_set_text(GTK_CLIST(widget), row, column, newline);
			if (((row+1) < GTK_CLIST(widget)->rows) &&
				!(event->state & (GDK_SHIFT_MASK | GDK_LOCK_MASK)))
			{
				gtk_clist_moveto(GTK_CLIST(widget), (row+1), -1, 0.5, 0);
			}
			g_free(time_stamp);
		}
		break;
	default:
		break;
	}

	singit_editor_view_modify(SINGIT_EDITOR_VIEW(user_data), TRUE);
	return TRUE;
}

static gboolean editor_clist_timestamps_keypress(GtkWidget *widget, GdkEventKey *event, gpointer data)
{
	gboolean result = TRUE;
/*	GdkEventButton *button_event = NULL;
	EditorClistTimestamps *ect = EDITOR_CLIST_TIMESTAMPS(data);*/

	switch(event->keyval) {
/*
		case GDK_Insert:
		case GDK_Delete:
			button_event = g_malloc(sizeof(GdkEventButton));
			if (event->keyval == GDK_Insert) { button_event->state = 0; }
			else { button_event->state = (event->state | GDK_CONTROL_MASK); }
			insert_remove_timestamp_event(ect->clist, GTK_CLIST(ect->clist)->focus_row,
				0, button_event, NULL);
			g_free(button_event);
			break;
*/
/*		case GDK_Up:
		case GDK_Down:
		case GDK_Page_Up:
		case GDK_Page_Down:
			gtk_clist_select_row(GTK_CLIST(ect->clist), GTK_CLIST(ect->clist)->focus_row, 0);
			break;*/
		default:
			result = FALSE;
			// DEBUG(DLV_ALL, ("%x - %x\n", event->state, event->keyval));
			break;
        }

        return result;
}

GtkWidget* editor_clist_timestamps_new(void)
{
	GtkWidget *editor;

#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("editor_clist_timestamps.c [editor_clist_timestamps_new]\n"));
#endif

	editor = gtk_type_new(TYPE_EDITOR_CLIST_TIMESTAMPS);

	return editor;
}

static void editor_clist_select_row_event(GtkWidget *widget, gint row, gint column, GdkEventButton *event, gpointer user_data)
{
	if (!event) { return; }

	// prevent visual focus
	if (GTK_CLIST(widget)->focus_row != -1)
		GTK_CLIST(widget)->focus_row = -1;
	gtk_clist_unselect_row(GTK_CLIST(widget), row, column);
}
