/*
 *	$COPYRIGHT$
 *
 *	$Id: xmpi_misc.cc,v 1.4 2001/07/05 17:02:28 bbarrett Exp $
 *
 *	Function:	- miscellaneous XMPI functions
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include <Xm/Form.h>
#include <Xm/Frame.h>
#include <Xm/Label.h>
#include <Xm/MessageB.h>
#include <Xm/PushB.h>
#include <Xm/SashP.h>
#include <X11/cursorfont.h>

#include "xmpi.h"
#include "xmpi_misc.h"
#include "xpm.h"

#include "Bitmaps/stipple.xbm"


/*
 * local functions
 */
static Pixmap make_stippled(Display*, Drawable, 
			    Widget, Pixel, Pixmap);

static void wait_cb(Widget, int*, XmAnyCallbackStruct*);

/*
 * local variables
 */
static Cursor busycursor = 0;	       /* busy cursor */

/*
 * local macros and defines
 */
#define YES	1
#define NO	2


/*
 *	xmpi_atexit
 *
 *	Function:	- wrapper for atexit()
 */
void
xmpi_atexit(xmpi_atexit_funct funct)
{

#if HAVE_ATEXIT
  atexit(funct);
#else
  on_exit((int ((*) ())) funct, (char *) 0);
#endif
}

/*
 *	xmpi_fail
 *
 *	Function:	- print error message and exit
 *	Accepts:	- error message
 */
void
xmpi_fail(char *msg)
{
  perror(msg);
  exit(errno);
}

/*
 *      xmpi_coll2pt
 *
 *      Function:       - change context ID from collective to pt2pt
 *      Accepts:        - collective context ID
 *      Returns:        - pt2pt context ID
 */
int
xmpi_coll2pt(int cid)
{
  return ((cid >= 0) ? cid : -(cid + 1));
}

/*
 * 	xmpi_flush
 *
 *	Function:	- flush and update display
 */
void
xmpi_flush()
{
  XFlush(XtDisplay(xmpi_shell));
  XmUpdateDisplay(xmpi_shell);
}

/*
 *	xmpi_busy_widget
 *
 *	Function:	- switches windows cursor to a busy indicator
 *	Accepts:	- any widget in the window
 */
void
xmpi_busy_widget(Widget w)
{
  if (busycursor == 0) {
    busycursor = XCreateFontCursor(XtDisplay(w), XC_watch);
  }
  XDefineCursor(XtDisplay(w), XtWindow(w), busycursor);
  xmpi_flush();
}

/*
 *	xmpi_unbusy_widget
 *
 *	Function:	- switches windoews cursor back to normal value
 *	Accepts:	- any widget in the window
 */
void
xmpi_unbusy_widget(Widget w)
{
  XUndefineCursor(XtDisplay(w), XtWindow(w));
}

/*
 *	xmpi_vstr_extent
 *
 *	Function:	- get extent of vertical text
 *	Accepts:	- text string
 *			- text font
 *			- text width (out)
 *			- text height (out)
 *	Returns:	-
 */
void
xmpi_vstr_extent(char *str, XFontStruct font, 
		 int *width, int *height)
{
  int dir;			       /* string direction */

  int ascent;			       /* font ascent */

  int descent;			       /* font descent */

  XCharStruct fmtinfo;		       /* string font info */

  XTextExtents(&font, str, strlen(str),
	       &dir, &ascent, &descent, &fmtinfo);

  *height = fmtinfo.width;
  *width = fmtinfo.ascent + fmtinfo.descent;
}

/*
 *	xmpi_vstr_draw
 *
 *	Function:	- draw vertical text
 *	Accepts:	- display
 *			- drawable text is to be drawn into
 *			- graphics context to draw with
 *			- x position of text in drawable
 *			- y position of text on drawable
 *			- text string
 *			- text font (must be that in the GC)
 */
void
xmpi_vstr_draw(Display *disp, Drawable draw, GC gc,
	       int x, int y, char *str, XFontStruct font)
{
  XImage *hori, *vert;

  int xp, yp;

  Pixmap pix;

  int w, h;

  int dir;			       /* string direction */

  int ascent;			       /* font ascent */

  int descent;			       /* font descent */

  XCharStruct fmtinfo;		       /* string font info */

  XGCValues gcval;		       /* graphic context info */

  Pixel fg;			       /* foreground */

  unsigned long pixel;

  unsigned int depth;

  memset(&gcval, 0, sizeof(gcval));

  depth = DefaultDepth(disp, DefaultScreen(disp));

  XTextExtents(&font, str, strlen(str),
	       &dir, &ascent, &descent, &fmtinfo);

  w = fmtinfo.width;
  h = fmtinfo.ascent + fmtinfo.descent;

  pix = XCreatePixmap(disp, draw, w, h, depth);

  XGetGCValues(disp, gc, GCForeground, &gcval);
  fg = gcval.foreground;

  gcval.foreground = gcval.background;
  XChangeGC(disp, gc, GCForeground, &gcval);

  XFillRectangle(disp, pix, gc, 0, 0, w, h);

  gcval.foreground = fg;
  XChangeGC(disp, gc, GCForeground, &gcval);

  XDrawImageString(disp, pix, gc, 0, fmtinfo.ascent, str, strlen(str));

  hori = XCreateImage(disp,
		      DefaultVisual(disp, DefaultScreen(disp)),
	   depth, XYPixmap, 0, XtMalloc(((w + 7) / 8) * (h + 1) * depth),
		      w, h, 8, 0);

  XGetSubImage(disp, pix, 0, 0, w, h, AllPlanes, XYPixmap, hori, 0, 0);

  vert = XCreateImage(disp,
		      DefaultVisual(disp, DefaultScreen(disp)),
	   depth, XYPixmap, 0, XtMalloc(((h + 7) / 8) * (w + 1) * depth),
		      h, w, 8, 0);

  for (xp = 0; xp < w; xp++) {
    for (yp = 0; yp < h; yp++) {
      pixel = XGetPixel(hori, xp, yp);
      XPutPixel(vert, yp, w - 1 - xp, pixel);
    }
  }

  XPutImage(disp, draw, gc, vert, 0, 0, x, y, h, w);

  XDestroyImage(hori);
  XDestroyImage(vert);
  XFreePixmap(disp, pix);
}

/*
 *	xmpi_mkpixbutton
 *
 *	Function:	- make a button with pixmap
 *			- creates pixmaps for the regular and armed
 *			  button, and for "insensitive" mode if needed
 *			- pixmaps are not created if "bits" is NULL
 *	Accepts:	- parent widget
 *			- activate callback function (or NULL)
 *			- activate argument (or NULL)
 *			- arm callback function (or NULL)
 *			- arm argument (or NULL)
 *			- disarm callback function (or NULL)
 *			- disarm argument (or NULL)
 *			- pixmap bits
 *			- pixmap width
 *			- pixmap height
 *			- insensitive flag
 *	Returns:	- button widget
 */
Widget
xmpi_mkpixbutton(Widget parent_w,  
		 void (*act_cb) (Widget, XtPointer, XtPointer), 
		 char *act_arg, 
		 void (*arm_cb) (Widget, XtPointer, XtPointer), 
		 char *arm_arg, 
		 void (*dis_cb) (Widget, XtPointer, XtPointer),
		 char *dis_arg, unsigned char *pixbits, 
		 int pixwidth, int pixheight, int fl_insensitive)
{
  Widget w;			       /* favourite widget */

  Pixmap pix;			       /* button pixmap */

  Pixmap clipmask;		       /* clipmask bitmap */

  Pixmap stipple;		       /* stipple bitmap */

  Pixel fg, bg, ag;		       /* button colours */

  GC pen;			       /* stipple pen */

  XGCValues val;		       /* pen values */

  Display *display;		       /* display pointer */

  Drawable root;		       /* root of screen */

  unsigned int depth;		       /* depth of screen */

/*
 * Create the button.
 */
  w = XtVaCreateManagedWidget("ctl_btn",
			      xmPushButtonWidgetClass, parent_w,
			      NULL);
/*
 * Set the callback functions.
 */
  if (act_cb)
    XtAddCallback(w, XmNactivateCallback, act_cb, act_arg);
  if (arm_cb)
    XtAddCallback(w, XmNarmCallback, arm_cb, arm_arg);
  if (dis_cb)
    XtAddCallback(w, XmNdisarmCallback, dis_cb, dis_arg);
/*
 * If needed, create/attach the pixmap.
 */
  if (pixbits) {
    XtVaGetValues(parent_w,
		  XmNforeground, &fg, XmNbackground, &bg, NULL);

    display = XtDisplay(parent_w);
    root = RootWindowOfScreen(XtScreen(parent_w));
    depth = DefaultDepthOfScreen(XtScreen(parent_w));

    pix = XCreatePixmapFromBitmapData(display, root,
		   (char *) pixbits, pixwidth, pixheight, fg, bg, depth);

    XtVaSetValues(w, XmNlabelType, XmPIXMAP,
		  XmNlabelPixmap, pix, NULL);
/*
 * Create the armed-mode pixmap.
 */
    XtVaGetValues(w, XmNarmColor, &ag, NULL);

    pix = XCreatePixmapFromBitmapData(display, root,
		   (char *) pixbits, pixwidth, pixheight, fg, ag, depth);

    XtVaSetValues(w, XmNarmPixmap, pix, NULL);
/*
 * Create the insensitive-mode pixmap if needed.
 */
    if (fl_insensitive) {

      pix = XCreatePixmap(display, root,
			  (int) pixwidth, pixheight, depth);

      val.foreground = bg;
      pen = XCreateGC(display, pix, GCForeground, &val);

      XFillRectangle(display, pix, pen,
		     0, 0, pixwidth, pixheight);

      clipmask = XCreateBitmapFromData(display, root,
				  (char *) pixbits, pixwidth, pixheight);

      stipple = XCreateBitmapFromData(display, root,
				    (char *) stipple_bits, stipple_width,
				      stipple_height);

      val.foreground = fg;
      val.background = bg;
      val.fill_style = FillStippled;
      val.stipple = stipple;
      val.clip_mask = clipmask;

      XChangeGC(display, pen,
		GCForeground | GCBackground |
		GCFillStyle | GCStipple | GCClipMask,
		&val);

      XFillRectangle(display, pix, pen,
		     0, 0, pixwidth, pixheight);

      XtVaSetValues(w, XmNlabelInsensitivePixmap, pix, NULL);

      XFreeGC(display, pen);
      XFreePixmap(display, stipple);
      XFreePixmap(display, clipmask);
    }
  }
  return (w);
}

/*
 *	xmpi_mkcolpixbutton
 *
 *	Function:	- make a button with coloured pixmap
 *			- creates pixmaps for the regular and armed
 *			  button, and for "insensitive" mode if needed
 *	Accepts:	- parent widget
 *			- activate callback function (or NULL)
 *			- activate argument (or NULL)
 *			- arm callback function (or NULL)
 *			- arm argument (or NULL)
 *			- disarm callback function (or NULL)
 *			- disarm argument (or NULL)
 *			- pixmap xpm
 *			- insensitive flag
 *	Returns:	- button widget or 0 if pixmap could not be created
 */
Widget
xmpi_mkcolpixbutton(Widget parent_w, 
		    void (*act_cb) (Widget, XtPointer, XtPointer),
		    char *act_arg, 
		    void (*arm_cb) (Widget, XtPointer, XtPointer), 
		    char *arm_arg, 
		    void (*dis_cb) (Widget, XtPointer, XtPointer),
		    char *dis_arg, const char *xpm[],
		    int fl_insensitive)
{
  Widget w;			       /* favourite widget */

  Pixmap pix;			       /* button pixmap */

  Pixel fg, bg;			       /* button colours */

  Display *display;		       /* display pointer */

  Drawable root;		       /* root of screen */

  int width;			       /* width of button pixmap */

  int height;			       /* height of button pixmap */

/*
 * Create the button.
 */
  w = XtVaCreateManagedWidget("ctl_btn",
			      xmPushButtonWidgetClass, parent_w, NULL);
/*
 * Set the callback functions.
 */
  if (act_cb)
    XtAddCallback(w, XmNactivateCallback, act_cb, act_arg);
  if (arm_cb)
    XtAddCallback(w, XmNarmCallback, arm_cb, arm_arg);
  if (dis_cb)
    XtAddCallback(w, XmNdisarmCallback, dis_cb, dis_arg);
/*
 * Create/attach the pixmap.
 */
  display = XtDisplay(parent_w);
  root = RootWindowOfScreen(XtScreen(parent_w));

  XtVaGetValues(parent_w, XmNforeground, &fg, XmNbackground, &bg, NULL);

  xpm_build(w, xpm, bg, &pix, &width, &height);
/*
 * Destroy the widget and bail out if the pixmap couldn't be made.
 */
  if (pix == XmUNSPECIFIED_PIXMAP) {
    XtDestroyWidget(w);
    return (0);
  }
  XtVaSetValues(w, XmNlabelType, XmPIXMAP, XmNlabelPixmap, pix,
		XmNhighlightColor, bg, NULL);
/*
 * Create the insensitive-mode pixmap if needed.
 */
  if (fl_insensitive) {
    pix = make_stippled(display, root, w, bg, pix);
    XtVaSetValues(w, XmNlabelInsensitivePixmap, pix, NULL);
  }
  return (w);
}

/*
 *	xmpi_formattach
 *
 *	Function:	- attach widget to Form (fractionBase) positions
 *			- work around SUN SDK Motif bug
 *			- negative positions are not attached
 *	Accepts:	- widget
 *			- top/bottom/left/right positions
 */
void
xmpi_formattach(Widget w, int top, int bottom, int left, int right)
{
  int frac;			       /* Form fractionBase */

  XtVaGetValues(XtParent(w), XmNfractionBase, &frac, NULL);

  if (top == 0) {
    XtVaSetValues(w, XmNtopAttachment, XmATTACH_FORM, NULL);
  } else if (top > 0) {
    XtVaSetValues(w, XmNtopAttachment, XmATTACH_POSITION,
		  XmNtopPosition, top, NULL);
  }
  if (left == 0) {
    XtVaSetValues(w, XmNleftAttachment, XmATTACH_FORM, NULL);
  } else if (left > 0) {
    XtVaSetValues(w, XmNleftAttachment, XmATTACH_POSITION,
		  XmNleftPosition, left, NULL);
  }
  if (bottom == frac) {
    XtVaSetValues(w, XmNbottomAttachment, XmATTACH_FORM, NULL);
  } else if (bottom > 0) {
    XtVaSetValues(w, XmNbottomAttachment, XmATTACH_POSITION,
		  XmNbottomPosition, bottom, NULL);
  }
  if (right == frac) {
    XtVaSetValues(w, XmNrightAttachment, XmATTACH_FORM, NULL);
  } else if (right > 0) {
    XtVaSetValues(w, XmNrightAttachment, XmATTACH_POSITION,
		  XmNrightPosition, right, NULL);
  }
}

/*
 *	xmpi_mklabel
 *
 *	Function:	- create a label with or without a frame
 *	Accepts:	- parent widget
 *			- label string
 *			- label alignment
 *			- frame flag
 *	Returns:	- label widget
 */

Widget
xmpi_mklabel(Widget parent_w, char *string, int align, int fl_frame)
{
  Widget lbl_w;			       /* label widget */

  Widget frm_w;			       /* frame widget */

  if (string == 0)
    string = (char*) " ";

  if (fl_frame) {
    frm_w = XtVaCreateManagedWidget("label_frame",
				    xmFrameWidgetClass, parent_w,
				    XmNshadowType, XmSHADOW_IN,
				    XmNshadowThickness, 2,
				    NULL);

    parent_w = frm_w;
  }
  lbl_w = XtVaCreateManagedWidget(string,
				  xmLabelWidgetClass, parent_w,
				  XmNalignment, align,
				  NULL);

  return (lbl_w);
}

/*
 *	xmpi_setlabel
 *
 *	Function:	- set the contents of a label
 *	Accepts:	- label widget
 *			- string
 */
void
xmpi_setlabel( Widget lbl_w, char *string)
{
  XtVaSetValues(lbl_w, XtVaTypedArg, XmNlabelString, XmRString,
		string, strlen(string) + 1, NULL);
}

/*
 *	make_stippled
 *
 *	Function:	- create a stippled version of a pixmap
 *	Accepts:	- X display
 *			- root drawable
 *			- pushbutton on which pixmap is to be used
 *			- background colour
 *			- pixmap to be stippled
 *	Returns:	- stippled pixmap
 */
static Pixmap
make_stippled(Display *display, Drawable root, 
	      Widget pushbutton, Pixel bg, Pixmap pix)
{
  static unsigned char stipplebitmap[8] = {0x55, 0xAA, 0x55, 0xAA,
  0x55, 0xAA, 0x55, 0xAA};

  Pixmap stipple;

  Pixmap inspix;

  GC gc;

  XGCValues gcvalues;

  Window rootwindow;

  int x, y;

  unsigned int width, height;

  unsigned int border;

  unsigned int depth;

  stipple = XCreateBitmapFromData(display, root, (char *) stipplebitmap, 8, 8);

  gcvalues.foreground = bg;
  gcvalues.fill_style = FillStippled;
  gcvalues.stipple = stipple;

  gc = XtGetGC(pushbutton, GCForeground | GCFillStyle | GCStipple,
	       &gcvalues);

  XGetGeometry(display, pix, &rootwindow, &x, &y, &width, &height,
	       &border, &depth);

  inspix = XCreatePixmap(display, root, width, height, depth);

  XCopyArea(display, pix, inspix, gc, 0, 0, width, height, 0, 0);

  XFillRectangle(display, inspix, gc, 0, 0, width, height);

  XtReleaseGC(pushbutton, gc);
  XFreePixmap(display, stipple);

  return (inspix);
}

/*
 *	xmpi_nosash
 *
 *	Function:	- removes sashes from paned manager
 *	Accepts:	- paned widget
 */
void
xmpi_nosash(Widget paned)
{
  Widget *pchildren;		       /* children of paned mgr */

  int nchildren;		       /* # paned children */

/*
 * Remove separators and make sashes microscopic.
 */

#if 0
  XtVaSetValues(paned,
		XmNsashWidth, 1,
		XmNsashHeight, 1,
		XmNseparatorOn, False,
		NULL);
#endif

/*
 * Remove the sashes from tab traversal.
 */
  XtVaGetValues(paned,
		XmNchildren, &pchildren,
		XmNnumChildren, &nchildren,
		NULL);

  while (nchildren-- > 0) {

    if (XmIsSash(pchildren[nchildren])) {
      XtVaSetValues(pchildren[nchildren],
		    XmNtraversalOn, False,
		    NULL);
    }
  }
}

/*
 *	xmpi_yesno
 *
 *	Function:	- pop up dialog asking a yes/no type question, wait
 *			  for and return the users response
 *	Accepts:	- parent widget
 *			- text of the question
 *	Returns:	- 1 if user responds YES, else 0
 */
int
xmpi_yesno(Widget parent, char *question)
{
  Widget dialog_w;		       /* question dialog */

  XmString str1, str2, str3;	       /* various strings */

  Arg args[5];

  int response = 0;		       /* response to question */

  str1 = XmStringCreateSimple(question);
  str2 = XmStringCreateSimple((char*) "Yes");
  str3 = XmStringCreateSimple((char*) "No");

  XtSetArg(args[0], XmNmessageString, str1);
  XtSetArg(args[1], XmNdialogStyle, XmDIALOG_FULL_APPLICATION_MODAL);
  XtSetArg(args[2], XmNautoUnmanage, False);
  XtSetArg(args[3], XmNokLabelString, str2);
  XtSetArg(args[4], XmNcancelLabelString, str3);

  dialog_w = XmCreateQuestionDialog(xmpi_wmshell(parent),
				    (char*) "wait_pop", args, 5);

  XmStringFree(str1);
  XmStringFree(str2);
  XmStringFree(str3);

  XtAddCallback(dialog_w, XmNokCallback, 
		(XtCallbackProc) wait_cb, &response);
  XtAddCallback(dialog_w, XmNcancelCallback, 
		(XtCallbackProc) wait_cb, &response);

  XtUnmanageChild(XmMessageBoxGetChild(dialog_w, XmDIALOG_HELP_BUTTON));
  XtManageChild(dialog_w);
  XtPopup(XtParent(dialog_w), XtGrabNone);

/*
 * Wait until a response has been selected.
 */
  while (!response) {
    XtAppProcessEvent(XtWidgetToApplicationContext(parent),
		      XtIMAll);
  }

  return (response == YES);
}

/*
 *	wait_cb
 *
 *	Function:	- set users response
 *			  callback for xmpi_yesno
 *	Accepts:	- callback parms
 */
static void
wait_cb(Widget w, int  *answer, XmAnyCallbackStruct *cbs)
{
  switch (cbs->reason) {

  case XmCR_OK:
    *answer = YES;
    break;
  default:
    *answer = NO;
  }

  XtDestroyWidget(w);
}
