/*

    xpuyopuyo - pmanip.c      Copyright(c) 1999,2000 Justin David Smith
    justins(at)chaos2.org     http://chaos2.org/
    
    Manipulation of the playing field.
    

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


#include <stdio.h>
#include <stdlib.h>
#include <config.h>
#include <pmanip.h>


static void p_manip_clear_counted(pfield *field) {
/* p_manip_clear_counted */
   
   int size;
   int *p;
   
   size = P_FIELD_SIZE(field);
   p = P_FIELD_XY_P(field, 0, 0);
   while(size > 0) {
      *p = *p & (~P_FLAG_COUNTED);
      ++p;
      --size;
   }

}
   
   
int p_manip_num_adjacent(pfield *field, int x, int y, int color) {
/* p_manip_num_adjacent

   This determines the number of cells adjacent to (x, y) of the same
   color as <color>.  If (x, y) are outside the field, zero is returned.
   If the blob at position (x, y) is not the same color as specified
   in color, then zero is also returned.  Otherwise, one is added to
   the return value and the function recursively calls itself on its
   neighbors.  In this fashion, if (x, y) is part of a chain of the same
   color, then all members of that chain are counted and returned (so
   this function goes beyond counting the immediate members).  This
   function needs write access to pfield so it can temporarily change
   the color value at (x, y).  */

   int sum;       /* Number of members of this chain */
   int *cell;     /* Pointer to the current cell */

   /* Check to see if (x, y) is within its boundaries */
   if(x < 0 || y < 0 || x >= field->width || y >= field->height) return(0);
   
   sum = 0;
   cell = P_FIELD_XY_P(field, x, y);

   if(!(*cell & P_FLAG_COUNTED) && P_IS_COLOR(*cell) && P_COLOR_VALUE(*cell) == P_COLOR_VALUE(color)) {
      /* Count in all directions, somewhat like a floodfill :) */
      sum++;
      *cell = *cell | P_FLAG_COUNTED;
      sum += p_manip_num_adjacent(field, x - 1, y, color);
      sum += p_manip_num_adjacent(field, x + 1, y, color);
      sum += p_manip_num_adjacent(field, x, y - 1, color);
      sum += p_manip_num_adjacent(field, x, y + 1, color);
   }
   return(sum);

}


static int p_manip_explode(pfield *field, int x, int y, int color) {
/* p_manip_explode

   This function explodes a chain of the color <color>, starting at
   coordinate (x, y).  The process of exploding replaces all members
   of the chain with the cell P_STAR.  A later call can replace the
   P_STAR's created with P_CLEAR tiles.  If the cell at (x, y) does
   not match the color, nothing occurs and zero is returned.  Else,
   the number of exploded tiles is returned.  (x, y) must be within
   the bounds of the field.  */

   int sum;
   int *cell;

   /* Check boundaries */
   if(x < 0 || y < 0 || x >= field->width || y >= field->height) return(0);

   sum = 0;
   cell = P_FIELD_XY_P(field, x, y);
   if(P_IS_STAR(*cell)) {
      /* Do nothing.  This cell is already an explosion */
      /* CAUTION the test below will succeed with P_STAR types, this
         is very undesirable! */
   } else if(P_IS_ROCK(*cell)) {
      /* Rocks may be destroyed, but we cannot continue past a rock.
         Rocks should not be counted in the final tally to decide how
         many rocks to dump on the opponent, FYI  :)  --  use the sum
         from p_manip_num_adjacent to determine that!   */
      sum++;
      #ifdef DEBUG_FIELD
      printf("%d (field):  Annihilated rock  %8x at (%2d, %2d)\n", field->number, *cell, x, y);
      #endif
      /* CAUTION YOU MUST READ COMMENTS WITH p_manip_clear_explosions BEFORE
         MODIFYING THIS, THIS _MUST_ BE SET TO P_ROCK_STAR!!! */
      *cell = P_ROCK_STAR;
      p_field_redraw_point(field, x, y);
   } else if(P_IS_COLOR(*cell) && P_COLOR_VALUE(*cell) == P_COLOR_VALUE(color)) {
      /* Colors match.  Call recursively to deal with all neighbours. */
      sum++;
      #ifdef DEBUG_FIELD
      printf("%d (field):  Annihilated block %8x at (%2d, %2d)\n", field->number, *cell, x, y);
      #endif
      *cell = P_STAR;
      p_field_redraw_point(field, x, y);
      sum += p_manip_explode(field, x - 1, y, color);
      sum += p_manip_explode(field, x + 1, y, color);
      sum += p_manip_explode(field, x, y - 1, color);
      sum += p_manip_explode(field, x, y + 1, color);
   }
   return(sum);

}


static int p_manip_clear(pfield *field, int x, int y, int color) {
/* p_manip_clear

   This clears a chain matching the same color as <color>, starting at
   (x, y).  The total number of tiles cleared is returned.  This function
   is otherwise very similar to the above function.  */

   int sum;
   int *cell;

   /* Check bounds */
   if(x < 0 || y < 0 || x >= field->width || y >= field->height) return(0);

   sum = 0;
   cell = P_FIELD_XY_P(field, x, y);
   if(P_IS_STAR(*cell)) {
      /* Do nothing for explosions */
      /* CAUTION the test below will succeed with P_STAR types, this
         is very undesirable! */
   } else if(P_IS_ROCK(*cell)) {
      /* Clear rocks, but do not continue in recursion */
      sum++;
      *cell = P_CLEAR;
      p_field_redraw_point(field, x, y);
   } else if(P_IS_COLOR(*cell) && P_COLOR_VALUE(*cell) == P_COLOR_VALUE(color)) {
      /* Clear this cell and continue in the recursion */
      sum++;
      *cell = P_CLEAR;
      p_field_redraw_point(field, x, y);
      sum += p_manip_clear(field, x - 1, y, color);
      sum += p_manip_clear(field, x + 1, y, color);
      sum += p_manip_clear(field, x, y - 1, color);
      sum += p_manip_clear(field, x, y + 1, color);
   }
   return(sum);

}


int p_manip_explode_matches(pfield *field, int mintomatch) {
/* p_manip_explode_matches

   This function "detonates" all matches currently existing in the playing
   field.  Any chain of 4 or more matching blobs will be counted and replaced
   by P_STAR tiles.  The total number of matches found is returned.  This
   function is NOT counting the total number of blobs removed.  If nonzero
   is returned, the field will need to be cleared with p_manip_clear_explosions,
   you will need to collapse the field with p_field_collapse/
   p_field_collapse_once, and you will need to recall this function in the case
   that more matches may have formed during the collapse.  */

   int i;
   int j;
   int sum;       /* Number of matches found */
   int score;     /* Number of blobs in current chain */
   int color;     /* Current color we are working with */

   sum = 0;

   j = field->height;
   while(j) {
      j--;
      i = field->width;
      while(i) {
         i--;
         color = P_FIELD_XY(field, i, j);
         p_manip_clear_counted(field);
         score = p_manip_num_adjacent(field, i, j, color);
         if(score >= mintomatch) {
            /* Chain of length >= 4 found, explode it! */
            /* FIXME  Should we make the value four configurable?  */
            p_manip_explode(field, i, j, color);
            sum++;
         }
      }
   }

   #ifdef DEBUG_FIELD
   printf("%d (field):  In explosion, found %d matches\n", field->number, sum);
   #endif
   return(sum);

}


int p_manip_clear_explosions(pfield *field) {
/* p_manip_clear_explosions

   This functions clears _explosions_ and returns the total
   number of colored blobs exploded (not counting rocks).
   This should be called after a healthy call to 
   p_manip_explode_matches.  The value returned here is the
   one you want to use when deciding how many rocks to dump
   on the opponent.  NOTE, in particular, that there are
   two types of STARS.  The P_ROCK_STAR is used for rocks
   that were exploded; these should not be counted in the
   final sum.   Be careful when working with above two 
   functions for this reason,  rocks must be exploded to
   the P_ROCK_STAR type, and these tiles will be picked up
   by both the P_IS_STAR() macro and the P_IS_ROCK() macro!  */

   int i;
   int j;
   int sum;
   int *cell;

   sum = 0;

   for(j = 0; j < field->height; j++) {
      for(i = 0; i < field->width; i++) {
         cell = P_FIELD_XY_P(field, i, j);
         if(P_IS_STAR(*cell)) {
            /* Explosion found, clear it and count it if it wasn't a rock. */
            if(!P_IS_ROCK(*cell)) sum++;
            *cell = P_CLEAR;
            p_field_redraw_point(field, i, j);
         }
      }
   }

   #ifdef DEBUG_FIELD
   printf("%d (field):  In clearing, found %d colored blobs (not rocks)\n", field->number, sum);
   #endif
   return(sum);

}


int p_manip_clear_matches(pfield *field, int mintomatch) {
/* p_manip_clear_matches

   This function clears matches from the field.  It is a shortcut
   for the above two functions if you don't want to display explosions
   at all but want to immediately clear the matches (useful for the
   AI simulations, in particular).  The value returned here is the
   same as p_manip_explode_matches;  if nonzero, you still need to
   call one of the collapse functions and then re-call this function. */

   int i;
   int j;
   int sum;       /* Total number of non-rocks cleared */
   int score;     /* Number of blobs in current chain */
   int color;     /* Current color we are wroking with */

   sum = 0;

   j = field->height;
   while(j) {
      j--;
      i = field->width;
      while(i) {
         i--;
         color = P_FIELD_XY(field, i, j);
         score = p_manip_num_adjacent(field, i, j, color);
         if(score >= mintomatch) {
            /* Match found, clear it and count it! */
            p_manip_clear(field, i, j, color);
            sum++;
         }
      }
   }

   return(sum);

}


int p_manip_collapse_once(pfield *field) {
/* p_manip_collapse_once

   Each blob in the playing field which is "hovering"
   is dropped one row vertically.  This should be used
   for the visible playing fields, as it adds animation
   to the blobs as they fall (they don't just immediately
   drop 10 rows down :).  Returned is the total number
   of blobs that fell, or zero if no blobs fell.  This
   function should be called repeatedly until it returns
   zero.  */

   int i;
   int j;
   int width;
   int fell;
   int *src;
   int *dest;
   
   fell  = 0;
   width = field->width;

   for(i = 0; i < field->width; i++) {
      j = field->height - 1;
      src = P_FIELD_XY_P(field, i, j);
      while(j >= 0 && *src != P_CLEAR) {
         src -= width;
         j--;
      }
      while(j >= 0) {
         if(*src != P_CLEAR) {
            fell++;
            dest = src + width;
            *dest = *src;
            *src = P_CLEAR;
            p_field_redraw_rect(field, i, j, i, j + 1);
         }
         src -= width;
         j--;
      }
   }

   return(fell);

}


int p_manip_collapse(pfield *field) {
/* p_manip_collapse

   Collapses everything at once.  Useful for internal functions,
   where animating the blobs while they fall is not needed.  This
   function returns the total number of blobs that fell.  This
   function does not need to be called recursively;  every blob
   will be in its proper place once this function completes.  */

   int i;
   int j;
   int width;
   int *src;
   int *dest;
   int fell;

   fell = 0;   
   width = field->width;

   for(i = 0; i < field->width; i++) {
      j = field->height - 1;
      src = P_FIELD_XY_P(field, i, j);
      while(j >= 0 && *src != P_CLEAR) {
         src -= width;
         j--;
      }
      dest = src;
      while(j >= 0) {
         if(*src != P_CLEAR) {
            fell++;
            *dest = *src;
            *src = P_CLEAR;
            p_field_redraw_rect(field, i, j, i, j + field->height);
            dest -= width;
         }
         src -= width;
         j--;
      }
   }

   return(fell);

}


void p_manip_lock(pfield *field) {
/* p_manip_lock

   The current piece actually "floats" on top of the playing field until this
   function is called.  This function locks the piece into the field permanently.
   Once locked, the piece cannot be unlocked and the player will no longer be
   able to control it, so don't call this until you're sure (preferably when one
   of the blobs hits an obstacle, or possibly when the player pushes down).
   Once the piece is locked, you will still need to call a collapse function,
   as one or both of the blobs mihgt be hovering.  */

   int x;
   int y;
   int i;
   int j;
   int width;
   int height;
   int *src;
   int *dest;
   
   x = field->piece->x;
   y = field->piece->y;
   width = field->piece->width;
   height = field->piece->height;
   src = field->piece->data;
   dest = P_FIELD_XY_P(field, x, y);
   
   for(j = 0; j < height; j++) {
      for(i = 0; i < width; i++) {
         if(*src != P_CLEAR) *dest = *src;
         dest++;
         src++;
      }
      dest += (field->width - width);
   }

}


void p_manip_update_joins(pfield *field) {
/* p_manip_update_joins

   Update the joins between blobs.  This is best called 
   sometime shortly after all matches have been cleared
   away, and before the next piece is placed on the field. */

   int i;
   int j;
   int width;
   int height;
   int *src;
   int color1;
   int color2;

   src = field->data;
   width = field->width;
   height = field->height;
   j = 0;
   while(j < height) {
      i = 0;
      while(i < width) {
         color1 = *src;
         if(P_IS_COLOR(color1)) {
            color1 = ((~P_JOIN_MASK) & color1);
            color2 = P_COLOR_VALUE(color1);
            if(i > 0          && P_COLOR_VALUE(*(src - 1)) == color2)      color1 = color1 | P_JOIN_WEST;
            if(j > 0          && P_COLOR_VALUE(*(src - width)) == color2)  color1 = color1 | P_JOIN_NORTH;
            if(i < width - 1  && P_COLOR_VALUE(*(src + 1)) == color2)      color1 = color1 | P_JOIN_EAST;
            if(j < height - 1 && P_COLOR_VALUE(*(src + width)) == color2)  color1 = color1 | P_JOIN_SOUTH;
            if(color1 != *src) {
               *src = color1;
               p_field_redraw_point(field, i, j);
            }
         }
         src++;
         i++;
      }
      j++;
   }

}


