/* $Header: /fridge/cvs/xscorch/sgame/sshield.c,v 1.31 2001/08/15 05:34:35 jacob Exp $ */
/*
   
   xscorch - sshield.c        Copyright(c) 2001,2000 Justin David Smith
                              Copyright(c) 2001,2000 Jacob Luna Lundberg
                              Copyright(c) 2001 Jacob Post
   justins(at)chaos2.org      http://chaos2.org/
   jacob(at)chaos2.org        http://chaos2.org/~jacob

   Scorched shields
    

   This program is free software; you can redistribute it and/or modify 
   it under the terms of the GNU General Public License as published by 
   the Free Software Foundation; either version 2 of the License, or 
   (at your option) any later version.

   This program is distributed in the hope that it will be useful, 
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation, 
   Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <stdlib.h>
#include <sconfig.h>
#include <sshield.h>
#include <saccessory.h>
#include <sexplosion.h>
#include <stankpro.h>
#include <sphysics.h>
#include <splayer.h>
#include <sland.h>
#include <sutil/srand.h>



sc_shield *sc_shield_new(sc_accessory_info *acc) {
/* sc_shield_new
   Create a new shield. */

   sc_shield *sh;
   
   if(!SC_ACCESSORY_IS_SHIELD(acc)) return(NULL);
   
   sh = (sc_shield *)malloc(sizeof(sc_shield));
   if(sh == NULL) return(NULL);

   sh->info = acc;
   sh->life = acc->shield;
   
   return(sh);

}



void sc_shield_free(sc_shield **sh) {
/* sc_shield_free
   Free an old (dead) shield. */

   if(sh == NULL || *sh == NULL) return;
   free(*sh);
   *sh = NULL;

}



sc_accessory_info *sc_shield_find_best(const sc_config *c, const sc_player *p) {
/* sc_shield_find_best
   Find the best shield in a player's inventory.
   This is very accurate, but also very sensitive to shield definition changes. */

   int count;        /* Iterator variable */
   sc_accessory_info *info, *best = NULL;

   /* Sanity check */
   if(c == NULL || p == NULL) return(false);

   /* Search for the best shield in inventory */
   count = sc_accessory_count(c->accessories, SC_ACCESSORY_LIMIT_ALL);
   info = sc_accessory_last(c->accessories, SC_ACCESSORY_LIMIT_ALL);
   for(; count > 0; --count) {
      if(SC_ACCESSORY_IS_SHIELD(info) && info->inventories[p->index] > 0) {
         if(best == NULL) {
            best = info;
         } else {
            if(info->shield > best->shield) {
               best = info;
            } else if(info->shield == best->shield) {
               if(SC_ACCESSORY_SHIELD_IS_FORCE(info)) {
                  best = info;
               } else if(SC_ACCESSORY_SHIELD_IS_STANDARD(info) && SC_ACCESSORY_SHIELD_IS_MAGNETIC(best)) {
                  best = info;
               } else if(SC_ACCESSORY_SHIELD_IS_MAGNETIC(info) && SC_ACCESSORY_SHIELD_IS_MAGNETIC(best)) {
                  if(info->repulsion > best->repulsion) {
                     best = info;
                  } /* shield has more repulsion? */
               } /* shield better by type? */
            } /* shield obviously better, or just maybe better? */
         } /* found a shield already? */
      } /* shield we own? */
      info = sc_accessory_prev(c->accessories, info, SC_ACCESSORY_LIMIT_ALL);
   }

   /* done */
   return(best);

}



void sc_shield_init_turn(struct _sc_player *p) {
/* sc_shield_init_turn
   Prepare a player's shield for the next turn. */

   /* How can you expect us to recharge a nonexistant shield? */
   if(p == NULL || p->shield == NULL) return;

   /* If the player has a shield recharger (like a Solar Panel), recharge! */
   if(p->ac_state & SC_ACCESSORY_STATE_RECHARGE) {
      p->shield->life += SC_SHIELD_RECHARGE_RATE;
      if(p->shield->life > p->shield->info->shield)
         p->shield->life = p->shield->info->shield;
   }

}



bool sc_shield_would_impact(const sc_config *c, const sc_player *owner, const sc_player *p, int traj_flags, double x, double y, double nextx, double nexty) {
/* sc_shield_would_impact
   When a shield will, to the best of our knowledge, take a hit in the
   next trajectory step.  We look to see that we will be passing from
   the outside of a player's shield to the inside. */

   double dx, dy, nextdx, nextdy, rad2, nextrad2;

   /* Sanity checks */   
   if(c == NULL || p == NULL) return(false);

   /* Make sure we have a shield here */
   if(p->shield == NULL)
      return(false);

   /* Make sure the shield is absorptive, unless we must impact it anyway */
   if(!(traj_flags & SC_TRAJ_HIT_SHIELDS) && SC_ACCESSORY_SHIELD_IS_MAGNETIC(p->shield->info))
      return(false);

   /* A player's own shield is disallowed from stopping their missiles.
      I guess you've gotta let them past in order to fire them after all... */
   if(owner != NULL && p->index == owner->index) return(false);

   /* Compute the distance to the missile's present and future. */
   if(!sc_land_calculate_deltas_d(c->land, &dx, &dy, (double)p->x, (double)p->y, x, y) ||
      !sc_land_calculate_deltas_d(c->land, &nextdx, &nextdy, (double)p->x, (double)p->y, nextx, nexty))
      return(false);

   /* Square radii, whee... */
   rad2 = SQR(dx) + SQR(dy);
   nextrad2 = SQR(nextdx) + SQR(nextdy);

   /* We must start outside the tank's shield range, and end up inside it to impact it. */
   return((rad2 >= (double)SQR(p->tank->radius + 1)) && (nextrad2 < (double)SQR(p->tank->radius + 1)));

}



bool sc_shield_absorb_hit(sc_player *p, bool sapper) {
/* sc_shield_absorb_hit
   When a shield takes a direct hit. */

   /* Don't accept no substitutes! */
   if(p->shield == NULL)
      return(false);

   /* Shield sappers take more life out of a shield. */
   if(sapper)
      p->shield->life -= SC_SHIELD_ABSORB_HIT * SC_SHIELD_SAPPER_RATE;
   else
      p->shield->life -= SC_SHIELD_ABSORB_HIT;

   /* If the shield was completely obliterated, get rid of it. */
   if(p->shield->life <= 0)
      sc_shield_free(&p->shield);

   return(true);

}



int sc_shield_absorb_explosion(sc_player *p, const sc_explosion *e, int damage) {
/* sc_shield_absorb_explosion
   Try to absorb damage into a player's shield.  The damage undealt
   is returned, or zero if all damage absorbed.  */

   /* Must have a shield for it to take damage */
   if(p->shield == NULL) return(damage);

   /* Find out how much of the damage it took */
   p->shield->life -= damage;
   if(p->shield->life <= 0) {
      damage = -p->shield->life;
      sc_shield_free(&p->shield);
      if(e->type == SC_EXPLOSION_NAPALM) {
         damage = 0;
      }
   } else {
      damage = 0;
   }

   /* Return damage to the actual tank */
   return(damage);

}



bool sc_shield_get_deflection(sc_config *c, const sc_player *owner, int traj_flags, double x, double y, double *vx, double *vy) {
/* sc_shield_get_deflection
   Find the total deflection in velocity of a missile by magnetic shields. */

   double dx, dy, sdist;
   double force = 0, vdx = 0, vdy = 0;
   bool changed = false;
   sc_player *p;
   int index;

   /* Look at each player. */
   for(index = 0; index < c->numplayers; ++index) {
      p = c->players[index];

      /* There are THREE continue statements below. */

      /* A player's mag shield is tuned so as not to affect his own weapons. */
      if(p == NULL || p->index == owner->index)
         continue;  /* NOTE this is a flow control point! */

      /* Determine whether this player has a magnetic shield. */
      if(p->shield == NULL || !SC_ACCESSORY_SHIELD_IS_MAGNETIC(p->shield->info))
         continue;  /* NOTE this is a flow control point! */

      /* Try and get a proper distance estimate. */
      if(!sc_land_calculate_deltas_d(c->land, &dx, &dy, (double)p->x, (double)p->y, x, y))
         continue;  /* This is a rare (perhaps nonexistant) failure case. */

      /* Determine whether the tank is close enough, but not too close. */
      sdist = SQR(dx) + SQR(dy);
      if(sdist < SQR(SC_SHIELD_MAG_MAX_DIST) && sdist > SQR(p->tank->radius)) {
         /* The weapon and tank are in range, so push the weapon. */
         changed = true;

         /*
            Find the force to be imparted to the missile, and scale it.
            We avoid using the evil sqrt() function by performing most
            of the calculations here working with squares.

            This is where the LORC reside.  :)  To be specific:

            o  The repulsion strength is p->shield->info->repulsion.
            o  The repulsion strength must be attenuated by distance.
               To do this, we use the square of the distance, which
               is stored in sdist, and the attenuation rate constant,
               from sshield.h, SC_SHIELD_MAG_ATTENUATION.
            o  Finally we scale the result to units of tank power with
               sshield.h's SC_SHIELD_MAG_TO_POWER constant, and from
               there to velocity units with SC_PHYSICS_VELOCITY_SCL,
               from sphysics.h.

            The result will be multiplied by the squares of the x and y
            distances and divided by the square of the total distance
            to get velocity differentials which can then be added into
            the current weapon velocities, *vx and *vy.
         */
         force  = p->shield->info->repulsion;
         force *= (double)SQR(SC_SHIELD_MAG_ATTENUATION) / sdist;
         force *= (double)(SC_SHIELD_MAG_TO_POWER * SC_PHYSICS_VELOCITY_SCL);

         /* 1. Set the new x velocity, accelerated by the magnetic shield. */
         vdx = force * SQR(dx) / sdist;
         if(dx > 0)
            *vx += vdx;
         else
            *vx -= vdx;

         /* 2. Set the new y velocity, accelerated by the magnetic shield. */
         vdy = force * SQR(dy) / sdist;
         if(dy > 0)
            *vy += vdy;
         else
            *vy -= vdy;

         /* 3. Sap energy off the player's shield, based on the acceleration given. */
         if(!(traj_flags & SC_TRAJ_NO_MODIFY)) {
            p->shield->life -= rint((vdx + vdy) / (double)(SC_PHYSICS_VELOCITY_SCL * SC_SHIELD_MAG_TO_COST));
            if(p->shield->life <= 0) {
               sc_shield_free(&p->shield);
            }
         }
      }
   }

   return(changed);

}



bool sc_shield_get_reflection(sc_config *c, const sc_player *owner, sc_player *p, int traj_flags, double *x, double *y, double *velx, double *vely) {
/* sc_shield_get_reflection
   Find the new reflection in velocity of a missile off a force shield. */

   double dx, dy, hand, dot, hyprod, comml, commr, tx;

   /* This is only for force shields. */
   if(p->shield == NULL || !SC_ACCESSORY_SHIELD_IS_FORCE(p->shield->info))
      return(false);

   /* If calculate_deltas fails somehow, so do we. */
   if(!sc_land_calculate_deltas_d(c->land, &dx, &dy, (double)p->x, (double)p->y, (*x), (*y)))
      return(false);

   /*
      Reflect the missile.
      This is an excerpt from Jacob Post's quick solution documentation:

      Let's say we have initial velocity <vx, vy> and impact at <dx, dy>
      relative to the center of the tank.  We want reflected velocity
      <x, y>.

      Basically, if you want to rotate by theta radians counterclockwise
      (left-handedness), you do a matrix multiplication:

      / x \ = / cos(theta)     -sin(theta) \ / vx \
      \ y /   \ sin(theta)      cos(theta) / \ vy /

      x = vx*cos(theta) - vy*sin(theta)
      y = vx*sin(theta) + vy*cos(theta)

      BUT beware as the above is a tail rotation, like so:

      ------->
      \
       \
        \
         _|

      And <vx, vy>'s HEAD connects to <x, y>'s TAIL:

      <---------
      \
       \
        \
         _|

      So we need to reverse <vx, vy>.

      x = -(vx*cos(theta) - vy*sin(theta))
      y = -(vx*sin(theta) + vy*cos(theta))

      Now, what is theta?  Well - based off of the scanned sketch - it's
      either 2*omega or -2*omega, based on the handedness of
      <-vx, -vy> ==> <x, y>.

      I've stipulated that this is the same as the handedness of
      <dx, dy>  ==> <vx, vy>
      I can explain this graphically, and will upon request.

      Jacob tells me we find the handedness by examining
      <dx, dy> x <vx, vy> = dx*vy - vx*dy

      So, that's what I've done.

      So, if we're left-hand, theta =  2*omega, and therefore:
        x = -(vx*cos( 2*omega) - vy*sin( 2*omega))
        y = -(vx*sin( 2*omega) + vy*cos( 2*omega))

      If we're right-hand,    theta = -2*omega, and therefore:
        x = -(vx*cos(-2*omega) - vy*sin(-2*omega))
        y = -(vx*sin(-2*omega) + vy*cos(-2*omega))

      But cos(-u) = cos(u) and sin(-u) = -sin(u).  So,
        x = -(vx*cos( 2*omega) + vy*sin( 2*omega))
        y = -(vy*cos( 2*omega) - vx*sin( 2*omega))

      So, as shown above in the comments, I set
      A' = sin( 2*omega)
      B' = cos( 2*omega)

      and fussed about with trig until I came up with

      A' = [2*(dy*vx - dx*vy)*(dx*vx + dy*vy)] / [(dx*dx + dy*dy)*(vx*vx + vy*vy)]
      B' = [(dx*vx+dy*vy)^2 - (dy*vx-dx*vy)^2] / [(dx*dx + dy*dy)*(vx*vx + vy*vy)]

      which is quite messy, yes.  But there's a lot of common elements, which is
      why I chose

      M = dy*vx - dx*vy
      N = dx*vx + dy*vy
      Z = (dx*dx + dy*dy)*(vx*vx + vy*vy)

      And then
      A = ZA' = 2*N*M;
      B = ZB' = N*N - M*M

      Z is "pulled out" because for x and y, we're always adding A' and B'
      with some coefficients, so the end result has denominator Z.

      Finally, I use the handedness to determine how to combine
      vx, vy, A, B, and Z to find x and y.
   */

   /* Find the handedness of the reflection. */
   hand = dx * (*vely) - (*velx) * dy;

   if(hand == 0) {
      /* Special case out the mirror image reflections. */
      *velx = -(*velx);
      *vely = -(*vely);
   } else {
      /* No special case means lots more math...  :) */

      /* Set up some useful constants. */
      dot = dx * (*velx) + dy * (*vely);
      hyprod = (SQR(dx) + SQR(dy)) * (SQR(*velx) + SQR(*vely));
      comml = SQR(dot) - SQR(hand);
      commr = -2.0 * dot * hand;

      /* Calculate the reflection based on the handedness. */
      if(hand > 0) {
         /* left-handedness */
         tx      = -((*velx) * comml - (*vely) * commr) / hyprod;
         (*vely) = -((*vely) * comml + (*velx) * commr) / hyprod;
         (*velx) = tx;
      } else {
         /* right-handedness */
         tx      = -((*velx) * comml + (*vely) * commr) / hyprod;
         (*vely) = -((*vely) * comml - (*velx) * commr) / hyprod;
         (*velx) = tx;
      }
   }

   /* Move the weapon just (0.1 pixels) outside the shield (p->tank->radius + 1).
      It's difficult to get gcc to do this with the required precision.  :(  */
   *x = (double)p->x + dx * SQR((double)1.1 + (double)p->tank->radius) / (SQR(dx) + SQR(dy));
   *y = (double)p->y + dy * SQR((double)1.1 + (double)p->tank->radius) / (SQR(dx) + SQR(dy));

   /* Sap the shield. */
   if(!(traj_flags & SC_TRAJ_NO_MODIFY))
      sc_shield_absorb_hit(p, (traj_flags & SC_TRAJ_HIT_SHIELDS));

   return(true);

}
