/*
   
   libj - reg.c               Copyright(c) 2001-2000 Justin David Smith
   justins(at)chaos2.org      http://chaos2.org/
    
   Config file processing
    

   This program is free software; you can redistribute it and/or modify 
   it under the terms of the GNU General Public License as published by 
   the Free Software Foundation; either version 2 of the License, or 
   (at your option) any later version.

   This program is distributed in the hope that it will be useful, 
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU 
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation, 
   Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
   
   This file is part of LIBJ.

*/
#include <_reg.h>



/* Waiting for robotnik */



reg *reg_new(const char *filename) {
/* reg_new
   Create a new registry structure, with the associated filename. 
   The filename is used by reg_load() and reg_save() functions,
   and may be modified later.  */

   reg *r;        /* Newly constructed reg variable */
   reg_class *tc; /* Class registry for top class. */

   /* Create the new object */
   r = (reg *)malloc(sizeof(reg));
   if(r == NULL) return(NULL);

   /* Initialise class registry */
   r->classes = NULL;
   reg_class_add(r, REG_TOP);
   tc = reg_class_lookup(r->classes, REG_TOP);

   /* Initialise toplevel block */
   r->top = reg_var_new_block(r, REG_TOP, tc);
   if(r->top == NULL) {
      /* Failed to allocate toplevel */
      free(r);
      return(false);
   } /* Do we have toplevel block? */

   /* Initialise associated filename */
   r->filename = NULL;
   reg_set_name(r, filename);
   
   /* Return the register structure */
   return(r);

}



void reg_free(reg **r) {
/* reg_free
   Releases a registry, and all associated data.  */

   /* release the registry structure */   
   if(r == NULL || *r == NULL) return;
   reg_var_free(&(*r)->top);
   reg_class_release(&(*r)->classes);
   free((*r)->filename);
   free(*r);
   *r = NULL;

}



static inline int _reg_text_to_index(const char *text, const char **list) {
/* reg_text_to_index 
   Determine which index the specified text string occurs at.
   <list> is a NULL-terminated list of strings.  */

   int index = 0;
   while(list[index] != NULL) {
      if(strcmpn(text, list[index])) return(index);
      ++index;
   }
   return(-1);

}



static inline int _reg_value_to_index(int value, const char **list, const int *values) {
/* reg_value_to_index 
   Determine which index the specified value lives at.  <list>
   is required since it has the NULL terminator; although it
   is the <values> list we compare <value> to.  */

   int index = 0;
   while(list[index] != NULL) {
      if(value == values[index]) return(index);
      ++index;
   }
   return(-1);

}



int reg_get_integer_from_values(reg *r, reg_var *v, const char *path, int defvalue, const char **names, const int *values) {
/* reg_get_integer_from_values
   Read the variable <path>, relative to <v>, and attempt to translate
   its string content to an enumerated type value.  If the value cannot
   be obtained (variable undefined, not a string, etc) then <defvalue>
   will be returned.  The enumerated types and corresponding names are 
   given as arguments to this function.  

   If <path> == NULL, then <v> is read.  */

   char buf[REG_BUFFER];   /* Temporary read-buffer */
   int index;              /* Index to matching name-value pair */

   /* Attempt to retrieve the string */   
   if(reg_get_string(r, v, path, buf, REG_BUFFER)) {
      /* We have a string to work with, yay.  Does it 
         have a corresponding entry in the values list? */
      trim(buf);
      index = _reg_text_to_index(buf, names);
      if(index >= 0) {
         /* Assign the new default value */
         defvalue = values[index];
      } /* Text matches? */
   } /* Is variable a valid string? */

   /* Return the enumerated value */
   return(defvalue);

}



int reg_get_bitmask_from_values(reg *r, reg_var *v, const char *path, int defvalue, const char **names, const int *values) {
/* reg_get_bitmask_from_values
   Read the variable <path>, relative to <v>, and attempt to translate its
   string content to a bitmask value.  If the value cannot be obtained
   (variable undefined, not a string, etc) then <defvalue> will be returned. 
   The enumerated types and corresponding names are given as arguments to
   this function.  If <path> == NULL, then <v> is read.  */

   char buf[REG_BUFFER];   /* Temporary read-buffer */
   char *ps;               /* Start of `current' bit */
   char *p;                /* Start of `next' bit */
   bool negate;            /* True if this bit negated */
   int  index;             /* Matching name-value pair */

   /* Attempt to retrieve the string */
   if(reg_get_string(r, v, path, buf, REG_BUFFER)) {
      /* Trim-o-rama, and loop while the string isn't empty */
      trim(buf);
      p = buf;
      while(*p != '\0') {
         /* Look for a comma or end-of-string */
         ps = p;
         while(*p != '\0' && *p != ',') ++p;

         /* If we hit a comma, there is another bit to parse */
         if(*p == ',') *(p++) = '\0';
         
         /* Look for a negation flag (if present) */
         trim(ps);
         negate = (*buf == '!');
         if(negate) trim(++ps);
         
         /* Attempt to find bitvalue corresponding to text */
         index = _reg_text_to_index(ps, names);
         if(index >= 0) {
            /* We have a matching bitvalue; mix it into default */
            index = values[index];
            if(negate) defvalue = defvalue & (~index);
            else defvalue = defvalue | index;
         } /* Found a matching bitvalue? */

      } /* Looping through named bits */
   } /* Was variable a valid string? */
   
   /* Return the newly constructed bitfield */
   return(defvalue);

}



bool reg_set_string_from_values(reg *r, reg_var *v, const char *path, int value, const char **names, const int *values) {
/* reg_set_string_from_values
   Write the enumerated value <value>, as its corresponding string to
   the variable <path> (relative to <v>).  The enumerated types and 
   corresponding names are also given.  
   
   If <path> == NULL, then modifications occur directly on <v>.  */

   int index;              /* Index to matching name-value pair */

   /* Attempt to get index matching up with value */   
   index = _reg_value_to_index(value, names, values);
   if(index >= 0) {
      /* We have it; try to set this as a string value. */
      return(reg_set_string(r, v, path, names[index]));
   } /* Match found? */
   
   /* No matches; cannot handle this case */
   return(false);

}



bool reg_set_bitmask_from_values(reg *r, reg_var *v, const char *path, int value, const char **names, const int *values) {
/* reg_set_bitmask_from_values
   Write the bitmask <value>, as its corresponding list of strings to the
   variable <path> (relative to <v>).  Enumerated types and corresponding
   names are also given.  If <path> == NULL, then modifications occur
   directly on <v>.  */

   char buf[REG_BUFFER];   /* Temp variable to construct value in */
   char *p;                /* Pointer into buffer */
   bool needcomma;         /* True if need preceding comma */
   int size;               /* Size of buffer remaining */
   int len;                /* Length of current bitname */
   int i;                  /* Index of matching element */

   *buf = '\0';   
   size = REG_BUFFER;
   needcomma = false;
   p = buf;

   i = 0;
   while(names[i] != NULL) {
      if((values[i] & value) == values[i]) {
         value = value & (~values[i]);
         len = strlen(names[i]) + 1;
         if(len > size) len = size;
         if(needcomma) strncatn(p, ", ", len);
         needcomma = true;
         strncatn(p, names[i], len);
         p += len - 1;
      }
      ++i;
   }      

   if(value != 0) return(false);
   
   return(reg_set_string(r, v, path, buf));

}



bool reg_get_integer(reg *r, reg_var *v, const char *path, int *value) {
/* reg_get_integer
   Reads the integer value <path>, relative to <v>.  True is returned
   on success.  If <path> == NULL, then <v> is read.  */

   reg_var *var;
   
   var = reg_block_resolve(r, v, path);
   if(var == NULL || var->type != REG_INTEGER) return(false);

   if(value != NULL) *value = var->value.integer;
   return(true);

}



bool reg_get_doublev(reg *r, reg_var *v, const char *path, double *value) {
/* reg_get_doublev
   Reads the double value <path>, relative to <v>.  True is returned
   on success.  If <path> == NULL, then <v> is read.  */

   reg_var *var;
   
   var = reg_block_resolve(r, v, path);
   if(var == NULL || var->type != REG_DOUBLEV) return(false);

   if(value != NULL) *value = var->value.doublev;
   return(true);

}



bool reg_get_boolean(reg *r, reg_var *v, const char *path, bool *value) {
/* reg_get_boolean
   Reads the boolean value <path>, relative to <v>.  True is returned
   on success.  If <path> == NULL, then <v> is read.  */

   reg_var *var;
   
   var = reg_block_resolve(r, v, path);
   if(var == NULL || var->type != REG_BOOLEAN) return(false);

   if(value != NULL) *value = var->value.boolean.value;
   return(true);

}



bool reg_get_string(reg *r, reg_var *v, const char *path, char *value, int size) {
/* reg_get_string
   Reads the string value <path>, relative to <v>.  The value is copied
   to the buffer indicated in <value>, of size <size>. True is returned
   on success.  If <path> == NULL, then <v> is read.  */

   reg_var *var;
   
   var = reg_block_resolve(r, v, path);
   if(var == NULL || var->type != REG_STRING) return(false);

   if(value != NULL && size > 0) {
      strncpyn(value, var->value.string, size);
   }
   return(true);

}



bool reg_set_integer(reg *r, reg_var *v, const char *path, int value) {
/* reg_set_integer
   Writes the integer <value> into the variable <path>, relative to <v>. 
   True is returned on success.  If <path> == NULL, then modifications 
   occur directly on <v>.  */

   char varname[REG_SYMBOL_SIZE];
   reg_var *block;
   
   block = reg_block_resolve_container(r, v, path, varname);
   if(block == NULL) return(false);

   return(reg_var_set_integer(r, block, varname, value));

}



bool reg_set_doublev(reg *r, reg_var *v, const char *path, double value) {
/* reg_set_doublev
   Writes the double <value> into the variable <path>, relative to <v>. 
   True is returned on success.  If <path> == NULL, then modifications 
   occur directly on <v>.  */

   char varname[REG_SYMBOL_SIZE];
   reg_var *block;
   
   block = reg_block_resolve_container(r, v, path, varname);
   if(block == NULL) return(false);

   return(reg_var_set_doublev(r, block, varname, value));

}



bool reg_set_boolean_f(reg *r, reg_var *v, const char *path, bool value, reg_format_bool format) {
/* reg_set_boolean_f
   Writes the boolean <value> into the variable <path>, relative to <v>. 
   True is returned on success.  If <path> == NULL, then modifications 
   occur directly on <v>.  */

   char varname[REG_SYMBOL_SIZE];
   reg_var *block;
   
   block = reg_block_resolve_container(r, v, path, varname);
   if(block == NULL) return(false);

   return(reg_var_set_boolean(r, block, varname, value, format));

}



bool reg_set_boolean(reg *r, reg_var *v, const char *path, bool value) {
/* reg_set_boolean */
   
   return(reg_set_boolean_f(r, v, path, value, REG_FORMAT_BOOL_DEFAULT));

}



bool reg_set_string(reg *r, reg_var *v, const char *path, const char *value) {
/* reg_set_string
   Writes the NULL-terminated string buffer <value> into the variable
   <path>, relative to <v>.  True is returned on success.  In case
   <path> == NULL, the modifications will occur directly on <v>.  */

   char varname[REG_SYMBOL_SIZE];
   reg_var *block;
   
   block = reg_block_resolve_container(r, v, path, varname);
   if(block == NULL) return(false);

   return(reg_var_set_string(r, block, varname, value));

}



bool reg_set_block(reg *r, reg_var *v, const char *path, const char *klass) {
/* reg_set_block
   Constructs a new block with the class named in <klass>.  If <path>
   is NULL, then <r> becomes a new block.  If <klass> is null, then
   an NULL-classed object will be created.  */

   char varname[REG_SYMBOL_SIZE];
   reg_var *block;
   reg_var *tmp;
   
   block = reg_block_resolve_container(r, v, path, varname);
   if(block == NULL) return(false);

   tmp = reg_var_new_block(r, varname, reg_class_lookup(r->classes, klass));
   return(reg_var_set_block(r, block, varname, &tmp));

}



