/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <unistd.h>
#include "AudioDriverSDL.h"
#include "AudioConfig.h"
#include <SDL/SDL_audio.h>
#include <SDL/SDL.h>
#include <SDL/SDL_timer.h>

#ifdef XSID_SDL_DEBUG
#include <iostream>
#include <iomanip>
using namespace std;
#endif

AudioDriverSDL::AudioDriverSDL()
{
#ifdef XSID_SDL_DEBUG
  cout << "AudioDriverSDL::AudioDriverSDL" << endl;
#endif
  SDL_Init(SDL_INIT_AUDIO | SDL_INIT_TIMER);
  outOfOrder();
  sdlTime=SDL_GetTicks();
  swapEndian = false;
}

AudioDriverSDL::~AudioDriverSDL()
{ 
#ifdef XSID_SDL_DEBUG
  cout << "AudioDriverSDL::~AudioDriverSDL" << endl;
#endif
  reset();
  unload();
  SDL_Quit();
}

void AudioDriverSDL::outOfOrder()
{
#ifdef XSID_SDL_DEBUG
  cout << "AudioDriverSDL::outOfOrder" << endl;
#endif
  // Reset everything.
  errorString = "None";
  haveStream = false;
}

bool AudioDriverSDL::isThere()
{
#ifdef XSID_SDL_DEBUG
    cout << "AudioDriverSDL::isThere" << endl;
#endif

  return true;
}

bool AudioDriverSDL::reset()
{
#ifdef XSID_SDL_DEBUG
    cout << "AudioDriverSDL::reset" << endl;
#endif

    SDL_PauseAudio(1);
    return false;  // not possible
}

bool AudioDriverSDL::open( const AudioConfig& inConfig )
{
#ifdef XSID_SDL_DEBUG
    cout << "AudioDriverSDL::open" << endl;
#endif

    // remove the samples there are still in the queue
    SDL_LockAudio();
    memoryDeque.clear();
    SDL_UnlockAudio();
    
    static AudioConfig lastConfig;
	
    if ( haveStream && inConfig==config && inConfig==lastConfig ) return true;
    else unload();
    
    // Copy input parameters. May later be replaced with driver defaults.
    config = inConfig;

    SDL_AudioSpec spec;

    memset(&spec, 0, sizeof(spec));
    spec.freq = config.frequency;
    spec.callback = sdl_callback;
    spec.channels = config.channels;
    spec.userdata=NULL;

    // set precision/encoding
    if (config.precision==AudioConfig::BITS_8) {
      spec.samples = 4096;
      if (config.encoding==AudioConfig::UNSIGNED_PCM) spec.format = AUDIO_U8; 
      else spec.format = AUDIO_S8;
    } else {
        spec.samples = 4096;
        if (config.encoding==AudioConfig::UNSIGNED_PCM) spec.format = AUDIO_U16; 
        else spec.format = AUDIO_S16; 
      }

    // init the SDL audio
    if(SDL_OpenAudio(&spec, &sdl_spec)) {
      errorString = SDL_GetError();
      return false;
    }

#ifdef XSID_SDL_DEBUG
    cout << "sdl_init() done." << endl;
#endif

    // be sure to have the true setting available
    if (sdl_spec.format != spec.format || sdl_spec.channels != spec.channels) {
#ifdef XSID_SDL_DEBUG
      cout << "Not able to have SDL with your specific format" << endl;
#endif
      errorString = "Not able to have SDL with your specific format";
      SDL_CloseAudio();
      return false;
    }

#ifdef XSID_SDL_DEBUG
    cout << spec.freq << endl;
    cout << spec.format << endl;
    cout << (int)spec.channels << endl;
    cout << spec.samples << endl;
    cout << spec.size << endl;
#endif

    SDL_PauseAudio(0);

    errorString = "OK";
#ifdef XSID_SDL_DEBUG
    cout << "return" << endl;
#endif

    // Assume CPU and soundcard have same endianess.
    swapEndian = false;
    haveStream=true;

    return true;
}

void AudioDriverSDL::unload()
{
#ifdef XSID_SDL_DEBUG
  cout << "AudioDriverSDL::unload" << endl;
#endif

  if ( !haveStream ) return;

  outOfOrder();
}

void AudioDriverSDL::close()
{
#ifdef XSID_SDL_DEBUG
    cout << "AudioDriverSDL::close" << endl;
#endif
  SDL_PauseAudio(1);
  SDL_CloseAudio();
}

void AudioDriverSDL::play(void* buffer, unsigned long int bufferSize)
{
  unsigned char* memory=(unsigned char*)buffer;
  SDL_LockAudio();
  SDL_PauseAudio(0);

#ifdef XSID_SDL_DEBUG
  cout << "AudioDriverSDL::play " << endl;
#endif

  // insert the data at the end of queue
  for (int i=0; i<bufferSize; i++) {
    memoryDeque.push_back(*memory);
    memory++;
  }

  SDL_UnlockAudio();

  int nSize;  // size of one sample
  if (sdl_spec.format == AUDIO_U8 || sdl_spec.format == AUDIO_S8) nSize=1;
  else nSize=2;
  nSize*=sdl_spec.channels;

  // delay needed by actual configuration
  Uint32 delay=1000*(bufferSize/nSize)/sdl_spec.freq;

  // compensate for not integer delay to use
  if (memoryDeque.size() > 3*bufferSize) delay++;

  // passed time from last call
  Uint32 passedTime=SDL_GetTicks()-sdlTime;

  // calculate new delay to use
  Sint32 newDelay=delay-passedTime;

  if (newDelay>0) SDL_Delay(newDelay);

  // remember the actual time
  sdlTime=SDL_GetTicks();
}

static void sdl_callback(void *userdata, Uint8 *stream, int len) {
#ifdef XSID_SDL_DEBUG
  cout << "sdl_callback " << endl;
#endif
  unsigned char value;

  int upTo=len;
  // if 16 bit, divide by 2 the lenght of buffer to fill
  if (sdl_spec.format != AUDIO_U8 && sdl_spec.format != AUDIO_S8) {
    upTo/=2;
    if (memoryDeque.size()<len) upTo=memoryDeque.size()/2;
  } else if (memoryDeque.size()<len) upTo=memoryDeque.size();

  // retrieve the data from the top of the list
  for (int i=0; i<upTo; i++) {
    value=memoryDeque.front();
    if (sdl_spec.format == AUDIO_U8 || sdl_spec.format == AUDIO_S8) {
     *stream++=value;
    } else { 
       *stream++=value;
       memoryDeque.pop_front();
       value=memoryDeque.front();
       *stream++=value;
      }
    memoryDeque.pop_front();
  }
}
