/**
 * @(#)sidid.h 2007/11/04
 *
 * @author Stefano Tognon (ice00)
 * @version 1.00
 *
 * Note: This code is based onto Cadaver sidid:
 * SIDId V1.07 - Quick & dirty HVSC playroutine identity scanner
 * Cadaver 9/2007 (loorni@gmail.com)
 * 
 * Copyright (C) 2006-2007 by the author & contributors. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ''AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO
 * EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER
 * IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 *
 * This file is part of XSidplay2.
 * See README for copyright notice.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef SIDID_H
#define SIDID_H

#include <list>
#include <string>
#include <iostream>

/**
 * Store the information about a signature (pattern)
 */
struct sidSigRecord {
  int *bytes;
  sidSigRecord(): bytes(NULL) {}
  //~sidSigRecord() { std::cerr << bytes << "dest\n";if (bytes!=NULL) delete[] bytes;}
  //~sidSigRecord() { std::cerr << "dest\n";}
}; 

typedef std::list<sidSigRecord> sidSigList;

/**
 * Store the information about a SID Id signature (patterns)
 */
struct sidIdRecord {
  std::string name;
  sidSigList list;

  sidIdRecord(): name("") {}
};

typedef std::list<sidIdRecord> sidIdList;

/**
 * SidId class
 *
 * This class implements the search of sid engines using the Cadaver pattern's
 * of SidId
 *
 * @author Stefano Tognon (ice00)
 * @version 1.00 04/11/2007
 */
class SidId {
  public:
    /**
     * Constructor
     */
    SidId();

    /**
     * Destructor
     */
    ~SidId();

    /**
     * Get the version of SIDID as a formatted string
     *
     * @return the version of SIDID
     */
    const char *getVersion() const {
      return m_versionString;
    }

    /**
     * Read the patterns id file
     *
     * @param name the name (with path) of the file to read
     * @return true if read is ok
     */
    bool readConfig(const char *name);

    /**
     * Identify the IDs of the given buffer
     *
     * @param buffer the buffed with the data to identify
     * @param length length of the buffer
     * @return the identified engines as string
     */
    std::string identifyBuffer(const unsigned char *buffer, int length);

    /**
     * Get the number of players recognized
     *
     * @return the number of players recognized
     */ 
    unsigned int getNumberOfPlayers();

    /**
     * Get the number of patterns used
     *
     * @return the number of patterns used
     */ 
    unsigned int getNumberOfPatterns();

  private:
    /**
     * Identify the bytes of ID and buffer according to the pattern rules
     *
     * @param bytes the ID bytes pattern
     * @param buffer the SID file buffer
     * @param length length of the buffer
     * @return true if ID is matching 
     */
    bool identifyBytes(int *bytes, const unsigned char *buffer, int length);

    /**
     * Determine if the passed char is an exe number
     * 
     * @param c the char to test
     * @return true if the char is an hex number
     */
    int isHex(char c);

    /**
     * Get the decimal number of the passed exe number char
     * 
     * @param c the char to use
     * @return the decimal number of the hex char passed
     */
    int getHex(char c);

    /** list of Sid IDs */
    sidIdList m_sidIdList;

    // Version number/copyright string
    static const char * const m_versionString;
};

#endif  /* SIDID_H */
