/* $Id: tide_db.c,v 1.17 2003/11/17 01:22:26 flaterco Exp $ */

#include "tcd.h"
#include "tide_db_header.h"
#include "tide_db_default.h"
#include "tide_db_version.h"


/*****************************************************************************\

                            DISTRIBUTION STATEMENT

    This source file is unclassified, distribution unlimited, public
    domain.  It is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

\*****************************************************************************/




/*****************************************************************************\

    Tide Constituent Database API


    Author  : Jan C. Depner (depnerj@navo.navy.mil)

    Date    : 08/01/02 
              (First day of Micro$oft's "Licensing 6" policy - P.T. Barnum was 
              right!!!)

    Purpose : To replace the ASCII/XML formatted harmonic constituent data
              files, used in Dave Flater's (http://www.flaterco.com/xtide/) 
              exceptionally fine open-source XTide program, with a fast, 
              efficient binary format.  In addition, we wanted to replace the 
              Naval Oceanographic Office's (http://www.navo.navy.mil) 
              antiquated ASCII format harmonic constituent data file due to 
              problems with configuration management of the file.  The 
              resulting database will become a Navy OAML (Oceanographic and
              Atmospheric Master Library) standard harmonic tide constituent
              database.

    Design  : The following describes the database file and some of the 
              rationale behind the design.

    First question - Why didn't I use PostgreSQL or MySQL?  Mostly 
    portability.  What?  PostgreSQL runs on everything!  Yes, but it doesn't
    come installed on everything.  This would have meant that the poor,
    benighted Micro$oft borgs would have had to actually load a software
    package.  In addition, the present harmonics/offset files only contain
    a total of 6,409 stations.  It hardly seemed worth the effort (or overhead)
    of a fullblown RDBMS to handle this.  Second question - Why binary and not
    ASCII or XML?  This actually gets into philosophy.  At NAVO we have used an
    ASCII harmonic constituent file for years (we were founded in 1830 and I
    think that that's when they designed the file).  We have about fifty 
    million copies floating around and each one is slightly different.  Why?
    Because they're ASCII and everyone thinks they know what they're doing so
    they tend to modify the file.  Same problem with XML, it's still ASCII.
    We wanted a file that people weren't going to mess with and that we could
    version control.  We also wanted a file that was small and fast.  This is 
    very difficult to do with ASCII.  The big slowdown with the old format
    was I/O and parsing.  Third question - will it run on any OS?  Hopefully,
    yes.  After twenty-five years of working with low bidder systems I've 
    worked on almost every OS known to man.  Once you've been bitten by big
    endian vs little endian or IEEE floating point format vs VAX floating 
    point format or byte addressable memory vs word addressable memory or 32 
    bit word vs 36 bit word vs 48 bit word vs 64 bit word sizes you get the 
    message.  All of the data in the file is stored either as ASCII text or
    scaled integers (32 bits or smaller), bit-packed and stuffed into an 
    unsigned character buffer for I/O.  No endian issues, no floating point 
    issues, no word size issues, no memory mapping issues.  I will be testing 
    this on x86 Linux, HP-UX, and Micro$oft Windoze.  By the time you read 
    this it will be portable to those systems at least.

    Now, on to the file layout.  As much as I dislike ASCII it is occasionally
    handy to be able to see some information about a file without having to
    resort to a special purpose program.  With that in mind I made the first 
    part of the header of the file ASCII.  The following is an example of the 
    ASCII portion of the header:

        [VERSION] = PFM Software - tide_db V1.00 - 08/01/02
        [LAST MODIFIED] = Thu Aug  1 02:46:29 2002
        [HEADER SIZE] = 4096
        [NUMBER OF RECORDS] = 10652
        [START YEAR] = 1970
        [NUMBER OF YEARS] = 68
        [SPEED BITS] = 31
        [SPEED SCALE] = 10000000
        [SPEED OFFSET] = -410667
        [EQUILIBRIUM BITS] = 16
        [EQUILIBRIUM SCALE] = 100
        [EQUILIBRIUM OFFSET] = 0
        [NODE BITS] = 15
        [NODE SCALE] = 10000
        [NODE OFFSET] = -3949
        [AMPLITUDE BITS] = 19
        [AMPLITUDE SCALE] = 10000
        [EPOCH BITS] = 16
        [EPOCH SCALE] = 100
        [RECORD TYPE BITS] = 4
        [LATITUDE BITS] = 25
        [LATITUDE SCALE] = 100000
        [LONGITUDE BITS] = 26
        [LONGITUDE SCALE] = 100000
        [RECORD SIZE BITS] = 12
        [STATION BITS] = 18
        [DATUM OFFSET BITS] = 32
        [DATUM OFFSET SCALE] = 10000
        [DATE BITS] = 27
        [MONTHS ON STATION BITS] = 10
        [CONFIDENCE VALUE BITS] = 4
        [TIME BITS] = 13
        [LEVEL ADD BITS] = 16
        [LEVEL ADD SCALE] = 100
        [LEVEL MULTIPLY BITS] = 16
        [LEVEL MULTIPLY SCALE] = 1000
        [DIRECTION BITS] = 9
        [LEVEL UNIT BITS] = 3
        [LEVEL UNIT TYPES] = 6
        [LEVEL UNIT SIZE] = 15
        [DIRECTION UNIT BITS] = 2
        [DIRECTION UNIT TYPES] = 3
        [DIRECTION UNIT SIZE] = 15
        [RESTRICTION BITS] = 4
        [RESTRICTION TYPES] = 2
        [RESTRICTION SIZE] = 30
        [PEDIGREE BITS] = 6
        [PEDIGREE TYPES] = 13
        [PEDIGREE SIZE] = 60
        [DATUM BITS] = 7
        [DATUM TYPES] = 61
        [DATUM SIZE] = 70
        [CONSTITUENT BITS] = 8
        [CONSTITUENTS] = 173
        [CONSTITUENT SIZE] = 10
        [COUNTRY BITS] = 9
        [COUNTRIES] = 240
        [COUNTRY SIZE] = 50
        [TZFILE BITS] = 10
        [TZFILES] = 449
        [TZFILE SIZE] = 30
        [END OF FILE] = 2585170
        [END OF ASCII HEADER DATA]

    Most of these values will make sense in the context of the following
    description of the rest of the file.  Some caveats on the data storage - 
    if no SCALE is listed for a field, the scale is 1.  If no BITS field is 
    listed, this is a variable length character field and is stored as 8 bit 
    ASCII characters.  If no OFFSET is listed, the offset is 0.  Offsets are 
    scaled.  All SIZE fields refer to the maximum length, in characters, of a 
    variable length character field.  Some of the BITS fields are calculated 
    while others are hardwired (see code).  For instance, [DIRECTION BITS] is 
    hardwired because it is an integer field whose value can only be from 0 to
    361 (361 = no direction flag).  [NODE BITS], on the other hand, is 
    calculated on creation by checking the min, max, and range of all of the 
    node factor values.  The number of bits needed is easily calculated by 
    taking the log of the adjusted, scaled range, dividing by the log of 2 and
    adding 1.  Immediately following the ASCII portion of the header is a 32 
    bit checksum of the ASCII portion of the header.  Why?  Because some 
    genius always gets the idea that he/she can modify the header with a text 
    or hex editor.  Go figure.  

    The rest of the header is as follows :

        [LEVEL UNIT TYPES] fields of [LEVEL UNIT SIZE] characters, each field 
            is internally 0 terminated (anything after the zero is garbage)

        [DIRECTION UNIT TYPES] fields of [DIRECTION UNIT SIZE] characters, 0
            terminated

        [RESTRICTION TYPES] fields of [RESTRICTION SIZE] characters, 0 
            terminated

        [PEDIGREE TYPES] fields of [PEDIGREE SIZE] characters, 0 terminated

        [TZFILES] fields of [TZFILE SIZE] characters, 0 terminated

        [COUNTRIES] fields of [COUNTRY SIZE] characters, 0 terminated

        [DATUM TYPES] fields of [DATUM SIZE] characters, 0 terminated

        [CONSTITUENTS] fields of [CONSTITUENT SIZE] characters, 0 terminated
            Yes, I know, I wasted some space with these fields but I wasn't 
            worried about a couple of hundred bytes.

        [CONSTITUENTS] fields of [SPEED BITS], speed values (scaled and offset)

        [CONSTITUENTS] groups of [NUMBER OF YEARS] fields of 
            [EQUILIBRIUM BITS], equilibrium arguments (scaled and offset)

        [CONSTITUENTS] groups of [NUMBER OF YEARS] fields of [NODE BITS], node
            factors (scaled and offset)


    Finally, the data.  At present there are two types of records in the file.
    These are reference stations (record type 1) and subordinate stations 
    (record type 2).  Reference stations contain a set of constituents while
    subordinate stations contain a number of offsets to be applied to the
    reference station that they are associated with.  Note that reference 
    stations (record type 1) may, in actuality, be subordinate stations, albeit
    with a set of constituents.  All of the records have the following subset
    of information stored as the first part of the record:

        [RECORD SIZE BITS] - record size in bytes
        [RECORD TYPE BITS] - record type (1 or 2)
        [LATITUDE BITS] - latitude (degrees, south negative, scaled & offset)
        [LONGITUDE BITS] - longitude (degrees, west negative, scaled & offset)
        [TZFILE BITS] - index into timezone array (retrieved from header)
        variable size - station name, 0 terminated
        [STATION BITS] - record number of reference station or -1
        [COUNTRY_BITS] index into country array (retrieved from header)
        [PEDIGREE BITS] - index into pedigree array (retrieved from header)
        variable size - source, 0 terminated
        [RESTRICTION BITS] - index into restriction array
        variable size - comments, may contain LFs to indicate newline (no CRs)


    These are the rest of the fields for record type 1:

        [LEVEL UNIT BITS] - index into level units array
        [DATUM OFFSET BITS] - datum offset (scaled)
        [DATUM BITS] - index into datum name array
        [TIME BITS] - time zone offset from GMT0 (meridian, integer +/-HHMM)
        [DATE BITS] - expiration date, (integer YYYYMMDD, default is 0)
        [MONTHS ON STATION BITS] - months on station
        [DATE BITS] - last date on station, default is 0
        [CONFIDENCE BITS] - confidence value (TBD)
        [CONSTITUENT BITS] - "N", number of constituents for this station

        N groups of:
            [CONSTITUENT BITS] - constituent number
            [AMPLITUDE BITS] - amplitude (scaled & offset)
            [EPOCH BITS] - epoch (scaled & offset)


    These are the rest of the fields for record type 2:

        [LEVEL UNIT BITS] - leveladd units, index into level_units array
        [DIRECTION UNIT BITS] - direction units, index into dir_units array
        [LEVEL UNIT BITS] - avglevel units, index into level_units array
        [TIME BITS] - min timeadd (integer +/-HHMM) or 0
        [LEVEL ADD BITS] - min leveladd (scaled) or 0
        [LEVEL MULTIPLY BITS] - min levelmultiply (scaled) or 0
        [LEVEL ADD BITS] - min avglevel (scaled) or 0
        [DIRECTION BITS] - min direction (0-360 or 361 for no direction)
        [TIME BITS] - max timeadd (integer +/-HHMM) or 0
        [LEVEL ADD BITS] - max leveladd (scaled) or 0
        [LEVEL MULTIPLY BITS] - max levelmultiply (scaled) or 0
        [LEVEL ADD BITS] - max avglevel (scaled) or 0
        [DIRECTION BITS] - max direction (0-360 or 361 for no direction)
        [TIME BITS] - floodbegins (integer +/-HHMM) or 0
        [TIME BITS] - ebbbegins (integer +/-HHMM) or 0


    Back to philosophy!  When you design a database of any kind the first
    thing you should ask yourself is "Self, how am I going to access this
    data most of the time?".  If you answer yourself out loud you should 
    consider seeing a shrink.  99 and 44/100ths percent of the time this 
    database is going to be read to get station data.  The other 66/100ths 
    percent of the time it will be created/modified.  Variable length records 
    are no problem on retrieval.  They are no problem to create.  They can be 
    a major pain in the backside if you have to modify/delete them.  Since we 
    shouldn't be doing too much editing of the data (usually just adding 
    records) this is a pretty fair design.  At some point though we are going 
    to want to modify or delete a record.  There are two possibilities here.
    We can dump the database to an ASCII file or files using restore_tide_db, 
    use a text editor to modify them, and then rebuild the database.  The 
    other possibility is to modify the record in place.  This is OK if you
    don't change a variable length field but what if you want to change the 
    station name or add a couple of constituents?  With the design as is we 
    have to read the remainder of the file from the end of the record to be
    modified, write the modified record, rewrite the remainder of the file,
    and then change the end_of_file pointer in the header.  So, which fields 
    are going to be a problem?  Changes to station name, source, comments, or 
    the number of constituents for a station will require a resizing of the 
    database.  Changes to any of the other fields can be done in place.  The 
    worst thing that you can do though is to delete a record.  Not just 
    because the file has to be resized but because it might be a reference 
    record with subordinate stations.  These would have to be deleted as well.
    The delete_tide_record function will do just that so make sure you check
    before you call it.  You might not want to do that.

    Another point to note is that when you open the database the records are
    indexed at that point.  This takes about half a second on a dual 450.
    Most applications use the header part of the record very often and
    the rest of the record only if they are going to actually produce
    predicted tides.  For instance, XTide plots all of the stations on a 
    world map or globe and lists all of the station names.  It also needs the
    timezone up front.  To save re-indexing to get these values I save them
    in memory.  The only time an application needs to actually read an entire
    record is when you want to do the prediction.  Otherwise just use 
    get_partial_tide_record or get_next_partial_tide_record to yank the good 
    stuff out of memory.

    'Nuff said?

\*****************************************************************************/


/*  Function prototypes.  */

NV_U_INT32 calculate_bits (NV_U_INT32 range);
void bit_pack (NV_U_BYTE *, NV_U_INT32, NV_U_INT32, NV_INT32);
NV_INT32 bit_unpack (NV_U_BYTE *, NV_U_INT32, NV_U_INT32);
NV_INT32 signed_bit_unpack (NV_U_BYTE buffer[], NV_U_INT32 start, 
                            NV_U_INT32 numbits);



/*  Global variables.  */

typedef struct
{
    NV_INT32                address;
    NV_U_INT16              record_size;
    NV_U_INT16              tzfile;
    NV_INT32                reference_station;
    NV_INT32                lat;
    NV_INT32                lon;
    NV_U_BYTE               record_type;
    NV_CHAR                 *name;
} TIDE_INDEX;


static FILE                 *fp;
static TIDE_INDEX           *tindex = NULL;
static NV_BOOL              modified = NVFalse;
static NV_INT32             current_record, current_index;
static NV_CHAR              filename[512];


/*****************************************************************************\

    Function        dump_tide_record - prints out all of the fields in the 
                    input tide record

    Synopsis        dump_tide_record (rec);

                    TIDE_RECORD *rec        pointer to the tide record

    Returns         void

    Author          Jan C. Depner
    Date            08/01/02

\*****************************************************************************/

void dump_tide_record (TIDE_RECORD *rec)
{
    NV_INT32              i;


    fprintf (stderr, "\n\nRecord Number = %d\n", rec->header.record_number);
    fprintf (stderr, "Record Size = %d\n", rec->header.record_size);
    fprintf (stderr, "Record Type = %d\n", rec->header.record_type);
    fprintf (stderr, "Latitude = %f\n", rec->header.latitude);
    fprintf (stderr, "Longitude = %f\n", rec->header.longitude);
    fprintf (stderr, "Name = %s\n", rec->header.name);
    fprintf (stderr, "Reference Station = %d\n", 
        rec->header.reference_station);
    fprintf (stderr, "Country = %s\n", get_country (rec->country));
    fprintf (stderr, "Tzfile = %s\n", get_tzfile (rec->header.tzfile));
    fprintf (stderr, "Pedigree = %s\n", get_pedigree (rec->pedigree));
    fprintf (stderr, "Source = %s\n", rec->source);
    fprintf (stderr, "Restriction = %s\n", get_restriction (rec->restriction));
    fprintf (stderr, "Comments = %s\n", rec->comments);

    if (rec->header.record_type == 1)
    {
        fprintf (stderr, "Units = %s\n", get_level_units (rec->units));
        fprintf (stderr, "Datum = %s\n", get_datum (rec->datum));
        fprintf (stderr, "Datum Offset = %f\n", rec->datum_offset);
        fprintf (stderr, "Zone Offset = %d\n", rec->zone_offset);
        fprintf (stderr, "Use Until Date = %d\n", rec->expiration_date);
        fprintf (stderr, "Months On Station = %d\n", rec->months_on_station);
        fprintf (stderr, "Last Date On Station = %d\n", 
            rec->last_date_on_station);
        fprintf (stderr, "Confidence = %d\n", rec->confidence);
        for (i = 0 ; i < hd.pub.constituents ; i++)
        {
            if (rec->amplitude[i] != 0.0 || rec->epoch[i] != 0.0)
            {
                fprintf (stderr, "Amplitude[%d] = %f\n", i, rec->amplitude[i]);
                fprintf (stderr, "Epoch[%d] = %f\n", i, rec->epoch[i]);
            }
        }
    }

    if (rec->header.record_type == 2)
    {
        fprintf (stderr, "Level Units = %s\n", 
            get_level_units (rec->level_units));
        fprintf (stderr, "Average Level Units = %s\n", 
            get_level_units (rec->avg_level_units));
        fprintf (stderr, "Direction Units = %s\n", 
            get_dir_units (rec->direction_units));
        fprintf (stderr, "Min Time Add = %d\n", rec->min_time_add);
        fprintf (stderr, "Min Level Add = %f\n", rec->min_level_add);
        fprintf (stderr, "Min Level Multiply = %f\n", rec->min_level_multiply);
        fprintf (stderr, "Min Average Level = %f\n", rec->min_avg_level);
        fprintf (stderr, "Min Direction = %d\n", rec->min_direction);
        fprintf (stderr, "Max Time Add = %d\n", rec->max_time_add);
        fprintf (stderr, "Max Level Add = %f\n", rec->max_level_add);
        fprintf (stderr, "Max Level Multiply = %f\n", rec->max_level_multiply);
        fprintf (stderr, "Max Average Level = %f\n", rec->max_avg_level);
        fprintf (stderr, "Max Direction = %d\n", rec->max_direction);
        fprintf (stderr, "Flood Begins = %d\n", rec->flood_begins);
        fprintf (stderr, "Ebb Begins %d\n", rec->ebb_begins);
    }
}





/*****************************************************************************\

    Function        get_country - gets the country field for record "num"

    Synopsis        get_country (num);

                    NV_INT32 num            tide record number         

    Returns         NV_CHAR *               country name (associated with 
                                            ISO 3166-1:1999 2-character 
                                            country code

    Author          Jan C. Depner
    Date            08/01/02

\*****************************************************************************/

NV_CHAR *get_country (NV_INT32 num)
{
    if (num >= 0 && num < hd.pub.countries) return (hd.country[num]);

    return ("Unknown");
}





/*****************************************************************************\

    Function        get_tzfile - gets the time zone name for record "num"

    Synopsis        get_tzfile (num); 

                    NV_INT32 num            tide record number         

    Returns         NV_CHAR *               time zone name used in TZ variable

    Author          Jan C. Depner
    Date            08/01/02

\*****************************************************************************/

NV_CHAR *get_tzfile (NV_INT32 num)
{
    if (num >= 0 && num < hd.pub.tzfiles) return (hd.tzfile[num]);

    return ("Unknown");
}





/*****************************************************************************\

    Function        get_station - get the name of the station for record "num"

    Synopsis        get_station (num);

                    NV_INT32 num            tide record number         

    Returns         NV_CHAR *               station name                      

    Author          Jan C. Depner
    Date            08/01/02

\*****************************************************************************/

NV_CHAR *get_station (NV_INT32 num)
{
    if (num >= 0 && num < hd.pub.number_of_records) return (tindex[num].name);

    return ("Unknown");
}






/*****************************************************************************\

    Function        get_constituent - get the constituent name for constituent
                    number "num"

    Synopsis        get_constituent (num);

                    NV_INT32 num            constituent number         

    Returns         NV_CHAR *               constituent name                  

    Author          Jan C. Depner
    Date            08/01/02

\*****************************************************************************/

NV_CHAR *get_constituent (NV_INT32 num)
{
    if (num >= 0 && num < hd.pub.constituents) return (hd.constituent[num]);

    return ("Unknown");
}






/*****************************************************************************\

    Function        get_level_units - get the level units for level units
                    number "num"

    Synopsis        get_level_units (num); 

                    NV_INT32 num            level units number         

    Returns         NV_CHAR *               units (ex. "meters");             

    Author          Jan C. Depner
    Date            08/01/02

\*****************************************************************************/

NV_CHAR *get_level_units (NV_INT32 num)
{
    if (num >= 0 && num < hd.pub.level_unit_types) return (hd.level_unit[num]);

    return ("Unknown");
}






/*****************************************************************************\

    Function        get_dir_units - get the direction units for direction
                    units number "num"

    Synopsis        get_dir_units (num); 

                    NV_INT32 num            direction units number         

    Returns         NV_CHAR *               units (ex. "degrees true");

    Author          Jan C. Depner
    Date            08/01/02

\*****************************************************************************/

NV_CHAR *get_dir_units (NV_INT32 num)
{
    if (num >= 0 && num < hd.pub.dir_unit_types) return (hd.dir_unit[num]);

    return ("Unknown");
}






/*****************************************************************************\

    Function        get_restriction - gets the restriction description for 
                    restriction number "num"

    Synopsis        get_restriction (num);

                    NV_INT32 num            restriction number         

    Returns         NV_CHAR *               restriction (ex. "PUBLIC DOMAIN");

    Author          Jan C. Depner
    Date            08/01/02

\*****************************************************************************/

NV_CHAR *get_restriction (NV_INT32 num)
{
    if (num >= 0 && num < hd.pub.restriction_types) 
        return (hd.restriction[num]);

    return ("Unknown");
}






/*****************************************************************************\

    Function        get_pedigree - gets the pedigree description for pedigree
                    number "num"

    Synopsis        get_pedigree (num);

                    NV_INT32 num            pedigree number         

    Returns         NV_CHAR *               pedigree description

    Author          Jan C. Depner
    Date            08/01/02

\*****************************************************************************/

NV_CHAR *get_pedigree (NV_INT32 num)
{
    if (num >= 0 && num < hd.pub.pedigree_types) return (hd.pedigree[num]);

    return ("Unknown");
}






/*****************************************************************************\

    Function        get_datum - gets the datum name for datum number "num"

    Synopsis        get_datum (num);

                    NV_INT32 num            datum number

    Returns         NV_CHAR *               datum name

    Author          Jan C. Depner
    Date            08/01/02

\*****************************************************************************/

NV_CHAR *get_datum (NV_INT32 num)
{
    if (num >= 0 && num < hd.pub.datum_types) return (hd.datum[num]);

    return ("Unknown");
}






/*****************************************************************************\

    Function        get_speed - gets the speed value for constituent number 
                    "num"

    Synopsis        get_speed (num);   

                    NV_INT32 num            constituent number         

    Returns         NV_FLOAT64              speed   

    Author          Jan C. Depner
    Date            08/01/02

\*****************************************************************************/

NV_FLOAT64 get_speed (NV_INT32 num)
{
    if (num >= 0 && num < hd.pub.constituents) return (hd.speed[num]);

    return (-1.0);
}


  



/*****************************************************************************\

    Function        get_equilibrium - gets the equilibrium value for 
                    constituent number "num" and year "year"

    Synopsis        get_equilibrium (num, year);

                    NV_INT32 num            constituent number 
                    NV_INT32 year           year

    Returns         NV_FLOAT32              equilibrium argument

    Author          Jan C. Depner
    Date            08/01/02

\*****************************************************************************/

NV_FLOAT32 get_equilibrium (NV_INT32 num, NV_INT32 year)
{
    if (num >= 0 && num < hd.pub.constituents && 
        year < hd.pub.number_of_years) return (hd.equilibrium[num][year]);

    return (-1.0);
}






/*****************************************************************************\

    Function        get_node_factor - gets the node factor value for 
                    constituent number "num" and year "year"

    Synopsis        get_node_factor (num, year);

                    NV_INT32 num            constituent number 
                    NV_INT32 year           year

    Returns         NV_FLOAT32              node factor

    Author          Jan C. Depner
    Date            08/01/02

\*****************************************************************************/

NV_FLOAT32 get_node_factor (NV_INT32 num, NV_INT32 year)
{
    if (num >= 0 && num < hd.pub.constituents && 
        year < hd.pub.number_of_years) return (hd.node_factor[num][year]);

    return (-1.0);
}






/*****************************************************************************\

    Function        get_partial_tide_record - gets "header" portion of record
                    "num" from the index that is stored in memory.  This is
                    way faster than reading it again and we have to read it
                    to set up the index.  This costs a bit in terms of 
                    memory but most applications use this data far more than
                    the rest of the record.

    Synopsis        get_partial_tide_record (num, rec);

                    NV_INT32 num              record number 
                    TIDE_STATION_HEADER *rec  header portion of the record

    Returns         NV_BOOL                   NVTrue if successful

    Author          Jan C. Depner
    Date            08/01/02

\*****************************************************************************/

NV_BOOL get_partial_tide_record (NV_INT32 num, TIDE_STATION_HEADER *rec)
{
    if (num < 0 || num >= hd.pub.number_of_records) return (NVFalse);

    rec->record_number = num;
    rec->record_size = tindex[num].record_size;
    rec->record_type = tindex[num].record_type;
    rec->latitude = (NV_FLOAT64) tindex[num].lat / hd.latitude_scale;
    rec->longitude = (NV_FLOAT64) tindex[num].lon / hd.longitude_scale;
    rec->reference_station = tindex[num].reference_station;
    rec->tzfile = tindex[num].tzfile;
    strcpy (rec->name, tindex[num].name);

    current_index = num;

    return (NVTrue);
}






/*****************************************************************************\

    Function        get_next_partial_tide_record - gets "header" portion of 
                    the next record from the index that is stored in memory.  

    Synopsis        get_next_partial_tide_record (rec);

                    TIDE_STATION_HEADER *rec  header portion of the record

    Returns         NV_INT32                  record number or -1 on failure

    Author          Jan C. Depner
    Date            08/01/02

\*****************************************************************************/

NV_INT32 get_next_partial_tide_record (TIDE_STATION_HEADER *rec)
{
    if (!get_partial_tide_record (current_index + 1, rec)) return (-1);

    return (current_index);
}






/*****************************************************************************\

    Function        get_nearest_partial_tide_record - gets "header" portion of 
                    the record closest geographically to the input position.

    Synopsis        get_nearest_partial_tide_record (lat, lon, rec);

                    NV_FLOAT64 lat            latitude
                    NV_FLOAT64 lon            longitude
                    TIDE_STATION_HEADER *rec  header portion of the record

    Returns         NV_INT32                  record number or -1 on failure

    Author          Jan C. Depner
    Date            08/01/02

\*****************************************************************************/

NV_INT32 get_nearest_partial_tide_record (NV_FLOAT64 lat, NV_FLOAT64 lon,
TIDE_STATION_HEADER *rec)
{
    NV_FLOAT64           diff, min_diff, lt, ln;
    NV_INT32             i, shortest = 0;


    min_diff = 999999999.9;
    for (i = 0 ; i < hd.pub.number_of_records ; i++)
    {
        lt = (NV_FLOAT64) tindex[i].lat / hd.latitude_scale;
        ln = (NV_FLOAT64) tindex[i].lon / hd.longitude_scale;

        diff = sqrt ((lat - lt) * (lat - lt) + (lon - ln) * (lon - ln));

        if (diff < min_diff)
        {
            min_diff = diff;
            shortest = i;
        }
    }


    if (!get_partial_tide_record (shortest, rec)) return (-1);

    return (shortest);
}






/*****************************************************************************\

    Function        get_time - converts a time string in +/-HH:MM form to an
                    integer in +/-HHMM form

    Synopsis        get_time (string);

                    NV_CHAR *string         time string

    Returns         NV_INT32                time

    Author          Jan C. Depner
    Date            08/01/02

\*****************************************************************************/

NV_INT32 get_time (NV_CHAR *string)
{
    NV_INT32        hour, minute, hhmm;


    sscanf (string, "%d:%d", &hour, &minute);


    /*  Trying to deal with negative 0 (-00:45).  */

    if (string[0] == '-')
    {
        if (hour < 0) hour = -hour;

        hhmm = -(hour * 100 + minute);
    }
    else
    {
        hhmm = hour * 100 + minute;
    }

    return (hhmm);
}






/*****************************************************************************\

    Function        ret_time - converts a time value in +/-HHMM form to a
                    time string in +/-HH:MM form

    Synopsis        ret_time (time);

                    NV_INT32                time

    Returns         NV_CHAR *               time string

    Author          Jan C. Depner
    Date            08/01/02

\*****************************************************************************/

NV_CHAR *ret_time (NV_INT32 time)
{
    NV_INT32          hour, minute;
    static NV_CHAR    tname[10];


    hour = abs (time) / 100;
    minute = abs (time) % 100;

    if (time < 0)
    {
        sprintf (tname, "-%02d:%02d", hour, minute);
    }
    else
    {
        sprintf (tname, "+%02d:%02d", hour, minute);
    }

    return (tname);
}






/*****************************************************************************\

    Function        get_tide_db_header - gets the public portion of the tide
                    database header

    Synopsis        get_tide_db_header ();

    Returns         DB_HEADER_PUBLIC        public tide header

    Author          Jan C. Depner
    Date            08/01/02

\*****************************************************************************/

DB_HEADER_PUBLIC get_tide_db_header ()
{
    return (hd.pub);
}






/*****************************************************************************\

    Function        clip_string - removes leading and trailing spaces from 
                    search strings.

    Synopsis        clip_string (string);

                    NV_CHAR *string         search string

    Returns         NV_CHAR *               clipped string

    Author          Jan C. Depner
    Date            09/16/02

\*****************************************************************************/

static NV_CHAR *clip_string (NV_CHAR *string)
{
    static NV_CHAR        new_string[256];
    NV_INT32              i, start = 0, end = 0;


    for (i = 0 ; i < strlen (string) ; i++) 
    {
        if (string[i] != ' ')
        {
            start = i;
            break;
        }
    }


    for (i = strlen (string) ; i >= 0 ; i--) 
    {
        if (string[i] && string[i] != ' ' && string[i] != 10 && 
            string[i] != 13)
        {
            end = i;
            break;
        }
    }


    memset (new_string, 0, sizeof (new_string));

    for (i = start ; i <= end ; i++) new_string[i - start] = string[i];

    new_string[end + 1] = 0;


    return (new_string);
}






/*****************************************************************************\

    Function        search_station - returns record numbers of all stations
                    that have the string "string" anywhere in the station
                    name.  This search is case insensitive.  When no more 
                    records are found it returns -1;

    Synopsis        search_station (string);

                    NV_CHAR *string         search string

    Returns         NV_INT32                record number or -1 when no more
                                            matches

    Author          Jan C. Depner
    Date            08/01/02

\*****************************************************************************/

NV_INT32 search_station (NV_CHAR *string)
{
    static NV_CHAR        last_search[NAME_LENGTH];
    static NV_INT32       j;
    NV_INT32              i;
    NV_CHAR               name[NAME_LENGTH], search[NAME_LENGTH];


    for (i = 0 ; i < strlen (string) + 1 ; i++) 
        search[i] = tolower (string[i]);

    if (strcmp (search, last_search)) j = 0;

    strcpy (last_search, search);

    while (j < hd.pub.number_of_records)
    {
        for (i = 0 ; i < strlen (tindex[j].name) + 1 ; i++) 
            name[i] = tolower (tindex[j].name[i]);


        if (strstr (name, search)) 
        {
            j++;
            return (j - 1);
        }

        j++;
    }

    j = 0;

    return (-1);
}






/*****************************************************************************\

    Function        find_station - finds the record number of the station
                    that has name "name"

    Synopsis        find_station (name);

                    NV_CHAR *name           station name               

    Returns         NV_INT32                record number                     

    Author          Jan C. Depner
    Date            08/01/02

\*****************************************************************************/

NV_INT32 find_station (NV_CHAR *name)
{
    NV_INT32              i;


    for (i = 0 ; i < hd.pub.number_of_records ; i++)
    {
        if (!strcmp (name, tindex[i].name)) return (i);
    }

    return (-1);
}






/*****************************************************************************\

    Function        find_tzfile - gets the timezone number (index into
                    tzfile array) given the tzfile name

    Synopsis        find_tzfile (tname);

                    NV_CHAR *tname          tzfile name

    Returns         NV_INT32                tzfile number

    Author          Jan C. Depner
    Date            08/01/02

\*****************************************************************************/

NV_INT32 find_tzfile (NV_CHAR *tname)
{
    NV_INT32                 i, j;


    j = -1;
    for (i = 0 ; i < hd.pub.tzfiles ; i++)
    {
        if (!strcmp (tname, get_tzfile (i)))
        {
            j = i;
            break;
        }
    }

    return (j);
}






/*****************************************************************************\

    Function        find_country - gets the timezone number (index into
                    country array) given the country name

    Synopsis        find_country (tname);

                    NV_CHAR *tname          country name

    Returns         NV_INT32                country number

    Author          Jan C. Depner
    Date            08/01/02

\*****************************************************************************/

NV_INT32 find_country (NV_CHAR *tname)
{
    NV_INT32                 i, j;
    NV_CHAR	*temp;

    temp = clip_string(tname);

    j = -1;
    for (i = 0 ; i < hd.pub.countries ; i++)
    {
        if (!strcmp (temp, get_country (i)))
        {
            j = i;
            break;
        }
    }

    return (j);
}






/*****************************************************************************\

    Function        find_level_units - gets the index into the level_units 
                    array given the level units name

    Synopsis        find_level_units (tname);

                    NV_CHAR *tname          units name (ex. "meters")

    Returns         NV_INT32                units number

    Author          Jan C. Depner
    Date            08/01/02

\*****************************************************************************/

NV_INT32 find_level_units (NV_CHAR *tname)
{
    NV_INT32                 i, j;


    j = -1;
    for (i = 0 ; i < hd.pub.level_unit_types ; i++)
    {
        if (!strcmp (get_level_units (i), clip_string (tname)))
        {
            j = i;
            break;
        }
    }

    return (j);
}






/*****************************************************************************\

    Function        find_dir_units - gets the index into the dir_units 
                    array given the direction units name

    Synopsis        find_dir_units (tname);

                    NV_CHAR *tname          units name (ex. "degrees true")

    Returns         NV_INT32                units number

    Author          Jan C. Depner
    Date            08/01/02

\*****************************************************************************/

NV_INT32 find_dir_units (NV_CHAR *tname)
{
    NV_INT32                 i, j;


    j = -1;
    for (i = 0 ; i < hd.pub.dir_unit_types ; i++)
    {
        if (!strcmp (get_dir_units (i), clip_string (tname)))
        {
            j = i;
            break;
        }
    }

    return (j);
}






/*****************************************************************************\

    Function        find_pedigree - gets the index into the pedigree array 
                    given the pedigree name

    Synopsis        find_pedigree (tname);

                    NV_CHAR *tname          pedigree name

    Returns         NV_INT32                pedigree number

    Author          Jan C. Depner
    Date            08/01/02

\*****************************************************************************/

NV_INT32 find_pedigree (NV_CHAR *tname)
{
    NV_INT32                 i, j;


    j = -1;
    for (i = 0 ; i < hd.pub.pedigree_types ; i++)
    {
        if (!strcmp (get_pedigree (i), clip_string (tname)))
        {
            j = i;
            break;
        }
    }
    return (j);
}






/*****************************************************************************\

    Function        find_datum - gets the index into the datum array given the
                    datum name

    Synopsis        find_datum (tname);

                    NV_CHAR *tname          datum name

    Returns         NV_INT32                datum number

    Author          Jan C. Depner
    Date            08/01/02

\*****************************************************************************/

NV_INT32 find_datum (NV_CHAR *tname)
{
    NV_INT32                 i, j;


    j = -1;
    for (i = 0 ; i < hd.pub.datum_types ; i++)
    {
        if (!strcmp (get_datum (i), clip_string (tname)))
        {
            j = i;
            break;
        }
    }

    return (j);
}






/*****************************************************************************\

    Function        find_constituent - gets the index into the constituent 
                    arrays for the named constituent.

    Synopsis        find_constituent (name);

                    NV_CHAR *name           constituent name (ex. M2)

    Returns         NV_INT32                index into constituent arrays or -1
                                            on failure

    Author          Jan C. Depner
    Date            08/01/02

\*****************************************************************************/

NV_INT32 find_constituent (NV_CHAR *name)
{
    NV_INT32               i;


    for (i = 0 ; i < hd.pub.constituents ; i++)
    {
        if (!strncmp (get_constituent (i), name, sizeof (name))) return (i);
    }

    return (-1);
}






/*****************************************************************************\

    Function        find_restriction - gets the index into the restriction 
                    array given the restriction name

    Synopsis        find_restriction (tname);

                    NV_CHAR *tname          restriction name

    Returns         NV_INT32                restriction number

    Author          Jan C. Depner
    Date            08/01/02

\*****************************************************************************/

NV_INT32 find_restriction (NV_CHAR *tname)
{
    NV_INT32                 i, j;


    j = -1;
    for (i = 0 ; i < hd.pub.restriction_types ; i++)
    {
        if (!strcmp (get_restriction (i), clip_string (tname)))
        {
            j = i;
            break;
        }
    }
    return (j);
}






/*****************************************************************************\

    Function        set_speed - sets the speed value for constituent "num"

    Synopsis        set_speed (num, value);

                    NV_INT32 num            constituent number
                    NV_FLOAT64 value        speed value

    Returns         void

    Author          Jan C. Depner
    Date            08/01/02

\*****************************************************************************/

void set_speed (NV_INT32 num, NV_FLOAT64 value)
{
    if (num >= 0 && num < hd.pub.constituents)
    {
        hd.speed[num] = value;
        modified = NVTrue;
    }
}


  



/*****************************************************************************\

    Function        set_equilibrium - sets the equilibrium argument for
                    constituent "num" and year "year"

    Synopsis        set_equilibrium (num, year, value);

                    NV_INT32 num            constituent number
                    NV_INT32 year           year
                    NV_FLOAT64 value        equilibrium argument

    Returns         void

    Author          Jan C. Depner
    Date            08/01/02

\*****************************************************************************/

void set_equilibrium (NV_INT32 num, NV_INT32 year, NV_FLOAT32 value)
{
    if (num >= 0 && num < hd.pub.constituents && 
        year < hd.pub.number_of_years) 
    {
        hd.equilibrium[num][year] = value;
        modified = NVTrue;
    }
}


  



/*****************************************************************************\

    Function        set_node_factor - sets the node factor for constituent 
                    "num" and year "year"

    Synopsis        set_node_factor (num, year, value);

                    NV_INT32 num            constituent number
                    NV_INT32 year           year
                    NV_FLOAT64 value        node factor

    Returns         void

    Author          Jan C. Depner
    Date            08/01/02

\*****************************************************************************/

void set_node_factor (NV_INT32 num, NV_INT32 year, NV_FLOAT32 value)
{
    if (num >= 0 && num < hd.pub.constituents && 
        year < hd.pub.number_of_years) 
    {
        hd.node_factor[num][year] = value;
        modified = NVTrue;
    }
}






/*****************************************************************************\

    Function        add_pedigree - adds a new pedigree to the database

    Synopsis        add_pedigree (tname, db);

                    NV_CHAR *tname          new pedigree string
                    DB_HEADER_PUBLIC *db    modified header

    Returns         NV_INT32                new pedigree index

    Author          Jan C. Depner
    Date            09/20/02

\*****************************************************************************/

NV_INT32 add_pedigree (NV_CHAR *tname, DB_HEADER_PUBLIC *db)
{
    NV_CHAR       c_tname[256];


    if (hd.pub.pedigree_types == hd.max_pedigree_types)
    {
        fprintf (stderr, 
            "You have exceeded the maximum number of pedigree types!\n");
        fprintf (stderr, 
            "You cannot add any new pedigree types.\n");
        fprintf (stderr, 
            "Modify the DEFAULT_PEDIGREE_BITS and rebuild the database.\n");
        exit (-1);
    }


    strcpy (c_tname, clip_string (tname));

    hd.pedigree[hd.pub.pedigree_types] = 
        (NV_CHAR *) calloc (strlen (c_tname) + 1, sizeof (NV_CHAR));

    if (hd.pedigree[hd.pub.pedigree_types] == NULL)
    {
        perror ("Allocating new pedigree string");
        exit (-1);
    }


    strcpy (hd.pedigree[hd.pub.pedigree_types], c_tname);

    hd.pub.pedigree_types++;
    *db = hd.pub;

    modified = NVTrue;

    return (hd.pub.pedigree_types - 1);
}






/*****************************************************************************\

    Function        add_tzfile - adds a new tzfile to the database

    Synopsis        add_tzfile (tname, db);

                    NV_CHAR *tname          new tzfile string
                    DB_HEADER_PUBLIC *db    modified header

    Returns         NV_INT32                new tzfile index

    Author          Jan C. Depner
    Date            09/20/02

\*****************************************************************************/

NV_INT32 add_tzfile (NV_CHAR *tname, DB_HEADER_PUBLIC *db)
{
    NV_CHAR       c_tname[256];


    if (hd.pub.tzfiles == hd.max_tzfiles)
    {
        fprintf (stderr, 
            "You have exceeded the maximum number of tzfile types!\n");
        fprintf (stderr, 
            "You cannot add any new tzfile types.\n");
        fprintf (stderr, 
            "Modify the DEFAULT_TZFILE_BITS and rebuild the database.\n");
        exit (-1);
    }


    strcpy (c_tname, clip_string (tname));

    hd.tzfile[hd.pub.tzfiles] = (NV_CHAR *) calloc (strlen (c_tname) + 1, 
        sizeof (NV_CHAR));

    if (hd.tzfile[hd.pub.tzfiles] == NULL)
    {
        perror ("Allocating new tzfile string");
        exit (-1);
    }


    strcpy (hd.tzfile[hd.pub.tzfiles], c_tname);

    hd.pub.tzfiles++;
    *db = hd.pub;


    modified = NVTrue;

    return (hd.pub.tzfiles - 1);
}






/*****************************************************************************\

    Function        add_country - adds a new country to the database

    Synopsis        add_country (tname, db);

                    NV_CHAR *tname          new country string
                    DB_HEADER_PUBLIC *db    modified header

    Returns         NV_INT32                new country index

    Author          Jan C. Depner
    Date            09/20/02

\*****************************************************************************/

NV_INT32 add_country (NV_CHAR *tname, DB_HEADER_PUBLIC *db)
{
    NV_CHAR       c_tname[256];


    if (hd.pub.countries == hd.max_countries)
    {
        fprintf (stderr, 
            "You have exceeded the maximum number of country names!\n");
        fprintf (stderr, 
            "You cannot add any new country names.\n");
        fprintf (stderr, 
            "Modify the DEFAULT_COUNTRY_BITS and rebuild the database.\n");
        exit (-1);
    }

    /* H2 added clip_string(tname) to take care of cases when the country name
       had leading/trailing spaces. */
    strcpy (c_tname, clip_string (tname));

    hd.country[hd.pub.countries] = (NV_CHAR *) calloc (strlen (c_tname) + 1,
        sizeof (NV_CHAR));

    if (hd.country[hd.pub.countries] == NULL)
    {
        perror ("Allocating new country string");
        exit (-1);
    }


    strcpy (hd.country[hd.pub.countries], c_tname);

    hd.pub.countries++;
    *db = hd.pub;


    modified = NVTrue;

    return (hd.pub.countries - 1);
}






/*****************************************************************************\

    Function        add_datum - adds a new datum to the database

    Synopsis        add_datum (tname, db);

                    NV_CHAR *tname          new datum string
                    DB_HEADER_PUBLIC *db    modified header

    Returns         NV_INT32                new datum index

    Author          Jan C. Depner
    Date            09/20/02

\*****************************************************************************/

NV_INT32 add_datum (NV_CHAR *tname, DB_HEADER_PUBLIC *db)
{
    NV_CHAR       c_tname[256];


    if (hd.pub.datum_types == hd.max_datum_types)
    {
        fprintf (stderr, 
            "You have exceeded the maximum number of datum types!\n");
        fprintf (stderr, 
            "You cannot add any new datum types.\n");
        fprintf (stderr, 
            "Modify the DEFAULT_DATUM_BITS and rebuild the database.\n");
        exit (-1);
    }


    strcpy (c_tname, clip_string (tname));

    hd.datum[hd.pub.datum_types] = (NV_CHAR *) calloc (strlen (c_tname) + 1,
        sizeof (NV_CHAR));

    if (hd.datum[hd.pub.datum_types] == NULL)
    {
        perror ("Allocating new datum string");
        exit (-1);
    }


    strcpy (hd.datum[hd.pub.datum_types], c_tname);

    hd.pub.datum_types++;
    *db = hd.pub;

    modified = NVTrue;

    return (hd.pub.datum_types - 1);
}






/*****************************************************************************\

    Function        add_restriction - adds a new restriction to the database

    Synopsis        add_restriction (tname, db);

                    NV_CHAR *tname          new restriction string
                    DB_HEADER_PUBLIC *db    modified header

    Returns         NV_INT32                new restriction index

    Author          Jan C. Depner
    Date            09/20/02

\*****************************************************************************/

NV_INT32 add_restriction (NV_CHAR *tname, DB_HEADER_PUBLIC *db)
{
    NV_CHAR       c_tname[256];


    if (hd.pub.restriction_types == hd.max_restriction_types)
    {
        fprintf (stderr, 
            "You have exceeded the maximum number of restriction types!\n");
        fprintf (stderr, 
            "You cannot add any new restriction types.\n");
        fprintf (stderr, 
            "Modify the DEFAULT_RESTRICTION_BITS and rebuild the database.\n");
        exit (-1);
    }


    strcpy (c_tname, clip_string (tname));

    hd.restriction[hd.pub.restriction_types] = 
        (NV_CHAR *) calloc (strlen (c_tname) + 1, sizeof (NV_CHAR));

    if (hd.restriction[hd.pub.restriction_types] == NULL)
    {
        perror ("Allocating new restriction string");
        exit (-1);
    }


    strcpy (hd.restriction[hd.pub.restriction_types], c_tname);

    hd.pub.restriction_types++;
    *db = hd.pub;

    modified = NVTrue;

    return (hd.pub.restriction_types - 1);
}






/*****************************************************************************\

    Function        check_simple - checks tide record to see if it is a 
                    "simple" subordinate station.  

    Synopsis        check_simple (rec);

                    TIDE_RECORD rec         tide record

    Returns         NV_BOOL                 NVTrue if "simple"

    Author          Jan C. Depner
    Date            08/01/02

    Modified        David Flater
    Date            2003-03-27

    "Simplified" type 2 records were done away with 2003-03-27 per the
    discussion in http://www.flaterco.com/xtide/tcd_notes.html.  This
    function is now only a convenience to check whether the record
    *could* be simplified, and is used only by restore_tide_db to
    determine whether it should output a shorthand XML notation.

\*****************************************************************************/

NV_BOOL check_simple (TIDE_RECORD rec)
{
  if (rec.max_time_add       == rec.min_time_add        &&
      rec.max_level_add      == rec.min_level_add       &&
      rec.max_level_multiply == rec.min_level_multiply  &&
      rec.max_avg_level == 0   &&
      rec.min_avg_level == 0   &&
      rec.max_direction == 361 &&
      rec.min_direction == 361 &&
      rec.flood_begins  == NULLSLACKOFFSET &&
      rec.ebb_begins    == NULLSLACKOFFSET)
    return (NVTrue);

    return (NVFalse);
}






/*****************************************************************************\

    Function        header_checksum - compute the checksum for the ASCII 
                    portion of the database header

    Synopsis        header_checksum ();

    Returns         NV_U_INT32              checksum value

    Author          Jan C. Depner
    Date            08/01/02

\*****************************************************************************/

static NV_U_INT32 header_checksum ()
{
    NV_U_INT32          checksum, i, save_pos;
    NV_U_BYTE           *buf;
    NV_U_INT32		crc_table[256] = 
      {0x00000000,0x77073096,0xEE0E612C,0x990951BA,0x076DC419,0x706AF48F,
       0xE963A535,0x9E6495A3,0x0EDB8832,0x79DCB8A4,0xE0D5E91E,0x97D2D988,
       0x09B64C2B,0x7EB17CBD,0xE7B82D07,0x90BF1D91,0x1DB71064,0x6AB020F2,
       0xF3B97148,0x84BE41DE,0x1ADAD47D,0x6DDDE4EB,0xF4D4B551,0x83D385C7,
       0x136C9856,0x646BA8C0,0xFD62F97A,0x8A65C9EC,0x14015C4F,0x63066CD9,
       0xFA0F3D63,0x8D080DF5,0x3B6E20C8,0x4C69105E,0xD56041E4,0xA2677172,
       0x3C03E4D1,0x4B04D447,0xD20D85FD,0xA50AB56B,0x35B5A8FA,0x42B2986C,
       0xDBBBC9D6,0xACBCF940,0x32D86CE3,0x45DF5C75,0xDCD60DCF,0xABD13D59,
       0x26D930AC,0x51DE003A,0xC8D75180,0xBFD06116,0x21B4F4B5,0x56B3C423,
       0xCFBA9599,0xB8BDA50F,0x2802B89E,0x5F058808,0xC60CD9B2,0xB10BE924,
       0x2F6F7C87,0x58684C11,0xC1611DAB,0xB6662D3D,0x76DC4190,0x01DB7106,
       0x98D220BC,0xEFD5102A,0x71B18589,0x06B6B51F,0x9FBFE4A5,0xE8B8D433,
       0x7807C9A2,0x0F00F934,0x9609A88E,0xE10E9818,0x7F6A0DBB,0x086D3D2D,
       0x91646C97,0xE6635C01,0x6B6B51F4,0x1C6C6162,0x856530D8,0xF262004E,
       0x6C0695ED,0x1B01A57B,0x8208F4C1,0xF50FC457,0x65B0D9C6,0x12B7E950,
       0x8BBEB8EA,0xFCB9887C,0x62DD1DDF,0x15DA2D49,0x8CD37CF3,0xFBD44C65,
       0x4DB26158,0x3AB551CE,0xA3BC0074,0xD4BB30E2,0x4ADFA541,0x3DD895D7,
       0xA4D1C46D,0xD3D6F4FB,0x4369E96A,0x346ED9FC,0xAD678846,0xDA60B8D0,
       0x44042D73,0x33031DE5,0xAA0A4C5F,0xDD0D7CC9,0x5005713C,0x270241AA,
       0xBE0B1010,0xC90C2086,0x5768B525,0x206F85B3,0xB966D409,0xCE61E49F,
       0x5EDEF90E,0x29D9C998,0xB0D09822,0xC7D7A8B4,0x59B33D17,0x2EB40D81,
       0xB7BD5C3B,0xC0BA6CAD,0xEDB88320,0x9ABFB3B6,0x03B6E20C,0x74B1D29A,
       0xEAD54739,0x9DD277AF,0x04DB2615,0x73DC1683,0xE3630B12,0x94643B84,
       0x0D6D6A3E,0x7A6A5AA8,0xE40ECF0B,0x9309FF9D,0x0A00AE27,0x7D079EB1,
       0xF00F9344,0x8708A3D2,0x1E01F268,0x6906C2FE,0xF762575D,0x806567CB,
       0x196C3671,0x6E6B06E7,0xFED41B76,0x89D32BE0,0x10DA7A5A,0x67DD4ACC,
       0xF9B9DF6F,0x8EBEEFF9,0x17B7BE43,0x60B08ED5,0xD6D6A3E8,0xA1D1937E,
       0x38D8C2C4,0x4FDFF252,0xD1BB67F1,0xA6BC5767,0x3FB506DD,0x48B2364B,
       0xD80D2BDA,0xAF0A1B4C,0x36034AF6,0x41047A60,0xDF60EFC3,0xA867DF55,
       0x316E8EEF,0x4669BE79,0xCB61B38C,0xBC66831A,0x256FD2A0,0x5268E236,
       0xCC0C7795,0xBB0B4703,0x220216B9,0x5505262F,0xC5BA3BBE,0xB2BD0B28,
       0x2BB45A92,0x5CB36A04,0xC2D7FFA7,0xB5D0CF31,0x2CD99E8B,0x5BDEAE1D,
       0x9B64C2B0,0xEC63F226,0x756AA39C,0x026D930A,0x9C0906A9,0xEB0E363F,
       0x72076785,0x05005713,0x95BF4A82,0xE2B87A14,0x7BB12BAE,0x0CB61B38,
       0x92D28E9B,0xE5D5BE0D,0x7CDCEFB7,0x0BDBDF21,0x86D3D2D4,0xF1D4E242,
       0x68DDB3F8,0x1FDA836E,0x81BE16CD,0xF6B9265B,0x6FB077E1,0x18B74777,
       0x88085AE6,0xFF0F6A70,0x66063BCA,0x11010B5C,0x8F659EFF,0xF862AE69,
       0x616BFFD3,0x166CCF45,0xA00AE278,0xD70DD2EE,0x4E048354,0x3903B3C2,
       0xA7672661,0xD06016F7,0x4969474D,0x3E6E77DB,0xAED16A4A,0xD9D65ADC,
       0x40DF0B66,0x37D83BF0,0xA9BCAE53,0xDEBB9EC5,0x47B2CF7F,0x30B5FFE9,
       0xBDBDF21C,0xCABAC28A,0x53B39330,0x24B4A3A6,0xBAD03605,0xCDD70693,
       0x54DE5729,0x23D967BF,0xB3667A2E,0xC4614AB8,0x5D681B02,0x2A6F2B94,
       0xB40BBE37,0xC30C8EA1,0x5A05DF1B,0x2D02EF8D};


    save_pos = ftell (fp);

    fseek (fp, 0, SEEK_SET);

    if ((buf = (NV_U_BYTE *) calloc (hd.header_size, sizeof (NV_U_BYTE))) == 
        NULL)
    {
        perror ("Allocating checksum buffer");
        exit (-1);
    }

    checksum = ~0;

    fread (buf, hd.header_size, 1, fp);
    for (i = 0 ; i < hd.header_size ; i++) 
    {
	checksum = crc_table[(checksum ^ buf[i]) & 0xff] ^ (checksum >> 8);
    }
    checksum ^= ~0;
    
    free (buf);

    fseek (fp, save_pos, SEEK_SET);

    return (checksum);
}






/*****************************************************************************\

    Function        old_header_checksum - compute the old-style checksum for 
                    the ASCII portion of the database header just in case this
                    is a pre 1.02 file.

    Synopsis        old_header_checksum ();

    Returns         NV_U_INT32              checksum value

    Author          Jan C. Depner
    Date            11/15/02

\*****************************************************************************/

static NV_U_INT32 old_header_checksum ()
{
    NV_U_INT32          checksum, i, save_pos;
    NV_U_BYTE           *buf;


    save_pos = ftell (fp);

    checksum = 0;

    fseek (fp, 0, SEEK_SET);

    if ((buf = (NV_U_BYTE *) calloc (hd.header_size, sizeof (NV_U_BYTE))) == 
        NULL)
    {
        perror ("Allocating checksum buffer");
        exit (-1);
    }

    fread (buf, hd.header_size, 1, fp);

    for (i = 0 ; i < hd.header_size ; i++) checksum += buf[i];

    free (buf);

    fseek (fp, save_pos, SEEK_SET);

    return (checksum);
}






/*****************************************************************************\

    Function        write_tide_db_header - writes the database header to the
                    file

    Synopsis        write_tide_db_header ();

    Returns         void

    Author          Jan C. Depner
    Date            08/01/02

\*****************************************************************************/

static void write_tide_db_header ()
{
    time_t              systemtime;
    NV_INT32            i, j, start, temp_int, pos, size;
    static NV_CHAR      zero = 0;
    NV_U_BYTE           *buf, checksum_c[4];

    /* NV_U_INT32 save,utemp; */

    systemtime = time (&systemtime);


    fseek (fp, 0, SEEK_SET);


    fprintf (fp, "[VERSION] = %s\n", VERSION);

    fprintf (fp, "[LAST MODIFIED] = %s", asctime (gmtime (&systemtime)));

    fprintf (fp, "[HEADER SIZE] = %d\n", hd.header_size);
    fprintf (fp, "[NUMBER OF RECORDS] = %d\n", hd.pub.number_of_records);

    fprintf (fp, "[START YEAR] = %d\n", hd.pub.start_year);
    fprintf (fp, "[NUMBER OF YEARS] = %d\n", hd.pub.number_of_years);

    fprintf (fp, "[SPEED BITS] = %d\n", hd.speed_bits);
    fprintf (fp, "[SPEED SCALE] = %d\n", hd.speed_scale);
    fprintf (fp, "[SPEED OFFSET] = %d\n", hd.speed_offset);
    fprintf (fp, "[EQUILIBRIUM BITS] = %d\n", hd.equilibrium_bits);
    fprintf (fp, "[EQUILIBRIUM SCALE] = %d\n", hd.equilibrium_scale);
    fprintf (fp, "[EQUILIBRIUM OFFSET] = %d\n", hd.equilibrium_offset);
    fprintf (fp, "[NODE BITS] = %d\n", hd.node_bits);
    fprintf (fp, "[NODE SCALE] = %d\n", hd.node_scale);
    fprintf (fp, "[NODE OFFSET] = %d\n", hd.node_offset);
    fprintf (fp, "[AMPLITUDE BITS] = %d\n", hd.amplitude_bits);
    fprintf (fp, "[AMPLITUDE SCALE] = %d\n", hd.amplitude_scale);
    fprintf (fp, "[EPOCH BITS] = %d\n", hd.epoch_bits);
    fprintf (fp, "[EPOCH SCALE] = %d\n", hd.epoch_scale);

    fprintf (fp, "[RECORD TYPE BITS] = %d\n", hd.record_type_bits);
    fprintf (fp, "[LATITUDE BITS] = %d\n", hd.latitude_bits);
    fprintf (fp, "[LATITUDE SCALE] = %d\n", hd.latitude_scale);
    fprintf (fp, "[LONGITUDE BITS] = %d\n", hd.longitude_bits);
    fprintf (fp, "[LONGITUDE SCALE] = %d\n", hd.longitude_scale);
    fprintf (fp, "[RECORD SIZE BITS] = %d\n", hd.record_size_bits);

    fprintf (fp, "[STATION BITS] = %d\n", hd.station_bits);

    fprintf (fp, "[DATUM OFFSET BITS] = %d\n", hd.datum_offset_bits);
    fprintf (fp, "[DATUM OFFSET SCALE] = %d\n", hd.datum_offset_scale);
    fprintf (fp, "[DATE BITS] = %d\n", hd.date_bits);
    fprintf (fp, "[MONTHS ON STATION BITS] = %d\n", 
             hd.months_on_station_bits);
    fprintf (fp, "[CONFIDENCE VALUE BITS] = %d\n", hd.confidence_value_bits);


    fprintf (fp, "[TIME BITS] = %d\n", hd.time_bits);
    fprintf (fp, "[LEVEL ADD BITS] = %d\n", hd.level_add_bits);
    fprintf (fp, "[LEVEL ADD SCALE] = %d\n", hd.level_add_scale);
    fprintf (fp, "[LEVEL MULTIPLY BITS] = %d\n", hd.level_multiply_bits);
    fprintf (fp, "[LEVEL MULTIPLY SCALE] = %d\n", hd.level_multiply_scale);
    fprintf (fp, "[DIRECTION BITS] = %d\n", hd.direction_bits);

    fprintf (fp, "[LEVEL UNIT BITS] = %d\n", hd.level_unit_bits);
    fprintf (fp, "[LEVEL UNIT TYPES] = %d\n", hd.pub.level_unit_types);
    fprintf (fp, "[LEVEL UNIT SIZE] = %d\n", hd.level_unit_size);

    fprintf (fp, "[DIRECTION UNIT BITS] = %d\n", hd.dir_unit_bits);
    fprintf (fp, "[DIRECTION UNIT TYPES] = %d\n", hd.pub.dir_unit_types);
    fprintf (fp, "[DIRECTION UNIT SIZE] = %d\n", hd.dir_unit_size);

    fprintf (fp, "[RESTRICTION BITS] = %d\n", hd.restriction_bits);
    fprintf (fp, "[RESTRICTION TYPES] = %d\n", hd.pub.restriction_types);
    fprintf (fp, "[RESTRICTION SIZE] = %d\n", hd.restriction_size);

    fprintf (fp, "[PEDIGREE BITS] = %d\n", hd.pedigree_bits);
    fprintf (fp, "[PEDIGREE TYPES] = %d\n", hd.pub.pedigree_types);
    fprintf (fp, "[PEDIGREE SIZE] = %d\n", hd.pedigree_size);

    fprintf (fp, "[DATUM BITS] = %d\n", hd.datum_bits);
    fprintf (fp, "[DATUM TYPES] = %d\n", hd.pub.datum_types);
    fprintf (fp, "[DATUM SIZE] = %d\n", hd.datum_size);

    fprintf (fp, "[CONSTITUENT BITS] = %d\n", hd.constituent_bits);
    fprintf (fp, "[CONSTITUENTS] = %d\n", hd.pub.constituents);
    fprintf (fp, "[CONSTITUENT SIZE] = %d\n", hd.constituent_size);

    fprintf (fp, "[TZFILE BITS] = %d\n", hd.tzfile_bits);
    fprintf (fp, "[TZFILES] = %d\n", hd.pub.tzfiles);
    fprintf (fp, "[TZFILE SIZE] = %d\n", hd.tzfile_size);

    fprintf (fp, "[COUNTRY BITS] = %d\n", hd.country_bits);
    fprintf (fp, "[COUNTRIES] = %d\n", hd.pub.countries);
    fprintf (fp, "[COUNTRY SIZE] = %d\n", hd.country_size);

    fprintf (fp, "[END OF FILE] = %d\n", hd.end_of_file);
    fprintf (fp, "[END OF ASCII HEADER DATA]\n");


    /*  Fill the remainder of the [HEADER SIZE] ASCII header with zeroes.  */

    start = ftell (fp);
    for (i = start ; i < hd.header_size ; i++) fwrite (&zero, 1, 1, fp);
    fflush (fp);


    /*  Compute and save the checksum. */

    bit_pack (checksum_c, 0, 32, header_checksum ());
    fwrite (checksum_c, 4, 1, fp);


    /*  NOTE : Using strcpy for character strings (no endian issue).  */

    /*  Write level units.  */

    pos = 0;
    size = hd.pub.level_unit_types * hd.level_unit_size;

    if ((buf = (NV_U_BYTE *) calloc (size, sizeof (NV_U_BYTE))) == NULL)
    {
        perror ("Allocating unit write buffer");
        exit (-1);
    }
    memset (buf, 0, size);

    for (i = 0 ; i < hd.pub.level_unit_types ; i++)
    {
        strcpy ((NV_CHAR *) &buf[pos], hd.level_unit[i]);
        pos += hd.level_unit_size;
    }

    fwrite (buf, pos, 1, fp);
    free (buf);


    /*  Write direction units.  */

    pos = 0;
    size = hd.pub.dir_unit_types * hd.dir_unit_size;

    if ((buf = (NV_U_BYTE *) calloc (size, sizeof (NV_U_BYTE))) == NULL)
    {
        perror ("Allocating unit write buffer");
        exit (-1);
    }
    memset (buf, 0, size);

    for (i = 0 ; i < hd.pub.dir_unit_types ; i++)
    {
        strcpy ((NV_CHAR *) &buf[pos], hd.dir_unit[i]);
        pos += hd.dir_unit_size;
    }

    fwrite (buf, pos, 1, fp);
    free (buf);


    /*  Write restrictions.  */

    pos = 0;
    size = hd.max_restriction_types * hd.restriction_size;

    if ((buf = (NV_U_BYTE *) calloc (size, sizeof (NV_U_BYTE))) == NULL)
    {
        perror ("Allocating restriction write buffer");
        exit (-1);
    }
    memset (buf, 0, size);

    for (i = 0 ; i < hd.max_restriction_types ; i++)
    {
        if (i == hd.pub.restriction_types) break;

        strcpy ((NV_CHAR *) &buf[pos], hd.restriction[i]);
        pos += hd.restriction_size;
    }
    memcpy (&buf[pos], "__END__", 7);

    fwrite (buf, size, 1, fp);
    free (buf);


    /*  Write pedigrees.  */

    pos = 0;
    size = hd.max_pedigree_types * hd.pedigree_size;

    if ((buf = (NV_U_BYTE *) calloc (size, sizeof (NV_U_BYTE))) == NULL)
    {
        perror ("Allocating pedigree write buffer");
        exit (-1);
    }
    memset (buf, 0, size);

    for (i = 0 ; i < hd.max_pedigree_types ; i++)
    {
        if (i == hd.pub.pedigree_types) break;

        strcpy ((NV_CHAR *) &buf[pos], hd.pedigree[i]);
        pos += hd.pedigree_size;
    }
    memcpy (&buf[pos], "__END__", 7);

    fwrite (buf, size, 1, fp);
    free (buf);


    /*  Write tzfiles.  */

    pos = 0;
    size = hd.max_tzfiles * hd.tzfile_size;

    if ((buf = (NV_U_BYTE *) calloc (size, sizeof (NV_U_BYTE))) == NULL)
    {
        perror ("Allocating tzfile write buffer");
        exit (-1);
    }
    memset (buf, 0, size);

    for (i = 0 ; i < hd.max_tzfiles ; i++)
    {
        if (i == hd.pub.tzfiles) break;

        strcpy ((NV_CHAR *) &buf[pos], hd.tzfile[i]);
        pos += hd.tzfile_size;
    }
    memcpy (&buf[pos], "__END__", 7);

    fwrite (buf, size, 1, fp);
    free (buf);


    /*  Write countries.  */

    pos = 0;
    size = hd.max_countries * hd.country_size;

    if ((buf = (NV_U_BYTE *) calloc (size, sizeof (NV_U_BYTE))) == NULL)
    {
        perror ("Allocating country write buffer");
        exit (-1);
    }
    memset (buf, 0, size);

    for (i = 0 ; i < hd.max_countries ; i++)
    {
        if (i == hd.pub.countries) break;

        strcpy ((NV_CHAR *) &buf[pos], hd.country[i]);
        pos += hd.country_size;
    }
    memcpy (&buf[pos], "__END__", 7);

    fwrite (buf, size, 1, fp);
    free (buf);


    /*  Write datums.  */

    pos = 0;
    size = hd.max_datum_types * hd.datum_size;

    if ((buf = (NV_U_BYTE *) calloc (size, sizeof (NV_U_BYTE))) == NULL)
    {
        perror ("Allocating datum write buffer");
        exit (-1);
    }
    memset (buf, 0, size);

    for (i = 0 ; i < hd.max_datum_types ; i++)
    {
        if (i == hd.pub.datum_types) break;

        strcpy ((NV_CHAR *) &buf[pos], hd.datum[i]);
        pos += hd.datum_size;
    }
    memcpy (&buf[pos], "__END__", 7);

    fwrite (buf, size, 1, fp);
    free (buf);


    /*  Write constituent names.  */

    pos = 0;
    size = hd.pub.constituents * hd.constituent_size;

    if ((buf = (NV_U_BYTE *) calloc (size, sizeof (NV_U_BYTE))) == NULL)
    {
        perror ("Allocating constituent write buffer");
        exit (-1);
    }
    memset (buf, 0, size);

    for (i = 0 ; i < hd.pub.constituents ; i++)
    {
        strcpy ((NV_CHAR *) &buf[pos], hd.constituent[i]);
        pos += hd.constituent_size;
    }

    fwrite (buf, pos, 1, fp);
    free (buf);


    /*  NOTE : Using bit_pack for integers.  */

    /*  Write speeds.  */

    pos = 0;
    size = ((hd.pub.constituents * hd.speed_bits) / 8) + 1;

    if ((buf = (NV_U_BYTE *) calloc (size, sizeof (NV_U_BYTE))) == NULL)
    {
        perror ("Allocating speed write buffer");
        exit (-1);
    }
    memset (buf, 0, size);

    for (i = 0 ; i < hd.pub.constituents ; i++)
    {
        temp_int = NINT (hd.speed[i] * hd.speed_scale) - hd.speed_offset;
        bit_pack (buf, pos, hd.speed_bits, temp_int);
        pos += hd.speed_bits;
    }

    fwrite (buf, size, 1, fp);
    free (buf);


    /*  Write equilibrium arguments.  */

    pos = 0;
    size = ((hd.pub.constituents *  hd.pub.number_of_years * 
        hd.equilibrium_bits) / 8) + 1;

    if ((buf = (NV_U_BYTE *) calloc (size, sizeof (NV_U_BYTE))) == NULL)
    {
        perror ("Allocating equilibrium write buffer");
        exit (-1);
    }
    memset (buf, 0, size);

    for (i = 0 ; i < hd.pub.constituents ; i++)
    {
        for (j = 0 ; j < hd.pub.number_of_years ; j++)
        {
            temp_int = NINT (hd.equilibrium[i][j] * hd.equilibrium_scale) - 
                hd.equilibrium_offset;
            bit_pack (buf, pos, hd.equilibrium_bits, temp_int);
            pos += hd.equilibrium_bits;
        }
    }

    fwrite (buf, size, 1, fp);
    free (buf);


    /*  Write node factors.  */

    pos = 0;
    size = ((hd.pub.constituents * hd.pub.number_of_years * 
        hd.node_bits) / 8) + 1;

    if ((buf = (NV_U_BYTE *) calloc (size, sizeof (NV_U_BYTE))) == NULL)
    {
        perror ("Allocating node write buffer");
        exit (-1);
    }
    memset (buf, 0, size);

    for (i = 0 ; i < hd.pub.constituents ; i++)
    {
        for (j = 0 ; j < hd.pub.number_of_years ; j++)
        {
            temp_int = NINT (hd.node_factor[i][j] * hd.node_scale) -
                hd.node_offset;
            bit_pack (buf, pos, hd.node_bits, temp_int);
            pos += hd.node_bits;
        }
    }

    fwrite (buf, size, 1, fp);
    free (buf);
}






/***************************************************************************\
*                                                                           *
*   Module Name:        get_string                                          *
*                                                                           *
*   Programmer(s):      Jan C. Depner                                       *
*                                                                           *
*   Date Written:       December 1994                                       *
*                                                                           *
*   Purpose:            Parses the input string for the equals sign and     *
*                       returns everything to the right.                    *
*                                                                           *
*   Arguments:          *in     -   Input string                            *
*                       *out    -   Output string                           *
*                                                                           *
*   Return Value:       None                                                *
*                                                                           *
\***************************************************************************/

static void get_string (NV_CHAR *in, NV_CHAR *out)
{
    NV_INT32            i, start, length;


    start = 0;
    length = 0;

    strcpy (out, (strchr (in, '=') + 1));

    /*  Search for first non-blank character.   */

    for (i = 0 ; i < strlen (out) ; i++)
    {
        if (out[i] != ' ')
        {
            start = i;
            break;
        }
    }

    /*  Search for last non-blank, non-zero, non-LF, non-CR character.    */

    for (i = strlen (out) ; i >= 0 ; i--)
    {
        if (out[i] != ' ' && out[i] != 0 && out[i] != 10 && out[i] != 13)
        {
            length = (i + 1) - start;
            break;
        }
    }

    strncpy (out, &out[start], length);
    out[length] = 0;
}






/*****************************************************************************\

    Function        unpack_partial_tide_record - unpacks the "header" portion
                    of a tide record from the supplied buffer

    Synopsis        unpack_partial_tide_record (buf, rec, pos);

                    NV_U-CHAR *buf          input buffer
                    TIDE_RECORD *rec        tide record
                    NV_INT32 *pos           final position in buffer after
                                            unpacking the header

    Returns         void

    Author          Jan C. Depner
    Date            08/01/02

\*****************************************************************************/

static void unpack_partial_tide_record (NV_U_BYTE *buf, TIDE_RECORD *rec, 
    NV_INT32 *pos)
{
    NV_INT32                i, temp_int;


    *pos = 0;

    rec->header.record_size = bit_unpack (buf, *pos, hd.record_size_bits);
    *pos += hd.record_size_bits;

    rec->header.record_type = bit_unpack (buf, *pos, hd.record_type_bits);
    *pos += hd.record_type_bits;

    temp_int = signed_bit_unpack (buf, *pos, hd.latitude_bits);
    rec->header.latitude = (NV_FLOAT64) temp_int / hd.latitude_scale;
    *pos += hd.latitude_bits;

    temp_int = signed_bit_unpack (buf, *pos, hd.longitude_bits);
    rec->header.longitude = (NV_FLOAT64) temp_int / hd.longitude_scale;
    *pos += hd.longitude_bits;

    rec->header.tzfile = bit_unpack (buf, *pos, hd.tzfile_bits);
    *pos += hd.tzfile_bits;

    for (i = 0 ; i < NAME_LENGTH ; i++)
    {
        rec->header.name[i] = bit_unpack (buf, *pos, 8);
        (*pos) += 8;
        if (rec->header.name[i] == 0) break;
    }

    rec->header.reference_station = 
        signed_bit_unpack (buf, *pos, hd.station_bits);
    *pos += hd.station_bits;
}






/*****************************************************************************\

    Function        read_partial_tide_record - reads the "header" portion
                    of a tide record from the database.  This is used to index
                    the database on opening.

    Synopsis        read_partial_tide_record (num, rec);

                    NV_INT32 num            record number
                    TIDE_RECORD *rec        tide record

    Returns         NV_INT32                record number read

    Author          Jan C. Depner
    Date            08/01/02

\*****************************************************************************/

static NV_INT32 read_partial_tide_record (NV_INT32 num, TIDE_RECORD *rec)
{
    NV_U_BYTE               *buf;
    NV_INT32                size, pos;


    /*  Read just the record size, record type, position, time zone, and
        name.  */

    size = hd.record_size_bits + hd.record_type_bits + hd.latitude_bits +
        hd.longitude_bits + hd.tzfile_bits + (NAME_LENGTH * 8) +
        hd.station_bits;

    size = size / 8 + 1;

    if ((buf = (NV_U_BYTE *) calloc (size, sizeof (NV_U_BYTE))) == NULL)
    {
        perror ("Allocating partial tide record buffer");
        exit (-1);
    }


    current_record = num;


    fseek (fp, tindex[num].address, SEEK_SET);

    fread (buf, size, 1, fp);


    pos = 0;
    unpack_partial_tide_record (buf, rec, &pos);


    free (buf);

    return (num);
}






/*****************************************************************************\

    Function        read_tide_db_header - reads the tide database header

    Synopsis        read_tide_db_header (file);

                    NV_CHAR *file           database file name

    Returns         NV_BOOL                 NVTrue if header is correct

    Author          Jan C. Depner
    Date            08/01/02

    Modified        David Flater
    Date            2003-11-16

    Added handling for zero tide records, which happens on new
    database create.

\*****************************************************************************/

static NV_BOOL read_tide_db_header (NV_CHAR *file)
{
    NV_INT32            i, j, pos, size, temp_int, key_count, offset;
    NV_CHAR             varin[1024], info[1024], ctemp[1024];
    NV_F64_COORD2       central;
    NV_I32_COORD2       coord;
    NV_U_INT32          utemp;
    NV_U_BYTE           *buf, checksum_c[4];
    TIDE_RECORD         rec;



    strcpy (hd.pub.version, "NO VERSION");


    /*  Compute the number of key phrases there are to match.  */

    key_count = sizeof (keys) / sizeof (KEY);


    /*  Zero out the header structure.  */

    memset (&hd, 0, sizeof (hd));


    /*  Handle the ASCII header data.  */

    while (fgets (varin, sizeof (varin), fp) != NULL)
    {
        if (strstr (varin, "[END OF ASCII HEADER DATA]")) break;


        /*  Put everything to the right of the equals sign in 'info'.   */

        if (strchr (varin, '=') != NULL) strcpy (info, (strchr (varin, '=') + 
            1));


        /*  Check against all possible keys.  */

        for (i = 0 ; i < key_count ; i++)        
        {
            /*  Set the array offset for repeating fields.  */
                
            if (keys[i].count == NULL)
            {
                offset = 0;
            }
            else
            {
                offset = *keys[i].count;
            }


            /*  Check input for matching strings and load values if 
                found.  */

            if (strstr (varin, keys[i].keyphrase) != NULL)
            {
                /*  Strings.  */

                if (!strcmp (keys[i].datatype, "string"))
                {
                    get_string (varin, ctemp);
                    strcpy ((keys[i].address.c + offset), ctemp);
                }


                /*  Signed 16 bit ints.  */

                if (!strcmp (keys[i].datatype, "i16"))
                {
                    sscanf (info, "%hd", keys[i].address.i16 + offset);
                }



                /*  Signed 32 bit ints.  */

                if (!strcmp (keys[i].datatype, "i32"))
                {
                    sscanf (info, "%d", keys[i].address.i32 + offset);
                }


                /*  Unsigned 8 bit ints.  */

                if (!strcmp (keys[i].datatype, "uc"))
                {
                    sscanf (info, "%u", &utemp);
                    *(keys[i].address.uc + offset) = utemp;
                }


                /*  Unsigned 16 bit ints.  */

                if (!strcmp (keys[i].datatype, "ui16"))
                {
                    sscanf (info, "%hd", keys[i].address.ui16 + offset);
                }


                /*  Unsigned 32 bit ints.  */

                if (!strcmp (keys[i].datatype, "ui32"))
                {
                    sscanf (info, "%d", keys[i].address.ui32 + offset);
                }


                /*  32 bit floats.  */

                if (!strcmp (keys[i].datatype, "f32"))
                {
                    sscanf (info, "%f", keys[i].address.f32 + offset);
                }


                /*  64 bit floats.  */

                if (!strcmp (keys[i].datatype, "f64"))
                {
                    sscanf (info, "%lf", (keys[i].address.f64 + offset));
                }


                /*  64 bit float coordinate pairs.  */

                if (!strcmp (keys[i].datatype, "f64c2"))
                {
                    sscanf (info, "%lf,%lf", &central.y, &central.x);
                    *(keys[i].address.f64c2 + offset) = central;
                }


                /*  32 bit signed int coordinate pairs.  */

                if (!strcmp (keys[i].datatype, "i32c2"))
                {
                    sscanf (info, "%d,%d", &coord.x, &coord.y);
                    *(keys[i].address.i32c2 + offset) = coord;
                }


                /*  Booleans.  */

                if (!strcmp (keys[i].datatype, "b"))
                {
                    sscanf (info, "%d", &temp_int);
                    *(keys[i].address.b + offset) = temp_int;
                }


                /*  64 bit signed ints.  */

                if (!strcmp (keys[i].datatype, "i64"))
                {
                    sscanf (info, "%lld", (keys[i].address.i64 + offset));
                }
                

                /*  Increment the repeating field counter.  */

                if (keys[i].count != NULL) (*keys[i].count)++;
            }
        }
    }


    /*  We didn't get a valid version string.  */

    if (!strcmp (hd.pub.version, "NO VERSION")) 
    {
        fclose (fp); /* DWF */
        return (NVFalse);
    }



    /*  Move to end of ASCII header.  */

    fseek (fp, hd.header_size, SEEK_SET);


    /*  Read and check the checksum. */

    fread (checksum_c, 4, 1, fp);
    utemp = bit_unpack (checksum_c, 0, 32);

    if (utemp != header_checksum ())
    {
        /*  Check for an old-style checksum.  */

        if (utemp != old_header_checksum ())
        {
            /* DWF: made error text less insulting */
            /* JCD: Aw, Dave, you're no fun. */
            /* DWF: No good deed goes unpunished. */
            fprintf (stderr, 
                "\nlibtcd error:  The header checksum is incorrect.\n");
            fprintf (stderr, 
                "Either the file has become corrupted, the HFILE_PATH environment\n");
            fprintf (stderr, 
                "variable is pointing to the wrong file, or someone has\n");
            fprintf (stderr, 
                "modified the ASCII portion of the header (don't do that).\n\n");
            fclose (fp);
            return NVFalse; /* DWF -- instead of exiting */
        }
    }
    fseek (fp, hd.header_size + 4, SEEK_SET);


    /*  Set the max possible restriction types based on the number of bits
        used.  */

    hd.max_restriction_types = NINT (pow (2.0, 
        (NV_FLOAT64) hd.restriction_bits));


    /*  Set the max possible pedigree types based on the number of bits
        used.  */

    hd.max_pedigree_types = NINT (pow (2.0, (NV_FLOAT64) hd.pedigree_bits));


    /*  Set the max possible tzfiles based on the number of bits used.  */

    hd.max_tzfiles = NINT (pow (2.0, (NV_FLOAT64) hd.tzfile_bits));


    /*  Set the max possible countries based on the number of bits used.  */

    hd.max_countries = NINT (pow (2.0, (NV_FLOAT64) hd.country_bits));


    /*  Set the max possible datum types based on the number of bits
        used.  */

    hd.max_datum_types = NINT (pow (2.0, (NV_FLOAT64) hd.datum_bits));


    /*  NOTE : Using strcpy for character strings (no endian issue).  */

    /*  Read level units.  */

    hd.level_unit = (NV_CHAR **) calloc (hd.pub.level_unit_types, 
        sizeof (NV_CHAR *));

    if ((buf = (NV_U_BYTE *) calloc (hd.level_unit_size, 
        sizeof (NV_U_BYTE))) == NULL)
    {
        perror ("Allocating level unit read buffer");
        exit (-1);
    }

    for (i = 0 ; i < hd.pub.level_unit_types ; i++)
    {
        fread (buf, hd.level_unit_size, 1, fp);
        hd.level_unit[i] = (NV_CHAR *) calloc (strlen (buf) + 1, 
            sizeof (NV_CHAR));
        strcpy (hd.level_unit[i], (NV_CHAR *) buf);
    }
    free (buf);



    /*  Read direction units.  */

    hd.dir_unit = (NV_CHAR **) calloc (hd.pub.dir_unit_types, 
        sizeof (NV_CHAR *));

    if ((buf = (NV_U_BYTE *) calloc (hd.dir_unit_size, sizeof (NV_U_BYTE))) ==
        NULL)
    {
        perror ("Allocating dir unit read buffer");
        exit (-1);
    }

    for (i = 0 ; i < hd.pub.dir_unit_types ; i++)
    {
        fread (buf, hd.dir_unit_size, 1, fp);
        hd.dir_unit[i] = (NV_CHAR *) calloc (strlen (buf) + 1, 
            sizeof (NV_CHAR));
        strcpy (hd.dir_unit[i], (NV_CHAR *) buf);
    }
    free (buf);



    /*  Read restrictions.  */
 
    utemp = ftell (fp);
    hd.restriction = (NV_CHAR **) calloc (hd.max_restriction_types, 
        sizeof (NV_CHAR *));

    if ((buf = (NV_U_BYTE *) calloc (hd.restriction_size, 
        sizeof (NV_U_BYTE))) == NULL)
    {
        perror ("Allocating restriction read buffer");
        exit (-1);
    }

    hd.pub.restriction_types = 0;
    for (i = 0 ; i < hd.max_restriction_types ; i++)
    {
        fread (buf, hd.restriction_size, 1, fp);
        if (!strcmp (buf, "__END__"))
        { 
            hd.pub.restriction_types = i;
            break;
        }
        hd.restriction[i] = (NV_CHAR *) calloc (strlen (buf) + 1, 
            sizeof (NV_CHAR));
        strcpy (hd.restriction[i], (NV_CHAR *) buf);
    }
    free (buf);
    fseek (fp, utemp + hd.max_restriction_types * hd.restriction_size, 
        SEEK_SET);



    /*  Read pedigrees.  */
 
    utemp = ftell (fp);
    hd.pedigree = (NV_CHAR **) calloc (hd.max_pedigree_types,
        sizeof (NV_CHAR *));

    if ((buf = (NV_U_BYTE *) calloc (hd.pedigree_size, sizeof (NV_U_BYTE))) ==
        NULL)
    {
        perror ("Allocating pedigree read buffer");
        exit (-1);
    }

    hd.pub.pedigree_types = 0;
    for (i = 0 ; i < hd.max_pedigree_types ; i++)
    {
        fread (buf, hd.pedigree_size, 1, fp);
        if (!strcmp (buf, "__END__"))
        { 
            hd.pub.pedigree_types = i;
            break;
        }
        hd.pedigree[i] = (NV_CHAR *) calloc (strlen (buf) + 1, 
            sizeof (NV_CHAR));
        strcpy (hd.pedigree[i], (NV_CHAR *) buf);
    }
    free (buf);
    fseek (fp, utemp + hd.max_pedigree_types * hd.pedigree_size, SEEK_SET);



    /*  Read tzfiles.  */

    utemp = ftell (fp);
    hd.tzfile = (NV_CHAR **) calloc (hd.max_tzfiles, sizeof (NV_CHAR *));

    if ((buf = (NV_U_BYTE *) calloc (hd.tzfile_size, sizeof (NV_U_BYTE))) == 
        NULL)
    {
        perror ("Allocating tzfile read buffer");
        exit (-1);
    }

    hd.pub.tzfiles = 0;
    for (i = 0 ; i < hd.max_tzfiles ; i++)
    {
        fread (buf, hd.tzfile_size, 1, fp);
        if (!strcmp (buf, "__END__")) 
        {
            hd.pub.tzfiles = i;
            break;
        }
        hd.tzfile[i] = (NV_CHAR *) calloc (strlen (buf) + 1, sizeof (NV_CHAR));
        strcpy (hd.tzfile[i], (NV_CHAR *) buf);
    }
    free (buf);
    fseek (fp, utemp + hd.max_tzfiles * hd.tzfile_size, SEEK_SET);


    /*  Read countries.  */

    utemp = ftell (fp);
    hd.country = (NV_CHAR **) calloc (hd.max_countries, sizeof (NV_CHAR *));

    if ((buf = (NV_U_BYTE *) calloc (hd.country_size, sizeof (NV_U_BYTE))) == 
        NULL)
    {
        perror ("Allocating country read buffer");
        exit (-1);
    }

    hd.pub.countries = 0;
    for (i = 0 ; i < hd.max_countries ; i++)
    {
        fread (buf, hd.country_size, 1, fp);
        if (!strcmp (buf, "__END__")) 
        {
            hd.pub.countries = i;
            break;
        }
        hd.country[i] = (NV_CHAR *) calloc (strlen (buf) + 1,  
            sizeof (NV_CHAR));
        strcpy (hd.country[i], (NV_CHAR *) buf);
    }
    free (buf);
    fseek (fp, utemp + hd.max_countries * hd.country_size, SEEK_SET);


    /*  Read datums.  */
 
    utemp = ftell (fp);
    hd.datum = (NV_CHAR **) calloc (hd.max_datum_types, sizeof (NV_CHAR *));

    if ((buf = (NV_U_BYTE *) calloc (hd.datum_size, sizeof (NV_U_BYTE))) == 
        NULL)
    {
        perror ("Allocating datum read buffer");
        exit (-1);
    }

    hd.pub.datum_types = 0;
    for (i = 0 ; i < hd.max_datum_types ; i++)
    {
        fread (buf, hd.datum_size, 1, fp);
        if (!strcmp (buf, "__END__"))
        { 
            hd.pub.datum_types = i;
            break;
        }
        hd.datum[i] = (NV_CHAR *) calloc (strlen (buf) + 1, sizeof (NV_CHAR));
        strcpy (hd.datum[i], (NV_CHAR *) buf);
    }
    free (buf);
    fseek (fp, utemp + hd.max_datum_types * hd.datum_size, SEEK_SET);



    /*  Read constituent names.  */

    hd.constituent = 
        (NV_CHAR **) calloc (hd.pub.constituents, sizeof (NV_CHAR *));

    if ((buf = (NV_U_BYTE *) calloc (hd.constituent_size, 
        sizeof (NV_U_BYTE))) == NULL)
    {
        perror ("Allocating constituent read buffer");
        exit (-1);
    }

    for (i = 0 ; i < hd.pub.constituents ; i++)
    {
        fread (buf, hd.constituent_size, 1, fp);
        hd.constituent[i] = (NV_CHAR *) calloc (strlen (buf) + 1, 
            sizeof (NV_CHAR));
        strcpy (hd.constituent[i], (NV_CHAR *) buf);
    }
    free (buf);



    /*  NOTE: Using bit_unpack to get integers.  */

    /*  Read speeds.  */

    hd.speed = (NV_FLOAT64 *) calloc (hd.pub.constituents, 
        sizeof (NV_FLOAT64));

    pos = 0;
    size = ((hd.pub.constituents * hd.speed_bits) / 8) + 1;

    if ((buf = (NV_U_BYTE *) calloc (size, sizeof (NV_U_BYTE))) == NULL)
    {
        perror ("Allocating speed read buffer");
        exit (-1);
    }

    fread (buf, size, 1, fp);

    for (i = 0 ; i < hd.pub.constituents ; i++)
    {
        temp_int = bit_unpack (buf, pos, hd.speed_bits);
        hd.speed[i] = (NV_FLOAT64) (temp_int + hd.speed_offset) / 
            hd.speed_scale;
        pos += hd.speed_bits;
    }
    free (buf);



    /*  Read equilibrium arguments.  */

    hd.equilibrium = (NV_FLOAT32 **) calloc (hd.pub.constituents,
        sizeof (NV_FLOAT32 *));

    for (i = 0 ; i < hd.pub.constituents ; i++)
    {
        hd.equilibrium[i] = (NV_FLOAT32 *) calloc (hd.pub.number_of_years,
            sizeof (NV_FLOAT32));
    }



    pos = 0;
    size = ((hd.pub.constituents * hd.pub.number_of_years * 
        hd.equilibrium_bits) / 8) + 1;

    if ((buf = (NV_U_BYTE *) calloc (size, sizeof (NV_U_BYTE))) == NULL)
    {
        perror ("Allocating equilibrium read buffer");
        exit (-1);
    }

    fread (buf, size, 1, fp);

    for (i = 0 ; i < hd.pub.constituents ; i++)
    {
        for (j = 0 ; j < hd.pub.number_of_years ; j++)
        {
            temp_int = bit_unpack (buf, pos, hd.equilibrium_bits);
            hd.equilibrium[i][j] = (NV_FLOAT32) (temp_int + 
                hd.equilibrium_offset) / hd.equilibrium_scale;
            pos += hd.equilibrium_bits;
        }
    }
    free (buf);



    /*  Read node factors.  */

    hd.node_factor = (NV_FLOAT32 **) calloc (hd.pub.constituents,
        sizeof (NV_FLOAT32 *));

    for (i = 0 ; i < hd.pub.constituents ; i++)
    {
        hd.node_factor[i] = 
            (NV_FLOAT32 *) calloc (hd.pub.number_of_years, 
            sizeof (NV_FLOAT32));
    }
    

    pos = 0;
    size = ((hd.pub.constituents * hd.pub.number_of_years * 
        hd.node_bits) / 8) + 1;

    if ((buf = (NV_U_BYTE *) calloc (size, sizeof (NV_U_BYTE))) == NULL)
    {
        perror ("Allocating node read buffer");
        exit (-1);
    }

    fread (buf, size, 1, fp);

    for (i = 0 ; i < hd.pub.constituents ; i++)
    {
        for (j = 0 ; j < hd.pub.number_of_years ; j++)
        {
            temp_int = bit_unpack (buf, pos, hd.node_bits);
            hd.node_factor[i][j] = (NV_FLOAT32) (temp_int + 
                hd.node_offset) / hd.node_scale;
            pos += hd.node_bits;
        }
    }
    free (buf);


    /*  Read the header portion of all of the records in the file and save
        the record size, address, and name.  */

    /* DWF added test for zero 2003-11-16 -- happens on create new db */
    if (hd.pub.number_of_records) {
      if ((tindex = (TIDE_INDEX *) calloc (hd.pub.number_of_records, 
          sizeof (TIDE_INDEX))) == NULL) {
          perror ("Allocating tide index");
          exit (-1);
      }
      /*  Set the first address to be immediately after the header  */
      tindex[0].address = ftell (fp);
    } else tindex = NULL; /* May as well be explicit... */

    for (i = 0 ; i < hd.pub.number_of_records ; i++)
    {
        /*  Set the address for the next record so that 
            read_partial_tide_record will know where to go.  */

        if (i) tindex[i].address = tindex[i - 1].address + 
            rec.header.record_size;

        read_partial_tide_record (i, &rec);


        /*  Save the header info in the index.  */

        tindex[i].record_size = rec.header.record_size;
        tindex[i].record_type = rec.header.record_type;
        tindex[i].reference_station = rec.header.reference_station;
        tindex[i].tzfile = rec.header.tzfile;
        tindex[i].lat = NINT (rec.header.latitude * hd.latitude_scale);
        tindex[i].lon = NINT (rec.header.longitude * hd.longitude_scale);

        if ((tindex[i].name = 
            (NV_CHAR *) calloc (strlen (rec.header.name) + 1, 
            sizeof (NV_CHAR))) == NULL)
        {
            perror ("Allocating index name memory");
            exit (-1);
        }

        strcpy (tindex[i].name, rec.header.name);
    }


    current_record = -1;
    current_index = -1;

    return (NVTrue);
}






/*****************************************************************************\

    Function        open_tide_db - opens the tide database

    Synopsis        open_tide_db (file);

                    NV_CHAR *file           database file name

    Returns         NV_BOOL                 NVTrue if file opened

    Author          Jan C. Depner
    Date            08/01/02

    Modified        David Flater
    Date            2003-10-14

    Incorporated patch from Phil Thornton that closes a memory leak
    and improves performance on repeat calls to open_tide_db.

    Modified        David Flater
    Date            2003-11-16

    Added check of modified flag to 2003-10-14 code.

\*****************************************************************************/

NV_BOOL open_tide_db (NV_CHAR *file)
{
    current_record = -1;
    current_index = -1;
    if (tindex) {
        if (!strcmp(file,filename) && !modified) return NVTrue;
        else close_tide_db();
    }
    if ((fp = fopen (file, "rb+")) == NULL) {
        if ((fp = fopen (file, "rb")) == NULL) return (NVFalse);
    }
    assert (strlen(file) < sizeof(filename)); /* DWF */
    strcpy (filename, file);
    return (read_tide_db_header (file));
}






/*****************************************************************************\

    Function        close_tide_db - closes the tide database

    Synopsis        close_tide_db ();

    Returns         void

    Author          Jan C. Depner
    Date            08/01/02

    Modified        David Flater
    Date            2003-10-14

    Incorporated patch from Phil Thornton that closes a memory leak
    and improves performance on repeat calls to open_tide_db.

    Modified        David Flater
    Date            2003-11-16

    Deleted repeat free of tindex introduced 2003-10-14.
    Cleared modified flag on close.

\*****************************************************************************/

void close_tide_db ()
{
    NV_INT32            i;


    /*  If we've changed something in the file, write the header to reset 
        the last modified time.  */

    if (modified) write_tide_db_header ();


    /*  Free all of the temporary memory.  */

    for (i = 0 ; i < hd.pub.constituents ; i++)
    {
        if (hd.constituent[i] != NULL) free (hd.constituent[i]);
    }
    if (hd.constituent != NULL) free (hd.constituent);


    if (hd.speed != NULL) free (hd.speed);


    for (i = 0 ; i < hd.pub.constituents ; i++)
    {
        if (hd.equilibrium[i] != NULL) free (hd.equilibrium[i]);
    }
    if (hd.equilibrium != NULL) free (hd.equilibrium);


    for (i = 0 ; i < hd.pub.constituents ; i++)
    {
        if (hd.node_factor[i] != NULL) free (hd.node_factor[i]);
    }
    if (hd.node_factor != NULL) free (hd.node_factor);


    for (i = 0 ; i < hd.pub.level_unit_types ; i++)
    {
        if (hd.level_unit[i] != NULL) free (hd.level_unit[i]);
    }
    if (hd.level_unit != NULL) free (hd.level_unit);


    for (i = 0 ; i < hd.pub.dir_unit_types ; i++)
    {
        if (hd.dir_unit[i] != NULL) free (hd.dir_unit[i]);
    }
    if (hd.dir_unit != NULL) free (hd.dir_unit);


    for (i = 0 ; i < hd.max_restriction_types ; i++)
    {
        if (hd.restriction[i] != NULL) free (hd.restriction[i]);
    }
    if (hd.restriction != NULL) free (hd.restriction);


    for (i = 0 ; i < hd.max_pedigree_types ; i++)
    {
        if (hd.pedigree[i] != NULL) free (hd.pedigree[i]);
    }
    if (hd.pedigree != NULL) free (hd.pedigree);

    for (i = 0 ; i < hd.max_tzfiles ; i++)
    {
        if (hd.tzfile[i] != NULL) free (hd.tzfile[i]);
    }
    if (hd.tzfile != NULL) free (hd.tzfile);


    for (i = 0 ; i < hd.max_countries ; i++)
    {
        if (hd.country[i] != NULL) free (hd.country[i]);
    }
    if (hd.country != NULL) free (hd.country);


    for (i = 0 ; i < hd.max_datum_types ; i++)
    {
        if (hd.datum[i] != NULL) free (hd.datum[i]);
    }
    if (hd.datum != NULL) free (hd.datum);


    for (i = 0 ; i < hd.pub.number_of_records ; i++)
    {
        if (tindex[i].name) free (tindex[i].name);
    }

    if (tindex) free (tindex);
    tindex = NULL;
    filename[0] = '\0';
    fclose (fp);
    modified = NVFalse;
}






/*****************************************************************************\

    Function        create_tide_db - creates the tide database

    Synopsis        create_tide_db (file, constituents, constituent, speed, 
                        start_year, num_years, equilibrium, node_factor);

                    NV_CHAR *file              database file name
                    NV_INT32 constituents      number of constituents
                    NV_CHAR *constituent[]     constituent names
                    NV_FLOAT64 *speed          speed values
                    NV_INT32 start_year        start year
                    NV_INT32 num_years         number of years
                    NV_FLOAT32 *equilibrium[]  equilibrium arguments
                    NV_FLOAT32 *node_factor[]  node factors

    Returns         NV_BOOL                 NVTrue if file created

    Author          Jan C. Depner
    Date            08/01/02

    Modified        David Flater
    Date            2003-11-16

    Fixed horrible bug:  offsets for speeds, equilibrium args, and node
    factors were sign-reversed with respect to their usage in
    read_tide_db_header and write_tide_db_header, resulting in
    possible overflows.

\*****************************************************************************/

NV_BOOL create_tide_db (NV_CHAR *file, NV_INT32 constituents, 
NV_CHAR *constituent[], NV_FLOAT64 *speed, NV_INT32 start_year, 
NV_INT32 num_years, NV_FLOAT32 *equilibrium[], NV_FLOAT32 *node_factor[])
{
    NV_INT32              i, j;
    NV_FLOAT64            min_value, max_value, range;


    if ((fp = fopen (file, "wb+")) == NULL) return (NVFalse);


    /*  Zero out the header structure.  */

    memset (&hd, 0, sizeof (hd));


    hd.header_size = DEFAULT_HEADER_SIZE;
    hd.pub.number_of_records = DEFAULT_NUMBER_OF_RECORDS;

    hd.pub.start_year = start_year;
    hd.pub.number_of_years = num_years;

    hd.pub.constituents = constituents;


    /*  Constituent names.  */

    hd.constituent = 
        (NV_CHAR **) calloc (hd.pub.constituents, sizeof (NV_CHAR *));
    for (i = 0 ; i < hd.pub.constituents ; i++)
    {
        hd.constituent[i] = (NV_CHAR *) calloc (strlen (constituent[i]) + 1, 
            sizeof (NV_CHAR));
        strcpy (hd.constituent[i], constituent[i]);
    }


    hd.constituent_bits = calculate_bits (hd.pub.constituents);


    /*  Set all of the speed attributes.  */

    hd.speed =  (NV_FLOAT64 *) calloc (hd.pub.constituents,
        sizeof (NV_FLOAT64));

    hd.speed_scale = DEFAULT_SPEED_SCALE;
    min_value = 99999999.0;
    max_value = -99999999.0;
    for (i = 0 ; i < hd.pub.constituents ; i++)
    {
        if (speed[i] < min_value) min_value = speed[i];
        if (speed[i] > max_value) max_value = speed[i];

        hd.speed[i] = speed[i];
    }

    range = max_value - min_value;

    /* DWF fixed sign reversal 2003-11-16 */
    hd.speed_offset = (NINT (min_value * hd.speed_scale));
    hd.speed_bits = calculate_bits (range * hd.speed_scale);


    /*  Set all of the equilibrium attributes.  */

    hd.equilibrium = (NV_FLOAT32 **) calloc (hd.pub.constituents,
        sizeof (NV_FLOAT32 *));

    hd.equilibrium_scale = DEFAULT_EQUILIBRIUM_SCALE;
    min_value = 99999999.0;
    max_value = -99999999.0;
    for (i = 0 ; i < hd.pub.constituents ; i++)
    {
        hd.equilibrium[i] = (NV_FLOAT32 *) calloc (hd.pub.number_of_years,
            sizeof (NV_FLOAT32));
        for (j = 0 ; j < hd.pub.number_of_years ; j++)
        {
            if (equilibrium[i][j] < min_value) min_value = equilibrium[i][j];
            if (equilibrium[i][j] > max_value) max_value = equilibrium[i][j];

            hd.equilibrium[i][j] = equilibrium[i][j];
        }
    }

    range = max_value - min_value;

    /* DWF fixed sign reversal 2003-11-16 */
    hd.equilibrium_offset = (NINT (min_value * hd.equilibrium_scale));
    hd.equilibrium_bits = calculate_bits (range * hd.equilibrium_scale);


    /*  Set all of the node factor attributes.  */

    hd.node_factor = (NV_FLOAT32 **) calloc (hd.pub.constituents,
        sizeof (NV_FLOAT32 *));

    hd.node_scale = DEFAULT_NODE_SCALE;
    min_value = 99999999.0;
    max_value = -99999999.0;
    for (i = 0 ; i < hd.pub.constituents ; i++)
    {
        hd.node_factor[i] = (NV_FLOAT32 *) calloc (hd.pub.number_of_years,
            sizeof (NV_FLOAT32));
        for (j = 0 ; j < hd.pub.number_of_years ; j++)
        {
            if (node_factor[i][j] < min_value) min_value = 
                node_factor[i][j];
            if (node_factor[i][j] > max_value) max_value = 
                node_factor[i][j];

            hd.node_factor[i][j] = node_factor[i][j];
        }
    }

    range = max_value - min_value;

    /* DWF fixed sign reversal 2003-11-16 */
    hd.node_offset = (NINT (min_value * hd.node_scale));
    hd.node_bits = calculate_bits (range * hd.node_scale);


    /*  Default city.  */

    hd.amplitude_bits = DEFAULT_AMPLITUDE_BITS;
    hd.amplitude_scale = DEFAULT_AMPLITUDE_SCALE;
    hd.epoch_bits = DEFAULT_EPOCH_BITS;
    hd.epoch_scale = DEFAULT_EPOCH_SCALE;

    hd.record_type_bits = DEFAULT_RECORD_TYPE_BITS;
    hd.latitude_bits = DEFAULT_LATITUDE_BITS;
    hd.latitude_scale = DEFAULT_LATITUDE_SCALE;
    hd.longitude_bits = DEFAULT_LONGITUDE_BITS;
    hd.longitude_scale = DEFAULT_LONGITUDE_SCALE;
    hd.record_size_bits = DEFAULT_RECORD_SIZE_BITS;

    hd.station_bits = DEFAULT_STATION_BITS;

    hd.datum_offset_bits = DEFAULT_DATUM_OFFSET_BITS;
    hd.datum_offset_scale = DEFAULT_DATUM_OFFSET_SCALE;
    hd.date_bits = DEFAULT_DATE_BITS;
    hd.months_on_station_bits = DEFAULT_MONTHS_ON_STATION_BITS;
    hd.confidence_value_bits = DEFAULT_CONFIDENCE_VALUE_BITS;

    hd.time_bits = DEFAULT_TIME_BITS;
    hd.level_add_bits = DEFAULT_LEVEL_ADD_BITS;
    hd.level_add_scale = DEFAULT_LEVEL_ADD_SCALE;
    hd.level_multiply_bits = DEFAULT_LEVEL_MULTIPLY_BITS;
    hd.level_multiply_scale = DEFAULT_LEVEL_MULTIPLY_SCALE;
    hd.direction_bits = DEFAULT_DIRECTION_BITS;

    hd.constituent_size = DEFAULT_CONSTITUENT_SIZE;
    hd.level_unit_size = DEFAULT_LEVEL_UNIT_SIZE;
    hd.dir_unit_size = DEFAULT_DIR_UNIT_SIZE;
    hd.restriction_size = DEFAULT_RESTRICTION_SIZE;
    hd.pedigree_size = DEFAULT_PEDIGREE_SIZE;
    hd.tzfile_size = DEFAULT_TZFILE_SIZE;
    hd.country_size = DEFAULT_COUNTRY_SIZE;
    hd.datum_size = DEFAULT_DATUM_SIZE;



    /*  Level units.  */

    hd.pub.level_unit_types = DEFAULT_LEVEL_UNIT_TYPES;
    hd.level_unit_bits = calculate_bits (hd.pub.level_unit_types);

    hd.level_unit = (NV_CHAR **) calloc (hd.pub.level_unit_types,
        sizeof (NV_CHAR *));
    for (i = 0 ; i < hd.pub.level_unit_types ; i++)
    {
        hd.level_unit[i] = (NV_CHAR *) calloc (strlen (level_unit[i]) + 1,
            sizeof (NV_CHAR));
        strcpy (hd.level_unit[i], level_unit[i]);
    }


    /*  Direction units.  */

    hd.pub.dir_unit_types = DEFAULT_DIR_UNIT_TYPES;
    hd.dir_unit_bits = calculate_bits (hd.pub.dir_unit_types);

    hd.dir_unit = (NV_CHAR **) calloc (hd.pub.dir_unit_types,
        sizeof (NV_CHAR *));
    for (i = 0 ; i < hd.pub.dir_unit_types ; i++)
    {
        hd.dir_unit[i] = (NV_CHAR *) calloc (strlen (dir_unit[i]) + 1, 
            sizeof (NV_CHAR));
        strcpy (hd.dir_unit[i], dir_unit[i]);
    }


    /*  Restrictions.  */

    hd.restriction_bits = DEFAULT_RESTRICTION_BITS;
    hd.max_restriction_types = NINT (pow (2.0, 
        (NV_FLOAT64) hd.restriction_bits));
    hd.pub.restriction_types = DEFAULT_RESTRICTION_TYPES;

    hd.restriction = (NV_CHAR **) calloc (hd.max_restriction_types,
        sizeof (NV_CHAR *));
    for (i = 0 ; i < hd.max_restriction_types ; i++)
    {
        if (i == hd.pub.restriction_types) break;

        hd.restriction[i] = (NV_CHAR *) calloc (strlen (restriction[i]) + 1,
            sizeof (NV_CHAR));
        strcpy (hd.restriction[i], restriction[i]);
    }


    /*  Pedigrees.  */

    hd.pedigree_bits = DEFAULT_PEDIGREE_BITS;
    hd.max_pedigree_types = NINT (pow (2.0, (NV_FLOAT64) hd.pedigree_bits));
    hd.pub.pedigree_types = DEFAULT_PEDIGREE_TYPES;

    hd.pedigree = (NV_CHAR **) calloc (hd.max_pedigree_types,
        sizeof (NV_CHAR *));
    for (i = 0 ; i < hd.max_pedigree_types ; i++)
    {
        if (i == hd.pub.pedigree_types) break;

        hd.pedigree[i] = (NV_CHAR *) calloc (strlen (pedigree[i]) + 1,
            sizeof (NV_CHAR));
        strcpy (hd.pedigree[i], pedigree[i]);
    }


    /*  Tzfiles.  */

    hd.tzfile_bits = DEFAULT_TZFILE_BITS;
    hd.max_tzfiles = NINT (pow (2.0, (NV_FLOAT64) hd.tzfile_bits));
    hd.pub.tzfiles = DEFAULT_TZFILES;

    hd.tzfile = (NV_CHAR **) calloc (hd.max_tzfiles, sizeof (NV_CHAR *));
    for (i = 0 ; i < hd.max_tzfiles ; i++)
    {
        if (i == hd.pub.tzfiles) break;

        hd.tzfile[i] = (NV_CHAR *) calloc (strlen (tzfile[i]) + 1,
            sizeof (NV_CHAR));
        strcpy (hd.tzfile[i], tzfile[i]);
    }


    /*  Countries.  */

    hd.country_bits = DEFAULT_COUNTRY_BITS;
    hd.max_countries = NINT (pow (2.0, (NV_FLOAT64) hd.country_bits));
    hd.pub.countries = DEFAULT_COUNTRIES;

    hd.country = (NV_CHAR **) calloc (hd.max_countries, sizeof (NV_CHAR *));
    for (i = 0 ; i < hd.max_countries ; i++)
    {
        if (i == hd.pub.countries) break;

        hd.country[i] = (NV_CHAR *) calloc (strlen (country[i]) + 1,
            sizeof (NV_CHAR));
        strcpy (hd.country[i], country[i]);
    }


    /*  Datums.  */

    hd.datum_bits = DEFAULT_DATUM_BITS;
    hd.max_datum_types = NINT (pow (2.0, (NV_FLOAT64) hd.datum_bits));
    hd.pub.datum_types = DEFAULT_DATUM_TYPES;

    hd.datum = (NV_CHAR **) calloc (hd.max_datum_types, sizeof (NV_CHAR *));
    for (i = 0 ; i < hd.max_datum_types ; i++)
    {
        if (i == hd.pub.datum_types) break;

        hd.datum[i] = (NV_CHAR *) calloc (strlen (datum[i]) + 1,
            sizeof (NV_CHAR));
        strcpy (hd.datum[i], datum[i]);
    }


    /*  Write the header to the file.  */

    write_tide_db_header ();


    /*  Close the file.  */

    close_tide_db ();


    /*  Re-open it and read the header from the file.  */

    i = (open_tide_db (file));



    /*  Set the correct end of file position since the one in the header is
        set to 0.  */

    hd.end_of_file = ftell(fp);


    return (i);
}






/*****************************************************************************\

    Function        write_tide_record - writes a tide record to the database

    Synopsis        write_tide_record (num, rec);

                    NV_INT32 num            record number, -1 for a new record
                    TIDE_RECORD *rec        tide record

    Returns         NV_BOOL                 NVTrue if successful

    Author          Jan C. Depner
    Date            08/01/02

\*****************************************************************************/

NV_BOOL write_tide_record (NV_INT32 num, TIDE_RECORD *rec)
{
    NV_U_BYTE               *buf = NULL;
    NV_INT32                i, j, pos, temp_int, name_length = 0, 
                            source_length = 0, comments_length = 0, count = 0;
    NV_BOOL                 ret;
    NV_CHAR                 name[NAME_LENGTH], source[SOURCE_LENGTH], 
                            comments[COMMENTS_LENGTH];


    ret = NVFalse;


    /*  If we come in with a -1 that means we are appending a new record.  */

    if (num < 0)
    {
        fseek (fp, hd.end_of_file, SEEK_SET);
    }
    else
    {
        fseek (fp, tindex[num].address, SEEK_SET);
    }


    /*  Figure out the lengths of the variable length ASCII fields so we'll
        know how much memory to allocate.  */

    for (i = 0 ; i < NAME_LENGTH ; i++)
    {
        /*  Strip CR and LF from input string.  */

        name[i] = rec->header.name[i];
        if (name[i] == 13 || name[i] == 10 || name[i] == 0) 
        {
            name[i] = 0;
            name_length = i + 1;
            break;
        }
    }

    for (i = 0 ; i < SOURCE_LENGTH ; i++)
    {
        /*  Strip CR and LF from input string.  */

        source[i] = rec->source[i];
        if (source[i] == 13 || source[i] == 10 || source[i] == 0) 
        {
            source[i] = 0;
            source_length = i + 1;
            break;
        }
    }

    for (i = 0 , j = 0 ; i < COMMENTS_LENGTH ; i++)
    {
        /*  Strip CRs from input string.  */

        if (rec->comments[i] != 13) 
        {
            comments[j] = rec->comments[i];

            if (comments[j] == 0)
            {
                comments_length = j + 1;
                break;
            }
            j++;
        }
    }


    /*  Figure out how many bits we'll need for this record.  This is the
        common section of the record.  */

    rec->header.record_size = 
        hd.record_size_bits +
        hd.record_type_bits +
        hd.latitude_bits +
        hd.longitude_bits +
        hd.tzfile_bits +
        (name_length * 8) +
        hd.station_bits +
        hd.country_bits +
        hd.pedigree_bits +
        (source_length  * 8) +
        hd.restriction_bits +
        (comments_length * 8);


    /*  Add stuff for record type 1.  */

    if (rec->header.record_type == 1)
    {
         rec->header.record_size += 
             hd.level_unit_bits +
             hd.datum_offset_bits +
             hd.datum_bits +
             hd.time_bits +
             hd.date_bits +
             hd.months_on_station_bits +
             hd.date_bits +
             hd.confidence_value_bits +
             hd.constituent_bits;
  
         count = 0;
         for (i = 0 ; i < hd.pub.constituents ; i++)
         {
             if (rec->amplitude[i] != 0.0 && rec->epoch[i] != 0.0) count++;
         }

         rec->header.record_size += 
             (count * hd.constituent_bits +
             count * hd.amplitude_bits +
             count * hd.epoch_bits);
    }


    /*  Add stuff for record type 2.  */

    else
    {
        rec->header.record_size += 
            hd.level_unit_bits +
            hd.dir_unit_bits +
            hd.level_unit_bits +
            hd.time_bits +
            hd.level_add_bits +
            hd.level_multiply_bits +
            hd.level_add_bits +
            hd.direction_bits +
            hd.time_bits +
            hd.level_add_bits +
            hd.level_multiply_bits +
            hd.level_add_bits +
            hd.direction_bits +
            hd.time_bits +
            hd.time_bits;
    }


    /*  Convert bits to bytes.  */

    rec->header.record_size = rec->header.record_size / 8 + 1;


    if ((buf = (NV_U_BYTE *) calloc (rec->header.record_size, 
        sizeof (NV_U_BYTE))) == NULL)
    {
        perror ("Allocating tide record buffer");
        exit (-1);
    }


    /*  Bit pack the common section.  "pos" is the bit position within the
        buffer "buf".  */

    pos = 0;


    bit_pack (buf, pos, hd.record_size_bits, rec->header.record_size);
    pos += hd.record_size_bits;

    bit_pack (buf, pos, hd.record_type_bits, rec->header.record_type);
    pos += hd.record_type_bits;

    temp_int = NINT (rec->header.latitude * hd.latitude_scale);
    bit_pack (buf, pos, hd.latitude_bits, temp_int);
    pos += hd.latitude_bits;

    temp_int = NINT (rec->header.longitude * hd.longitude_scale);
    bit_pack (buf, pos, hd.longitude_bits, temp_int);
    pos += hd.longitude_bits;

    bit_pack (buf, pos, hd.tzfile_bits, rec->header.tzfile);
    pos += hd.tzfile_bits;

    for (i = 0 ; i < name_length ; i++)
    {
        bit_pack (buf, pos, 8, name[i]);
        pos += 8;
    }

    bit_pack (buf, pos, hd.station_bits, rec->header.reference_station);
    pos += hd.station_bits;

    bit_pack (buf, pos, hd.country_bits, rec->country);
    pos += hd.country_bits;

    bit_pack (buf, pos, hd.pedigree_bits, rec->pedigree);
    pos += hd.pedigree_bits;

    for (i = 0 ; i < source_length ; i++)
    {
        bit_pack (buf, pos, 8, source[i]);
        pos += 8;
    }

    bit_pack (buf, pos, hd.restriction_bits, rec->restriction);
    pos += hd.restriction_bits;

    for (i = 0 ; i < comments_length ; i++)
    {
        bit_pack (buf, pos, 8, comments[i]);
        pos += 8;
    }


    /*  Bit pack record type 1 records.  */

    if (rec->header.record_type == 1)
    {
        bit_pack (buf, pos, hd.level_unit_bits, rec->units);
        pos += hd.level_unit_bits;

        temp_int = NINT (rec->datum_offset * hd.datum_offset_scale);
        bit_pack (buf, pos, hd.datum_offset_bits, temp_int);
        pos += hd.datum_offset_bits;

        bit_pack (buf, pos, hd.datum_bits, rec->datum);
        pos += hd.datum_bits;

        bit_pack (buf, pos, hd.time_bits, rec->zone_offset);
        pos += hd.time_bits;

        bit_pack (buf, pos, hd.date_bits, rec->expiration_date);
        pos += hd.date_bits;

        bit_pack (buf, pos, hd.months_on_station_bits, 
            rec->months_on_station);
        pos += hd.months_on_station_bits;

        bit_pack (buf, pos, hd.date_bits, rec->last_date_on_station);
        pos += hd.date_bits;
    
        bit_pack (buf, pos, hd.confidence_value_bits, rec->confidence);
        pos += hd.confidence_value_bits;

        bit_pack (buf, pos, hd.constituent_bits, count);
        pos += hd.constituent_bits;

        for (i = 0 ; i < hd.pub.constituents ; i++)
        {
            if (rec->amplitude[i] != 0.0 && rec->epoch[i] != 0.0)
            {
                bit_pack (buf, pos, hd.constituent_bits, i);
                pos += hd.constituent_bits;

                temp_int = NINT (rec->amplitude[i] * hd.amplitude_scale);
                bit_pack (buf, pos, hd.amplitude_bits, temp_int);
                pos += hd.amplitude_bits;

                temp_int = NINT (rec->epoch[i] * hd.epoch_scale);
                bit_pack (buf, pos, hd.epoch_bits, temp_int);
                pos += hd.epoch_bits;
            }
        }

        fwrite (buf, rec->header.record_size, 1, fp);

        free (buf);
        ret = NVTrue;
    }


    /*  Bit pack record type 2 records.  */

    else if (rec->header.record_type == 2)
    {
        bit_pack (buf, pos, hd.level_unit_bits, rec->level_units);
        pos += hd.level_unit_bits;

        bit_pack (buf, pos, hd.dir_unit_bits, rec->direction_units);
        pos += hd.dir_unit_bits;

        bit_pack (buf, pos, hd.level_unit_bits, rec->avg_level_units);
        pos += hd.level_unit_bits;

        bit_pack (buf, pos, hd.time_bits, rec->min_time_add);
        pos += hd.time_bits;

        temp_int = NINT (rec->min_level_add * hd.level_add_scale);
        bit_pack (buf, pos, hd.level_add_bits, temp_int);
        pos += hd.level_add_bits;

        temp_int = NINT (rec->min_level_multiply * hd.level_multiply_scale);
        bit_pack (buf, pos, hd.level_multiply_bits, temp_int);
        pos += hd.level_multiply_bits;

        temp_int = NINT (rec->min_avg_level * hd.level_add_scale);
        bit_pack (buf, pos, hd.level_add_bits, temp_int);
        pos += hd.level_add_bits;

        bit_pack (buf, pos, hd.direction_bits, rec->min_direction);
        pos += hd.direction_bits;

        bit_pack (buf, pos, hd.time_bits, rec->max_time_add);
        pos += hd.time_bits;

        temp_int = NINT (rec->max_level_add * hd.level_add_scale);
        bit_pack (buf, pos, hd.level_add_bits, temp_int);
        pos += hd.level_add_bits;

        temp_int = NINT (rec->max_level_multiply * hd.level_multiply_scale);
        bit_pack (buf, pos, hd.level_multiply_bits, temp_int);
        pos += hd.level_multiply_bits;

        temp_int = NINT (rec->max_avg_level * hd.level_add_scale);
        bit_pack (buf, pos, hd.level_add_bits, temp_int);
        pos += hd.level_add_bits;

        bit_pack (buf, pos, hd.direction_bits, rec->max_direction);
        pos += hd.direction_bits;

        bit_pack (buf, pos, hd.time_bits, rec->flood_begins);
        pos += hd.time_bits;

        bit_pack (buf, pos, hd.time_bits, rec->ebb_begins);
        pos += hd.time_bits;

        fwrite (buf, rec->header.record_size, 1, fp);

        free (buf);
        ret = NVTrue;
    }
    else
    {
        fprintf (stderr, "Record type %d is undefined\n", 
            rec->header.record_type);
    }


    if (ret) modified = NVTrue;


    return (ret);
}






/*****************************************************************************\

    Function        read_next_tide_record - reads the next tide record from
                    the database

    Synopsis        read_next_tide_record (rec);

                    TIDE_RECORD *rec        tide record

    Returns         NV_INT32                record number of the tide record

    Author          Jan C. Depner
    Date            08/01/02

\*****************************************************************************/

NV_INT32 read_next_tide_record (TIDE_RECORD *rec)
{
    return (read_tide_record (current_record + 1, rec));
}






/*****************************************************************************\

    Function        read_tide_record - reads tide record "num" from the
                    database

    Synopsis        read_tide_record (num, rec);

                    NV_INT32 num            record number
                    TIDE_RECORD *rec        tide record

    Returns         NV_INT32                record number of the tide record

    Author          Jan C. Depner
    Date            08/01/02

\*****************************************************************************/

NV_INT32 read_tide_record (NV_INT32 num, TIDE_RECORD *rec)
{
    NV_U_BYTE               *buf;
    NV_INT32                i, j, pos, temp_int, count;


    fseek (fp, tindex[num].address, SEEK_SET);


    if ((buf = (NV_U_BYTE *) calloc (tindex[num].record_size, 
        sizeof (NV_U_BYTE))) == NULL)
    {
        perror ("Allocating read tide record buffer");
        exit (-1);
    }


    current_record = num;

    rec->header.record_number = current_record;


    fread (buf, tindex[num].record_size, 1, fp);


    /*  Unpack the common section of the record.  "pos" is the bit position 
        within the buffer "buf".  */

    pos = 0;
    unpack_partial_tide_record (buf, rec, &pos);

    rec->country = bit_unpack (buf, pos, hd.country_bits);
    pos += hd.country_bits;

    rec->pedigree = bit_unpack (buf, pos, hd.pedigree_bits);
    pos += hd.pedigree_bits;

    for (i = 0 ; i < SOURCE_LENGTH ; i++)
    {
        rec->source[i] = bit_unpack (buf, pos, 8);
        pos += 8;
        if (rec->source[i] == 0) break;
    }

    rec->restriction = bit_unpack (buf, pos, hd.restriction_bits);
    pos += hd.restriction_bits;

    for (i = 0 ; i < COMMENTS_LENGTH ; i++)
    {
        rec->comments[i] = bit_unpack (buf, pos, 8);
        pos += 8;
        if (rec->comments[i] == 0) break;
    }


    /*  Unpack record type 1 data.  */

    if (rec->header.record_type == 1)
    {
        rec->units = bit_unpack (buf, pos, hd.level_unit_bits);
        pos += hd.level_unit_bits;

        temp_int = signed_bit_unpack (buf, pos, hd.datum_offset_bits);
        rec->datum_offset = (NV_FLOAT32) temp_int / hd.datum_offset_scale;
        pos += hd.datum_offset_bits;

        rec->datum = bit_unpack (buf, pos, hd.datum_bits);
        pos += hd.datum_bits;

        rec->zone_offset = signed_bit_unpack (buf, pos, hd.time_bits);
        pos += hd.time_bits;

        rec->expiration_date = bit_unpack (buf, pos, hd.date_bits);
        pos += hd.date_bits;

        rec->months_on_station = bit_unpack (buf, pos, 
            hd.months_on_station_bits);
        pos += hd.months_on_station_bits;

        rec->last_date_on_station = bit_unpack (buf, pos, hd.date_bits);
        pos += hd.date_bits;
    
        rec->confidence = bit_unpack (buf, pos, hd.confidence_value_bits);
        pos += hd.confidence_value_bits;

        for (i = 0 ; i < hd.pub.constituents ; i++)
        {
            rec->amplitude[i] = 0.0;
            rec->epoch[i] = 0.0;
        }


        count = bit_unpack (buf, pos, hd.constituent_bits);
        pos += hd.constituent_bits;

        for (i = 0 ; i < count ; i++)
        {
            j = bit_unpack (buf, pos, hd.constituent_bits);
            pos += hd.constituent_bits;

            rec->amplitude[j] = (NV_FLOAT32) bit_unpack (buf, pos, 
                hd.amplitude_bits) / hd.amplitude_scale;
            pos += hd.amplitude_bits;

            rec->epoch[j] = (NV_FLOAT32) bit_unpack (buf, pos, hd.epoch_bits) /
                hd.epoch_scale;
            pos += hd.epoch_bits;
        }
    }


    /*  Unpack record type 2 data.  */

    else if (rec->header.record_type == 2)
    {
        rec->level_units = bit_unpack (buf, pos, hd.level_unit_bits);
        pos += hd.level_unit_bits;

        rec->direction_units = bit_unpack (buf, pos, hd.dir_unit_bits);
        pos += hd.dir_unit_bits;

        rec->avg_level_units = bit_unpack (buf, pos, hd.level_unit_bits);
        pos += hd.level_unit_bits;

        rec->min_time_add = signed_bit_unpack (buf, pos, hd.time_bits);
        pos += hd.time_bits;

        temp_int = signed_bit_unpack (buf, pos, hd.level_add_bits); 
        rec->min_level_add = (NV_FLOAT32) temp_int / hd.level_add_scale;
        pos += hd.level_add_bits;

        temp_int = signed_bit_unpack (buf, pos, hd.level_multiply_bits);
        rec->min_level_multiply = (NV_FLOAT32) temp_int / 
            hd.level_multiply_scale;
        pos += hd.level_multiply_bits;

        temp_int = signed_bit_unpack (buf, pos, hd.level_add_bits); 
        rec->min_avg_level = (NV_FLOAT32) temp_int / hd.level_add_scale;
        pos += hd.level_add_bits;

        rec->min_direction = bit_unpack (buf, pos, hd.direction_bits);
        pos += hd.direction_bits;

        rec->max_time_add = signed_bit_unpack (buf, pos, hd.time_bits);
        pos += hd.time_bits;

        temp_int = signed_bit_unpack (buf, pos, hd.level_add_bits);
        rec->max_level_add = (NV_FLOAT32) temp_int / hd.level_add_scale;
        pos += hd.level_add_bits;

        temp_int = signed_bit_unpack (buf, pos, hd.level_multiply_bits);
        rec->max_level_multiply = (NV_FLOAT32) temp_int / 
            hd.level_multiply_scale;
        pos += hd.level_multiply_bits;

        temp_int = signed_bit_unpack (buf, pos, hd.level_add_bits); 
        rec->max_avg_level = (NV_FLOAT32) temp_int / hd.level_add_scale;
        pos += hd.level_add_bits;

        rec->max_direction = bit_unpack (buf, pos, hd.direction_bits);
        pos += hd.direction_bits;

        rec->flood_begins = signed_bit_unpack (buf, pos, hd.time_bits);
        pos += hd.time_bits;

        rec->ebb_begins = signed_bit_unpack (buf, pos, hd.time_bits);
        pos += hd.time_bits;
    }

    free (buf);

    return (num);
}






/*****************************************************************************\

    Function        add_tide_record - adds a tide record to the database

    Synopsis        add_tide_record (rec);

                    TIDE_RECORD *rec        tide record

    Returns         NV_BOOL                 NVTrue if successful

    Author          Jan C. Depner
    Date            08/01/02

\*****************************************************************************/

NV_BOOL add_tide_record (TIDE_RECORD *rec, DB_HEADER_PUBLIC *db)
{
    NV_INT32                pos;


    if (rec->header.latitude < -90.0 || rec->header.latitude > 90.0 || 
        rec->header.longitude < -180.0 || rec->header.longitude > 180.0)
    {
        fprintf (stderr, "Invalid position %f %f for record - %s\n", 
            rec->header.latitude, rec->header.longitude, rec->header.name);
        return (NVFalse);
    }


    fseek (fp, hd.end_of_file, SEEK_SET);
    pos = ftell (fp);

    rec->header.record_number = hd.pub.number_of_records;

    hd.pub.number_of_records++;

    if (write_tide_record (-1, rec))
    {
        if ((tindex = (TIDE_INDEX *) realloc (tindex, hd.pub.number_of_records *
            sizeof (TIDE_INDEX))) == NULL)
        {
            perror ("Allocating more index records");
            exit (-1);
        }

        tindex[rec->header.record_number].address = pos;
        tindex[rec->header.record_number].record_size = rec->header.record_size;
        tindex[rec->header.record_number].record_type = rec->header.record_type;
        tindex[rec->header.record_number].reference_station = 
            rec->header.reference_station;
        tindex[rec->header.record_number].tzfile = rec->header.tzfile;
        tindex[rec->header.record_number].lat = NINT (rec->header.latitude * 
            hd.latitude_scale);
        tindex[rec->header.record_number].lon = NINT (rec->header.longitude * 
            hd.longitude_scale);


        if ((tindex[rec->header.record_number].name = 
            (NV_CHAR *) calloc (strlen (rec->header.name) + 1, 
            sizeof (NV_CHAR))) == NULL)
        {
            perror ("Allocating index name memory");
            exit (-1);
        }


        strcpy (tindex[rec->header.record_number].name, rec->header.name);


        hd.end_of_file = ftell (fp);


        modified = NVTrue;


        /*  Get the new number of records.  */

        *db = hd.pub;


        return (NVTrue);
    }

    return (NVFalse);
}






/*****************************************************************************\

    Function        delete_tide_record - deletes a record and all subordinate
                    records from the database

    Synopsis        delete_tide_record (num);

                    NV_INT32 num            record number

    Returns         void                    NVTrue if successful

    Author          Jan C. Depner
    Date            08/01/02

\*****************************************************************************/

void delete_tide_record (NV_INT32 num, DB_HEADER_PUBLIC *db)
{
    NV_INT32                i, end_of_file, size;
    TIDE_RECORD             tmp_rec;
    NV_U_BYTE               *block;


    end_of_file = 0;


    /*  Loop through from the end to the beginning so we don't move a record
        before we try to delete it.  */

    for (i = hd.pub.number_of_records - 1 ; i >= 0 ; i--)
    {
        get_partial_tide_record (i, &tmp_rec.header);


        /*  If this is the record itself or a subordinate station, delete
            it.  */

        if (i == num || tmp_rec.header.reference_station == num)
        {
            hd.pub.number_of_records--;


            /*  If we're deleting the last record all we have to do is change 
                the end_of_file pointer in the header and re-index.  */

            if (i == hd.pub.number_of_records)
            {
                end_of_file = tindex[i].address;
            }
            else
            {
                /*  Figure out how big a block we need to move.  Yeah, I'm just
                    grabbing that much memory.  The file size will probably 
                    never exceed 10MB.  Right now, with 10,652 records, it is 
                    only 2.5MB.  If it gets too big someone can rewrite this 
                    section to use disk.  But seriously folks, with memory as 
                    cheap as it is, 10-20MB is in the noise and we only need 
                    it for a fraction of a second.  So I'm lazy, so sue me!  */

                size = hd.end_of_file - tindex[i + 1].address;


                /*  Move to the next record.  */

                fseek (fp, tindex[i + 1].address, SEEK_SET);


                /*  Allocate and read the big-ole-block.  */

                if ((block = (NV_U_BYTE *) calloc (size, 
                    sizeof (NV_U_BYTE))) == NULL)
                {
                    perror ("Allocating block");
                    exit (-1);
                }

                fread (block, size, 1, fp);


                /*  Move to where the record was.  */

                fseek (fp, tindex[i].address, SEEK_SET);


                /*  Write the big-ole-block.  */

                fwrite (block, size, 1, fp);


                free (block);


                /*  Set the new end_of_file to be written to the header when 
                    we are done.  */

                end_of_file = ftell (fp);
            }
        }
    }


    /*  Just in case we didn't find the record (I'm not sure how that could
        happen).  */

    if (end_of_file)
    {
        /*  The new end_of_file.  */

        hd.end_of_file = end_of_file;
        write_tide_db_header ();


        /*  Close the file and reopen it to index the records again.  */

        close_tide_db ();

        open_tide_db (filename);
    }


    /*  Send back the public part of the header (with the new end_of_file) to
        the caller.  */

    *db = hd.pub;
}






/*****************************************************************************\

    Function        update_tide_record - updates a tide record in the database

    Synopsis        update_tide_record (num, rec);

                    NV_INT32 num            record number
                    TIDE_RECORD *rec        tide record

    Returns         NV_BOOL                 NVTrue if successful

    Author          Jan C. Depner
    Date            08/01/02

\*****************************************************************************/

NV_BOOL update_tide_record (NV_INT32 num, TIDE_RECORD *rec)
{
    NV_INT32                i, tmp_count, count, pos, size;
    TIDE_RECORD             tmp_rec;
    NV_BOOL                 flush;
    NV_U_BYTE               *block = NULL;


    if (rec->header.latitude < -90.0 || rec->header.latitude > 90.0 || 
        rec->header.longitude < -180.0 || rec->header.longitude > 180.0)
    {
        fprintf (stderr, "Invalid position %f %f for record - %s\n", 
            rec->header.latitude, rec->header.longitude, rec->header.name);
        return (NVFalse);
    }


    read_tide_record (num, &tmp_rec);


    /*  Check for differences in those things that are going to give us 
        problems.  */

    flush = NVFalse;

    if (strlen (rec->header.name) != strlen (tmp_rec.header.name) ||
        strlen (rec->source) != strlen (tmp_rec.source) ||
        strlen (rec->comments) != strlen (tmp_rec.comments)) flush = NVTrue;

    count = 0;
    tmp_count = 0;
    for (i = 0 ; i < hd.pub.constituents ; i++)
    {
        if (rec->amplitude[i] != 0.0 || rec->epoch[i] != 0.0) count++;
        if (tmp_rec.amplitude[i] != 0.0 || tmp_rec.epoch[i] != 0.0) 
            tmp_count++;
    }

    if (count != tmp_count) flush = NVTrue;


    /*  Aaaaaaarrrrrgggggghhhh!!!!  We have to move stuff!  */

    if (flush)
    {
        /*  Save where we are - end of record being modified.  */

        pos = ftell (fp);


        /*  Figure out how big a block we need to move.  */

        size = hd.end_of_file - pos;


        /*  Allocate memory and read the block.  */

        if (size)
        {
            if ((block = (NV_U_BYTE *) calloc (size, sizeof (NV_U_BYTE))) == 
                NULL)
            {
                perror ("Allocating block");
                exit (-1);
            }
            fread (block, size, 1, fp);
        }


        /*  Write out the modified record.  */

        write_tide_record (num, rec);


        /*  If we weren't at the end of file, move the block.  */

        if (size)
        {
            fwrite (block, size, 1, fp);

            free (block);
        }


        hd.end_of_file = ftell (fp);

        write_tide_db_header ();


        /*  Close the file and reopen it to index the records again.  */

        close_tide_db ();

        open_tide_db (filename);
    }


    /*  The easy way.  No change to the record size.  */

    else
    {
        write_tide_record (num, rec);


        if (rec->header.record_size != tmp_rec.header.record_size)
        {
            fprintf (stderr, 
                "New record size %d does not match old record size %d!\n", 
                rec->header.record_size, tmp_rec.header.record_size);
            exit (-1);
        }


        /*  Save the header info in the index.  */

        tindex[num].record_size = rec->header.record_size;
        tindex[num].record_type = rec->header.record_type;
        tindex[num].reference_station = rec->header.reference_station;
        tindex[num].tzfile = rec->header.tzfile;
        tindex[num].lat = NINT (rec->header.latitude * hd.latitude_scale);
        tindex[num].lon = NINT (rec->header.longitude * hd.longitude_scale);

        strcpy (tindex[num].name, rec->header.name);
    }

    return (NVTrue);
}




/*****************************************************************************\

    Function        infer_constituents - computes inferred constituents when
                    M2, S2, K1, and O1 are given.  This function fills the 
                    remaining unfilled constituents.  The inferred constituents
                    are developed or decided based on article 230 of 
                    "Manual of Harmonic Analysis and Prediction of Tides",
                    Paul Schureman, C & GS special publication no. 98, 
                    October 1971.  This function is really just for NAVO
                    since we go to weird places and put in tide gages for
                    ridiculously short periods of time so we only get a
                    few major constituents developed.  This function was 
                    modified from the NAVO FORTRAN program pred_tide_corr, 
                    subroutine infer.ftn, 08-oct-86.

    Synopsis        infer_constituents (rec);

                    TIDE_RECORD rec         tide record

    Returns         NV_BOOL                 NVFalse if not enough constituents
                                            available to infer others

    Author          Jan C. Depner
    Date            08/01/02

\*****************************************************************************/

NV_BOOL infer_constituents (TIDE_RECORD *rec)
{
    NV_INT32          i, j, m2, s2, k1, o1;
    NV_FLOAT32        epoch_m2, epoch_s2, epoch_k1, epoch_o1;


    m2 = find_constituent ("M2");
    s2 = find_constituent ("S2");
    k1 = find_constituent ("K1");
    o1 = find_constituent ("O1");


    if (rec->amplitude[m2] == 0.0 || rec->amplitude[s2] == 0.0 ||
        rec->amplitude[k1] == 0.0 || rec->amplitude[o1] == 0.0) 
        return (NVFalse);


    epoch_m2 = rec->epoch[m2];
    epoch_s2 = rec->epoch[s2];
    epoch_k1 = rec->epoch[k1];
    epoch_o1 = rec->epoch[o1];


    for (i = 0 ; i < hd.pub.constituents ; i++)
    {
        if (rec->amplitude[i] == 0.0 && rec->epoch[i] == 0.0)
        {
            for (j = 0 ; j < INFERRED_SEMI_DIURNAL_COUNT ; j++)
            {
                if (!strcmp (inferred_semi_diurnal[j], get_constituent (i)))
                {
                    /*  Compute the inferred semi-diurnal constituent.  */

                    rec->amplitude[i] = (semi_diurnal_coeff[j] / coeff[0]) *
                        rec->amplitude[m2];

                    if (fabs ((NV_FLOAT64) (epoch_s2 - epoch_m2)) > 180.0)
                    {
                        if (epoch_s2 < epoch_m2)
                        {
                            epoch_s2 += 360.0;
                        }
                        else
                        {
                            epoch_m2 += 360.0;
                        }
                    }
                    rec->epoch[i] = epoch_m2 + ((hd.speed[i] - hd.speed[m2]) /
                        (hd.speed[s2] - hd.speed[m2])) * (epoch_s2 - epoch_m2);
                }
            }


            for (j = 0 ; j < INFERRED_DIURNAL_COUNT ; j++)
            {
                if (!strcmp (inferred_diurnal[j], get_constituent (i)))
                {
                    /*  Compute the inferred diurnal constituent.  */

                    rec->amplitude[i] = (diurnal_coeff[j] / coeff[1]) *
                        rec->amplitude[o1];

                    if (fabs ((NV_FLOAT64) (epoch_k1 - epoch_o1)) > 180.0)
                    {
                        if (epoch_k1 < epoch_o1)
                        {
                            epoch_k1 += 360.0;
                        }
                        else
                        {
                            epoch_o1 += 360.0;
                        }
                    }
                    rec->epoch[i] = epoch_o1 + ((hd.speed[i] - hd.speed[o1]) /
                        (hd.speed[k1] - hd.speed[o1])) * (epoch_k1 - epoch_o1);
                }
            }
        }
    }

    return (NVTrue);
}
