

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <dirent.h>
#include <sys/types.h>
#include <errno.h>

#include <X11/Xlib.h>           
#include <X11/Intrinsic.h>

#include "types.h"
#include "xwave.h"
#include "xwave_widget.h"
#include "audio_file.h"
#include "riff.h"
#include "aifc.h"
#include "au.h"

static short af_type(int fd);
static void af_perr(char *name,int err);

static char *type_strings[]={"RIFF","AIFC","AU","RAW"};
static char *comp_strings[]={"PCM","ALAW","MULAW","ADPCM"};


#define count_elements(arr) ((int) (sizeof(arr) / sizeof(arr[0])))

char *ft2string(int i)
{
   if (i<count_elements(type_strings)) return(type_strings[i]);
   else return(NULL);
}

char *comp2string(int i)
{
   if (i<count_elements(comp_strings)) return(comp_strings[i]);
   else return(NULL);
}

void af_perr(char *name,int err)
{
    printf("XWave error: ");
    switch (err) {
     case EISDIR: printf("file is a directory");break;
     case EACCES: printf("no access permissions");break;
     case EMFILE:
     case ENFILE: printf("SYSTEM prevents opening more files");break;
     case ENOSPC: printf("no space left on device");break;
    }
    printf(" while working on %s !\n",name);

}


int af_open(char *name,Audio_File *af,int mode)
{
    int type;
    
    if (mode==AF_NEW) {
	if ((af->fd=open(name,mode,0666))==-1) {
	    af_perr(name,errno);
	    return(AF_ERROR);
	}
	type=af->type;
	af->length=0;
    } else {
	if ((af->fd=open(name,mode))==-1) {
	    af_perr(name,errno);
	    return(AF_ERROR);
	}
	type=af_type(af->fd);
    }
    
    switch (type) {
	int o_type;
     case AF_RIFF:
	if ((o_type=riff_open(af,mode))!=AF_RIFF) {
	    af_perr(name,errno);
	    close(af->fd);
	    return(o_type);
	}
	break;
     case AF_AIFC:
      if ((o_type=aifc_open(af,mode))!=AF_AIFC) {
	  af_perr(name,errno);
	  close(af->fd);
	  return(o_type);
      }
	break;
     case AF_AU:
	if ((o_type=au_open(af,mode))!=AF_AU) {
	    af_perr(name,errno);
	    close(af->fd);
	    return(o_type);
	}
	break;
     case AF_RAW:
	af->type=AF_RAW;
	af->comp=AF_PCM;
	af->headoffs=0;
	if (mode!=AF_NEW) {
	    if ((af->length=lseek(af->fd,0,SEEK_END))==-1) { 
		af_perr(name,errno);
		close(af->fd);
		return(AF_ERROR);
	    }
	    if (lseek(af->fd,0,SEEK_SET)==-1) {
		af_perr(name,errno);
		close(af->fd);
		return(AF_ERROR);
	    }
	} else af->length=0;
	break;
    }
    af->name=malloc(strlen(name)+1);
    if (af->name!=NULL) strcpy(af->name,name);
    return(af->type);
}

int af_read(Audio_File af,char *buffer,int size) 
{
    int af_return=AF_ERROR;

    switch (af.type) {
     case AF_RIFF:
	af_return=riff_read(af,buffer,size);
	break;
     case AF_AIFC:
	af_return=aifc_read(af,buffer,size); 
	break;
     case AF_AU:
	af_return=au_read(af,buffer,size); 
	break;
     case AF_RAW:
	af_return=read(af.fd,buffer,size);
	break;
     default:
	printf("XWave error: file: %s Audio_Type %li not supported !\n",
	       af.name,af.type);
    }
    if (af_return==AF_ERROR) af_perr(af.name,errno);
	
    return(af_return);
}

int af_write(Audio_File af,char *buffer,int size)
{
    int af_return=AF_ERROR;

    switch (af.type) {
     case AF_RIFF:
	af_return=riff_write(af,buffer,size); 
	break;
     case AF_AIFC:
	af_return=aifc_write(af,buffer,size); 
	break;
     case AF_AU:
	af_return=au_write(af,buffer,size); 
	break;
     case AF_RAW:
	af_return=write(af.fd,buffer,size);
	break;
     default:
	printf("XWave error: file: %s Audio_Type %li not supported !\n",
	       af.name,af.type);
    }

    if (af_return==AF_ERROR) af_perr(af.name,errno);
    
    return(af_return);
}

int af_seek(Audio_File af,int pos,int mode)
{
    switch (af.type) {
     case AF_RIFF:
	return(riff_seek(af,pos,mode)); 
     case AF_AIFC:
	return(aifc_seek(af,pos,mode)); 
     case AF_AU:
	return(au_seek(af,pos,mode)); 
     case AF_RAW:
	return(lseek(af.fd,pos,mode)); 
	break;
    }
   return(AF_ERROR);
}

int af_rewind(Audio_File af)
{
    return(lseek(af.fd,af.headoffs,SEEK_SET));
}

int af_close(Audio_File af)
{
    if (af.name!=NULL) free(af.name);
    af.name=NULL;
    switch (af.type) {
     case AF_RIFF:
	return(riff_close(af)); 
     case AF_AIFC:
	return(aifc_close(af)); 
     case AF_AU:
	return(au_close(af)); 
     case AF_RAW:
	close(af.fd);
	break;
    }
    return(AF_ERROR);
}

short af_type(int fd)
{
    if (is_riff(fd)) return(AF_RIFF);
    if (is_aifc(fd)) return(AF_AIFC);
    if (is_au(fd)) return(AF_AU);
    return(AF_RAW);
}

char *af_info(char *name,short *af_type)
{
    Audio_File af;
    char *info;
    char *type;
    char *comp;
    float time;
    
    
   if ((info=malloc(MAX_NAMELENGTH))==NULL) {
       *af_type=AF_ERROR;
       return(NULL);
   }
    
    if (af_open(name,&af,O_RDONLY)==AF_ERROR) {
	sprintf(info,"Couldn't open file %s !",name);
	*af_type=AF_ERROR;
	return(info);
    }
    *af_type=af.type;
    
    if (af.type==AF_RAW) return(info);
    
    if (af.type>AF_MAX_TYPE) {
	sprintf(info,"Unknown Filetype");
	return(info);
    }
    
    type=ft2string(af.type);
    
    if (af.comp>AF_MAX_COMP) {
	sprintf(info,"Unsupported compression \"%i\" !",(int) af.comp);
	*af_type=AF_NOTSUPPORTED;
	return(info);
    }
    
    comp=comp2string(af.comp);
    
    time=(float) af.length/(float) (af.freq * af.channels * af.bps/8);
    sprintf(info,"%s (%s): %i bit, Ch:%i , %i hz\nLength: %i bytes (%.2f sec)",
	    type,comp,af.bps,af.channels,(int)af.freq,(int)af.length,time);
    
    return(info);
}

