/* 
 * xwave - an interactive audio player, recorder, editor 
 * for the XWindow System
 * 
 * Copyright (C) 1996 Kai Kollmorgen
 * (kkollmor@informatik.uni-rostock.de)
 *
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 */

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <dirent.h>
#include <unistd.h>
#include <ctype.h>
#include <math.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Xaw/Scrollbar.h>

#include "types.h"
#include "xwave.h"
#include "xwave_widget.h"
#include "graphics.h"
#include "button.h"
#include "menu.h"
#include "audio_file.h"
#include "misc.h"
#include "effects.h"
#include "status.h"
#include "edit.h"

extern Main_Data *MD;
extern AppResources app_resources;
extern char zoom_text[MAX_NUMLENGTH];
extern char length_text[MAX_NUMLENGTH];
extern char begin_text[MAX_NUMLENGTH];
extern bool button_2;

void begin_up(Widget w, XtPointer client_data, XtPointer call_data)
{
    Main_Data *md=(Main_Data*) client_data;
    int beg=0,x=0,nx=0;
    
    if (md->wd->markbeg==md->wd->tlength) return;
    
    x=md->wd->markbeg/md->mg->step;

    if (button_2) {
	nx=x+1;
	beg=ceil(nx*md->mg->step);
    } else {
	beg=md->wd->markbeg+md->cg->step;
	nx=beg/md->mg->step;
    }
    if (beg>md->wd->tlength) beg=md->wd->tlength;
    
    md->wd->markbeg=beg;
    
    if ((md->wd->marklength==0)||(md->wd->isplay)) {
	
	md->wd->isplay=True;
	md->wd->ismark=False;
	
	if (nx>x) {
	    clear_line(x);
	    set_line(nx);
	}
	
    } else {
	if (beg+md->wd->marklength>md->wd->tlength) {
	    md->wd->marklength=md->wd->tlength-beg;
	    set_lengthmark_label(md->wd->marklength);
	}
	if (nx>x) {
	    set_mark_beg(md,x);
	}
    }
    
    /* update canvas */
    md->cg->pos=md->wd->markbeg*md->wd->bpspl;
    if (md->wd->ismark) 
      md->cg->pos-=md->cg->width/2*md->wd->bpspl*md->cg->step;
    update_canvas(md,NULL);
    
    set_beginmark_label(beg);
}

void begin_down(Widget w, XtPointer client_data, XtPointer call_data)
{
    Main_Data *md=(Main_Data*) client_data;
    int beg=0,x=0,nx=0;
    
    if (md->wd->markbeg==0) return;
    
    x=md->wd->markbeg/md->mg->step;
    if (button_2) {
	nx=x-1;
	beg=ceil(nx*md->mg->step);
    } else {
	beg=md->wd->markbeg-md->cg->step;
	nx=beg/md->mg->step;
    }
    if (beg<0) beg=0;
    
    md->wd->markbeg=beg;
    
    if ((md->wd->marklength==0)||(md->wd->isplay)) {
	
	md->wd->isplay=True;
	
	if (nx<x) {
	    clear_line(x);
	    if (nx!=0) set_line(nx);
	    else md->wd->isplay=False;
	}
    } else {
	if (nx<x) {
	    set_mark_beg(md,nx);
	}
    }
    
    /* update canvas */
    md->cg->pos=md->wd->markbeg*md->wd->bpspl;
    if (md->wd->ismark) 
      md->cg->pos-=md->cg->width/2*md->wd->bpspl*md->cg->step;
    update_canvas(md,NULL);

    set_beginmark_label(beg);
}

void length_up(Widget w, XtPointer client_data, XtPointer call_data)
{
    Main_Data *md=(Main_Data*) client_data;
    int length=0,x=0,nx=0;
    
    if (md->wd->markbeg+md->wd->marklength==md->wd->tlength) return;
    
    x=md->wd->marklength/md->mg->step;

    md->wd->ismark=True;
    if (md->wd->marklength==0) {
	if (md->wd->isplay) {
	    clear_line(md->wd->markbeg/md->mg->step);
	    md->wd->isplay=False;
	}
    }

    if (button_2) {
	nx=x+1;
	length=ceil(nx*md->mg->step);
    } else {
	length=md->wd->marklength+md->cg->step;
	nx=length/md->mg->step;
    }
    if (length+md->wd->markbeg>md->wd->tlength) 
      length=md->wd->tlength-md->wd->markbeg;
    
    md->wd->marklength=length;
    
    if (nx>x) set_mark_length(md,md->wd->markbeg/md->mg->step+x);

    /* update canvas */
    md->cg->pos=(md->wd->markbeg+md->wd->marklength)*md->wd->bpspl;
    md->cg->pos-=md->cg->width/2*md->wd->bpspl*md->cg->step;
    update_canvas(md,NULL);
    
    set_lengthmark_label(length);
}

void length_down(Widget w, XtPointer client_data, XtPointer call_data)
{
    Main_Data *md=(Main_Data*) client_data;
    int length=0,x=0,nx=0;
    
    if (md->wd->marklength==0) return;

    x=md->wd->marklength/md->mg->step;
    
    if (button_2) {
	nx=x-1;
	length=ceil(nx*md->mg->step);
    } else {
	length=md->wd->marklength-md->cg->step;
	nx=length/md->mg->step;
    }

    if (length<0) length=0;
    
    md->wd->marklength=length;
    
    if (nx<x) set_mark_length(md,md->wd->markbeg/md->mg->step+nx);

    if ((length==0)&&(md->wd->markbeg!=0)) {
	md->wd->ismark=False;
	md->wd->isplay=True;
	set_line(md->wd->markbeg/md->mg->step);
    }

    /* update canvas */
    md->cg->pos=(md->wd->markbeg+md->wd->marklength)*md->wd->bpspl;
    md->cg->pos-=md->cg->width/2*md->wd->bpspl*md->cg->step;
    update_canvas(md,NULL);

    set_lengthmark_label(length);
} 

void zoom_up(Widget w, XtPointer client_data, XtPointer call_data)
{
    Main_Data *md=(Main_Data*) client_data;
    int zoom,maxzoom;
    Dimension width;
    XtVaGetValues(md->cw->graph,XtNwidth,&width,NULL);
    
    zoom=atoi(zoom_text);
    maxzoom=md->wd->tlength/width+1;
    if (zoom==maxzoom) return;
    zoom++;
    if (zoom==maxzoom) {
	float top=0.0;
	float shown=-1.0;
	XawScrollbarSetThumb(md->cw->scroll,top,shown);
	md->cg->pos=0;
    }
    md->cg->step=zoom;
    set_zoom_label(zoom);
    update_canvas(md,NULL);
}

void zoom_down(Widget w, XtPointer client_data, XtPointer call_data)
{
    Main_Data *md=(Main_Data*) client_data;
    int zoom;
    Dimension width;
    XtVaGetValues(md->cw->graph,XtNwidth,&width,NULL);
    
    zoom=atoi(zoom_text);
    if (zoom==1) return;
    zoom--;
    md->cg->step=zoom;
    set_zoom_label(zoom);
    update_canvas(md,NULL);
}

void begin_enter()
{
    int begin,i;
    Main_Data *md=MD;
    
    if (strlen(begin_text)==0) return;
    
    begin=atoi(begin_text);
    
    i=strlen(begin_text)-1;
    while (i>=0) if (!isdigit(begin_text[i--])) begin=-1;
    
    if ((begin<0)||(begin>md->wd->tlength)) {
	sprintf(md->mw->messages,"%s %i - %i",
		app_resources.err_input,0,md->wd->tlength);
	warn_popup(md->mw->form,md->mw->messages);
	return;
    }
    
    if (md->wd->ismark) {
	if (begin==md->wd->markbeg) return;
	set_whole_mark(md);
	if (begin+md->wd->marklength>md->wd->tlength) {
	    md->wd->marklength=md->wd->tlength-begin;
	    set_lengthmark_label(md->wd->marklength);
	}
	md->wd->markbeg=begin;
	set_whole_mark(md);
    } else {
	if (md->wd->isplay) 
	  clear_line(md->wd->markbeg/md->mg->step);
	md->wd->markbeg=begin;
	md->wd->isplay=True;
	if (begin>0) set_line((int)((float)md->wd->markbeg/md->mg->step));
    }
    
    /* update canvas */
    md->cg->pos=md->wd->markbeg*md->wd->bpspl;
    if (md->wd->ismark)
      md->cg->pos-=md->cg->width/2*md->wd->bpspl*md->cg->step;
    update_canvas(md,NULL);
    
    update_status(md);
}

void length_enter()
{
    int length,begin,i;
    Main_Data *md=MD;
    
    if (strlen(length_text)==0) return;
    
    length=atoi(length_text);
    
    i=strlen(length_text)-1;
    while (i>=0) if (!isdigit(length_text[i--])) length=-1;
    
    if (length==md->wd->marklength) return;
    
    begin=md->wd->markbeg;
    
    if (begin<0) begin=0;
    
    if ((length<0)||(begin+length>md->wd->tlength)) {
	sprintf(md->mw->messages,"%s %i - %i",app_resources.err_input,
		0,md->wd->tlength-begin);
	warn_popup(md->mw->form,md->mw->messages);
	return;
    }
    if (length<=md->wd->length) {
	/* clear last mark */
	if (md->wd->ismark) set_whole_mark(md);
	
	/* clear last line */
	if (md->wd->isplay) clear_line((int)((float)begin/md->mg->step));
	
	if (length>0) {
	    md->wd->isplay=False;
	    md->wd->ismark=True;
	    md->wd->marklength=length;
	    md->wd->markbeg=begin;
	    set_whole_mark(md);
	} else {
	    md->wd->markbeg=begin;
	    md->wd->isplay=True;
	    md->wd->ismark=False;
	    if (begin>0) set_line((int)((float)begin/md->mg->step));
	}
	
	/* update canvas */
	md->cg->pos=(md->wd->markbeg+md->wd->marklength)*md->wd->bpspl;
	if (md->wd->ismark)
	  md->cg->pos-=md->cg->width/2*md->wd->bpspl*md->cg->step;
	update_canvas(md,NULL);
	
	update_status(md);
    }
}

void zoom_enter()
{
    int zoom,i,maxzoom;
    Dimension width;
    
    if (strlen(zoom_text)==0) return;
    
    XtVaGetValues(MD->cw->graph,XtNwidth,&width,NULL);
    maxzoom=MD->wd->tlength/width+1;
    
    zoom=atoi(zoom_text);
    
    i=strlen(zoom_text)-1;
    while (i>=0) if (!isdigit(zoom_text[i--])) zoom=0;
    
    if (zoom==maxzoom) return;
    
    if ((zoom>maxzoom)||(zoom==0)) {
	sprintf(MD->mw->messages,"%s %i - %i",app_resources.err_input,
		1,maxzoom);
	warn_popup(MD->mw->form,MD->mw->messages);
	return;
    }
    MD->cg->step=zoom;
    update_canvas(MD,NULL);
}

void paste_merge_call(Widget w, XtPointer client_data, XtPointer call_data)
{
    Main_Data *md=(Main_Data *) client_data;
    Wave_Data *wd=md->wd;
    Wave_Data *clip_wd=md->cd->wd;
    int length,begin,peak,clip_peak;
    float factor1,factor2,maxval;
    
    if (!md->mb->isclip) return;
    
    /* destination is empty, call insert */
    if (wd->length==0) {
	paste_insert_call(w,client_data,call_data);
	return;
    }
    
    if ((clip_wd->res!=wd->res)  || (clip_wd->channels!=wd->channels)|| 
	(clip_wd->freq!=wd->freq)) {
	sprintf(md->mw->messages,"%s %i bit, %i chn. %i hz\n",
		app_resources.err_wrongformat,clip_wd->res,clip_wd->channels,
		clip_wd->freq);
	warn_popup(md->mw->form,md->mw->messages);
	return;
    }
    
    /* prepare error message for file error */
    sprintf(md->mw->messages,"%s\n\"%s\" !",app_resources.err_openfile,
	    wd->name);
    
    watch_cursor(True);
    
    find_peak(wd);
    find_peak(clip_wd);
    
    maxval=pow(2.0,(float)wd->res)/2.0;
    peak=wd->peak_l;
    clip_peak=clip_wd->peak_l;
    
    if (peak+clip_peak>(int)maxval) factor1=maxval/(float)(peak+clip_peak);
    else factor1=1.0;
    factor2=factor1;
    
    if (wd->channels==2) {
	peak=wd->peak_r;
	clip_peak=clip_wd->peak_r;
	if (peak+clip_peak>(int)maxval) factor2=maxval/(float)(peak+clip_peak);
	else factor2=1.0;
    }
    
    begin=0;
    length=clip_wd->length;
    if (wd->isplay) begin=wd->markbeg*wd->bpspl;
    if (wd->ismark) {
	begin=wd->markbeg*wd->bpspl;
	if (length>wd->marklength*wd->bpspl) 
	  length=wd->marklength*wd->bpspl;
    }
    
    if (begin+length>wd->length) length=wd->length-begin;
    
    if (wd->inmem && clip_wd->inmem) {
	switch (wd->res) {
	 case 8:
	    merge_8((char*)(wd->buffer+begin),(char*)clip_wd->buffer,
		    length,factor1,factor2);
	    break;
	 case 16:
	    merge_16((short*)(wd->buffer+begin),(short*)clip_wd->buffer,
		     length/2,factor1,factor2);
	    break;
	}
    } else {
	Audio_File af,new_af,clip_af;
	int i,toread;
	char *oldname=NULL,*outbuf,*clipbuf;
	
	
	if (!clip_wd->inmem) {
	    wd2af(clip_wd,&clip_af);
	    af_rewind(clip_af);
	    if (!wd->inmem) {
		/* need one more buffer for clipfile */
		if ((clipbuf=malloc(MAXUSHORT))==NULL) {
		    watch_cursor(False);
		    warn_popup(md->mw->form,app_resources.err_mem);
		    return;
		}
	    } else {
		clipbuf=(char*)md->mg->fbuf;
	    }
	} else {
	    clipbuf=(char*) clip_wd->buffer;
	}
	
	/* create new file and copy all until mark begin */
	if (!wd->inmem) {
	    wd2af(wd,&af);
	    af_rewind(af);
	    
	    outbuf=(char*)md->mg->fbuf;
	    oldname=wd->actual_name;
	    
	    if ((wd->actual_name=get_actual_name(wd))==NULL) {
		watch_cursor(False);
		warn_popup(md->mw->form,app_resources.err_mem);
		if (!clip_wd->inmem) free(clipbuf);
		return;
	    }
	    wd2af(wd,&new_af);
	    new_af.type=AF_RAW;
	    new_af.comp=AF_PCM;
	    if (af_open(wd->actual_name,&new_af,AF_NEW)==AF_ERROR) {
		watch_cursor(False);
		warn_popup(md->mw->form,md->mw->messages);
		XtFree(wd->actual_name);
		wd->actual_name=oldname;
		if (!clip_wd->inmem) free(clipbuf);
		return;
	    }
	    
	    i=begin;
	    while (i>0) {
		if (i>MAXUSHORT) toread=MAXUSHORT;
		else toread=i;
		toread=af_read(af,(char*)md->mg->fbuf,toread);
		toread=af_write(new_af,(char*)md->mg->fbuf,toread);
		if (toread==-1) {
		    watch_cursor(False);
		    warn_popup(md->mw->form,md->mw->messages);
		    XtFree(wd->actual_name);
		    md->wd->actual_name=oldname;
		    if (!clip_wd->inmem) free(clipbuf);
		    af_close(new_af);
		    return;
		}
		i-=toread;
		new_af.length+=toread;
	    }
	    /* ok new file is prepared */
	} else {
	    /* data are in buffer, so goto right position */
	    outbuf=(char*)(wd->buffer+begin);
	}
	
	i=length;
	while (i>0) {
	    if (i>MAXUSHORT) toread=MAXUSHORT;
	    else toread=i;
	    if (!wd->inmem) toread=af_read(af,outbuf,toread);
	    if (!clip_wd->inmem) toread=af_read(clip_af,clipbuf,toread);
	    
	    switch (wd->res) {
	     case 8:
		merge_8(outbuf,clipbuf,toread,factor1,factor2);
		break;
	     case 16:
		merge_16((short*)outbuf,(short*)clipbuf,toread/2,factor1,factor2);
		break;
	    }
	    
	    if (wd->inmem) outbuf+=toread;
	    else {
		toread=af_write(new_af,outbuf,toread);
		new_af.length+=toread;
	    }
	    
	    if (toread==-1) {
		watch_cursor(False);
		warn_popup(md->mw->form,md->mw->messages);
		XtFree(wd->actual_name);
		md->wd->actual_name=oldname;
		af_close(new_af);
		return;
	    }
	    i-=toread;
	    if (clip_wd->inmem) clipbuf+=toread;
	} /* ok merged all data */
	
	if (!wd->inmem) {
	    /* copy rest of old file to new file */
	    
	    i=wd->length-(begin+length);
	    while (i>0) {
		if (i>MAXUSHORT) toread=MAXUSHORT;
		else toread=i;
		toread=af_read(af,(char*)md->mg->fbuf,toread);
		toread=af_write(new_af,(char*)md->mg->fbuf,toread);
		if (toread==-1) {
		    watch_cursor(False);
		    warn_popup(md->mw->form,md->mw->messages);
		    XtFree(wd->actual_name);
		    md->wd->actual_name=oldname;
		    if (!clip_wd->inmem) free(clipbuf);
		    af_close(new_af);
		    return;
		}
		i-=toread;
		new_af.length+=toread;
	    }
	    
	    af_close(af);
	    XtFree(oldname);
	    
	    wd->length=new_af.length;
	    wd->fd=new_af.fd;
	    wd->headoffs=new_af.headoffs;
	    if (wd->type!=AF_RAW) {
		wd->oldtype=wd->type;
		wd->oldcomp=wd->oldcomp;
		wd->type=AF_RAW;
		wd->comp=AF_PCM;
	    }
	    wd->actual_no++;
	}
	
	if ((!clip_wd->inmem)&&(!wd->inmem)) free(clipbuf);
    }
    
    wd->tlength=wd->length/wd->bpspl;
    md->cb->modified=True;
    md->cb->update=True;
    update_canvas(md,NULL);
    update_display(md);
    watch_cursor(False);
}

void paste_insert_call(Widget w, XtPointer client_data, XtPointer call_data)
{
    Main_Data *md=(Main_Data *) client_data;
    Wave_Data *wd=md->wd;
    Wave_Data *clip_wd=md->cd->wd;
    int i,toread,length,begin;
    Audio_File clip_af;
    
    if (!md->mb->isclip) return;
    
    if ((clip_wd->res!=wd->res)  || (clip_wd->channels!=wd->channels)|| 
	(clip_wd->freq!=wd->freq)) {
	sprintf(md->mw->messages,"%s %i bit, %i chn. %i hz\n",
		app_resources.err_wrongformat,clip_wd->res,clip_wd->channels,
		clip_wd->freq);
	warn_popup(md->mw->form,md->mw->messages);
	return;
    }
    
    /* prepare error message for file error */
    sprintf(md->mw->messages,"%s\n\"%s\" !",app_resources.err_openfile,
	    md->wd->name);
    
    watch_cursor(True);
    
    begin=0;
    length=clip_wd->length;
    if ((wd->isplay)||(wd->ismark)) begin=wd->markbeg*wd->bpspl;
    
    if (!clip_wd->inmem) {
	wd2af(clip_wd,&clip_af);
	af_rewind(clip_af);
    }
    
    /* after new_call the length is 0, we can append without problems */
    if (wd->length==0) {
	if (length > md->maxmem) {
	    Audio_File new_af;
	    
	    wd2af(wd,&new_af);
	    if (af_open(wd->name,&new_af,AF_NEW)==AF_ERROR) {
		watch_cursor(False);
		warn_popup(md->mw->form,md->mw->messages);
		return;
	    }
	    
	    i=length;
	    while (i>0) {
		if (i>MAXUSHORT) toread=MAXUSHORT;
		else toread=i;
		toread=af_read(clip_af,(char*)md->mg->fbuf,toread);
		toread=af_write(new_af,(char*)md->mg->fbuf,toread);
		if (toread==-1) {
		    watch_cursor(False);
		    warn_popup(md->mw->form,md->mw->messages);
		    af_close(new_af);
		    return;
		}
		i-=toread;
		new_af.length+=toread;
	    }
	    af_close(new_af);
	    
	    af_open(wd->name,&new_af,AF_OPEN);
	    af2wd(new_af,md->wd);
	    wd->inmem=False;
	    
	} else {
	    if ((wd->buffer=malloc(length))==NULL) {
		watch_cursor(False);
		warn_popup(md->mw->form,app_resources.err_mem);
		return;
	    }
	    memcpy(wd->buffer,clip_wd->buffer,length);
	    md->cb->modified=True;
	    wd->length=length;
	    wd->inmem=True;
	}
    } else {
	/* try to insert clipboard in actual wave */
	if (wd->length+length > md->maxmem) {
	    /* changed wave is a file */
	    Audio_File af;
	    Audio_File new_af;
	    char *oldname;
	    
	    oldname=wd->actual_name;
	    
	    if ((wd->actual_name=get_actual_name(wd))==NULL) {
		watch_cursor(False);
		warn_popup(md->mw->form,app_resources.err_mem);
		XtFree(oldname);
		return;
	    }
	    
	    /* create new file (AF_RAW) */
	    wd2af(wd,&new_af);
	    new_af.type=AF_RAW;
	    new_af.comp=AF_PCM;
	    if (af_open(wd->actual_name,&new_af,AF_NEW)==AF_ERROR) {
		watch_cursor(False);
		warn_popup(md->mw->form,md->mw->messages);
		XtFree(wd->actual_name);
		wd->actual_name=oldname;
		return;
	    }
	    /* copy samples from 0 to insertion point to new file */
	    if (wd->inmem) {
		/* samples are in buffer */
		
		if ((toread=af_write(new_af,(char*)wd->buffer,begin))
		    ==AF_ERROR) {
		    watch_cursor(False);
		    warn_popup(md->mw->form,md->mw->messages);
		    af_close(new_af);
		    XtFree(wd->actual_name);
		    wd->actual_name=oldname;
		    return;
		}
		new_af.length+=toread;
		
	    } else {
		/* samples are in file */
		/* rewind old file */
		wd2af(wd,&af);
		af_rewind(af);
		
		i=begin;
		while (i>0) {
		    if (i>MAXUSHORT) toread=MAXUSHORT;
		    else toread=i;
		    toread=af_read(af,(char*)md->mg->fbuf,toread);
		    toread=af_write(new_af,(char*)md->mg->fbuf,toread);
		    if (toread==-1) {
			watch_cursor(False);
			warn_popup(md->mw->form,md->mw->messages);
			af_close(new_af);
			XtFree(wd->actual_name);
			wd->actual_name=oldname;
			return;
		    }
		    i-=toread;
		    new_af.length+=toread;
		}
	    } /* samples from 0 to insertion point copied */
	    
	    /* copy clipboard to new file */
	    if (clip_wd->inmem) {
		/* clipboard is in buffer */
		if ((toread=af_write(new_af,(char*)clip_wd->buffer,length))
		    ==AF_ERROR) {
		    watch_cursor(False);
		    warn_popup(md->mw->form,md->mw->messages);
		    af_close(new_af);
		    XtFree(wd->actual_name);
		    wd->actual_name=oldname;
		    return;
		}
		new_af.length+=toread;
	    } else {
		/* clipboard is in file */
		i=length;
		
		while (i>0) {
		    if (i>MAXUSHORT) toread=MAXUSHORT;
		    else toread=i;
		    toread=af_read(clip_af,(char*)md->mg->fbuf,toread);
		    toread=af_write(new_af,(char*)md->mg->fbuf,toread);
		    if (toread==-1) {
			watch_cursor(False);
			warn_popup(md->mw->form,md->mw->messages);
			af_close(new_af);
			XtFree(wd->actual_name);
			wd->actual_name=oldname;
			return;
		    }
		    i-=toread;
		    new_af.length+=toread;
		}
	    } /* samples from clipboard to new file copied */
	    
	    /* copy samples from insertion point to new file */
	    if (wd->inmem) {
		/* samples are in buffer */
		
		if ((toread=af_write(new_af,(char*)(wd->buffer+begin),
				     wd->length-begin))==AF_ERROR) {
		    watch_cursor(False);
		    warn_popup(md->mw->form,md->mw->messages);
		    af_close(new_af);
		    XtFree(wd->actual_name);
		    wd->actual_name=oldname;
		    return;
		}
		new_af.length+=toread;
		
	    } else {
		/* samples are in file */
		i=wd->length-begin;
		while (i>0) {
		    if (i>MAXUSHORT) toread=MAXUSHORT;
		    else toread=i;
		    toread=af_read(af,(char*)md->mg->fbuf,toread);
		    toread=af_write(new_af,(char*)md->mg->fbuf,toread);
		    if (toread==-1) {
			watch_cursor(False);
			warn_popup(md->mw->form,md->mw->messages);
			af_close(new_af);
			XtFree(wd->actual_name);
			wd->actual_name=oldname;
			return;
		    }
		    if (toread==0) break;
		    i-=toread;
		    new_af.length+=toread;
		}
		af_close(af);
	    } /* samples insertion point copied */
	    
	    XtFree(oldname);
	    
	    wd->fd=new_af.fd;
	    wd->headoffs=new_af.headoffs;
	    wd->length=new_af.length;
	    if (wd->type!=AF_RAW) {
		wd->oldtype=wd->type;
		wd->oldcomp=wd->oldcomp;
		wd->type=AF_RAW;
		wd->comp=AF_PCM;
	    }
	    wd->actual_no++;
	    free(wd->buffer);
	    wd->buffer=NULL;
	    wd->inmem=False;
	    
	    md->cb->modified=True;
	} else {
	    /* clipboard is in mem and changed wave also */
	    char *buffer=NULL;
	    
	    if ((buffer=XtMalloc(wd->length+length))==NULL) {
		warn_popup(md->mw->form,app_resources.err_mem);
		return;
	    }
	    
	    memcpy(buffer,wd->buffer,begin);
	    memcpy(buffer+begin,clip_wd->buffer,length);
	    memcpy(buffer+begin+length,wd->buffer+begin,wd->length-begin);
	    
	    free(wd->buffer);
	    wd->buffer=(unsigned char*) buffer;
	    wd->length+=length;
	    md->cb->modified=True;
	}
    }
    wd->tlength=wd->length/wd->bpspl;
    md->cb->update=True;
    update_canvas(md,NULL);
    update_display(md);
    watch_cursor(False);
}

void paste_new_call(Widget w, XtPointer client_data, XtPointer call_data)
{
    Next_Wave *nw;
    Wave_Data *wd;
    Main_Data *md=(Main_Data *) client_data;
    Wave_Data *clip_wd=md->cd->wd;
    
    if (!md->mb->isclip) return;

    watch_cursor(True);
    
    if (md->no>0) {
	nw=md->nw;
	while (nw->nw!=NULL) nw=nw->nw;
	nw->nw=XtNew(Next_Wave);
	nw=nw->nw;
    } else {
	md->nw=XtNew(Next_Wave);
	nw=md->nw;
    }
    
    if (nw==NULL) {
	watch_cursor(False);
	warn_popup(MD->mw->form,app_resources.err_mem);
	return;
    }
    
    nw->wd=XtNew(Wave_Data);
    nw->nw=NULL;

    if (nw->wd==NULL) { 
	watch_cursor(False);
	XtFree((char*)nw);
	warn_popup(MD->mw->form,app_resources.err_mem);
	return;
    }
    
    wd=nw->wd;
    if ((wd->name=new_name())==NULL) {
	watch_cursor(False);
	XtFree((char*)nw->wd);
	XtFree((char*)nw);
	warn_popup(MD->mw->form,app_resources.err_mem);
	return;
    }

    wd->actual_name=NULL;
    wd->actual_no=0;
    wd2wd(wd,clip_wd);
    wd->buffer=NULL;
    wd->lines_l=NULL;
    wd->lines_r=NULL;
    wd->type=app_resources.default_ftype;
    wd->comp=app_resources.default_comp;
    
    
    if (clip_wd->inmem) {
	if ((wd->buffer=malloc(clip_wd->length))==NULL) {
	    watch_cursor(False);
	    XtFree((char*)nw->wd);
	    XtFree((char*)nw);
	    warn_popup(MD->mw->form,app_resources.err_mem);
	    return;
	}
	memcpy(wd->buffer,clip_wd->buffer,clip_wd->length);
    } else {
	Audio_File af,clip_af;
	int count,i;
	
	/* prepare error message for file error */
	sprintf(md->mw->messages,"%s\n\"%s\" !",app_resources.err_openfile,
		md->wd->name);
	
	wd2af(clip_wd,&clip_af);
	af_rewind(clip_af);
	
	wd2af(wd,&af);
	
	if (af_open(wd->name,&af,AF_NEW)==AF_ERROR) {
	    watch_cursor(False);
	    XtFree((char*)nw->wd);
	    XtFree((char*)nw);
	    warn_popup(md->mw->form,md->mw->messages);
	    return;
	}
	
	count=clip_wd->length;
	
	while (True) {
	    i=MAXUSHORT;
	    if (count<MAXUSHORT) i=count;
	    i=af_read(clip_af,(char*) md->mg->fbuf,i);
	    i=af_write(af,(char*) md->mg->fbuf,i);
	    
	    if (i==-1) {
		watch_cursor(False);
		XtFree((char*)nw->wd);
		XtFree((char*)nw);
		warn_popup(md->mw->form,md->mw->messages);
		af_close(af);
		return;
	    }
	    count-=i;
	    af.length+=i;
	    if (count==0) break;
	}
	af2wd(af,wd);
    }

    nw->cw=XtNew(Canvas_Widget);
    nw->cb=XtNew(Canvas_Bool);
    nw->cg=XtNew(Canvas_Graphic);
    
    if (nw->cw==NULL || nw->cb==NULL || nw->cg==NULL) {
	watch_cursor(False);
	XtFree((char*)wd->buffer);
	XtFree((char*)nw->wd);
	XtFree((char*)nw);
	warn_popup(MD->mw->form,app_resources.err_mem);
	return;
    }
    
    nw->cb->canwrite=can_write(nw->wd->name);
    nw->cb->modified=wd->inmem;
    nw->cb->update=False;
    nw->cg->step=1;
    nw->cg->pos=0;
    
    md->no++;
    md->wd=nw->wd;
    md->cb=nw->cb;
    md->cw=nw->cw;
    md->cg=nw->cg;
     
    new_canvas(md->mw->form,nw);
    update_canvas(md,NULL);
    update_display(md);
    watch_cursor(False);
}

void copy_call(Widget w, XtPointer client_data, XtPointer call_data)
{
    int i,count;
    Main_Data *md=(Main_Data *) client_data;
    Wave_Data *wd=md->wd;
    Wave_Data *clip_wd=md->cd->wd;
    Audio_File af;
    
    watch_cursor(True);
    
    if (md->mb->isclip) {
	if (clip_wd->inmem) free(clip_wd->buffer);
	/* no need to af_close here 
	 * (clipfile is raw, and we discard it in any way) */
	else close(clip_wd->fd);
    }
    
    clip_wd->length=0;
    md->mb->isclip=False;
    
    wd2af(wd,&af);
    
    if (!wd->inmem) {
	af_rewind(af);
	af_seek(af,wd->markbeg*wd->bpspl,SEEK_SET);
    }
    
    if (wd->marklength*wd->bpspl > md->maxmem) {
	Audio_File clip_af;
	
	sprintf(md->mw->messages,"%s \n\"%s\" !",app_resources.err_openfile,
		md->wd->name);
	
	wd2af(wd,&clip_af);
	clip_wd->inmem=False;
	clip_af.type=AF_RAW;
	clip_af.comp=AF_PCM;
	if (af_open(clip_wd->name,&clip_af,AF_NEW)==AF_ERROR) {
	    watch_cursor(False);
	    warn_popup(md->mw->form,md->mw->messages);
	    return;
	}
	
	count=md->wd->marklength*md->wd->bpspl;
	
	while (True) {
	    i=MAXUSHORT;
	    if (count<MAXUSHORT) i=count;
	    i=af_read(af,(char*) md->mg->fbuf,i);
	    i=af_write(clip_af,(char*) md->mg->fbuf,i);
	    
	    if (i==-1) {
		watch_cursor(False);
		warn_popup(md->mw->form,md->mw->messages);
		af_close(clip_af);
		return;
	    }
	    count-=i;
	    clip_af.length+=i;
	    if (count==0) break;
	}
	af2wd(clip_af,clip_wd);
    } else {
	
	i=md->wd->marklength*md->wd->bpspl;
	
	if ((clip_wd->buffer=malloc(i))==NULL) {
	    watch_cursor(False);
	    warn_popup(md->mw->form,app_resources.err_mem);
	    return;
	}
	if (wd->inmem) {
	    memcpy(clip_wd->buffer,wd->buffer+wd->markbeg*wd->bpspl,i);
	} else {
	    i=af_read(af,(char*)clip_wd->buffer,i);
	}
	clip_wd->inmem=True;
    }
    
    clip_wd->res=wd->res;
    clip_wd->channels=wd->channels;
    clip_wd->freq=wd->freq;
    clip_wd->length=wd->marklength*wd->bpspl;
    clip_wd->ismark=clip_wd->isplay=False;
    clip_wd->bpspl=(clip_wd->res*clip_wd->channels)/8;
    clip_wd->tlength=clip_wd->length/clip_wd->bpspl;
    clip_wd->markbeg=0;
    clip_wd->marklength=0;
    clip_wd->type=AF_RAW;
    clip_wd->comp=AF_PCM;
    
    md->mb->isclip=True;
    
    update_status(md);
    
    watch_cursor(False);
}

void cut_call(Widget w, XtPointer client_data, XtPointer call_data)
{
    Main_Data *md=(Main_Data *) client_data;
    Wave_Data *wd=md->wd;
    int beg,length,offset;
    
    copy_call(w,client_data,call_data);
    
    watch_cursor(True);
    
    beg=wd->markbeg*wd->bpspl;
    length=wd->length-(wd->marklength*wd->bpspl);
    offset=(wd->markbeg+wd->marklength)*wd->bpspl;
    
    sprintf(md->mw->messages,"%s \n\"%s\" !",app_resources.err_openfile,
	    wd->name);
    
    /* after cut the samples fitting in memory ? */
    if ((wd->inmem)||(length<md->maxmem)) {
	char* newbuf;
	if ((newbuf=(char*) malloc(length))==NULL) {
	    watch_cursor(False);
	    warn_popup(md->mw->form,app_resources.err_mem);
	    return;
	}
	if (wd->inmem) {
	    memcpy(newbuf,wd->buffer,beg);
	    memcpy(newbuf+beg,wd->buffer+offset,length-beg);
	    free(wd->buffer);
	} else {
	    /* after cut the samples fitting in memory */
	    Audio_File af;
	    wd2af(wd,&af);
	    
	    af_rewind(af);
	    af_read(af,newbuf,beg);
	    af_seek(af,wd->marklength*wd->bpspl,SEEK_CUR);
	    af_read(af,newbuf+beg,length-beg);
	    close(wd->fd);
	    wd->inmem=True;
	}
	wd->buffer=newbuf;
	wd->length=length;
	
    } else {
	Audio_File af,new_af;
	char *oldname;
	int i,toread;
	
	oldname=wd->actual_name;
	
	if ((wd->actual_name=get_actual_name(wd))==NULL) {
	    watch_cursor(False);
	    warn_popup(md->mw->form,app_resources.err_mem);
	    return;
	}
	
	/* rewind old file */
	wd2af(wd,&af);
	af_rewind(af);
	
	/* create new file (AF_RAW) */
	wd2af(wd,&new_af);
	new_af.type=AF_RAW;
	new_af.comp=AF_PCM;
	if (af_open(wd->actual_name,&new_af,AF_NEW)==AF_ERROR) {
	    watch_cursor(False);
	    warn_popup(md->mw->form,md->mw->messages);
	    XtFree(wd->actual_name);
	    wd->actual_name=oldname;
	    return;
	}
	
	i=beg;
	while (i>0) {
	    if (i>MAXUSHORT) toread=MAXUSHORT;
	    else toread=i;
	    toread=af_read(af,(char*)md->mg->fbuf,toread);
	    toread=af_write(new_af,(char*)md->mg->fbuf,toread);
	    if (toread==-1) {
		watch_cursor(False);
		warn_popup(md->mw->form,md->mw->messages);
		XtFree(wd->actual_name);
		md->wd->actual_name=oldname;
		af_close(new_af);
		return;
	    }
	    i-=toread;
	    new_af.length+=toread;
	}
	
	/* seek behind the cut position */
	af_seek(af,wd->marklength*wd->bpspl,SEEK_CUR);
	
	i=length-beg;
	while (i>0) {
	    if (i>MAXUSHORT) toread=MAXUSHORT;
	    else toread=i;
	    toread=af_read(af,(char*)md->mg->fbuf,toread);
	    toread=af_write(new_af,(char*)md->mg->fbuf,toread);
	    if (toread==-1) {
		watch_cursor(False);
		warn_popup(md->mw->form,md->mw->messages);
		XtFree(wd->actual_name);
		wd->actual_name=oldname;
		af_close(new_af);
		return;
	    }
	    i-=toread;
	    new_af.length+=toread;
	}
	
	af_close(af);
	XtFree(oldname);
	
	wd->fd=new_af.fd;
	wd->headoffs=new_af.headoffs;
	wd->length=new_af.length;
	if (wd->type!=AF_RAW) {
	    wd->oldtype=wd->type;
	    wd->oldcomp=wd->oldcomp;
	    wd->type=AF_RAW;
	    wd->comp=AF_PCM;
	}
	wd->actual_no++;
    }
    
    wd->tlength=wd->length/wd->bpspl;
    wd->ismark=False;
    md->cb->modified=True;
    md->cb->update=True;
    
    update_canvas(md,NULL);
    update_display(md);
    watch_cursor(False);
}

void undo_call(Widget w, XtPointer client_data, XtPointer call_data)
{
}

