""" ExtFile product module """
###############################################################################
#
# Copyright (c) 2001 Gregor Heine <mac.gregor@gmx.de>. All rights reserved.
# ExtFile Home: http://www.zope.org/Members/MacGregor/ExtFile/index_html
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
#
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
# 3. The name of the author may not be used to endorse or promote products
#    derived from this software without specific prior written permission
#
# Disclaimer
#
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
# OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
# IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
# INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
# NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
# DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
# THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
# THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#   
#  In accordance with the license provided for by the software upon
#  which some of the source code has been derived or used, the following
#  acknowledgement is hereby provided :
#
#      "This product includes software developed by Digital Creations
#      for use in the Z Object Publishing Environment
#      (http://www.zope.org/)."
#
###############################################################################

__doc__ = """
              ExtFile product module.
              The ExtFile-Product works like the Zope File-product, but stores 
              the uploaded file externally in a repository-direcory.

$Id: ExtFile.py,v 1.37.2.1 2001/09/10 23:56:16 gregor Exp $
"""
__version__='$Release: 1.1.3 $'[10:-2]

import Globals
from __main__ import *
import Zope
from Products.ZCatalog.CatalogAwareness import CatalogAware
from OFS.SimpleItem import SimpleItem
from OFS.PropertyManager import PropertyManager
from Globals import HTMLFile, MessageDialog
from OFS.content_types import guess_content_type
from webdav.common import rfc1123_date
from DateTime import DateTime
import urllib, os, types, string
from os.path import join, isfile
try: from cStringIO import StringIO
except: from StringIO import StringIO

FLAT = 0
SYNC_ZODB = 1
SLICED = 2
REPOSITORY = FLAT

# format for the files in the repository:
# %u=user, %p=path, %n=file name, %e=file extension, %c=counter, %t=time
FILE_FORMAT = "%n%c%e"

BACKUP_ON_DELETE = 0
ALWAYS_BACKUP = 1
UNDO_POLICY = BACKUP_ON_DELETE

bad_chars =  ' ,;()[]{}ݟ'
good_chars = '_________AAAAAAaaaaaaCcEEEEEeeeeeIIIIiiiiNnOOOOOOooooooSssUUUUuuuuYYyyZz'
TRANSMAP = string.maketrans(bad_chars, good_chars)

manage_addExtFileForm = HTMLFile('extFileAdd', globals()) 

def manage_addExtFile(self, id='', title='', descr='', file='', 
                      content_type='', permission_check=0, REQUEST=None):
	""" Add a ExtFile to a folder. """
#	print self._d.getPhysicalPath()
	if not id and hasattr(file,'filename'): 
		# generate id from filename and make sure, it has no 'bad' chars
		id = file.filename
		title = title or id
		id = id[max(string.rfind(id,'/'), 
		            string.rfind(id,'\\'), 
		            string.rfind(id,':')
		           )+1:]
		id = string.translate(id, TRANSMAP)
	self = self.this()
	tempExtFile = ExtFile(id, title, descr, permission_check)
	self._setObject(id, tempExtFile)
	self._getOb(id).manage_file_upload(file, content_type)
	if REQUEST is not None:
		return MessageDialog(title = 'Created',
			message = 'The ExtFile %s was successfully created!' % id,
			action = './manage_main',)

class ExtFile(CatalogAware, SimpleItem, PropertyManager): 
	
	# what properties have we?
	_properties=(
		{'id':'title',							'type':'string'},
		{'id':'descr', 							'type':'string'},
		{'id':'content_type',					'type':'string'},
		{'id':'use_download_permission_check',	'type':'int'},
	)
	
	# what management options are there?
	manage_options = (
		{'label':'Edit',			'action': 'manage_main'				},
		{'label':'View/Download',	'action': ''						},
		{'label':'Upload',			'action': 'manage_uploadForm'		},
		{'label':'Properties',		'action': 'manage_propertiesForm'	},
		{'label':'Security',		'action': 'manage_access'			},
	) 
	
	# what permissions make sense for us? 
	__ac_permissions__=(
		('View management screens',	('manage_tabs',
									 'manage_main',
									 'manage_uploadForm')),
		('Change permissions',		('manage_access',)),
		('Change ExtFile/ExtImage',	('manage_editExtFile',
									 'manage_file_upload',
									 'manage_http_upload',
									 'PUT')),
		('FTP access',				('manage_FTPstat',
									 'manage_FTPget',
									 'manage_FTPlist')),
		('Download ExtFile/ExtImage',()),
		('View',					('index_html',
									 'icon_html',
									 'icon_gif',
									 'link',
									 'is_broken',
									 'get_size',
									 'getContentType',
									 '__str__'
									 )),
	)
	
	# what do people think they're adding? 
	meta_type = 'ExtFile'
	
	# location of the file-repository
	_repository = ['var','reposit']
	
	use_download_permission_check = 0
	
	# MIME-Type Dictionary. To add a MIME-Type, add a file in the directory 
	# icons/_category_/_subcategory-icon-file_
	# example: Icon tifficon.gif for the MIME-Type image/tiff goes to 
	# icons/image/tifficon.gif and the dictionary must be updated like this: 
	# 'image':{'tiff':'tifficon.gif','default':'default.gif'}, ...
	_types={'image':					
				{'default':'default.gif'},
			'text':
				{'html':'html.gif', 'xml':'xml.gif', 'default':'default.gif', 
				 'python':'py.gif'},
			'application':
				{'pdf':'pdf.gif', 'zip':'zip.gif', 'tar':'zip.gif', 
				 'msword':'doc.gif', 'excel':'xls.gif', 'powerpoint':'ppt.gif', 
				 'default':'default.gif'},
			'video':
				{'default':'default.gif'},
			'audio':
				{'default':'default.gif'},
			'default':'default.gif'
		}
	
	################################
	# Init method                  #
	################################
	
	def __init__(self, id, title='', descr='', permission_check=0): 
		""" initialize a new instance of ExtFile """
		self.id = id
		self.title = title
		self.descr = descr
		self.use_download_permission_check = permission_check
		self.__version__ = __version__
		self.filename = []
		self.content_type = ''
			
	################################
	# Public methods               #
	################################
	
	def __str__(self): return self.index_html()
	
	def __len__(self): return 1
	
	def index_html (self, icon=0, preview=0, width=None, height=None, 
	                REQUEST=None):
		""" return the file with it's corresponding MIME-type """
		# HTTP If-Modified-Since header handling. (Copied from OFS/Image.py)
		if REQUEST is None and hasattr(self,'REQUEST'): 
			REQUEST = self.REQUEST
		if REQUEST is not None:
			header = REQUEST.get_header('If-Modified-Since', None)
			if header is not None:
				header = string.split(header, ';')[0]
				try:    mod_since = long(DateTime(header).timeTime())
				except: mod_since = None
				if mod_since is not None:
					if self._p_mtime:
						last_mod = long(self._p_mtime)
					else:
						last_mod = long(0)
					if last_mod > 0 and last_mod < mod_since:
						RESPONSE.setStatus(304)
						return ''
		
		if hasattr(self,'has_preview') and self.has_preview: has_preview = 1
		else: has_preview = 0
		if not self._access_permitted(REQUEST): preview = 1
		if (preview and not has_preview): icon = 1
		
		if icon:
			filename = join(SOFTWARE_HOME, 'Products', 
			                'ExtFile', self.getIconPath())
			content_type = 'image/gif'
		elif preview:
			filename = self._get_filename(self.prev_filename)
			content_type = self.prev_content_type
		else:
			filename = self._get_filename(self.filename)
			content_type = self.content_type
		
		cant_read_exc = "Can't read: "
		if not isfile(filename): self._undo()
		if isfile(filename):
			size = os.stat(filename)[6] # file size
		else:
			filename = join(SOFTWARE_HOME, 'Products',
			              'ExtFile', 'icons', 'broken.gif')
			try: size = os.stat(filename)[6]
			except: raise cant_read_exc, ("%s (%s)" %(self.id, filename))
			content_type = 'image/gif'
			icon = 1
		data = StringIO()
		if icon==0 and width is not None and height is not None:
			try:
				from PIL import Image
				im = Image.open(filename) 
				if im.mode!='RGB' and im.mode!='CMYK': im = im.convert("RGB")
				im.draft(None,(int(width),int(height)))
				im.load()
				im = im.resize((int(width),int(height)), Image.BICUBIC)
				im.save(data, 'JPEG')
			except:
				self._copy(filename, data)
			else:
				data.seek(0,2)
				size = data.tell()
				content_type = 'image/jpeg'
		else:
			self._copy(filename, data)
		if REQUEST is not None:
			last_mod = rfc1123_date(self._p_mtime)
			REQUEST.RESPONSE.setHeader('Last-Modified', last_mod)
			REQUEST.RESPONSE.setHeader('Content-Type', content_type)
			REQUEST.RESPONSE.setHeader('Content-Length', size)
		return data.getvalue()
	
	def view_image_or_file(self):
		""" The default view of the contents of the File or Image. """
		raise 'Redirect', self.absolute_url()

	def link(self, text='', **args):
		""" return a HTML link tag to the file """
		if text=='': text = self.title_or_id()
		strg = '<a href="%s"' % (self.absolute_url())
		for key in args.keys():
			value = args.get(key)
			strg = '%s %s="%s"' % (strg, key, value)
		strg = '%s>%s</a>' % (strg, text)
		return strg
	
	def icon_gif(self):
		""" return an icon for the file's MIME-Type """
		raise 'Redirect', self.absolute_url()+'?icon=1'
	
	def icon_html(self):
		""" The icon embedded in html with a link to the real file """
		return '<img src="%s?icon=1" border="0">' % self.absolute_url()
	
	def is_broken(self):
		""" Check if external file exists and return true (1) or false (0) """
		fn = self._get_filename(self.filename)
		if not isfile(fn):
			self._undo()
			if not isfile(fn):
				return 1
		return 0
	
	def get_size(self):
		""" Returns the size of the file or image """
		fn = self._get_filename(self.filename)
		if not isfile(fn): self._undo()
		if isfile(fn): size = os.stat(fn)[6]
		else: size = 0
		return size
	
	rawsize = get_size
	getSize = get_size
	
	def size(self):
		""" Returns a formatted stringified version of the file size """
		return self._bytetostring(self.get_size())
	
	def getContentType(self):
		""" Get the content type of a file or image.
		    Returns the content type (MIME type) of a file or image.
		"""
		return self.content_type
		
	def getIconPath(self):
		""" Depending on the MIME Type of the file/image an icon
		    can be displayed. This function determines which
		    image in the lib/python/Products/ExtFile/icons/...
		    directory shold be used as icon for this file/image
		"""
		cat, sub = self._getMIMECatAndSub(self.content_type)
		if hasattr(self,'has_preview') and self.has_preview: cat = 'image'
		if self._types.has_key(cat):
			file = self._types[cat]['default']
			for item in self._types[cat].keys():
				if string.find(sub,item)>=0:
					file = self._types[cat][item]
					break
			return join('icons',cat,file)
		else:
			return join('icons',self._types['default'])
	
	################################
	# Protected management methods #
	################################
	
	# Management Interface
	manage_main = HTMLFile('extFileEdit', globals())	
	
	def manage_editExtFile(self, title='', descr='', REQUEST=None): 
		""" Manage the edited values """
		if self.title!=title: self.title = title
		if self.descr!=descr: self.descr = descr
		# update ZCatalog
		self.reindex_object()
		if REQUEST is not None:
			return MessageDialog(
				title = 'Edited',
				message = "The properties of %s have been changed!" % self.id,
				action = './manage_main',
			)
	
	# File upload Interface
	manage_uploadForm = HTMLFile('extFileUpload', globals())
	
	def manage_file_upload(self, file='', content_type='', REQUEST=None):
		""" Upload file from local directory """
		new_fn = self._get_ufn(self.filename)
		self._copy(file, self._get_filename(new_fn))
		self.content_type = self._get_content_type(file, file.read(100), 
		                    self.id, content_type or self.content_type)
		self.filename = new_fn
		if REQUEST is not None:
			return MessageDialog(title = 'Uploaded',
				message = "The file was uploaded successfully!", 
				action = './manage_main',)
	
	def manage_http_upload(self, url, REQUEST=None):
		""" Upload file from http-server """
		url = urllib.quote(url,'/:')
		new_fn = self._get_ufn(self.filename)
		cant_read_exc = "Can't open: "
		try: fp_in = urllib.urlopen(url)
		except: raise cant_read_exc, url
		self._copy(fp_in, self._get_filename(new_fn))
		try:
			instream = open(self._get_filename(new_fn), 'rb')
			self.content_type = self._get_content_type(instream, 
			                    instream.read(100),
			                    self.id, self.content_type)
			instream.close()
		except:
			if UNDO_POLICY==ALWAYS_BACKUP: 
				os.remove(self._get_filename(new_fn))
		else:
			self.filename = new_fn
		if REQUEST is not None:
			return MessageDialog(title = 'Uploaded',
				message = "The file was uploaded successfully!", 
				action = './manage_main',)
	
	manage_FTPget = index_html
	
	def PUT(self, REQUEST, RESPONSE):
		""" Handle HTTP PUT requests """
		self.dav__init(REQUEST, RESPONSE)
		content_type = REQUEST.get_header('content-type', None)
		instream = REQUEST['BODYFILE']
		new_fn = self._get_ufn(self.filename)
		self._copy(instream, self._get_filename(new_fn))
		try:
			self.content_type = self._get_content_type(instream, 
			                    instream.read(100),
			                    self.id, content_type or self.content_type)
		except:
			if UNDO_POLICY==ALWAYS_BACKUP: 
				os.remove(self._get_filename(new_fn))
		else:
			self.filename = new_fn
		RESPONSE.setStatus(204)
		return RESPONSE
	
	################################
	# Private methods              #
	################################
	
	def _access_permitted(self, REQUEST):
		""" check if the user is allowed to download the file """
		if hasattr(self,'use_download_permission_check') and \
		   self.use_download_permission_check and \
		   (REQUEST is None or not REQUEST.has_key('AUTHENTICATED_USER') or 
		    not REQUEST['AUTHENTICATED_USER'].has_permission(
		                                'Download ExtFile/ExtImage', self)
		   ):
			return 0
		else: 
			return 1
	
	def _get_content_type(self, file, body, id, content_type=None):
		""" determine the mime-type """
		headers = getattr(file, 'headers', None)
		if headers and headers.has_key('content-type'):
			content_type = headers['content-type']
		else:
			if type(body) is not type(''): body = body.data
			content_type, enc = guess_content_type(getattr(file,'filename',id),
			                                       body, content_type)
		return content_type
	
	def _getMIMECatAndSub(self, mime_string):
		""" Split MIME String into Category and Subcategory """
		cat = mime_string[:string.find(mime_string, '/')]   # MIME-category
		sub = mime_string[string.find(mime_string, '/')+1:] # sub-category
		return cat, sub
	
	def _copy(self, infile, outfile):
		""" read binary data from infile and write it to outfile
		    infile and outfile my be strings, in which case a file with that
		    name is opened, or filehandles, in which case they are accessed
		    directly.
		"""
		if type(infile) is types.StringType: 
			try:
				instream = open(infile, 'rb')
			except IOError:
				self._undo()
				try:
					instream = open(infile, 'rb')
				except IOError:
					raise IOError, ("%s (%s)" %(self.id, infile))
			close_in = 1
		else:
			instream = infile
			close_in = 0
		if type(outfile) is types.StringType: 
			try:
				outstream = open(outfile, 'wb')
			except IOError:
				raise IOError, ("%s (%s)" %(self.id, outfile))
			close_out = 1
		else:
			outstream = outfile
			close_out = 0
		try:
			blocksize = 2<<16
			block = instream.read(blocksize)
			outstream.write(block)
			while len(block)==blocksize:
				block = instream.read(blocksize)
				outstream.write(block)
		except IOError:
			raise IOError, ("%s (%s)" %(self.id, filename))
		try: instream.seek(0)
		except: pass
		if close_in: instream.close()
		if close_out: outstream.close()
	
	def _bytetostring (self, value):
		""" Convert an int-value (file-size in bytes) to an String
		    with the file-size in Byte, KB or MB
		"""
		bytes = float(value)
		if bytes>=1000:
			bytes = bytes/1024
			if bytes>=1000:
				bytes = bytes/1024
				typ = ' MB'
			else:
				typ = ' KB'
		else:
			typ = ' Bytes'
		strg = '%4.2f'%bytes
		strg = strg[:4]
		if strg[3]=='.': strg = strg[:3]
		strg = strg+typ
		return strg
		
	def _undo (self):
		""" restore filename after undo or copy-paste """
		fn = self._get_filename(self.filename)
		if not isfile(fn) and isfile(fn+'.undo'): 
			os.rename(fn+'.undo', fn)
	
	def _get_filename(self, filename=''):
		""" Generate the full filename, incuding directories from 
		    self._repository and self.filename
		"""
		path = INSTANCE_HOME
		for item in self._repository:
			path = join(path,item)
		if type(filename)==types.ListType:
			for item in filename:
				path = join(path,item)
		elif filename!='':
			path = join(path,filename)
		return path
	
	def _get_ufn(self, filename):
		""" If no unique filename has been generated, generate one
		    otherwise, return the existing one.
		"""
		if UNDO_POLICY==ALWAYS_BACKUP or filename==[]: 
			new_fn = self._get_new_ufn()
		else: 
			new_fn = filename[:]
		if filename:
			old_fn = self._get_filename(filename)
			if UNDO_POLICY==ALWAYS_BACKUP: 
				try: os.rename(old_fn, old_fn+'.undo')
				except: pass
			else:
				try: os.rename(old_fn+'.undo', old_fn)
				except: pass
		return new_fn
	
	def _get_new_ufn(self):
		""" Create a new unique filename """
		rel_url_list = string.split(self.absolute_url(1), '/')[:-1]
		rel_url_list = filter(None, rel_url_list)
		pos = string.rfind(self.id, '.')
		if (pos+1):
			id_name = self.id[:pos]
			id_ext = self.id[pos:]
		else:
			id_name = self.id
			id_ext = ''
		
		# generate directory structure
		dirs = []
		if REPOSITORY==SYNC_ZODB: 
			dirs = rel_url_list
		elif REPOSITORY==SLICED: 
			slice_depth = 2 # modify here, if you want a different slice depth
			slice_width = 1 # modify here, if you want a different slice width
			temp = id_name
			for i in range(slice_depth):
				if len(temp)<slice_width*(slice_depth-i): 
					dirs.append(slice_width*'_')
				else:
					dirs.append(temp[:slice_width])
					temp=temp[slice_width:]
		
		# generate file format
		# time/counter (%t)
		fileformat = FILE_FORMAT
		if string.find(fileformat, "%t")>=0:
			fileformat = string.replace(fileformat, "%t", "%c")
			counter = int(DateTime().strftime('%m%d%H%M%S'))
		else:
			counter = 0
		invalid_format_exc = "Invalid file format: "
		if string.find(fileformat, "%c")==-1:
			raise invalid_format_exc, FILE_FORMAT
		# user (%u)
		if string.find(fileformat, "%u")>=0:
			if (hasattr(self, "REQUEST") and 
			   self.REQUEST.has_key('AUTHENTICATED_USER')):
				user = self.REQUEST['AUTHENTICATED_USER'].name 
				fileformat = string.replace(fileformat, "%u", user)
			else:
				fileformat = string.replace(fileformat, "%u", "")
		# path (%p)
		if string.find(fileformat, "%p")>=0:
			temp = string.joinfields (rel_url_list, "_")
			fileformat = string.replace(fileformat, "%p", temp)
		# file and extension (%n and %e)
		if string.find(fileformat,"%n")>=0 or string.find(fileformat,"%e")>=0:
			fileformat = string.replace(fileformat, "%n", id_name)
			fileformat = string.replace(fileformat, "%e", id_ext)
		
		# make the directories
		path = self._get_filename(dirs)
		if not os.path.isdir(path):
			mkdir_exc = "Can't create directory: "
			try:
				os.makedirs(path)
			except:
				raise mkdir_exc, path
		
		# search for unique filename
		if counter: 
			fn = join(path, string.replace(fileformat, "%c", `counter`))
		else: 
			fn = join(path, string.replace(fileformat, "%c", ''))
		while (isfile(fn) or isfile(fn+'.preview') 
		       or isfile(fn+'.undo') or isfile(fn+'.preview.undo')):
			counter = counter+1
			fn = join(path, string.replace(fileformat, "%c", `counter`))
		if counter: fileformat = string.replace(fileformat, "%c", `counter`)
		else: fileformat = string.replace(fileformat, "%c", '')
		dirs.append(fileformat)
		return dirs
	
	################################
	# Special management methods   #
	################################
	
	def manage_afterClone(self, item, new_fn=None):
		""" When a copy of the object is created (zope copy-paste-operation),
		    this function is called by CopySupport.py. A copy of the external 
		    file is created and self.filename is changed.
		"""
		try: 
			self.absolute_url(1) # this raises an exception, if no context
		except: 
			self._v_has_been_cloned=1
		else:
			old_fn = self._get_filename(self.filename)
			new_fn = new_fn or self._get_new_ufn()
			if isfile(old_fn):
				self._copy(old_fn, self._get_filename(new_fn))
			self.filename = new_fn
		return ExtFile.inheritedAttribute ("manage_afterClone") (self, item)
		
	def manage_afterAdd(self, item, container):
		""" This method is called, whenever _setObject in ObjectManager gets 
		    called. This is the case after a normal add and if the object is a 
		    result of cut-paste- or rename-operation. In the first case, the
		    external files doesn't exist yet, otherwise it was renamed to .undo
		    by manage_beforeDelete before and must be restored by _undo().
		"""
		self._undo()
		if hasattr(self, "_v_has_been_cloned"):
			delattr(self, "_v_has_been_cloned")
			self.manage_afterClone(item)
		return ExtFile.inheritedAttribute ("manage_afterAdd") \
		       (self, item, container)
	
	def manage_beforeDelete(self, item, container):
		""" This method is called, when the object is deleted. To support 
		    undo-functionality and because this happens too, when the object 
		    is moved (cut-paste) or renamed, the external file is not deleted. 
		    It are just renamed to filename.undo and remains in the 
		    repository, until it is deleted manually.
		"""
		fn = self._get_filename(self.filename)
		try:
			os.rename(fn, fn+'.undo')
		except OSError:
			pass
		return ExtFile.inheritedAttribute ("manage_beforeDelete") \
		       (self, item, container)

Globals.default__class_init__(ExtFile)
