/**********************************************************************
zyGrib: meteorological GRIB file viewer
Copyright (C) 2008-2010 - Jacques Zaninetti - http://www.zygrib.org

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
***********************************************************************/

#ifndef DATADEFINES_H
#define DATADEFINES_H
#include <stdint.h>

#define GRIB_NOTDEF -999999999

//--------------------------------------------------------
// dataTypes      cf. GribRecord::translateDataType()
//--------------------------------------------------------
#define GRB_PRESSURE        2   /* Pa     */
#define GRB_GEOPOT_HGT      7   /* gpm    */
#define GRB_TEMP           11   /* K      */
#define GRB_TEMP_POT       13   /* K      */
#define GRB_TMAX           15   /* K      */
#define GRB_TMIN           16   /* K      */
#define GRB_DEWPOINT       17   /* K      */
#define GRB_WIND_VX        33   /* m/s    */
#define GRB_WIND_VY        34   /* m/s    */
#define GRB_HUMID_SPEC     51   /* kg/kg  */
#define GRB_HUMID_REL      52   /* %      */
#define GRB_PRECIP_RATE    59   /* l/m2/s */
#define GRB_PRECIP_TOT     61   /* l/m2   */
#define GRB_SNOW_DEPTH     66   /* m      */
#define GRB_CLOUD_TOT      71   /* %      */
#define GRB_FRZRAIN_CATEG 141   /* 1=yes 0=no */
#define GRB_SNOW_CATEG    143   /* 1=yes 0=no */
#define GRB_CAPE 		  157   /* J/kg   */

#define GRB_WIND_DIR       249   /* private: wind direction in degrees */
#define GRB_WIND_XY2D      250   /* private: GRB_WIND_VX+GRB_WIND_VX */
#define GRB_DIFF_TEMPDEW   251   /* private: GRB_TEMP-GRB_DEWPOINT */
#define GRB_THETA_E   	   252   /* K   */
#define GRB_WIND_GUST 	   253   /* m/s */
#define GRB_PRECIP_PROB	   254   /* %   */

#define GRB_TYPE_NOT_DEFINED 0   /* private */

//--------------------------------------------------------
// Levels types (altitude reference)
//--------------------------------------------------------
#define LV_GND_SURF    1
#define LV_ISOTHERM0   4
#define LV_ISOBARIC  100
#define LV_MSL       102
#define LV_ABOV_GND  105
#define LV_SIGMA     107
#define LV_ATMOS_ALL  200
#define LV_ATMOS_LOW  214
#define LV_ATMOS_MID  224
#define LV_ATMOS_HIGH 234


#define LV_TYPE_NOT_DEFINED 0   /* private */

// altitude index in tables
#define H850 0
#define H700 1
#define H500 2
#define H300 3
#define H200 4

#define GEOPOTidx(h) ((h)==850?0:(h)==700?1:(h)==500?2:(h)==300?3:(h)==200?4:-1)
#define GEOPOThgt(i) ((i)==0?850:(i)==1?700:(i)==2?500:(i)==3?300:(i)==4?200:-1)

//--------------------------------------------------------
// Data definition
//--------------------------------------------------------
class Altitude
{
	public:
		Altitude (int levelType=-1, int levelValue=-1)
			{  this->levelType  = levelType; 
			   this->levelValue = levelValue;  }
			
		int levelType;
		int levelValue;
		
		void set (int levelType=-1, int levelValue=-1)
			{  this->levelType  = levelType; 
			   this->levelValue = levelValue;  }
			   
		int index () const  { if (levelType==LV_ISOBARIC) 
								return GEOPOTidx(levelValue);
							  else return -1; }
			
		bool equals (int levelType, int levelValue) const
			{  return this->levelType==levelType && 
					  this->levelValue==levelValue;  }
		bool operator== (const Altitude &alt) const
			{ return alt.levelType==levelType
						&& alt.levelValue==levelValue; }
		bool operator!= (const Altitude &alt) const
			{ return alt.levelType!=levelType
						|| alt.levelValue!=levelValue; }
		bool operator< (const Altitude &alt) const
			{ return alt.levelType<levelType
						|| 
					(alt.levelType==levelType && alt.levelValue<levelValue); }
};
//--------------------------------------------------------
class DataCode
{
	public:
		DataCode (int dataType=GRB_TYPE_NOT_DEFINED, int levelType=-1, int levelValue=-1)
			{  this->dataType   = dataType; 
			   this->levelType  = levelType; 
			   this->levelValue = levelValue;  }
			
		DataCode (int dataType, const Altitude &alt)
			{  this->dataType   = dataType; 
			   this->levelType  = alt.levelType; 
			   this->levelValue = alt.levelValue;  }
			
		DataCode (uint32_t v)   { fromInt32 (v); }
			
		int dataType;
		int levelType;
		int levelValue;
		
		// int32 = #aabbccdd    aabb=levelValue cc=levelType dd=dataCode
		uint32_t toInt32 () {
			return ((levelValue&0xFFFF)<<16)+((levelType&0xFF)<<8)+(dataType&0xFF);
		}
		void fromInt32 (uint32_t v) {
			levelValue = (v>>16) & 0xFFFF;
			levelType  = (v>>8) & 0xFF;
			dataType   =  v     & 0xFF;
		}
		
		void set (int dataType=GRB_TYPE_NOT_DEFINED, int levelType=-1, int levelValue=-1)
			{  this->dataType   = dataType; 
			   this->levelType  = levelType; 
			   this->levelValue = levelValue;  }
			   
		Altitude getAltitude () const
			{ return Altitude(levelType, levelValue); }
			
		bool equals (int dataType, int levelType, int levelValue) const
			{  return this->dataType==dataType && 
			          this->levelType==levelType && 
					  this->levelValue==levelValue;  }
		bool operator== (const DataCode &dtc) const
			{ return dtc.dataType==dataType 
						&& dtc.levelType==levelType
						&& dtc.levelValue==levelValue; }
		bool operator!= (const DataCode &dtc) const
			{ return dtc.dataType!=dataType 
						|| dtc.levelType!=levelType
						|| dtc.levelValue!=levelValue; }
		bool operator< (const DataCode &dtc) const
			{ return dataType<dtc.dataType
						|| 
					(dtc.dataType==dataType && getAltitude()<dtc.getAltitude()); }
};

#endif
