/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */

#ifndef QTSS_H
#define QTSS_H

#ifdef __cplusplus
extern "C" {
#endif


#include "OSHeaders.h"


//MAJOR TYPEDEFS
typedef void* QTSS_FileDesc;
typedef void* QTSS_MediaSrcRef;

//*******************************
// ENUMERATED TYPES

//errorcodes
enum {
	QTSS_NoErr = 0,
	QTSS_BadArgument = -1,
	QTSS_BadCommandPtr = -2,
	QTSS_MemAllocFailed = -3,
	QTSS_ObjectNotFound = -4,
	QTSS_RequestFailed = -5,
	QTSS_InvalidParameter = -6,
	QTSS_IllegalValue = -7,
	QTSS_ReadOnlyParam = -8,
	QTSS_FileNotOpen = -9,
	QTSS_DupName = -10,
	QTSS_BadURLFormat = -11,
	QTSS_NotEnoughSpace = -12,
	QTSS_TooMuchSpace = -13,
	QTSS_Unimplemented = -14,
	QTSS_RequestArrived = -15,
	QTSS_OutOfState = -16,
	QTSS_FileExists = -17
};
typedef SInt32 QTSS_ErrorCode;

//parameters
typedef enum 
{
	//request-specific
	qtssMethodParam = 0,
	qtssFilePathParam = 1,
	qtssURLParam = 2,
	qtssFilePathTruncParam = 3, //file path without last element of path
	qtssFileNameParam = 4, //file path without last element of path
	qtssClientPortA = 5,
	qtssClientPortB = 6,
	qtssFileDigitParam = 7,
	qtssAbsoluteURLParam = 8,
	qtssFullRequestParam = 9,
	qtssFirstLineParam = 10,
	qtssTruncAbsoluteURLParam = 11, //absolute URL without last element of path
	//global server parameters
	
	//RTP media src parameters
	qtssFirstSequenceNumber = 500,
	qtssSeqNumRandomOffset = 501,
	qtssTimescale = 502,
	qtssFirstTimestamp = 503
	
} QTSS_ParamKeywords;

//important constants used by the server
typedef enum {
	kQTSSNumRequestParameters = 12,
	kQTSSMinRequestParamIndex = 0,
	kQTSSMaxRequestParamIndex = 11,
	kQTSSNumServerParameters = 0,
	kQTSSNumMediaSrcParameters = 4,
	kMinMediaSrcParamIndex = 500,
	kMaxMediaSrcParamIndex = 503
} QTSS_Private_Constants;


typedef enum {
	QTSS_IO_Closed, 	//connection is terminated or file is closed.
	QTSS_IO_Open, 		//connection/file is open
	QTSS_IO_Data, 		//socket/file is open and has unread data on it
	QTSS_IO_Unknown 	//an unknown or unspecified error occurred
} QTSS_IOStatus;


/********************************************************************/
//QTSS Callback Interface

// You may call any of the following functions from inside your plug-in to 
// have AppleShare IP perform the requested function.

/*************************/
/**  Memory Management  **/

void *QTSS_AllocateMemory (long size);
/* Allocates size bytes. Returns NULL if not enough memory is available. */

void QTSS_FreeMemory (void *data);
/*	Deallocates memory previously allocated with QTSS_AllocateMemory. */


/*************************/
/**  Parameters  **/

QTSS_ErrorCode QTSS_GetParameter (QTSS_ParamKeywords whichParam, char **buffer, int *len);
/*	QTSS_GetParameter returns a pointer to the parameter requested and its length in bytes. 
	Because this pointer references internal server data, you might want to copy the data into
	a private buffer. The server guarantees that the returned pointer will remain valid for
	the duration of the request, if it is a request related parameter, or the duration of the
	server, if it is a global server parameter.
*/

QTSS_ErrorCode QTSS_SetParameter (QTSS_ParamKeywords whichParam, char *buffer, int len);
/*	Sets param parameter to new value provided by buffer. The value in the buffer is 
	copied into the servers internal buffer, so you may delete buffer immediately.
*/

/*************************/
/**  Synchronous I/O  **/

QTSS_ErrorCode QTSS_Read (QTSS_FileDesc fd, void *buffer, long bufferLen, long *dataLen);
/*	Reads data from a file or socket. (This call allows reading of the log.)  You must supply
	storage for data and pass in the maximum amount of data to be returned in the len parameter.
	QTSS will set lenRead to the actual amount of data returned.
*/

QTSS_ErrorCode QTSS_Write (QTSS_FileDesc fd, void *data, long bytesIn, long* bytesSent);
/*	Writes an arbitrarily large buffer of data into a file, log or socket. This call implements
	a raw data stream. This call may block and, depending on the nature of the i/o device, it
	may buffer data internally. 

	In the case of the HTTP connection: you can write your own header and content. Also, if this
	call ever returns a non-zero value other than values defined below, you should terminate any
	processing and return control to AppleShare IP immediately. This is an indication that the
	client has disconnected and AppleShare IP needs to clean up the connection.
*/

QTSS_ErrorCode QTSS_WriteMsg (QTSS_FileDesc fd,char* msg);
/*	Writes NULL terminated string into a file, log or socket. This call may block and,
	depending on the nature of the i/o device, it may buffer data internally.
*/

QTSS_ErrorCode QTSS_Close (QTSS_FileDesc fd);
/*	Gracefully closes a connection/file. */


QTSS_ErrorCode QTSS_Status (QTSS_FileDesc fd, QTSS_IOStatus *status);
/*	Returns the status of the specified socket or file. Obviously, this is an invalid call\
	to make from any message handler except the QTSS_Run message. Returns the appropriate
	QTSS_ConnectionStatus in the status parameter.
*/

#ifdef __cplusplus
}
#endif

#endif
