/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
/*
	File:		RTPModule.h

	Contains:	Base class that defines the interface that RTP modules can override.
				The primary job of an RTP module is to serve a specific form of RTP
				content (Moov files, camera, reflector, etc). So the methods that a
				module must implement are very much a part of that goal. 
					
	$Log: RTPModule.h,v $
	Revision 1.2  1999/02/19 23:07:23  ds
	Created
	

*/

#ifndef _RTPMODULE_H_
#define _RTPMODULE_H_

#include "RTSPRequestInterface.h"
#include "RTSPProtocol.h"
#include "QTSS.h"


class RTCPModule;
class RTPSession;
class RTPStream;

class RTPModule
{
	public:
	
		RTPModule(UInt32 inRoleMask = 0, char* inModuleName = NULL) : fName(inModuleName), fRoleMask(inRoleMask) {}
		virtual ~RTPModule() {}
		
		//WHAT TYPE OF MODULE IS THIS???
		enum
		{
			kPreProcessingModule = 			0x00000001,	//All are UInt32s
			kRequestProcessingModule = 		0x00000002,
			kPostProcessingModule = 		0x00000004,
			kAllRTCPProcessingModule =		0x00000008,
			kAllTimeoutProcessingModule =	0x00000010,
			kRTSPSessionClosingModule =		0x00000020
		};
		
		bool	IsPreProcessingModule()		{ return fRoleMask & kPreProcessingModule; }
		bool	IsRequestProcessingModule() { return fRoleMask & kRequestProcessingModule; }
		bool	IsPostProcessingModule() 	{ return fRoleMask & kPostProcessingModule; }
		bool	IsRTCPProcessingModule() 	{ return fRoleMask & kAllRTCPProcessingModule; }
		bool	IsTimeoutProcessingModule() { return fRoleMask & kAllTimeoutProcessingModule; }
		bool	IsRTSPSessionClosingModule(){ return fRoleMask & kRTSPSessionClosingModule; }
		
		//The key methods that a module must implement are prototyped here
		
		//Initialize gets called once for each module at startup time
		virtual bool  Initialize() {return true;}
		
		//Shutdown gets called once for each module at shutdown time
		virtual void		Shutdown(){}
		
		//Each RTP module will be invoked when a Describe is sent by the client
		//(until one of the modules responds)
		virtual RTSPProtocol::RTSPStatusCode Describe(RTSPRequestInterface* /*inRequest*/,
											void** /*outCookie*/) { return RTSPProtocol::kSuccessOK; }
		
		//Returning NoErr & no session means "I'm not the right module for this
		//request". Returning an Error & no session means, "I'm the right module,
		//but I can't handle this request.". Returning noErr & a session means,
		//I'm the right module, here is the session
		virtual RTSPProtocol::RTSPStatusCode  	NewSession(RTSPRequestInterface* /*inRequest*/,
											RTPSession* /*inSessionP*/,
											void* /*inCookie*/) { return RTSPProtocol::kSuccessOK; }
		
		//Allows the module to clean up resources allocated for this session before
		//it goes away. At this point, the session is completely valid, but you are
		//guarenteed to have no other activity on this session.
		virtual void			DestroySession(RTPSession* /*inSessionP*/) {}
		virtual void			DestroyCookie(void* /*inCookieP*/) {}
		
		//This gets called immediately before ProcessRTSPRequest
		virtual void	PreProcessRTSPRequest(RTSPRequestInterface* /*inRequest*/,
												RTPSession* /*inSession*/)
														{}
		
		//After taking control of an RTPSession, the module becomes responsible
		//for processing all RTSP requests on that session. It does so here.						
		virtual RTSPProtocol::RTSPStatusCode	ProcessRTSPRequest(RTSPRequestInterface* /*inRequest*/,
												RTPSession* /*inSession*/)
														{ return RTSPProtocol::kSuccessOK; }
		
		//This gets called immediately after ProcessRTSPRequest
		virtual void	PostProcessRTSPRequest(RTSPRequestInterface* /*inRequest*/,
												RTPSession* /*inSession*/)
														{}
														
		//Gets called each time an RTCP packet arrives
		virtual void	ProcessRTCPPacket(RTPStream* /*inStream*/, StrPtrLen* /*inPacket*/) {}
		
		virtual void	ProcessTimeout(RTPSession* /*inRTPSession*/) {}
		
		virtual void	ProcessRTSPSessionClosing(RTSPSessionInterface* /*inRTSPSession*/) {}
		
		//The server may sometimes query the module for information on the media src.
		//It does so here. The param passed in must be in the media src range of param
		//keywords												
		virtual QTSS_ErrorCode	GetParameter(QTSS_ParamKeywords ,
												RTPSession*,
												RTPStream*,
												void* /*outResult*/,
												UInt32* /*ioLen*/)
												{ return QTSS_BadArgument; }
												
		//when playing, the server will invoke the module from time to time asking for
		//packets. The packets sent must be a complete RTP packet, except for the
		//RTP version #, the seq # & timestamp random offsets, & ssrc.
		//Return # milliseconds before the next invocation on this session. 0 means
		//don't invoke me again
		virtual SInt64			SendPackets(RTPSession*) { return 0; }
												
		StrPtrLen*				GetName() { return &fName; }
	
	protected:
	
		StrPtrLen	fName;
		static StrPtrLen sDefaultName;

		//What roles does this module want to be invoked in?
		UInt32	fRoleMask;
											
};



class RTPModuleInterface
{
	public:
	
		RTPModule* GetRTPModuleByIndex(UInt32 inIndex);

		static void ExecuteRTCPModules(RTPStream* inStream, StrPtrLen* inPacket);
		static RTSPProtocol::RTSPStatusCode DoDescribe(RTSPRequestInterface* inRequest);
		static RTSPProtocol::RTSPStatusCode	NewSession(RTSPRequestInterface* inRequest,
													RTPSession* inSessionP);

		static RTSPProtocol::RTSPStatusCode DoSetup(RTSPRequestInterface *inRTSPRequest, StrPtrLen* inSessionID);
		static RTSPProtocol::RTSPStatusCode ProcessRequest( RTSPRequestInterface *inRTSPRequest, StrPtrLen* inSessionID);		
		static void  	SessionClosing( RTSPSessionInterface *session);
		static void 	PreProcessRTSPRequest(RTSPRequestInterface* inRTSPRequest, RTPSession* inSession);
		static void 	PostProcessRTSPRequest(RTSPRequestInterface* inRTSPRequest, RTPSession* inSession);
		static void		ProcessTimeout(RTPSession* inRTPSession);
		
		void InitModules();

		enum
		{
			kMaxSessionIDLength = 36//session IDs are at most this long (UInt32)
		};

	protected:
	
		//creates a new session ID. Pass in a buffer at least kSessionIDLength in length,
		//and this function will create a unique sessionID.
		//returns the length of that sessionID
		static UInt32 GenerateNewSessionID(char* ioBuffer);

		static RTSPProtocol::RTSPStatusCode SendToRTPModule(RTSPRequestInterface *inRTSPRequest, StrPtrLen* inSessionID);
	
		static RTPModule**		sRTPModuleArray;
		static UInt32			sNumRTPModules;		
	
		//for responding to OPTIONS requests
		static StrPtrLen	sPublicHeader;
};


#endif //_RTPMODULE_H_
