/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
/*
	File:		RTPReflectorModule.cpp

	Contains:	Implementation of RTPReflectorModulex class. 
					
	
	$Log: RTPReflectorModule.cpp,v $
	Revision 1.2  1999/02/19 23:08:25  ds
	Created
	

*/

#include "RTPReflectorModule.h"
#include "RTPSession.h"
#include "RTPStream.h"
#include "RTPServerInterface.h"
#include "RTSPProtocol.h"

#include "StringParser.h"
#include "OSFileSource.h"
#include "OS.h"

#pragma mark _RTP_REFLECTOR_MODULE_

StrPtrLen	RTPReflectorModule::sSDPSuffix(".sdp", 4);
OSRefTable 	RTPReflectorModule::sSessionMap;

RTSPProtocol::RTSPStatusCode RTPReflectorModule::Describe(RTSPRequestInterface* inRequest, void** outCookie)
{
	//Reflector module requires an .sdp file and needs to rewrite it on the fly
	//before sending it back to the client.
	
	//check to see if this is a reflector request.
	StrPtrLen theSDPData;
	theSDPData.Ptr = this->GetSDPData(inRequest, qtssFilePathParam, &theSDPData.Len);
	if (theSDPData.Ptr == NULL)
		return RTSPProtocol::kSuccessOK;
	
	//This object takes over responsibility for destroying the actual SDP data
	SDPParser* theSDPParser = new ('sdpp') SDPParser();
	theSDPParser->Parse(theSDPData.Ptr, theSDPData.Len);
			
	//make sure that this sdp file is in fact reflectable. If it aint, abort
	if (!theSDPParser->IsReflectable())
	{
		delete theSDPParser;
		return RTSPProtocol::kSuccessOK;
	}
		
	//ok, this request is the reflectors responsibility. 
	//make the SDPParser the cookie so we don't have to repeat the same work later
	*outCookie = theSDPParser;
	
	//Unfortunately we can't send the sdp file back as-is, because we need to make
	//it look just like the sdp file for a local movie. We can use the SDP parser utility.
	UInt32 theLocalSDPLen;
	OSCharArrayDeleter theLocalSDP(SDPParser::ConvertToLocalSDP(&theSDPData, &theLocalSDPLen));
	
	//above function has signalled that this request belongs to us, so let's respond
	iovec theDescribeVec[2] = { 0 };
	theDescribeVec[1].iov_base = theLocalSDP.GetObject();
	theDescribeVec[1].iov_len = theLocalSDPLen;
	RTPServerInterface::SendDescribeResponse(inRequest, &theDescribeVec[0], 2, theLocalSDPLen);

	return RTSPProtocol::kSuccessOK;
}

char*	RTPReflectorModule::GetSDPData(RTSPRequestInterface* inRequest,
												QTSS_ParamKeywords inParam, UInt32* outLen)
{
	Assert(outLen != NULL);
	
	//locate the SDP file
	OSCharArrayDeleter sdpPath(inRequest->GetFullPath(inParam, &sSDPSuffix));
	OSFileSource theSDPFile(sdpPath.GetObject());
	if (theSDPFile.GetLength() == 0)
		return NULL;
	
	//yup, there's an sdp file. Return the contents and assume that it is a reflector sdp file
	char* sdpData = new ('rsdp') char[theSDPFile.GetLength() + 2];
	UInt32 recvLen = 0;
	QTSS_ErrorCode err = theSDPFile.Read(sdpData, theSDPFile.GetLength(), &recvLen);
	if ((err != QTSS_NoErr) || (recvLen != theSDPFile.GetLength()))
	{
		Assert(0);//can we ever get here?
		delete [] sdpData;
		return NULL;//just as if the file doesn't exist
	}
	//return the file data and its length
	*outLen = theSDPFile.GetLength();
	return sdpData;
}

RTSPProtocol::RTSPStatusCode RTPReflectorModule::NewSession(RTSPRequestInterface* inRequest,
											RTPSession* inSessionP, void* inCookie)
{
	Assert(inRequest != NULL);
	Assert(inSessionP != NULL);
	
	//if we didn't assign the cookie on the describe, there is NO WAY we should do anything here
	if (inCookie == NULL)
		return RTSPProtocol::kSuccessOK;
	
	ReflectorSession* theSession = NULL;
	SDPParser* theSDPParser = (SDPParser*)inCookie;
		
	//Build a sessionIDStr. Reflector sessions are uniquely identified by the (multicast)
	//source IP address and port #.
	char theSessionIDStr[sizeof(UInt32) + sizeof(UInt16)];
	::memcpy(theSessionIDStr, &theSDPParser->GetStreamInfo(0)->fIPAddr, sizeof(UInt32));
	::memcpy(&theSessionIDStr[sizeof(UInt32)], &theSDPParser->GetStreamInfo(0)->fPort, sizeof(UInt16));
	StrPtrLen theSessionID(theSessionIDStr, sizeof(UInt32) + sizeof(UInt16));
	
	//Check to see if there is already a reflector session for this URL. If there is,
	//we are all set, just refcount it and return.
	{
		OSMutexLocker locker(sSessionMap.GetMutex());
		OSRef* theSessionRef = sSessionMap.Resolve(&theSessionID);
		
		if (theSessionRef == NULL)
		{
			//If this URL doesn't already have a reflector session, we must make a new
			//one. The first step is to parse the SDP file and see what addresses we
			//can find the multicast at.
			Assert(theSDPParser->IsReflectable());
			
			//create a reflector session, and bind the sockets
			theSession = new ('refs') ReflectorSession(theSDPParser, &theSessionID);
			RTSPProtocol::RTSPStatusCode error = theSession->BindReflectorSockets(inRequest);
			if (error != RTSPProtocol::kSuccessOK)
			{
				//clean up the session safely. I'm not sure this is 100% necessary,
				//but better safe than sorry
				theSession->CloseSourceSockets();
				delete theSession;
				delete theSDPParser;
				return error;
			}
			
			//put the session's ID into the session map.
			QTSS_ErrorCode mapErr = sSessionMap.Register(theSession->GetRef());
			Assert(mapErr == QTSS_NoErr);

			//unless we do this, the refcount won't increment (and we'll delete the session prematurely
			OSRef* debug = sSessionMap.Resolve(&theSessionID);
			Assert(debug == theSession->GetRef());
		}
		else
			theSession = (ReflectorSession*)theSessionRef->GetObject();
	}
	
	//delete the sdp data and the parser
	
	delete theSDPParser;
	Assert(theSession != NULL);

	//ok, we've found or setup the proper reflector session, add this RTP session to
	//its list of sessions			
	theSession->AddSession(inSessionP);
	
	//ok, we sucessfully setup this movie. Bind the movie object to the session to tell
	//the server we are taking over this session.
	inSessionP->Bind(this, (QTSS_MediaSrcRef)theSession, ReflectorSession::kNumQualityLevels);
	return RTSPProtocol::kSuccessOK;
}

void RTPReflectorModule::DestroyCookie(void* inCookie)
{
	Assert(inCookie != NULL);
	SDPParser* theParser = (SDPParser*)inCookie;
	delete theParser;
}

void RTPReflectorModule::DestroySession(RTPSession* inSession)
{
	ReflectorSession* session = (ReflectorSession*)inSession->GetMediaSrcRef();
	
	//first remove the RTP streams from the ReflectorStream arrays
	session->RemoveSession(inSession);
		
	//check if the ReflectorSession should be deleted
	//(it should if its ref count has dropped to 0)
	OSMutexLocker locker (sSessionMap.GetMutex());
	//decrement the ref count
	sSessionMap.Release(session->GetRef());
	if (session->GetRef()->GetRefCount() == 0)
	{
		sSessionMap.UnRegister(session->GetRef());
		//deletion involves sending the ReflectorSession a kill signal.
		//But first, make sure the UDP sockets get closed. That way, if a
		//client is trying to reflect from this source RIGHT when this code
		//is executing (a possible race condition), the UDP port binding won't
		//be denied.
		session->CloseSourceSockets();
		delete session;
	}
}


RTSPProtocol::RTSPStatusCode RTPReflectorModule::ProcessRTSPRequest(RTSPRequestInterface* inRequest,
												RTPSession* inSession)
{
	//once we've bound a session, we'll start getting invoked here for RTSP requests.
	Assert(NULL != inRequest);
	Assert(NULL != inSession);

	switch (inRequest->GetMethod())
	{
		case RTSPProtocol::kSetupMethod:
			return DoSetup(inRequest, inSession);
		case RTSPProtocol::kPlayMethod:
			return DoPlay(inRequest, inSession);
		case RTSPProtocol::kTeardownMethod:
			inSession->SendTeardownResponse(inRequest);
			inSession->Signal(Task::kKillEvent);
			break;
		case RTSPProtocol::kPauseMethod:
			inSession->Pause();
			//decrement each stream's playing count
			((ReflectorSession*)inSession->GetMediaSrcRef())->ChangePlayCount(-1);
			inSession->SendPauseResponse(inRequest);
		default:
			break;
	}			
	return RTSPProtocol::kSuccessOK;
}

RTSPProtocol::RTSPStatusCode RTPReflectorModule::DoSetup(RTSPRequestInterface* inRTSPRequest, RTPSession* inSession)
{
	//find the track ID specified in the URL
	StrPtrLen* trackIDStr = inRTSPRequest->GetQTSSParameter(qtssFileDigitParam);
	if (trackIDStr->Len == 0)
		return inRTSPRequest->SendErrorResponse(RTSPProtocol::kClientBadRequest,
										RTSPMessages::kExpectedDigitFilename,
										inRTSPRequest->GetQTSSParameter(qtssURLParam));
	//conver the trackID from a string to an actual number. We are assuming that the
	//parameter returned here is NULL terminated, which is a safe assumption.
	UInt32 trackID = ::strtol(trackIDStr->Ptr, NULL, 10);
	
	//now find the ReflectorStream corresponding to this trackID
	ReflectorSession* reflectorSession = (ReflectorSession*)inSession->GetMediaSrcRef();
	if ((trackID == 0) || (reflectorSession->GetNumStreams() < trackID))
		return inRTSPRequest->SendErrorResponse(RTSPProtocol::kClientBadRequest,
												RTSPMessages::kNoReflectorStream);
	
	RTPStream* newStream = NULL;
	{
		//Because adding a stream
		//manipulates this RTPSession's stream queue, and because the reflector
		//uses that stream queue, we don't want the two running over each other.
		OSMutexLocker locker(reflectorSession->GetMutex());
		RTSPProtocol::RTSPStatusCode error = inSession->AddStream(trackID, reflectorSession->GetCodecName(trackID),
													reflectorSession->GetCodecType(trackID), inRTSPRequest, &newStream);
		if (error != RTSPProtocol::kSuccessOK)
			return error;
	}
	
	//send the SETUP response!
	Assert(newStream != NULL);
	newStream->AppendTransport(inRTSPRequest);
	inRTSPRequest->SendHeader();
	
	return RTSPProtocol::kSuccessOK;	
}

RTSPProtocol::RTSPStatusCode RTPReflectorModule::DoPlay(RTSPRequestInterface* inRequest, RTPSession* inSession)
{
	//Server shouldn't send RTCP (reflector does it), server shouldn't write rtp header either
	inSession->Play(inRequest, false, false);
	
	//increment each stream's playing count
	((ReflectorSession*)inSession->GetMediaSrcRef())->ChangePlayCount(1);
	
	//and send a standard play response
	inSession->SendPlayResponse(inRequest);
	return RTSPProtocol::kSuccessOK;
}
