/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
/*
	File:		RTPServer.cpp

	Contains:	Implementation of RTPServer class. 
					
	Change History (most recent first):

	$Log: RTPServer.cpp,v $
	Revision 1.2  1999/02/19 23:08:25  ds
	Created
	

*/

#include "RTPServer.h"
#include "RTPReflectorModule.h"
#include "RTPFileModule.h"
#include "RTCPFlowControlModule.h"
#include "RTCPDebugModule.h"
#include "RTPAccessLogModule.h"
#include "RTCPRemoteStatusModule.h"
#include "OS.h"
#include "SocketUtils.h"

#include "atomic.h"

RTPServer::~RTPServer()
{
	UInt32 moduleIter = 0;

	for (moduleIter = 0; moduleIter < sNumRTPModules; moduleIter++)
	{
		if (NULL != sRTPModuleArray[moduleIter])
		{
			sRTPModuleArray[moduleIter]->Shutdown();
			delete sRTPModuleArray[moduleIter];
		}
	}
}

bool	RTPServer::Initialize(RTSPPrefs* inPrefs, UInt16 inPortOverride)
{
	//Allocate static objects
	if (RTPServerInterface::sRTPMap == NULL)
		RTPServerInterface::sRTPMap = new OSRefTable(kRTPSessionMapSize);
	if (RTPServerInterface::sSocketPool == NULL)
		RTPServerInterface::sSocketPool = new RTPSocketPool();
	RTPServerInterface::sRTCPDataTask = &fRTCPTask;
	
	bool retval = fRTSPServer.Initialize(inPrefs, inPortOverride);
	if (!retval)
		return retval;
		
	retval = SetupUDPSockets();
	if (!retval)
	{
		RTSPModuleInterface::LogError(RTSPModule::kFatal, RTSPMessages::kNoUDPSockets, 0);
		return retval;
	}
	
	//initialize all RTP modules. Of course, right now we just have one,
	//so this code is pretty simple
	sNumRTPModules = 5;
#ifdef DEBUG_RTCP_PACKETS
	sNumRTPModules++;
#endif
    sRTPModuleArray = new ('rtpm') RTPModule*[sNumRTPModules];
	//currently, this order is actually important because the cache module
	//needs to grab requests before the moov module
	sRTPModuleArray[0] = new ('FLmd') RTPReflectorModule();
	sRTPModuleArray[1] = new ('RFmd') RTPFileModule();
	sRTPModuleArray[2] = new ('ALmd') RTPAccessLogModule();
	sRTPModuleArray[3] = new ('FCmd') RTCPFlowControlModule();
	sRTPModuleArray[4] = new ('RSmd') RTCPRemoteStatusModule();
#ifdef DEBUG_RTCP_PACKETS
	sRTPModuleArray[5] = new ('DBmd') RTCPDebugModule();
#endif

	this->InitModules();
		
	sServerState = kRunningState;
	
	//once the updater task gets this first signal, it will begin running continually.
	//Same thing for the RTCP data task
	fUpdater.Signal(Task::kStartEvent);
	fRTCPTask.Signal(Task::kStartEvent);
	
	return true;
}

bool	RTPServer::SetupUDPSockets()
{
	//function finds all IP addresses on this machine, and binds 1 RTP / RTCP
	//socket pair to a port pair on each address.
	
	UInt32 theNumAllocatedPairs = 0;
	for (UInt32 theNumPairs = 0; theNumPairs < SocketUtils::GetNumIPAddrs(); theNumPairs++)
	{
		if (sSocketPool->CreateUDPSocketPair(SocketUtils::GetIPAddr(theNumPairs), 0) != NULL)
			theNumAllocatedPairs++;
	}
	//only return an error if we couldn't allocate ANY pairs of sockets
	if (theNumAllocatedPairs == 0)
		return false;
	return true;
}

RTPSession* RTPServer::GetNewestSession()
{
	//Caller must lock down the RTP session map
	SInt64 theNewestPlayTime = 0;
	RTPSession* theNewestSession = NULL;
	
	//use the session map to iterate through all the sessions, finding the most
	//recently connected client
	for (OSRefHashTableIter theIter(sRTPMap->GetHashTable()); !theIter.IsDone(); theIter.Next())
	{
		OSRef* theRef = theIter.GetCurrent();
		RTPSession* theSession = (RTPSession*)theRef->GetObject();
		Assert(theSession->GetSessionCreateTime() > 0);
		if (theSession->GetSessionCreateTime() > theNewestPlayTime)
		{
			theNewestPlayTime = theSession->GetSessionCreateTime();
			theNewestSession = theSession;
		}
	}
	return theNewestSession;
}

SInt64 RTPTotalByteUpdater::Run()
{
	//First update total bytes. This must be done because total bytes is a 64 bit number,
	//so no atomic functions can apply
	unsigned int periodicBytes = RTPServerInterface::sPeriodicBytes;
	(void)atomic_sub(&RTPServerInterface::sPeriodicBytes, periodicBytes);
	
	RTPServerInterface::sTotalBytes += periodicBytes;
	
	//Now update the periodic packet tracker. Each time through this function, we reset
	//the value to 0, and whatever has accumilated we use as part of our packets per second computation
	unsigned int periodicPackets = RTPServerInterface::sPeriodicPackets;
	(void)atomic_sub(&RTPServerInterface::sPeriodicPackets, periodicPackets);

	SInt64 curTime = OS::Milliseconds();
	//also update current bandwidth statistic
	if (fLastBandwidthTime != 0)
	{
		Assert(curTime > fLastBandwidthTime);
		UInt32 delta = (UInt32)(curTime - fLastBandwidthTime);
		Assert(delta >= 1000);
		
		//do the bandwidth computation using floating point divides
		//for accuracy and speed.
		Float32 bits = (Float32)(periodicBytes * 8);
		Float32 theTime = (Float32)delta;
		theTime /= 1000;
		bits /= theTime;
		Assert(bits >= 0);
		RTPServerInterface::sCurrentBandwidthInBits = (UInt32)bits;
		
		//do the same computation for packets per second
		Float32 packetsPerSecond = (Float32)periodicPackets;
		packetsPerSecond /= theTime;
		Assert(packetsPerSecond >= 0);
		RTPServerInterface::sPacketsPerSecond = (UInt32)packetsPerSecond;
	}
	
	fLastBandwidthTime = curTime;
	
	//also compute average bandwidth, a much more smooth value. This is done with
	//the fLastBandwidthAvg, a timestamp of the last time we did an average, and
	//fLastBytesSent, the number of bytes sent when we last did an average.
	if ((fLastBandwidthAvg != 0) && (curTime > (fLastBandwidthAvg +
		(RTSPServerInterface::GetRTSPPrefs()->GetAvgBandwidthUpdateTimeInSecs() * 1000))))
	{
		UInt32 delta = (UInt32)(curTime - fLastBandwidthAvg);
		UInt64 bytesSent = RTPServerInterface::sTotalBytes - fLastBytesSent;

		//do the bandwidth computation using floating point divides
		//for accuracy and speed.
		Float32 bits = (Float32)(bytesSent * 8);
		Float32 theAvgTime = (Float32)delta;
		theAvgTime /= 1000;
		bits /= theAvgTime;
		Assert(bits >= 0);
		RTPServerInterface::sAvgBandwidthInBits = (UInt32)bits;

		fLastBandwidthAvg = curTime;
		fLastBytesSent = RTPServerInterface::sTotalBytes;
		
		//if the bandwidth is above the bandwidth setting, disconnect 1 user by sending them
		//a BYE RTCP packet.
		SInt32 maxKBits = RTSPServerInterface::GetRTSPPrefs()->GetMaxKBitsBandwidth();
		if ((maxKBits > -1) && (RTPServerInterface::sAvgBandwidthInBits > ((UInt32)maxKBits * 1024)))
		{
			//we need to make sure that all of this happens atomically wrt the session map
			OSMutexLocker locker(fServer->GetRTPSessionMap()->GetMutex());
			RTPSession* theSession = fServer->GetNewestSession();
			if (theSession != NULL)
				if ((curTime - theSession->GetSessionCreateTime()) <
						RTSPServerInterface::GetRTSPPrefs()->GetSafePlayDurationInSecs() * 1000)
					theSession->Signal(Task::kKillEvent);
		}
	}
	else if (fLastBandwidthAvg == 0)
	{
		fLastBandwidthAvg = curTime;
		fLastBytesSent = RTPServerInterface::sTotalBytes;
	}
	
	(void)this->GetEvents();//we must clear the event mask!
	return RTSPServerInterface::GetRTSPPrefs()->GetTotalBytesUpdateTimeInSecs() * 1000;
}
