/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
/*
	File:		RTSPPrefs.cpp

	Contains:	Implements class defined in RTSPPrefs.h.

	Change History (most recent first):
	
*/

#ifndef __MW_
#include <sys/types.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#endif

#include "RTSPPrefs.h"
#include "MyAssert.h"
#include "OS.h"
 
#pragma mark __STATIC_DATA__

StrPtrLen RTSPPrefs::sServerNameStr("QTSS", strlen("QTSS"));
StrPtrLen RTSPPrefs::sServerVersionStr("v96");
StrPtrLen RTSPPrefs::sServerBuildDateStr(__DATE__ " , "__TIME__, strlen(__DATE__ " , "__TIME__));

//These variables are used to store the default settings once
//we start altering the defaults by reading out of the file.
//It doesn't really matter what they are set to right now.
SInt32 	RTSPPrefs::sDefaultRTSPTimeout = 60;
SInt32 	RTSPPrefs::sDefaultRTPTimeout = 60;
SInt32 	RTSPPrefs::sDefaultMaximumConnections = 1000;
SInt32	RTSPPrefs::sDefaultMaxBandwidth = 100000;
char*  	RTSPPrefs::sDefaultMoviesFolder = "/Local/Library/QuickTimeStreaming/Movies";
bool 	RTSPPrefs::sDefaultTransferLogEnabled = false;
char*	RTSPPrefs::sDefaultTransferLogDir = "/Local/Library/QuickTimeStreaming/Logs/";
char*	RTSPPrefs::sDefaultTransferLogName = "QTStreamingServer";
SInt32	RTSPPrefs::sDefaultTransferLogSize = 50000000;
SInt32	RTSPPrefs::sDefaultTransferRollInterval = 0;
bool	RTSPPrefs::sDefaultErrorLogEnabled = true;
char*	RTSPPrefs::sDefaultErrorLogDir = "/Local/Library/QuickTimeStreaming/Logs/";
char*	RTSPPrefs::sDefaultErrorLogName = "QuickTimeStreamingError";
SInt32	RTSPPrefs::sDefaultErrorLogSize = 256000;
SInt32	RTSPPrefs::sDefaultErrorRollInterval = 0;
UInt32	RTSPPrefs::sDefaultErrorLogVerbosity = 1;
bool	RTSPPrefs::sDefaultScreenLoggingEnabled = true;
bool	RTSPPrefs::sDefaultBreakOnAssert = true;
UInt32	RTSPPrefs::sDefaultIPAddress = INADDR_ANY;
SInt32	RTSPPrefs::sDefaultHistoryInterval = 120;
UInt32	RTSPPrefs::sDefaultBufferSeconds = 8;
UInt32	RTSPPrefs::sDefaultTBUpdateTime = 1;
SInt64	RTSPPrefs::sDefaultDelayTime = 100;
UInt32	RTSPPrefs::sDefaultBucketSize = 16;
bool 	RTSPPrefs::sDefaultAutoRestart = true;
UInt32	RTSPPrefs::sDefaultABUpdateTime = 60;
UInt32	RTSPPrefs::sDefaultSafePlayDuration = 600;
char*	RTSPPrefs::sDefaultSDPURL = "";
char*	RTSPPrefs::sDefaultAdminEmail = "";
bool	RTSPPrefs::sDefaultDisableTimeout = true;
UInt32	RTSPPrefs::sDefaultMaxAdvanceTimeInMilSecs = 100;
char*	RTSPPrefs::sDefaultWebStatsURL = "";
UInt32	RTSPPrefs::sDefaultLossThinTolerance = 30;
UInt32	RTSPPrefs::sDefaultNumLossesToThin = 3;
UInt32	RTSPPrefs::sDefaultLossThickTolerance = 5;
UInt32	RTSPPrefs::sDefaultLossesToThick = 6;
UInt32	RTSPPrefs::sDefaultWorsesToThin = 3;
UInt32	RTSPPrefs::sDefaultRealRTSPTimeoutInSecs = 180;
char*	RTSPPrefs::sDefaultUsersMovieFolder = "";
bool	RTSPPrefs::sDefaultFlowControlLogEnabled = false;

char* RTSPPrefs::sEnabledString = "enabled";


char* RTSPPrefs::sPrefsKeyStrings[] = 
{	
	//kRTSPPortPref
	"rtsp_port",
	//kRTSPTimeout
	"rtsp_timeout",
	//kMaximumConnections
	"maximum_connections",
	//kMovieFolder
	"movie_folder",
	//kTransferLog
	"request_logging",
	//kTransferLogDir
	"request_logfile_dir",
	//kMaxTransferLogSize
	"request_logfile_size",
	//kTransferRollInterval
	"request_logfile_interval",
	//kErrorLog
	"error_logging",
	//kErrorLogDir
	"error_logfile_dir",
	//kMaxErrorLogSize
	"error_logfile_size",
	//kErrorRollInterval
	"error_logfile_interval",
	//kErrorLogVerbosity
	"error_logfile_verbosity",
	//kScreenLogging
	"screen_logging",
	//kBreakOnAssert
	"break_on_assert",
	//kBindToThisIPAddr
	"bind_ip_addr",
	//kHistoryUpdateInterval
	"history_update_interval",
	//kMaxBandwidth
	"maximum_bandwidth",
	//kBufferSeconds,
	"buffer_seconds",
	//kAutoRestart,
	"auto_restart",
	//kRTPTimeout,
	"rtp_timeout",
	//kTransferLogName
	"request_logfile_name",
	//kErrorLogName
	"error_logfile_name",
	//kTotalBytesUpdateTime
	"total_bytes_update",
	//kReflectorDelayTime
	"reflector_delay",
	//kReflectorBucketSize
	"reflector_bucket_size",
	//kABUpdateTime
	"average_bandwidth_update",
	//kSafePlayDuration
	"safe_play_duration",
	//kSDPURL
	"sdp_url",
	//kAdminEmail
	"admin_email",
	//kDisableTimeout
	"disable_timeout_on_play",
	//kMaxAdvanceTime
	"max_advance_send_time",
	//kWebStatsURL
	"web_stats_url",
	//kRemoteStatsAddr
	"remote_status_server_addr",
	//kRemoteStatsPort
	"remote_status_server_port",
	//kLossThinTolerance
	"loss_thin_tolerance",
	//kNumLossesToThin
	"num_losses_to_thin",
	//kLossThickTolerance
	"loss_thick_tolerance",
	//kNumLossesToThick
	"num_losses_to_thick",
	//kNumWorsesToThin
	"num_worses_to_thin",
	//kRealRTSPTimeout
	"real_rtsp_timeout",
	//kUsersMovieFolder
	"users_movie_folder",
	//kFlowControlLogEnabled
	"stream_thinning_log_enabled"
};


#pragma mark __RTSP_PREFS__

RTSPPrefs::RTSPPrefs(PrefsSource* inPrefsSource, bool inLogTimestamps, bool inAppendRandomOffsets,
						bool inKeyFramesOnly, bool inNoBFrames)
: 	fMutex('prfm'),
	fPrefsSource(inPrefsSource),
	fRTSPTimeoutInSecs(0),
	fRTPTimeoutInSecs(0),
	fMaximumConnections(0),
	fMaxBandwidthInKBits(0),
	fTransferLogEnabled(false),
	fTransferLogBytes(0),
	fTransferRollIntervalInDays(0),
	fErrorLogEnabled(false),
	fErrorLogBytes(0),
	fErrorRollIntervalInDays(0),
	fErrorLogVerbosity(0),
	fScreenLoggingEnabled(false),
	fBreakOnAssert(false),
	fIPAddress(0),
	fHistoryIntervalInSecs(0),
	fBufferSeconds(0),
	fAutoRestart(false),
	fTBUpdateTimeInSecs(0),
	fDelayTimeInMilSecs(0),
	fBucketSize(0),
	fABUpdateTimeInSecs(0),
	fSafePlayDurationInSecs(0),
	fDisableTimeout(false),
	fMaxAdvanceTimeInMilSecs(0),
	fRemoteStatsAddr(0),
	fRemoteStatsPort(0),
	fLogTimestamps(inLogTimestamps),
	fAppendRandomOffsets(inAppendRandomOffsets),
	fKeyFramesOnly(inKeyFramesOnly),
	fRTSPTimeoutString(fRTSPTimeoutBuf, 0),
	fNoBFrames(inNoBFrames),
	fLossThinTolerance(0),
	fNumLossesToThin(0),
	fLossThickTolerance(0),
	fLossesToThick(0),
	fWorsesToThin(0),
	fRealRTSPTimeoutInSecs(0),
	fFlowControlLogEnabled(0)
	
{
	RereadPreferences();
}

char* RTSPPrefs::GetMovieFolderPath()
{
	return GetStringPreference(fMoviesFolder.Ptr); 
}


void RTSPPrefs::RereadPreferences()
{
	static const UInt32 kMaxPrefValueSize = 1024;
	char thePrefValue[kMaxPrefValueSize];
	
	for (SInt32 x = 0; x < kNumPreferences; x++)
	{
		thePrefValue[0] = '\0';
		// function return 0 if it couldn't find the preference
		bool prefExists = (bool)fPrefsSource->GetValue(sPrefsKeyStrings[x], &thePrefValue[0]);
		
		/*
		each preferences requires setting a specific server global.
		do that here. If there was nothing in the NetInfo database
		for this setting, the buffer will remain an empty string,
		so each case in the case statement will set the proper setting
		back to its default. Most of this can be done pre-emptive safe
		without a mutex. For string preferences, we must grab the mutex.
		*/
		switch (x)
		{
			//integer or boolean preferences
			case kRTSPPortPref:
				break;//we no longer store any value for this preference
			case kRemoteStatsPort:
			{
				if (thePrefValue[0] == '\0')
					fRemoteStatsPort = 0;
				else
				{
					int tempPort2;
					::sscanf(thePrefValue, "%d", &tempPort2);
					fRemoteStatsPort = (UInt16)tempPort2;
				}
				break;
			}
			case kRTSPTimeout:
				this->ProcessSIntPref(thePrefValue, &fRTSPTimeoutInSecs, sDefaultRTSPTimeout);
				//also store the RTSP timeout as a string. We can't make the access to this
				//variable pre-emptive safe, however, so only set this variable once.
				if (fRTSPTimeoutString.Len == 0)
				{
					::sprintf(fRTSPTimeoutString.Ptr, "%lu", fRTSPTimeoutInSecs);
					fRTSPTimeoutString.Len = ::strlen(fRTSPTimeoutString.Ptr);
				}
				break;
			case kRTPTimeout:
				this->ProcessSIntPref(thePrefValue, &fRTPTimeoutInSecs, sDefaultRTPTimeout);
				break;
			case kMaximumConnections:
				this->ProcessSIntPref(thePrefValue, &fMaximumConnections, sDefaultMaximumConnections);
				break;
			case kMaxTransferLogSize:
				this->ProcessSIntPref(thePrefValue, &fTransferLogBytes, sDefaultTransferLogSize);
				break;
			case kTransferRollInterval:
				this->ProcessSIntPref(thePrefValue, &fTransferRollIntervalInDays, sDefaultTransferRollInterval);
				break;
			case kMaxErrorLogSize:
				this->ProcessSIntPref(thePrefValue, &fErrorLogBytes, sDefaultErrorLogSize);
				break;
			case kErrorRollInterval:
				this->ProcessSIntPref(thePrefValue, &fErrorRollIntervalInDays, sDefaultErrorRollInterval);
				break;
			case kErrorLogVerbosity:
				this->ProcessUIntPref(thePrefValue, &fErrorLogVerbosity, sDefaultErrorLogVerbosity);
				break;
			case kHistoryUpdateInterval:
				this->ProcessSIntPref(thePrefValue, &fHistoryIntervalInSecs, sDefaultHistoryInterval);
				break;
			case kMaxBandwidth:
				this->ProcessSIntPref(thePrefValue, &fMaxBandwidthInKBits, sDefaultMaxBandwidth);
				break;
			case kTotalBytesUpdateTime:
				this->ProcessUIntPref(thePrefValue, &fTBUpdateTimeInSecs, sDefaultTBUpdateTime);
				break;
			case kReflectorBucketSize:
				this->ProcessUIntPref(thePrefValue, &fBucketSize, sDefaultBucketSize);
				break;
			case kBufferSeconds:
				this->ProcessUIntPref(thePrefValue, &fBufferSeconds, sDefaultBufferSeconds);
				break;
			case kABUpdateTime:
				this->ProcessUIntPref(thePrefValue, &fABUpdateTimeInSecs, sDefaultABUpdateTime);
				break;
			case kSafePlayDuration:
				this->ProcessUIntPref(thePrefValue, &fSafePlayDurationInSecs, sDefaultSafePlayDuration);
				break;
			case kMaxAdvanceTime:
				this->ProcessUIntPref(thePrefValue, &fMaxAdvanceTimeInMilSecs, sDefaultMaxAdvanceTimeInMilSecs);
				break;
			case kLossThinTolerance:
				this->ProcessUIntPref(thePrefValue, &fLossThinTolerance, sDefaultLossThinTolerance);
				break;
			case kNumLossesToThin:
				this->ProcessUIntPref(thePrefValue, &fNumLossesToThin, sDefaultNumLossesToThin);
				break;
			case kLossThickTolerance:
				this->ProcessUIntPref(thePrefValue, &fLossThickTolerance, sDefaultLossThickTolerance);
				break;
			case kNumLossesToThick:
				this->ProcessUIntPref(thePrefValue, &fLossesToThick, sDefaultLossesToThick);
				break;
			case kNumWorsesToThin:
				this->ProcessUIntPref(thePrefValue, &fWorsesToThin, sDefaultWorsesToThin);
				break;
			case kRealRTSPTimeout:
				this->ProcessUIntPref(thePrefValue, &fRealRTSPTimeoutInSecs, sDefaultRealRTSPTimeoutInSecs);
			case kReflectorDelayTime:
			{
				if (thePrefValue[0] == '\0')
					fDelayTimeInMilSecs = sDefaultDelayTime;
				else
				{
					UInt32 tempDelayTime = 0;
					::sscanf(thePrefValue, "%ld", &tempDelayTime);
					fDelayTimeInMilSecs = tempDelayTime;
				}
				break;
			}
			
			//boolean settings
			case kTransferLog:
				this->ProcessBooleanPref(thePrefValue, &fTransferLogEnabled,
											sDefaultTransferLogEnabled);
				break;
			case kErrorLog:
				this->ProcessBooleanPref(thePrefValue, &fErrorLogEnabled,
											sDefaultErrorLogEnabled);
				break;
			case kScreenLogging:
				this->ProcessBooleanPref(thePrefValue, &fScreenLoggingEnabled,
											sDefaultScreenLoggingEnabled);
				break;
			case kBreakOnAssert:
				this->ProcessBooleanPref(thePrefValue, &fBreakOnAssert,
											sDefaultBreakOnAssert);
				break;
			case kAutoRestart:
				this->ProcessBooleanPref(thePrefValue, &fAutoRestart,
											sDefaultAutoRestart);
				break;
			case kDisableTimeout:
				this->ProcessBooleanPref(thePrefValue, &fDisableTimeout,
											sDefaultDisableTimeout);
				break;
			case kFlowControlLogEnabled:
				this->ProcessBooleanPref(thePrefValue, &fFlowControlLogEnabled,
											sDefaultFlowControlLogEnabled);
				break;

			//string preferences
			case kMovieFolder:
				this->ProcessStringPref(thePrefValue, &fMoviesFolder, sDefaultMoviesFolder);
				break;
			case kUsersMovieFolder:
				this->ProcessStringPref(thePrefValue, &fUsersMovieFolder, sDefaultUsersMovieFolder);
				break;
			case kTransferLogDir:
				this->ProcessStringPref(thePrefValue, &fTransferLogDir,
												sDefaultTransferLogDir);
				break;
			case kTransferLogName:
				this->ProcessStringPref(thePrefValue, &fTransferLogName,
												sDefaultTransferLogName);
				break;
			case kErrorLogDir:
				this->ProcessStringPref(thePrefValue, &fErrorLogDir,
												sDefaultErrorLogDir);
				break;
			case kErrorLogName:
				this->ProcessStringPref(thePrefValue, &fErrorLogName,
												sDefaultErrorLogName);
				break;
			case kSDPURL:
				this->ProcessStringPref(thePrefValue, &fSDPURL,
												sDefaultSDPURL);
				break;
			case kAdminEmail:
				this->ProcessStringPref(thePrefValue, &fAdminEmail,
												sDefaultAdminEmail);
				break;
			case kWebStatsURL:
			{
				OSMutexLocker locker(&fMutex);
				
				if ((fWebStatsURL.Ptr != NULL) && (fWebStatsURL.Ptr != sDefaultWebStatsURL))
					delete [] fWebStatsURL.Ptr;
					
				if (!prefExists)
				{
					fWebStatsURL.Ptr = sDefaultWebStatsURL;
					fWebStatsURL.Len = ::strlen(sDefaultWebStatsURL);
				}
				else
				{
					fWebStatsURL.Ptr = new ('sprf') char[::strlen(thePrefValue) + 2];
					::strcpy(fWebStatsURL.Ptr, thePrefValue);
					fWebStatsURL.Len = ::strlen(fWebStatsURL.Ptr);
				}			
			}
			break;
			
			//other random settings
			case kBindToThisIPAddr:
			{
				fIPAddress = sDefaultIPAddress;
				if (thePrefValue[0] != '\0')
				{
					struct in_addr theAddress;
					if (::inet_aton(thePrefValue, &theAddress) != 0)
	                	fIPAddress = ntohl(theAddress.s_addr);
	            }
	       		break;
			}

			case kRemoteStatsAddr:
			{
				fRemoteStatsAddr = 0;
				if (thePrefValue[0] != '\0')
				{
					struct in_addr theAddress;
					if (::inet_aton(thePrefValue, &theAddress) != 0)
	                	fRemoteStatsAddr = ntohl(theAddress.s_addr);
	            }
	       		break;
			}

			default:
				Assert(0);
			break;
		}
	}
}

char*	RTSPPrefs::GetMovieFolderPath(UInt32 inPadding, UInt32* outLength)
{
	Assert(outLength != NULL);
	
	OSMutexLocker locker(&fMutex);
	char* thePrefCopy = new ('mvpt') char[fMoviesFolder.Len + inPadding + 2];
	::memcpy(thePrefCopy, fMoviesFolder.Ptr, fMoviesFolder.Len);
	*outLength = fMoviesFolder.Len;
	return thePrefCopy;
}

#if !__MacOSX__
char*	RTSPPrefs::GetUsersMovieFolderPath(UInt32 inPadding, UInt32* outLength)
{
	Assert(outLength != NULL);
	
	OSMutexLocker locker(&fMutex);
	char* thePrefCopy = new ('umpt') char[fUsersMovieFolder.Len + inPadding + 2];
	::memcpy(thePrefCopy, fUsersMovieFolder.Ptr, fUsersMovieFolder.Len);
	*outLength = fUsersMovieFolder.Len;
	return thePrefCopy;
}
#endif

void RTSPPrefs::ProcessStringPref(char* inBuffer, StrPtrLen* inPref, char* inDefault)
{
	OSMutexLocker locker(&fMutex);
	
	if ((inPref->Ptr != NULL) && (inPref->Ptr != inDefault))
		delete [] inPref->Ptr;
		
	if (inBuffer[0] == '\0')
	{
		inPref->Ptr = inDefault;
		inPref->Len = strlen(inDefault);
	}
	else
	{
		inPref->Ptr = new ('sprf') char[::strlen(inBuffer) + 2];
		::strcpy(inPref->Ptr, inBuffer);
		inPref->Len = strlen(inPref->Ptr);
	}
}

void RTSPPrefs::ProcessBooleanPref(char* inBuffer, bool* inPref, bool inDefault)
{
	if (inBuffer[0] == '\0')
		*inPref = inDefault;
	else if (::strcmp(inBuffer, sEnabledString) == 0)
		*inPref = true;
	else
		*inPref = false;
}

void RTSPPrefs::ProcessUIntPref(char* inBuffer, UInt32* inPref, UInt32 inDefault)
{
	if (inBuffer[0] == '\0')
		*inPref = inDefault;
	else
		::sscanf(inBuffer, "%ld", inPref);
}

void RTSPPrefs::ProcessSIntPref(char* inBuffer, SInt32* inPref, SInt32 inDefault)
{
	if (inBuffer[0] == '\0')
		*inPref = inDefault;
	else
		::sscanf(inBuffer, "%ld", inPref);
}

char*	RTSPPrefs::GetStringPreference(char* inPref)
{
	OSMutexLocker locker(&fMutex);
	char* thePrefCopy = new ('spcp') char[::strlen(inPref) + 2];
	::strcpy(thePrefCopy, inPref);
	return thePrefCopy;
}
