/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
/*
	File:		RTSPValidationModule.cpp

	Contains:	Implements super-duper validation module.

	$Log: RTSPValidationModule.cpp,v $
	Revision 1.2  1999/02/19 23:08:41  ds
	Created
	
	
*/

#include "RTSPValidationModule.h"

char *sIPAddr = "17.221.41.108";
char *sSyntax1 = "DescRIBE /my/strange/url.bar RTSP/1.0\r\nConnection: boo\r\nFoop\r\n\r\n";
char *sSyntax2 = "FEOLED rtsp://17.221.41.108/this/is/totally/legit RTSP/1.0\r\n\r\n";
char *sSyntax3 = "options rtsp://17.221.41.108/this%29is/not%234a/%4legit/%54url RTSP/1.0\r\n\r\n";
char *sSyntax4 = "PLAy rtsp://17.221.41.108/this/is/totally/legit/but/URL/is/way/too/long/but/URL/is/way/too/long/but/URL/is/way/too/long/but/URL/is/way/too/long/but/URL/is/way/too/long/but/URL/is/way/too/long/but/URL/is/way/too/long/this/is/totally/legit/but/URL/is/way/too/long/but/URL/is/way/too/long/but/URL/is/way/too/long/but/URL/is/way/too/long/but/URL/is/way/too/long/but/URL/is/way/too/long/but/URL/is/way/too/long RTSP/1.0\r\n\r\n";
char *sSyntax5 = "SETUP rtsp://17.221.41.108/this/is/totally/legit/but/URL/is/way/ RTSP/1.0\r\n\r\n\r\n";
char *sSyntax6 = "ANNOUNCdE    /this/is/totally/legit/but/URL/is/way/    RTSP/1.0\r\nTransport: 501\r\n\r\n";
char *sSyntax7 = "aTEArdown rtsp/this/is/totally/legit/but/URL/is/way/    RTSP/1.0\r\nTransport: 501\r\n\r\n";
char *sSyntax8 = "TEArdown rtsp/this/is/totally/legit/but/URL/is/way/    RTSP/1.0\r\nTransport: 501TEArdown rtsp/this/is/totally/legit/but/URL/is/way/    RTSP/1.0\r\nTransport: 501TEArdown rtsp/this/is/totally/legit/but/URL/is/way/    RTSP/1.0\r\nTransport: 501TEArdown rtsp/this/is/totally/legit/but/URL/is/way/    RTSP/1.0\r\nTransport: 501TEArdown rtsp/this/is/totally/legit/but/URL/is/way/    RTSP/1.0\r\nTransport: 501TEArdown rtsp/this/is/totally/legit/but/URL/is/way/    RTSP/1.0\r\nTransport: 501TEArdown rtsp/this/is/totally/legit/but/URL/is/way/    RTSP/1.0\r\nTransport: 501TEArdown rtsp/this/is/totally/legit/but/URL/is/way/    RTSP/1.0\r\nTransport: 501TEArdown rtsp/this/is/totally/legit/but/URL/is/way/    RTSP/1.0\r\nTransport: 501TEArdown rtsp/this/is/totally/legit/but/URL/is/way/    RTSP/1.0\r\nTransport: 501TEArdown rtsp/this/is/totally/legit/but/URL/is/way/    RTSP/1.0\r\nTransport: 501TEArdown rtsp/this/is/totally/legit/but/URL/is/way/    RTSP/1.0\r\nTransport: 501TEArdown rtsp/this/is/totally/legit/but/URL/is/way/    RTSP/1.0\r\nTransport: 501TEArdown rtsp/this/is/totally/legit/but/URL/is/way/    RTSP/1.0\r\nTransport: 501TEArdown rtsp/this/is/totally/legit/but/URL/is/way/    RTSP/1.0\r\nTransport: 501TEArdown rtsp/this/is/totally/legit/but/URL/is/way/    RTSP/1.0\r\nTransport: 501TEArdown rtsp/this/is/totally/legit/but/URL/is/way/    RTSP/1.0\r\nTransport: 501TEArdown rtsp/this/is/totally/legit/but/URL/is/way/    RTSP/1.0\r\nTransport: 501TEArdown rtsp/this/is/totally/legit/but/URL/is/way/    RTSP/1.0\r\nTransport: 501TEArdown rtsp/this/is/totally/legit/but/URL/is/way/    RTSP/1.0\r\nTransport: 501TEArdown rtsp/this/is/totally/legit/but/URL/is/way/    RTSP/1.0\r\nTransport: 501TEArdown rtsp/this/is/totally/legit/but/URL/is/way/    RTSP/1.0\r\nTransport: 501TEArdown rtsp/this/is/totally/legit/but/URL/is/way/    RTSP/1.0\r\nTransport: 501TEArdown rtsp/this/is/totally/legit/but/URL/is/way/    RTSP/1.0\r\nTransport: 501TEArdown rtsp/this/is/totally/legit/but/URL/is/way/    RTSP/1.0\r\nTransport: 501TEArdown rtsp/this/is/totally/legit/but/URL/is/way/    RTSP/1.0\r\nTransport: 501TEArdown rtsp/this/is/totally/legit/but/URL/is/way/    RTSP/1.0\r\nTransport: 501TEArdown rtsp/this/is/totally/legit/but/URL/is/way/    RTSP/1.0\r\nTransport: 501TEArdown rtsp/this/is/totally/legit/but/URL/is/way/    RTSP/1.0\r\nTransport: 501TEArdown rtsp/this/is/totally/legit/but/URL/is/way/    RTSP/1.0\r\nTransport: 501TEArdown rtsp/this/is/totally/legit/but/URL/is/way/    RTSP/1.0\r\nTransport: 501TEArdown rtsp/this/is/totally/legit/but/URL/is/way/    RTSP/1.0\r\nTransport: 501TEArdown rtsp/this/is/totally/legit/but/URL/is/way/    RTSP/1.0\r\nTransport: 501TEArdown rtsp/this/is/totally/legit/but/URL/is/way/    RTSP/1.0\r\nTransport: 501TEArdown rtsp/this/is/totally/legit/but/URL/is/way/    RTSP/1.0\r\nTransport: 501TEArdown rtsp/this/is/totally/legit/but/URL/is/way/    RTSP/1.0\r\nTransport: 501TEArdown rtsp/this/is/totally/legit/but/URL/is/way/    RTSP/1.0\r\nTransport: 501TEArdown rtsp/this/is/totally/legit/but/URL/is/way/    RTSP/1.0\r\nTransport: 501TEArdown rtsp/this/is/totally/legit/but/URL/is/way/    RTSP/1.0\r\nTransport: 501";

//encoded, pipelined
char *sLegit1 = "record     	/a%20url/%25with%23/%38encoded%347characters/  RTSP/1.0\r\nCSeq: 0\r\nHost: foop\r\nConnection: keep-alive\r\nHost: meow\r\naccept: \r\n\r\n\r\n\r\n";
//pipelined
char *sLegit2 = "SETUP rtsp://17.221.41.108/ RTSP/1.0\r\n\r\nDESCRIBE rtsp://myhouse.com/short/url/ RTSP/1.0\rCSeq: 14\rTransport: 5000-5001\rRange: npt=14.56-\r\r";
//special filenames
char *sLegit3 = "OPTIONS rtsp://dopey.com/%20%202020%20%20/this/filename.has.a.number=4 RTSP/1.0\r\nHOST: dopey.com\r\nhost: meow.com\r\ncseq: 4\r\n\r\n";
//really long request
char *sLegit4 = "OPTIONS rtsp://dopey.com/%20%202020%20%20/this/filename.has.a.number=4/is/totally/legit/but/URL/is/way/ RTSP/1.0\r\nHOST: dopey.com\r\nhost: meow.com\r\nrange: npt=0-45.76\r\ncseq: 4\r\ncseq: 4\r\ncseq: 4\r\naccept-language: bollocks!\r\nbandwidth: too much for your server!\r\nexpires: now!\r\nLooopy: meowieal\r\nif-modified-since: loopy\r\ncseq: 4\r\n\r\n";
//pipelined really short requests
char *sLegit5 = "SETUP /short/url RTSP/1.0\r\nCSeq: 1\r\n\r\nPlay /short/url/with/number23\r\n\r\nteardown /short/url RTSP/1.0\r\ncseq: 5\r\n\r\npause /vryshrt RTSP/1.0\r\n\r\n";

#if _VALIDATION_CLIENT_

#include <string.h>
#include <stdlib.h>

#ifndef __MW_
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/errno.h>
#include <netinet/in.h>
#include <fcntl.h>
#include <mach/mach.h>
#include <mach/cthreads.h>
#include <netinet/tcp.h>
#include <unistd.h>
#endif

class RTSPClient
{
	public:
	
		RTSPClient(char* inRequest, char* ipAddr, bool fragmentPackets);
		~RTSPClient() { ::close(fSocket); }
		
		char* ReceiveResponse();
	
	private:
	
		int fSocket;
		char fResponseBuffer[4096];
};

RTSPClient::RTSPClient(char* inRequest, char* ipAddr, bool fragmentPackets)
{
	//open, bind, and connect the socket
	fSocket = ::socket(PF_INET, SOCK_STREAM, 0);
	AssertV(fSocket > -1, errno);
	
	struct sockaddr_in	localAddr;
	::memset(&localAddr, 0, sizeof(localAddr));
	localAddr.sin_family = PF_INET;
	localAddr.sin_port = 0;
	localAddr.sin_addr.s_addr = INADDR_ANY;
	
	int err = ::bind(fSocket, (sockaddr *)&localAddr, sizeof(localAddr));
	AssertV(err == 0, errno);
	
	int one = 1;
	err = ::setsockopt(fSocket, IPPROTO_TCP, TCP_NODELAY, (char*)&one, sizeof(int));
	AssertV(err == 0, errno);
	
	::memset(&localAddr, 0, sizeof(localAddr));
	localAddr.sin_family = PF_INET;
	localAddr.sin_port = 554;
	//get the ip address as a long based on the text
	localAddr.sin_addr.s_addr = ::inet_addr(ipAddr);
	err = ::connect(fSocket, (sockaddr*)&localAddr, sizeof(localAddr));
	AssertV(err == 0, errno);

	UInt32 totLen = ::strlen(inRequest);
	UInt32 curLen = 0;
	while (curLen < totLen)
	{
		UInt32 packetLen = totLen - curLen;
		if ((fragmentPackets) && (packetLen > 10))
		{
			Float32 realLen = (Float32)packetLen;
			Float32 numerator = (Float32)::rand();
			Float32 denominator = RAND_MAX;
			realLen = packetLen * (numerator / denominator);
			packetLen = (UInt32)realLen;
		}
		
		err = ::send(fSocket, inRequest + curLen, packetLen, 0);
		AssertV(err == packetLen, errno);
		curLen += packetLen;
		thread_switch(THREAD_NULL, SWITCH_OPTION_WAIT, 100);
	}
}

char* RTSPClient::ReceiveResponse()
{
	UInt32 curOffset = 0;
	while (true)
	{
		SInt32 theRecvLen = ::recv(fSocket, fResponseBuffer + curOffset, 4096 - curOffset, 0);//flags??
		if (theRecvLen == -1)
			break;
		curOffset += theRecvLen;
		fResponseBuffer[curOffset] = '\0';
		if (::strstr(fResponseBuffer, "\r\n\r\n") != NULL)
			break;
	}
	return fResponseBuffer;
}

void terminate() {}

int main(void)
{
	//Client makes a whole slew of edge case requests to the server.
	
	//First, client sends syntax error requests:
	RTSPClient* syntax1 = new RTSPClient(sSyntax1, sIPAddr, true);
	char* response = syntax1->ReceiveResponse();
	printf("Server response 1: %s\n", response);
	RTSPClient*  syntax2 = new RTSPClient(sSyntax2, sIPAddr, true);
	response = syntax2->ReceiveResponse();
	printf("Server response 2: %s\n", response);
	RTSPClient*  syntax3 = new RTSPClient(sSyntax3, sIPAddr, true);
	response = syntax3->ReceiveResponse();
	printf("Server response 3: %s\n", response);
	RTSPClient*  syntax4 = new RTSPClient(sSyntax4, sIPAddr, true);
	response = syntax4->ReceiveResponse();
	printf("Server response 4: %s\n", response);
	RTSPClient*  syntax5 = new RTSPClient(sSyntax5, sIPAddr, true);
	response = syntax5->ReceiveResponse();
	printf("Server response 5: %s\n", response);
	RTSPClient*  syntax6 = new RTSPClient(sSyntax6, sIPAddr, true);
	response = syntax6->ReceiveResponse();
	printf("Server response 6: %s\n", response);
	RTSPClient*  syntax7 = new RTSPClient(sSyntax7, sIPAddr, true);
	response = syntax7->ReceiveResponse();
	printf("Server response 7: %s\n", response);
	RTSPClient*  syntax500 = new RTSPClient(sSyntax8, sIPAddr, true);
	response = syntax8->ReceiveResponse();
	printf("Server response 8: %s\n", response);

	RTSPClient* syntax8 = new RTSPClient(sSyntax1, sIPAddr, false);
	response = syntax8->ReceiveResponse();
	printf("Server response 8: %s\n", response);
	RTSPClient* syntax9 = new RTSPClient(sSyntax2, sIPAddr, false);
	response = syntax9->ReceiveResponse();
	printf("Server response 9: %s\n", response);
	RTSPClient* syntax10 = new RTSPClient(sSyntax3, sIPAddr, false);
	response = syntax10->ReceiveResponse();
	printf("Server response 10: %s\n", response);
	RTSPClient* syntax11 = new RTSPClient(sSyntax4, sIPAddr, false);
	response = syntax11->ReceiveResponse();
	printf("Server response 11: %s\n", response);
	RTSPClient* syntax12 = new RTSPClient(sSyntax5, sIPAddr, false);
	response = syntax12->ReceiveResponse();
	printf("Server response 12: %s\n", response);
	RTSPClient* syntax13 = new RTSPClient(sSyntax6, sIPAddr, false);
	response = syntax13->ReceiveResponse();
	printf("Server response 13: %s\n", response);
	RTSPClient* syntax14 = new RTSPClient(sSyntax7, sIPAddr, false);
	response = syntax14->ReceiveResponse();
	printf("Server response 14: %s\n", response);
	RTSPClient*  syntax501 = new RTSPClient(sSyntax8, sIPAddr, true);
	response = syntax8->ReceiveResponse();
	printf("Server response 15: %s\n", response);

	RTSPClient* legit1 = new RTSPClient(sLegit1, sIPAddr, true);
	response = legit1->ReceiveResponse();
	printf("Server legit response 1: %s\n", response);
	RTSPClient* legit2 = new RTSPClient(sLegit2, sIPAddr, true);
	response = legit2->ReceiveResponse();
	printf("Server legit response 2: %s\n", response);
	RTSPClient* legit3 = new RTSPClient(sLegit3, sIPAddr, true);
	response = legit3->ReceiveResponse();
	printf("Server legit response 3: %s\n", response);
	RTSPClient* legit4 = new RTSPClient(sLegit4, sIPAddr, true);
	response = legit4->ReceiveResponse();
	printf("Server legit response 4: %s\n", response);
	RTSPClient* legit5 = new RTSPClient(sLegit5, sIPAddr, true);
	response = legit5->ReceiveResponse();
	printf("Server legit response 5: %s\n", response);

	RTSPClient* legit6 = new RTSPClient(sLegit1, sIPAddr, false);
	response = legit6->ReceiveResponse();
	printf("Server legit response 6: %s\n", response);
	RTSPClient* legit7 = new RTSPClient(sLegit2, sIPAddr, false);
	response = legit7->ReceiveResponse();
	printf("Server legit response 7: %s\n", response);
	RTSPClient* legit8 = new RTSPClient(sLegit3, sIPAddr, false);
	response = legit8->ReceiveResponse();
	printf("Server legit response 8: %s\n", response);
	RTSPClient* legit9 = new RTSPClient(sLegit4, sIPAddr, false);
	response = legit9->ReceiveResponse();
	printf("Server legit response 9: %s\n", response);
	RTSPClient* legit10 = new RTSPClient(sLegit5, sIPAddr, false);

	response = legit10->ReceiveResponse();
	printf("Server legit response 10: %s\n", response);
}

#else

UInt32 RTSPValidationModule::sInvocationCount = 0;
SInt32 RTSPValidationModule::sRequestInvocationCount = -2;
OSMutex RTSPValidationModule::sMutex('vald');

void RTSPValidationModule::FilterRequest(RTSPRequestInterface *request)
{
	static char *sSyntax5Part1 = "SETUP rtsp://17.221.41.108/this/is/totally/legit/but/URL/is/way/ RTSP/1.0\r\n\r\n";
	static char *sSyntax5Part2 = "\r\n";
	static char *sRequest1 = "record     	/a%20url/%25with%23/%38encoded%347characters/  RTSP/1.0\r\nCSeq: 0\r\nHost: foop\r\nConnection: keep-alive\r\nHost: meow\r\naccept: \r\n\r\n";
	static char *sRequest2 = "\r\n\r\n";
	static char *sRequest3 = "SETUP rtsp://17.221.41.108/ RTSP/1.0\r\n\r\n";
	static char *sRequest4 = "DESCRIBE rtsp://myhouse.com/short/url/ RTSP/1.0\rCSeq: 14\rTransport: 5000-5001\rRange: npt=14.56-\r\r";
	static char *sRequest5 = "OPTIONS rtsp://dopey.com/%20%202020%20%20/this/filename.has.a.number=4 RTSP/1.0\r\nHOST: dopey.com\r\nhost: meow.com\r\ncseq: 4\r\n\r\n";
	static char *sRequest6 = "OPTIONS rtsp://dopey.com/%20%202020%20%20/this/filename.has.a.number=4/is/totally/legit/but/URL/is/way/ RTSP/1.0\r\nHOST: dopey.com\r\nhost: meow.com\r\nrange: npt=0-45.76\r\ncseq: 4\r\ncseq: 4\r\ncseq: 4\r\naccept-language: bollocks!\r\nbandwidth: too much for your server!\r\nexpires: now!\r\nLooopy: meowieal\r\nif-modified-since: loopy\r\ncseq: 4\r\n\r\n";
	static char *sRequest7 = "SETUP /short/url RTSP/1.0\r\nCSeq: 1\r\n\r\n";
	static char *sRequest8 = "Play /short/url/with/number23\r\n\r\n";
	static char *sRequest9 = "teardown /short/url RTSP/1.0\r\ncseq: 5\r\n\r\n";
	static char* sRequest10 = "pause /vryshrt RTSP/1.0\r\n\r\n";

	OSMutexLocker locker(&sMutex);
	StrPtrLen* fullRequest = request->GetQTSSParameter(qtssFullRequestParam);
	if (sInvocationCount == 0)
	{
		Assert(fullRequest->Len == ::strlen(sSyntax1));
		Assert(memcmp(fullRequest->Ptr, sSyntax1, ::strlen(sSyntax1)) == 0);
	}
	if (sInvocationCount == 1)
	{
		Assert(fullRequest->Len == ::strlen(sSyntax2));
		Assert(memcmp(fullRequest->Ptr, sSyntax2, ::strlen(sSyntax2)) == 0);
	}
	if (sInvocationCount == 2)
	{
		Assert(fullRequest->Len == ::strlen(sSyntax3));
		Assert(memcmp(fullRequest->Ptr, sSyntax3, ::strlen(sSyntax3)) == 0);
	}
	if (sInvocationCount == 3)
	{
		Assert(fullRequest->Len == ::strlen(sSyntax4));
		Assert(memcmp(fullRequest->Ptr, sSyntax4, ::strlen(sSyntax4)) == 0);
	}
	if (sInvocationCount == 4)
	{
		Assert(fullRequest->Len == ::strlen(sSyntax5Part1));
		Assert(memcmp(fullRequest->Ptr, sSyntax5Part1, ::strlen(sSyntax5Part1)) == 0);
	}
	if (sInvocationCount == 5)
	{
		Assert(fullRequest->Len == ::strlen(sSyntax6));
		Assert(memcmp(fullRequest->Ptr, sSyntax6, ::strlen(sSyntax6)) == 0);
	}
	if (sInvocationCount == 6)
	{
		Assert(fullRequest->Len == ::strlen(sSyntax7));
		Assert(memcmp(fullRequest->Ptr, sSyntax7, ::strlen(sSyntax7)) == 0);
	}		
	if (sInvocationCount == 7)
	{
		Assert(fullRequest->Len == ::strlen(sSyntax1));
		Assert(memcmp(fullRequest->Ptr, sSyntax1, ::strlen(sSyntax1)) == 0);
	}
	if (sInvocationCount == 8)
	{
		Assert(fullRequest->Len == ::strlen(sSyntax2));
		Assert(memcmp(fullRequest->Ptr, sSyntax2, ::strlen(sSyntax2)) == 0);
	}
	if (sInvocationCount == 9)
	{
		Assert(fullRequest->Len == ::strlen(sSyntax3));
		Assert(memcmp(fullRequest->Ptr, sSyntax3, ::strlen(sSyntax3)) == 0);
	}
	if (sInvocationCount == 10)
	{
		Assert(fullRequest->Len == ::strlen(sSyntax4));
		Assert(memcmp(fullRequest->Ptr, sSyntax4, ::strlen(sSyntax4)) == 0);
	}
	if (sInvocationCount == 11)
	{
		Assert(fullRequest->Len == ::strlen(sSyntax5Part1));
		Assert(memcmp(fullRequest->Ptr, sSyntax5Part1, ::strlen(sSyntax5Part1)) == 0);
	}
	if (sInvocationCount == 12)
	{
		Assert(fullRequest->Len == ::strlen(sSyntax6));
		Assert(memcmp(fullRequest->Ptr, sSyntax6, ::strlen(sSyntax6)) == 0);
	}
	if (sInvocationCount == 13)
	{
		Assert(fullRequest->Len == ::strlen(sSyntax7));
		Assert(memcmp(fullRequest->Ptr, sSyntax7, ::strlen(sSyntax7)) == 0);
	}		
	if (sInvocationCount == 14)
	{
		Assert(fullRequest->Len == ::strlen(sRequest1));
		Assert(memcmp(fullRequest->Ptr, sRequest1, ::strlen(sRequest1)) == 0);
	}
	if (sInvocationCount == 15)
	{
		Assert(fullRequest->Len == ::strlen(sRequest2));
		Assert(memcmp(fullRequest->Ptr, sRequest2, ::strlen(sRequest2)) == 0);
	}
	if (sInvocationCount == 16)
	{
		Assert(fullRequest->Len == ::strlen(sRequest3));
		Assert(memcmp(fullRequest->Ptr, sRequest3, ::strlen(sRequest3)) == 0);
	}
	if (sInvocationCount == 17)
	{
		Assert(fullRequest->Len == ::strlen(sRequest4));
		Assert(memcmp(fullRequest->Ptr, sRequest4, ::strlen(sRequest4)) == 0);
	}
	if (sInvocationCount == 18)
	{
		Assert(fullRequest->Len == ::strlen(sRequest5));
		Assert(memcmp(fullRequest->Ptr, sRequest5, ::strlen(sRequest5)) == 0);
	}
	if (sInvocationCount == 19)
	{
		Assert(fullRequest->Len == ::strlen(sRequest6));
		Assert(memcmp(fullRequest->Ptr, sRequest6, ::strlen(sRequest6)) == 0);
	}
	if (sInvocationCount == 20)
	{
		Assert(fullRequest->Len == ::strlen(sRequest7));
		Assert(memcmp(fullRequest->Ptr, sRequest7, ::strlen(sRequest7)) == 0);
	}
	if (sInvocationCount == 21)
	{
		Assert(fullRequest->Len == ::strlen(sRequest8));
		Assert(memcmp(fullRequest->Ptr, sRequest8, ::strlen(sRequest8)) == 0);
	}
	if (sInvocationCount == 22)
	{
		Assert(fullRequest->Len == ::strlen(sRequest9));
		Assert(memcmp(fullRequest->Ptr, sRequest9, ::strlen(sRequest9)) == 0);
	}
	if (sInvocationCount == 23)
	{
		Assert(fullRequest->Len == ::strlen(sRequest10));
		Assert(memcmp(fullRequest->Ptr, sRequest10, ::strlen(sRequest10)) == 0);
	}
	if (sInvocationCount == 24)
	{
		Assert(fullRequest->Len == ::strlen(sRequest1));
		Assert(memcmp(fullRequest->Ptr, sRequest1, ::strlen(sRequest1)) == 0);
	}
	if (sInvocationCount == 25)
	{
		Assert(fullRequest->Len == ::strlen(sRequest2));
		Assert(memcmp(fullRequest->Ptr, sRequest2, ::strlen(sRequest2)) == 0);
	}
	if (sInvocationCount == 26)
	{
		Assert(fullRequest->Len == ::strlen(sRequest3));
		Assert(memcmp(fullRequest->Ptr, sRequest3, ::strlen(sRequest3)) == 0);
	}
	if (sInvocationCount == 27)
	{
		Assert(fullRequest->Len == ::strlen(sRequest4));
		Assert(memcmp(fullRequest->Ptr, sRequest4, ::strlen(sRequest4)) == 0);
	}
	if (sInvocationCount == 28)
	{
		Assert(fullRequest->Len == ::strlen(sRequest5));
		Assert(memcmp(fullRequest->Ptr, sRequest5, ::strlen(sRequest5)) == 0);
	}
	if (sInvocationCount == 29)
	{
		Assert(fullRequest->Len == ::strlen(sRequest6));
		Assert(memcmp(fullRequest->Ptr, sRequest6, ::strlen(sRequest6)) == 0);
	}
	if (sInvocationCount == 30)
	{
		Assert(fullRequest->Len == ::strlen(sRequest7));
		Assert(memcmp(fullRequest->Ptr, sRequest7, ::strlen(sRequest7)) == 0);
	}
	if (sInvocationCount == 31)
	{
		Assert(fullRequest->Len == ::strlen(sRequest8));
		Assert(memcmp(fullRequest->Ptr, sRequest8, ::strlen(sRequest8)) == 0);
	}
	if (sInvocationCount == 32)
	{
		Assert(fullRequest->Len == ::strlen(sRequest9));
		Assert(memcmp(fullRequest->Ptr, sRequest9, ::strlen(sRequest9)) == 0);
	}
	if (sInvocationCount == 33)
	{
		Assert(fullRequest->Len == ::strlen(sRequest10));
		Assert(memcmp(fullRequest->Ptr, sRequest10, ::strlen(sRequest10)) == 0);
	}
	sInvocationCount++;
	printf("Server passed filter invocation %d\n",sInvocationCount);
}

RTSPProtocol::RTSPStatusCode RTSPValidationModule::ProcessRequest(	RTSPRequestInterface *request)
{
	OSMutexLocker locker(&sMutex);

	if ((sRequestInvocationCount == -2) || (sRequestInvocationCount == -1))
	{
		Assert(request->GetMethod() == RTSPProtocol::kSetupMethod);
		Assert(request->GetVersion() == RTSPProtocol::k10Version);
		Assert(request->GetRequestKeepAlive() == true);
		StrPtrLen* param = request->GetHeaderValue(RTSPProtocol::kHostHeader);
		Assert(memcmp(param->Ptr, "17.221.41.108", 13) == 0);
		param = request->GetQTSSParameter(qtssMethodParam);
		Assert(param->Len == 5);
		Assert(memcmp(param->Ptr, "SETUP", 5) == 0);
		param = request->GetQTSSParameter(qtssURLParam);
		Assert(param->Len == strlen("/this/is/totally/legit/but/URL/is/way/"));
		Assert(memcmp(param->Ptr, "/this/is/totally/legit/but/URL/is/way/", strlen("/this/is/totally/legit/but/URL/is/way/")) == 0);
		param = request->GetQTSSParameter(qtssFilePathParam);
		Assert(param->Len == strlen("/this/is/totally/legit/but/URL/is/way/"));
		Assert(memcmp(param->Ptr, "/this/is/totally/legit/but/URL/is/way/", strlen("/this/is/totally/legit/but/URL/is/way/")) == 0);
		param = request->GetQTSSParameter(qtssFilePathTruncParam);
		Assert(param->Len == strlen("/this/is/totally/legit/but/URL/is/way"));
		Assert(memcmp(param->Ptr, "/this/is/totally/legit/but/URL/is/way", strlen("/this/is/totally/legit/but/URL/is/way")) == 0);
		param = request->GetQTSSParameter(qtssFileNameParam);
		Assert(param->Len == 0);
		param = request->GetQTSSParameter(qtssClientPortA);
		Assert(param->Len == 0);
		param = request->GetQTSSParameter(qtssClientPortB);
		Assert(param->Len == 0);
		param = request->GetQTSSParameter(qtssFileDigitParam);
		Assert(param->Len == 0);
		param = request->GetQTSSParameter(qtssAbsoluteURLParam);
		Assert(param->Len == strlen("rtsp://17.221.41.108/this/is/totally/legit/but/URL/is/way/"));
		Assert(memcmp(param->Ptr, "rtsp://17.221.41.108/this/is/totally/legit/but/URL/is/way/", strlen("rtsp://17.221.41.108/this/is/totally/legit/but/URL/is/way/")) == 0);		

		this->SendLongResponse(request);
	}
	if ((sRequestInvocationCount == 0) || (sRequestInvocationCount == 9))
	{
		Assert(request->GetMethod() == RTSPProtocol::kRecordMethod);
		Assert(request->GetVersion() == RTSPProtocol::k10Version);
		Assert(request->GetRequestKeepAlive() == true);
		StrPtrLen* param = request->GetHeaderValue(RTSPProtocol::kHostHeader);
		Assert(param->Len == 4);
		Assert(memcmp(param->Ptr, "meow", 4) == 0);
		param = request->GetQTSSParameter(qtssMethodParam);
		Assert(param->Len == 6);
		Assert(memcmp(param->Ptr, "record", 6) == 0);
		param = request->GetQTSSParameter(qtssURLParam);
		Assert(param->Len == strlen("/a%20url/%25with%23/%38encoded%347characters/"));
		Assert(memcmp(param->Ptr, "/a%20url/%25with%23/%38encoded%347characters/", strlen("/a%20url/%25with%23/%38encoded%347characters/")) == 0);
		param = request->GetQTSSParameter(qtssFilePathParam);
		Assert(param->Len == strlen("/a url/%with#/8encoded47characters/"));
		Assert(memcmp(param->Ptr, "/a url/%with#/8encoded47characters/", strlen("/a url/%with#/8encoded47characters/")) == 0);
		param = request->GetQTSSParameter(qtssFilePathTruncParam);
		Assert(param->Len == strlen("/a url/%with#/8encoded47characters"));
		Assert(memcmp(param->Ptr, "/a url/%with#/8encoded47characters", strlen("/a url/%with#/8encoded47characters")) == 0);
		param = request->GetQTSSParameter(qtssFileNameParam);
		Assert(param->Len == 0);
		param = request->GetQTSSParameter(qtssClientPortA);
		Assert(param->Len == 0);
		param = request->GetQTSSParameter(qtssClientPortB);
		Assert(param->Len == 0);
		param = request->GetQTSSParameter(qtssFileDigitParam);
		Assert(param->Len == 0);
		param = request->GetQTSSParameter(qtssAbsoluteURLParam);
		Assert(param->Len == strlen("/a%20url/%25with%23/%38encoded%347characters/"));
		Assert(memcmp(param->Ptr, "/a%20url/%25with%23/%38encoded%347characters/", strlen("/a%20url/%25with%23/%38encoded%347characters/")) == 0);
		
		this->SendLongResponse(request);
	}
	else if ((sRequestInvocationCount == 1) || (sRequestInvocationCount == 10))
	{
		Assert(request->GetMethod() == RTSPProtocol::kSetupMethod);
		Assert(request->GetVersion() == RTSPProtocol::k10Version);
		Assert(request->GetRequestKeepAlive() == true);
		Assert(request->GetStartTime() == -1);
		Assert(request->GetStopTime() == -1);
		StrPtrLen* param = request->GetHeaderValue(RTSPProtocol::kHostHeader);
		Assert(param->Len == strlen("17.221.41.108"));
		Assert(memcmp(param->Ptr, "17.221.41.108", strlen("17.221.41.108")) == 0);
		param = request->GetQTSSParameter(qtssMethodParam);
		Assert(param->Len == 5);
		Assert(memcmp(param->Ptr, "SETUP", 5) == 0);
		param = request->GetQTSSParameter(qtssURLParam);
		Assert(param->Len == 1);
		Assert(memcmp(param->Ptr, "/", 1) == 0);
		param = request->GetQTSSParameter(qtssFilePathParam);
		Assert(param->Len == 1);
		Assert(memcmp(param->Ptr, "/", 1) == 0);
		param = request->GetQTSSParameter(qtssFilePathTruncParam);
		Assert(param->Len == 0);
		param = request->GetQTSSParameter(qtssFileNameParam);
		Assert(param->Len == 0);
		param = request->GetQTSSParameter(qtssClientPortA);
		Assert(param->Len == 0);
		param = request->GetQTSSParameter(qtssClientPortB);
		Assert(param->Len == 0);
		param = request->GetQTSSParameter(qtssFileDigitParam);
		Assert(param->Len == 0);
		param = request->GetQTSSParameter(qtssAbsoluteURLParam);
		Assert(param->Len == strlen("rtsp://17.221.41.108/"));
		Assert(memcmp(param->Ptr, "rtsp://17.221.41.108/", strlen("rtsp://17.221.41.108/")) == 0);	

		this->SendLongResponse(request);
	}
	else if ((sRequestInvocationCount == 2) || (sRequestInvocationCount == 11))
	{
		Assert(request->GetMethod() == RTSPProtocol::kDescribeMethod);
		Assert(request->GetVersion() == RTSPProtocol::k10Version);
		Assert(request->GetRequestKeepAlive() == true);
		UInt32 temp = (UInt32)(request->GetStartTime() * 100);
		Assert(temp == 1456);
		Assert(request->GetStopTime() == -1);
		StrPtrLen* param = request->GetHeaderValue(RTSPProtocol::kHostHeader);
		Assert(param->Len == strlen("myhouse.com"));
		Assert(memcmp(param->Ptr, "myhouse.com", strlen("myhouse.com")) == 0);
		param = request->GetQTSSParameter(qtssMethodParam);
		Assert(param->Len == 8);
		Assert(memcmp(param->Ptr, "DESCRIBE", 8) == 0);
		param = request->GetQTSSParameter(qtssURLParam);
		Assert(param->Len == strlen("/short/url/"));
		Assert(memcmp(param->Ptr, "/short/url/", strlen("/short/url/")) == 0);
		param = request->GetQTSSParameter(qtssFilePathParam);
		Assert(param->Len == strlen("/short/url/"));
		Assert(memcmp(param->Ptr, "/short/url/", strlen("/short/url/")) == 0);
		param = request->GetQTSSParameter(qtssFilePathTruncParam);
		Assert(param->Len == strlen("/short/url"));
		Assert(memcmp(param->Ptr, "/short/url", strlen("/short/url")) == 0);
		param = request->GetQTSSParameter(qtssFileNameParam);
		Assert(param->Len == 0);
		param = request->GetQTSSParameter(qtssClientPortA);
		Assert(param->Len == 4);
		Assert(request->GetClientPortA() == 5000);
		Assert(request->GetClientPortB() == 5001);
		Assert(request->GetStartTime() == 14.56);
		Assert(request->GetStopTime() == -1);
		Assert(memcmp(param->Ptr, "5000", 4) == 0);
		param = request->GetQTSSParameter(qtssClientPortB);
		Assert(param->Len == 4);
		Assert(memcmp(param->Ptr, "5001", 4) == 0);
		param = request->GetQTSSParameter(qtssFileDigitParam);
		Assert(param->Len == 0);
		param = request->GetQTSSParameter(qtssAbsoluteURLParam);
		Assert(param->Len == strlen("rtsp://myhouse.com/short/url/"));
		Assert(memcmp(param->Ptr, "rtsp://myhouse.com/short/url/", strlen("rtsp://myhouse.com/short/url/")) == 0);	

		this->SendLongResponse(request);
	}
	else if ((sRequestInvocationCount == 3) || (sRequestInvocationCount == 12))
	{
		Assert(request->GetMethod() == RTSPProtocol::kOptionsMethod);
		Assert(request->GetVersion() == RTSPProtocol::k10Version);
		Assert(request->GetRequestKeepAlive() == true);
		Assert(request->GetStartTime() == -1);
		Assert(request->GetStopTime() == -1);
		StrPtrLen* param = request->GetHeaderValue(RTSPProtocol::kHostHeader);
		Assert(param->Len == strlen("meow.com"));
		Assert(memcmp(param->Ptr, "meow.com", strlen("meow.com")) == 0);
		param = request->GetQTSSParameter(qtssMethodParam);
		Assert(param->Len == 7);
		Assert(memcmp(param->Ptr, "OPTIONS", 7) == 0);
		param = request->GetQTSSParameter(qtssURLParam);
		Assert(param->Len == strlen("/%20%202020%20%20/this/filename.has.a.number=4"));
		Assert(memcmp(param->Ptr, "/%20%202020%20%20/this/filename.has.a.number=4", strlen("/%20%202020%20%20/this/filename.has.a.number=4")) == 0);
		param = request->GetQTSSParameter(qtssFilePathParam);
		Assert(param->Len == strlen("/  2020  /this/filename.has.a.number=4"));
		Assert(memcmp(param->Ptr, "/  2020  /this/filename.has.a.number=4", strlen("/  2020  /this/filename.has.a.number=4")) == 0);
		param = request->GetQTSSParameter(qtssFilePathTruncParam);
		Assert(param->Len == strlen("/  2020  /this"));
		Assert(memcmp(param->Ptr, "/  2020  /this", strlen("/  2020  /this")) == 0);
		param = request->GetQTSSParameter(qtssFileNameParam);
		Assert(param->Len == strlen("filename.has.a.number=4"));
		Assert(memcmp(param->Ptr, "filename.has.a.number=4", strlen("filename.has.a.number=4")) == 0);
		param = request->GetQTSSParameter(qtssClientPortA);
		Assert(param->Len == 0);
		param = request->GetQTSSParameter(qtssClientPortB);
		Assert(param->Len == 0);
		param = request->GetQTSSParameter(qtssFileDigitParam);
		Assert(param->Len == 1);
		Assert(memcmp(param->Ptr, "4", 1) == 0);
		param = request->GetQTSSParameter(qtssAbsoluteURLParam);
		Assert(param->Len == strlen("rtsp://dopey.com/%20%202020%20%20/this/filename.has.a.number=4"));
		Assert(memcmp(param->Ptr, "rtsp://dopey.com/%20%202020%20%20/this/filename.has.a.number=4", strlen("rtsp://dopey.com/%20%202020%20%20/this/filename.has.a.number=4")) == 0);	

		this->SendLongResponse(request);
	}
	else if ((sRequestInvocationCount == 4) || (sRequestInvocationCount == 13))
	{
		Assert(request->GetMethod() == RTSPProtocol::kOptionsMethod);
		Assert(request->GetVersion() == RTSPProtocol::k10Version);
		Assert(request->GetRequestKeepAlive() == true);
		Assert(request->GetStartTime() == 0);
		UInt32 test2 = (UInt32)(request->GetStopTime() * 100);
		Assert(test2 == 4575);
		StrPtrLen* param = request->GetHeaderValue(RTSPProtocol::kHostHeader);
		Assert(param->Len == strlen("meow.com"));
		Assert(memcmp(param->Ptr, "meow.com", strlen("meow.com")) == 0);
		param = request->GetQTSSParameter(qtssMethodParam);
		Assert(param->Len == 7);
		Assert(memcmp(param->Ptr, "OPTIONS", 7) == 0);
		param = request->GetQTSSParameter(qtssURLParam);
		Assert(param->Len == strlen("/%20%202020%20%20/this/filename.has.a.number=4/is/totally/legit/but/URL/is/way/"));
		Assert(memcmp(param->Ptr, "/%20%202020%20%20/this/filename.has.a.number=4/is/totally/legit/but/URL/is/way/", strlen("/%20%202020%20%20/this/filename.has.a.number=4/is/totally/legit/but/URL/is/way/")) == 0);
		param = request->GetQTSSParameter(qtssFilePathParam);
		Assert(param->Len == strlen("/  2020  /this/filename.has.a.number=4/is/totally/legit/but/URL/is/way/"));
		Assert(memcmp(param->Ptr, "/  2020  /this/filename.has.a.number=4/is/totally/legit/but/URL/is/way/", strlen("/  2020  /this/filename.has.a.number=4/is/totally/legit/but/URL/is/way/")) == 0);
		param = request->GetQTSSParameter(qtssFilePathTruncParam);
		Assert(param->Len == strlen("/  2020  /this/filename.has.a.number=4/is/totally/legit/but/URL/is/way"));
		Assert(memcmp(param->Ptr, "/  2020  /this/filename.has.a.number=4/is/totally/legit/but/URL/is/way", strlen("/  2020  /this/filename.has.a.number=4/is/totally/legit/but/URL/is/way")) == 0);
		param = request->GetQTSSParameter(qtssFileNameParam);
		Assert(param->Len == 0);
		param = request->GetQTSSParameter(qtssClientPortA);
		Assert(param->Len == 0);
		param = request->GetQTSSParameter(qtssClientPortB);
		Assert(param->Len == 0);
		param = request->GetQTSSParameter(qtssFileDigitParam);
		Assert(param->Len == 0);
		param = request->GetQTSSParameter(qtssAbsoluteURLParam);
		Assert(param->Len == strlen("rtsp://dopey.com/%20%202020%20%20/this/filename.has.a.number=4/is/totally/legit/but/URL/is/way/"));
		Assert(memcmp(param->Ptr, "rtsp://dopey.com/%20%202020%20%20/this/filename.has.a.number=4/is/totally/legit/but/URL/is/way/", strlen("rtsp://dopey.com/%20%202020%20%20/this/filename.has.a.number=4/is/totally/legit/but/URL/is/way/")) == 0);	

		this->SendLongResponse(request);
	}
	else if ((sRequestInvocationCount == 5) || (sRequestInvocationCount == 14))
	{
		Assert(request->GetMethod() == RTSPProtocol::kSetupMethod);
		Assert(request->GetVersion() == RTSPProtocol::k10Version);
		Assert(request->GetRequestKeepAlive() == true);
		Assert(request->GetStartTime() == -1);
		Assert(request->GetStopTime() == -1);
		StrPtrLen* param = request->GetHeaderValue(RTSPProtocol::kHostHeader);
		Assert(param->Len == 0);
		param = request->GetQTSSParameter(qtssMethodParam);
		Assert(memcmp(param->Ptr, "SETUP", 5) == 0);
		param = request->GetQTSSParameter(qtssURLParam);
		Assert(param->Len == strlen("/short/url"));
		Assert(memcmp(param->Ptr, "/short/url", strlen("/short/url")) == 0);
		param = request->GetQTSSParameter(qtssFilePathParam);
		Assert(param->Len == strlen("/short/url"));
		Assert(memcmp(param->Ptr, "/short/url", strlen("/short/url")) == 0);
		param = request->GetQTSSParameter(qtssFilePathTruncParam);
		Assert(param->Len == strlen("/short"));
		Assert(memcmp(param->Ptr, "/short", strlen("/short")) == 0);
		param = request->GetQTSSParameter(qtssFileNameParam);
		Assert(param->Len == 3);
		Assert(memcmp(param->Ptr, "url", 3) == 0);
		param = request->GetQTSSParameter(qtssClientPortA);
		Assert(param->Len == 0);
		param = request->GetQTSSParameter(qtssClientPortB);
		Assert(param->Len == 0);
		param = request->GetQTSSParameter(qtssFileDigitParam);
		Assert(param->Len == 0);
		param = request->GetQTSSParameter(qtssAbsoluteURLParam);
		Assert(param->Len == strlen("/short/url"));
		Assert(memcmp(param->Ptr, "/short/url", strlen("/short/url")) == 0);

		this->SendLongResponse(request);
	}
	else if ((sRequestInvocationCount == 6) || (sRequestInvocationCount == 15))
	{
		Assert(request->GetMethod() == RTSPProtocol::kPlayMethod);
		Assert(request->GetVersion() == RTSPProtocol::k10Version);
		Assert(request->GetRequestKeepAlive() == true);
		Assert(request->GetStartTime() == -1);
		Assert(request->GetStopTime() == -1);
		StrPtrLen* param = request->GetHeaderValue(RTSPProtocol::kHostHeader);
		Assert(param->Len == 0);
		param = request->GetQTSSParameter(qtssMethodParam);
		Assert(memcmp(param->Ptr, "Play", 4) == 0);
		param = request->GetQTSSParameter(qtssURLParam);
		Assert(param->Len == strlen("/short/url/with/number23"));
		Assert(memcmp(param->Ptr, "/short/url/with/number23", strlen("/short/url/with/number23")) == 0);
		param = request->GetQTSSParameter(qtssFilePathParam);
		Assert(param->Len == strlen("/short/url/with/number23"));
		Assert(memcmp(param->Ptr, "/short/url/with/number23", strlen("/short/url/with/number23")) == 0);
		param = request->GetQTSSParameter(qtssFilePathTruncParam);
		Assert(param->Len == strlen("/short/url/with"));
		Assert(memcmp(param->Ptr, "/short/url/with", strlen("/short/url/with")) == 0);
		param = request->GetQTSSParameter(qtssFileNameParam);
		Assert(param->Len == strlen("number23"));
		Assert(memcmp(param->Ptr, "number23", strlen("number23")) == 0);
		param = request->GetQTSSParameter(qtssClientPortA);
		Assert(param->Len == 0);
		param = request->GetQTSSParameter(qtssClientPortB);
		Assert(param->Len == 0);
		param = request->GetQTSSParameter(qtssFileDigitParam);
		Assert(param->Len == 2);
		Assert(memcmp(param->Ptr, "23", 2) == 0);
		param = request->GetQTSSParameter(qtssAbsoluteURLParam);
		Assert(param->Len == strlen("/short/url/with/number23"));
		Assert(memcmp(param->Ptr, "/short/url/with/number23", strlen("/short/url/with/number23")) == 0);

		this->SendLongResponse(request);
	}
	else if ((sRequestInvocationCount == 7) || (sRequestInvocationCount == 16))
	{
		Assert(request->GetMethod() == RTSPProtocol::kTeardownMethod);
		Assert(request->GetVersion() == RTSPProtocol::k10Version);
		Assert(request->GetRequestKeepAlive() == true);
		Assert(request->GetStartTime() == -1);
		Assert(request->GetStopTime() == -1);
		StrPtrLen* param = request->GetHeaderValue(RTSPProtocol::kHostHeader);
		Assert(param->Len == 0);
		param = request->GetQTSSParameter(qtssMethodParam);
		Assert(memcmp(param->Ptr, "teardown", 8) == 0);
		param = request->GetQTSSParameter(qtssURLParam);
		Assert(param->Len == strlen("/short/url"));
		Assert(memcmp(param->Ptr, "/short/url", strlen("/short/url")) == 0);
		param = request->GetQTSSParameter(qtssFilePathParam);
		Assert(param->Len == strlen("/short/url"));
		Assert(memcmp(param->Ptr, "/short/url", strlen("/short/url")) == 0);
		param = request->GetQTSSParameter(qtssFilePathTruncParam);
		Assert(param->Len == strlen("/short"));
		Assert(memcmp(param->Ptr, "/short", strlen("/short")) == 0);
		param = request->GetQTSSParameter(qtssFileNameParam);
		Assert(param->Len == strlen("url"));
		Assert(memcmp(param->Ptr, "url", strlen("url")) == 0);
		param = request->GetQTSSParameter(qtssClientPortA);
		Assert(param->Len == 0);
		param = request->GetQTSSParameter(qtssClientPortB);
		Assert(param->Len == 0);
		param = request->GetQTSSParameter(qtssFileDigitParam);
		Assert(param->Len == 0);
		param = request->GetQTSSParameter(qtssAbsoluteURLParam);
		Assert(param->Len == strlen("/short/url"));
		Assert(memcmp(param->Ptr, "/short/url", strlen("/short/url")) == 0);

		this->SendLongResponse(request);
	}
	else if ((sRequestInvocationCount == 8) || (sRequestInvocationCount == 17))
	{
		Assert(request->GetMethod() == RTSPProtocol::kPauseMethod);
		Assert(request->GetVersion() == RTSPProtocol::k10Version);
		Assert(request->GetRequestKeepAlive() == true);
		Assert(request->GetStartTime() == -1);
		Assert(request->GetStopTime() == -1);
		StrPtrLen* param = request->GetHeaderValue(RTSPProtocol::kHostHeader);
		Assert(param->Len == 0);
		param = request->GetQTSSParameter(qtssMethodParam);
		Assert(memcmp(param->Ptr, "pause", 5) == 0);
		param = request->GetQTSSParameter(qtssURLParam);
		Assert(param->Len == strlen("/vryshrt"));
		Assert(memcmp(param->Ptr, "/vryshrt", strlen("/vryshrt")) == 0);
		param = request->GetQTSSParameter(qtssFilePathParam);
		Assert(param->Len == strlen("/vryshrt"));
		Assert(memcmp(param->Ptr, "/vryshrt", strlen("/vryshrt")) == 0);
		param = request->GetQTSSParameter(qtssFilePathTruncParam);
		Assert(param->Len == 0);
		param = request->GetQTSSParameter(qtssFileNameParam);
		Assert(param->Len == strlen("vryshrt"));
		Assert(memcmp(param->Ptr, "vryshrt", strlen("vryshrt")) == 0);
		param = request->GetQTSSParameter(qtssClientPortA);
		Assert(param->Len == 0);
		param = request->GetQTSSParameter(qtssClientPortB);
		Assert(param->Len == 0);
		param = request->GetQTSSParameter(qtssFileDigitParam);
		Assert(param->Len == 0);
		param = request->GetQTSSParameter(qtssAbsoluteURLParam);
		Assert(param->Len == strlen("/vryshrt"));
		Assert(memcmp(param->Ptr, "/vryshrt", strlen("/vryshrt")) == 0);

		this->SendLongResponse(request);
	}
	sRequestInvocationCount++;
	printf("Server passed processRequest invocation %d\n",sInvocationCount);
	return RTSPProtocol::kSuccessOK;	
}

void RTSPValidationModule::SendLongResponse(RTSPRequestInterface* inRequest)
{
	static char* sResponse = "I want a donut so badly that I will try to\nTake all your money\nand go buy a donut with that money\n";
	
	for (UInt32 loopy = 0; loopy < 20; loopy++)
	{
		char buffer[10];
		sprintf(buffer,"%d\n",loopy);
		inRequest->Write(buffer, strlen(buffer));
		
		UInt32 length = ::strlen(sResponse);
		UInt32 curLen = 0;
				
		while ((length - curLen) > 10)
		{
			UInt32 packetLen = length - curLen;
			Float32 realLen = (Float32)packetLen;
			Float32 numerator = (Float32)::rand();
			Float32 denominator = RAND_MAX;
			realLen = packetLen * (numerator / denominator);
			packetLen = (UInt32)realLen;
			
			(void)inRequest->Write(sResponse + curLen, packetLen);
			curLen += packetLen;
		}
		(void)inRequest->Write(sResponse + curLen, length - curLen);
	}
	(void)inRequest->Write("\r\n\r\n", 4);
}
#endif
