/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */

#ifndef __MW_
#include <mach/mach.h>
#include <mach/cthreads.h>
#endif

#include "mycondition.h"
#include "stdlib.h"

#include "atomic.h"

struct MyCondition
{
	port_t				fWaitPort;
	unsigned int		fNumWaiting;
};

typedef struct MyCondition MyCondition;

MyCondition* MCAllocateCondition();
void MCDisposeCondition(MyCondition* theCondition);
void MCBroadcast(MyCondition* theCondition);
void MCSignal(MyCondition* theCondition);
void MCWait(MyCondition* theCondition, mymutex_t theMutex, int timeoutInMilSecs);
void MCBlockThread(MyCondition* theCondition, int timeoutInMilSecs);
void MCUnblockThread(MyCondition* theCondition);

mycondition_t mycondition_alloc()
{
	return (mycondition_t)MCAllocateCondition();
}

void mycondition_free(mycondition_t theCondition_t)
{
	MCDisposeCondition((MyCondition*)theCondition_t);
}

void mycondition_broadcast(mycondition_t theCondition_t)
{
	MCBroadcast((MyCondition*)theCondition_t);
}

void mycondition_signal(mycondition_t theCondition_t)
{
	MCSignal((MyCondition*)theCondition_t);
}

void mycondition_wait(mycondition_t theCondition_t, mymutex_t theMutex_t, int timeoutInMilSecs)
{
	MCWait((MyCondition*)theCondition_t, theMutex_t, timeoutInMilSecs);
}

MyCondition* MCAllocateCondition()
{
	kern_return_t ret;
	MyCondition* newCondition = (MyCondition*)malloc(sizeof(MyCondition));
	if (newCondition == NULL)
		return NULL;
		
	newCondition->fNumWaiting = 0;

	ret = port_allocate(task_self(), &newCondition->fWaitPort);
	if (ret != KERN_SUCCESS)
	{
		free(newCondition);
		return NULL;
	}
	
	return newCondition;
}

void MCDisposeCondition(MyCondition* theCondition)
{
	port_deallocate(task_self(), theCondition->fWaitPort);
	free(theCondition);
}

void MCBroadcast(MyCondition* theCondition)
{
	int numToSignal = theCondition->fNumWaiting;
	while (numToSignal > 0)
	{
		MCUnblockThread(theCondition);
		numToSignal--;
	}
}

void MCSignal(MyCondition* theCondition)
{
	MCUnblockThread(theCondition);
}

void MCWait(MyCondition* theCondition, mymutex_t theMutex, int timeoutInMilSecs)
{
	mymutex_unlock(theMutex);
	atomic_add(&theCondition->fNumWaiting, 1);
	MCBlockThread(theCondition, timeoutInMilSecs);
	atomic_sub(&theCondition->fNumWaiting, 1);
	mymutex_lock(theMutex);
}


void MCBlockThread(MyCondition* theCondition, int timeoutInMilSecs)
{
	kern_return_t ret;
	msg_header_t msg;

	msg.msg_size = sizeof(msg);
	msg.msg_local_port = theCondition->fWaitPort;

	if (timeoutInMilSecs > 0)
		ret = msg_receive(&msg, RCV_TIMEOUT, timeoutInMilSecs);
	else
		ret = msg_receive(&msg, MSG_OPTION_NONE, 0);
}

void MCUnblockThread(MyCondition* theCondition)
{
	kern_return_t ret;
	msg_header_t msg;

	msg.msg_simple = TRUE;
	msg.msg_size = sizeof(msg);
    msg.msg_type = MSG_TYPE_NORMAL;
	msg.msg_local_port = PORT_NULL; 
	msg.msg_remote_port = theCondition->fWaitPort;
    msg.msg_id = 0;
	ret = msg_send(&msg, SEND_TIMEOUT, 0);
}

